/*	CoRTOSiointMSP.c - I/O and interrupts for ti MSP430 processors

	2017.03.11 12:00	Interrupts stay enabled, remove rtos_int_xx,
							rename to timer interrupt control - which isn't
							used by CoRTOS
	2017.03.03 10:00	Initial release

	(c) 2017 Nicholas O. Lindan,
	Released under GPL V3 https://www.gnu.org/licenses/gpl.html */

// #include <msp430.h>
#include "msp430FR6989.h"

#include "CoRTOScomdefs.h"
#include "CoRTOStimer.h"
#include "CoRTOSioint.h"

static uint8_t intdis_nest_timer = 0;

/************************************************************************
*
*	Timer compare interrupt
*
************************************************************************/

/*	The interrupt is triggered 10mSec period on output
	compare A using timer 1.  Its function is to decrement
	the timers of any delayed tasks via a call to service_timers(). */

#pragma vector=TIMER1_A0_VECTOR
__interrupt void timerA1_CC0 (void)
{
	/*	This vector is shared with other timer interrupt sources.  At
		present there is only the one source triggered on overflow compare. */
	service_timers ();
	__enable_interrupt ();
}

/************************************************************************
*
*	I/O, only used for blinky & co.
*
************************************************************************/

/*	Change port as needed for your hardware - this works
	with ti's MSP-EXP30FR5969 Explorer board.

	0/bit0 - red led,
	1/bit1 - green led */

uint8_t led_status = 0;

void led_on (uint8_t led)
{
	if ((led & bit0) == 0)
	{
		P1OUT |= 0x01;
		led_status |= bit0;
	}
	else
	{
		P9OUT |= 0x80;
		led_status |= bit1;
	}
}

void led_off (uint8_t led)
{
	if ((led & bit0) == 0)
	{
		P1OUT &= ~0x01;
		led_status &= ~bit0;
	}
	else
	{
		P9OUT &= ~0x80;
		led_status &= ~bit1;
	}
}

void led_toggle (uint8_t led)
{
	if ((led & bit0) == 0)
	{
		P1OUT ^= 0x01;
		led_status ^= bit0;
	}
	else
	{
		P9OUT ^= 0x80;
		led_status ^= bit1;
	}
}

void write_leds (uint8_t leds)
{
	led_off (0);
	led_off (1);
	if ((leds & bit0) != 0)
		led_on (0);
	if ((leds & bit1) != 0)
		led_on (1);
}

uint8_t read_leds (void)
{
	return (led_status);
}

/************************************************************************
*
*	interrupt disable/enable routines
*
************************************************************************/

/*	Not used in CoRTOS, but timer needs to be enabled via main()
	when CoRTOS is started. */

void timer_int_disable (void)
{
	intdis_nest_timer++;
	TA1CCTL0 &= ~CCIE;
}

void timer_int_enable (boolean force)
{
	if (force == true)
		intdis_nest_timer = 0;
	if (intdis_nest_timer != 0)
		intdis_nest_timer--;
	else
		TA1CCTL0 |= CCIE;
}

/************************************************************************
*
*	Initialization
*
************************************************************************/

#define development_WDT 0x5A80
void initialize_io_and_interrupts (void)
{
	/*	Stop the watchdog timer. */
	WDTCTL = development_WDT;

	/*	Change port as needed - this is for the the
		ti's MSP-EXP430FR6989 Explorer board. */
	PM5CTL0 = 0xfffe;
	P1DIR = 0x01;
	P9DIR = 0x80;

	/*	Set one wait state for FRAM access, the a5 is the unlock code,
		10 is the number of wait states - in the high nibble.  This
		is needed as the FRAM can only run at 8MHz and so needs the
		wait state when the CPU runs at 16MHz. */
	FRCTL0 = 0xa510; FRCTL0 = 0xa510;

	/*	Setup XT1 */
	PJSEL0 |= BIT4 | BIT5;

	/*	Unlock CS registers */
	CSCTL0 = 0xa500;

	/*	Set DCO to 16MHz */
	CSCTL1 = DCOFSEL_4|DCORSEL;

	/*	set ACLK = XT1; SMCLK= MCLK = DCO = 16MHz */
	CSCTL2 = SELA__LFXTCLK | SELS__DCOCLK | SELM__DCOCLK;

	/*	Set all dividers */
	CSCTL3 = DIVA__1 | DIVS__1 | DIVM__1;
	CSCTL4 &= ~LFXTOFF;

	/*	Timer A1: SMCLK/8, MC_1 upmode, TASSEL_2 : 2 MHz clock source */
	TA1CTL = TASSEL_2 + MC_1 + ID_3;

	/*	10mSec period */
	TA1CCR0 =  20000;
}

