/*	CoRTOSresource.c - Resource/queue/mutex

	2017.03.03 10:00	Initial release

	(c) 2017 Nicholas O. Lindan,
	Released under GPL V3 https://www.gnu.org/licenses/gpl.html */

#include "CoRTOScomdefs.h"
#include "CoRTOSkernel.h"
#include "CoRTOSresource.h"

#define number_of_resources 3
#define resource_queue_length 4

static uint8_t resource_owner [number_of_resources];

/*	The resource queues holding the task numbers of the tasks trying to
	acquire the resource. */
static uint8_t resource_queue [number_of_resources][resource_queue_length];

/*	The add and remove indeces into the resource_que and a count of the
	number of queuing tasks. */
static uint8_t resource_queue_add_index [number_of_resources];
static uint8_t resource_queue_remove_index [number_of_resources];
static uint8_t resource_queue_n [number_of_resources];

/*	Called to acquire resource rn.  If the resource is owned by another
	task then the caller may either wait in a	relinquish loop or return
	early.  Return values are:

		status_ok		task has acquired resource
		status_busy		resource in use by another task
		status_full		the resource queue is full, programmer error!
		status_timeout	A timeout was started and has expired, the task does not own
							the resource
*/

status_t acquire_resource (uint8_t rn, boolean wait_for_resource)
{
	uint8_t add_index;

	if (resource_owner[rn] == 0xff)
	{
		resource_owner[rn] = current_task;
		return (status_ok);
	}
	if (wait_for_resource == false)
		return (status_busy);
	if (resource_queue_n[rn] == resource_queue_length)
		return (status_full);

	add_index = resource_queue_add_index[rn];
	resource_queue[rn][add_index] = current_task;
	if (++resource_queue_add_index[rn] == resource_queue_length)
		resource_queue_add_index[rn] = 0;
	resource_queue_n[rn]++;

	suspend ();
	relinquish ();

	if (resource_owner[rn] != current_task)
		return (status_timeout);

	return (status_ok);
}

/*	After a task is done using a resource it releases it to either
	the free state or it advances the queue and gives the resource
	over to a new owner. */
void release_resource (uint8_t rn)
{
	uint8_t remove_index;

	/*	Nobody waiting, make the resource available. */
	if (resource_queue_n[rn] == 0)
	{
		resource_owner[rn] = 0xff;
		return;
	}

	/*	Task waiting, pull its task number from the queue and set
		the task as the new resource owner. */
	remove_index = resource_queue_remove_index[rn];
	resource_owner[rn] = resource_queue[rn][remove_index];
	resume_task (resource_owner[rn]);
	if (++remove_index == resource_queue_length)
		remove_index = 0;
	resource_queue_remove_index[rn] = remove_index;
	resource_queue_n[rn]--;
}

void initialize_resource_module (void)
{
	uint8_t rn;

	for (rn = 0; rn < number_of_resources; rn++)
	{
		resource_owner[rn] = 0xff;
		resource_queue_add_index[rn] = 0;
		resource_queue_remove_index[rn] = 0;
		resource_queue_n[rn] = 0;
	}
}

uint8_t query_resource_owner (uint8_t rn)
{
	return (resource_owner[rn]);
}



