/*	CoRTOSiointAVR.c - I/O and interrupts for AVR processors

	2017.03.11 12:00	Interrupts stay enabled, remove rtos_int_xx,
							rename to timer interrupt control - which isn't
							used by CoRTOS
	2017.03.03 10:00	Initial release

	(c) 2017 Nicholas O. Lindan,
	Released under GPL V3 https://www.gnu.org/licenses/gpl.html */

#include <avr/io.h>
#include <avr/interrupt.h>

#include "CoRTOScomdefs.h"
#include "CoRTOStimer.h"
#include "CoRTOSioint.h"

static uint8_t intdis_nest_timer = 0;

/************************************************************************
*
*	Timer compare interrupt
*
************************************************************************/

ISR (TIMER1_COMPA_vect)
{
	sei ();	
	service_timers ();

	/*	Reset the hardware for the next timer compare interrupt.
		The manual says the hardware does this on executing the
		vector, but this doesn't always seem to be the case. */
	TIFR1 = 0b00000010;
	TIMSK1 = 0b00000010;
}

/************************************************************************
*
*	I/O, only used for blinky & co.
*
************************************************************************/

/*	Flip _on and _off and use ~leds when setting the port if using an STK500 */

void led_off (uint8_t led)
{
	PORTD &= ~(1 << led);
}

void led_on (uint8_t led)
{
	PORTD |= 1 << led;
}

void led_toggle (uint8_t led)
{
	PORTD ^= 1 << led;
}

void write_leds (uint8_t leds)
{
	PORTD = leds;
}

uint8_t read_leds (void)
{
	return (PORTD);
}

/************************************************************************
*
*	interrupt disable/enable routines
*
************************************************************************/

/*	Not used in CoRTOS, but timer needs to be enabled via main()
	when CoRTOS is started. */

void timer_int_disable (void)
{
	intdis_nest_timer++;
	TIMSK1 &= ~bit1;
}

void timer_int_enable (boolean force)
{
	if (force == true)
		intdis_nest_timer = 0;
	if (intdis_nest_timer != 0)
		intdis_nest_timer--;
	else
		TIMSK1 |= bit1;
}

/************************************************************************
*
*	Initialization
*
************************************************************************/

void initialize_io_and_interrupts (void)
{

/*	Power reduction register, 1 turns subsys off.  Turn on subsys before
	setting it up.

		7	1	PRTWI		2-wire interface
		6	1	PRTIM2	Timer/counter 2
		5	1	PRTIM0	Timer/counter 0
		4	0	-
		3	0	PRTIM1	Timer/counter 1
		2	0	PRSPI		Serial peripheral interface - NEEDED FOR DEBUGING
		1	1	PRUSART0	UART 0
		0	1	PRADC		ADC

			  76543210 */
//	PRR = 0b11100011;
	
/*	Port B

		PB0	
		PB1	
		PB2	
		PB3	
		PB4	
		PB5	XPlained LED	
		PB6	
		PB7	XPlained switch

				 76543210 */
	PORTB = 0b00000000;
	DDRB =  0b11111111;

/*	Port C

		PC0	
		PC1	
		PC2	
		PC3	
		PC4	
		PC5	
		PC6	Reset/Debug wire

				 76543210 */
	PORTC = 0b00000000;
	DDRC =  0b11111111;

/*	Port D set to outputs - LEDs on the STK500 & on the LED sheild for
	the Atmel XPlained board

		PD0	LED0
		PD1	LED1
		PD2	LED2
		PD3	LED3
		PD4	LED4
		PD5	LED5
		PD6	LED6
		PD7	LED7

			    76543210 */
	PORTD = 0b00000000;
	DDRD =  0b11111111;

/*	Digital Input Disable register, set if the input is only for analog */

	DIDR0 = 0b00000000;

/*	TC1 Control Register A

						For non-PWM:
	7	0	COM1A1	00 - OC1A/1B output pins disconected
	6	0	COM1A0	01 - Toggled on match
	5	0	COM1B1	10 - Clear on match
	4	0	COM1B0	11 - Set on match
	3	0	
	2	0	
	1	0	WGM11		00 - Normal / Clear timer on compare
	0	0	WGM10		xx - PWM modes

				  76543210 */
	TCCR1A = 0b00000000;

/*	TC1 Control Register B

	Timer is clocked at 1MHz, 1x system clock
	Reloaded from OCR1A

	7	0	ICNC1		1 - Cancel noise on ICP1 pin
	6	0	ICES1		1 - Rising edge on ICP1 pin
	5	0	
	4	0	WGM13		01 - OCR1A is top value
	3	1	WGM12		11 - ICR1A is top value
	2	0	CS12		000->101 0, 1, 8, 64, 256, 1024 sys clock
	1	0	CS11		110 T0 pin falling
	0	1	CS10  	111 T0 pin rising

	Divide by 64 - @8MHz gives 8 uSec.

				  76543210 */
	TCCR1B = 0b00001011;

/*	TC1 Control Register C

	7	0	FOC1A		Force output compare channel A
	6	0	FOC1B		Force output compare channel B
	5	0	
	4	0	
	3	0	
	2	0	
	1	0	
	0	0	

				  76543210 */
	TCCR1C = 0b00000000;

/*	Set up timer 1 for 10 mSec. overflow & interrupt

	8MHz internal clock	=>      0.125 uS
	prescale / 64 			=>      8.0
	1,250 (0x04e2) OCRA	=> 10,000.0 */

	OCR1AH = 0x04;
	OCR1AL = 0xe2;

/*	TC1 Interupt Mask Register

	7	0	
	6	0	
	5	0	ICIE1		1 - Interupt on TIFR1.ICF1 set
	4	0	
	3	0	
	2	0	OCIE1B	1 - Interupt on TIFR1.OCF1B set
	1	0	OCIE1A	1 - Interupt on TIFR1.OCF1A set
	0	0	TOIE1		1 - Interupt on TIFR1.TOV1 set

				  76543210 */
	TIMSK1 = 0b00000000;

/*	TC1 Interupt Flag Register

	7	0	
	6	0	
	5	0	ICF1		1 - TC1 == top & WGM1x set to use ICR1 as top
	4	0	
	3	0	
	2	0	OCF1B		1 - TC1 == OCR1B
	1	0	OCF1A		1 - TC1 == OCR1A			Supposedly cleared on isr vectoring
	0	0	TOV1		1 - TC1 == 0xffff -> 0

				 76543210 */
	TIFR1 = 0b00000000;
}

