/*	CoRTOSiointPIC.c - I/O and interrupts for Microchip PIC24 processors

	2017.03.11 12:00	Interrupts stay enabled, remove rtos_int_xx,
							rename to timer interrupt control - which isn't
							used by CoRTOS
	2017.03.03 10:00	Initial release

	(c) 2017 Nicholas O. Lindan,
	Released under GPL V3 https://www.gnu.org/licenses/gpl.html */

#include "p24FJ128GA204.h"

#include "CoRTOScomdefs.h"
#include "CoRTOStimer.h"
#include "CoRTOSioint.h"

static uint8_t intdis_nest_timer = 0;

/************************************************************************
*
*	Timer compare interrupt
*
************************************************************************/

/*	The interrupt is triggered 10mSec period on output
	compare using timer 1.  Its function is to decrement
	the timers of any delayed tasks via a call to service_timers(). */

void __attribute__((__interrupt__, no_auto_psv)) _T1Interrupt (void)
{
	IFS0bits.T1IF = 0;
	service_timers ();
}

/************************************************************************
*
*	I/O, only used for blinky & co.
*
************************************************************************/

/*	Change I/O assignments as needed for your hardware -
	this works with a Microchip DM240004 "Curiosity" PIC24F
	development board. */

void led_on (uint8_t led)
{
	LATC |= 1 << (led + 5);
}

void led_off (uint8_t led)
{
	LATC &= ~(1 << (led + 5));
}

void led_toggle (uint8_t led)
{
	LATC ^= 1 << (led + 5);
}

void write_leds (uint8_t leds)
{
	LATC = (LATC & 0xff1f) | ((leds & 0x07) << 5);
}

uint8_t read_leds (void)
{
	return (((uint8_t)LATC >> 5) & 0x07);
}

/************************************************************************
*
*	interrupt disable/enable routines
*
************************************************************************/

/*	Not used in CoRTOS, but timer needs to be enabled via main()
	when CoRTOS is started. */

void timer_int_disable (void)
{
	intdis_nest_timer++;
	IEC0bits.T1IE = 0;
}

void timer_int_enable (boolean force)
{
	if (force == true)
		intdis_nest_timer = 0;
	if (intdis_nest_timer != 0)
		intdis_nest_timer--;
	else
		IEC0bits.T1IE = 1;
}

/************************************************************************
*
*	Initialization
*
************************************************************************/

void initialize_io_and_interrupts (void)
{
    /*  8MHz processor, 4 cycles/Hz => 2MHz / 8 = 250kHz */
 	T1CON = 0x0010;
	TMR1 = 0x0000;
    /*  250kHz / 2,500 = 100Hz*/
	PR1 = 2500;

	IPC0bits.T1IP = 0x01;
	IFS0bits.T1IF = 0;
   IEC0bits.T1IE = 1;
	T1CONbits.TON = 1;

	/*	LED I/O pins to outputs */
	TRISC &= 0xff1f;
}

