/*	CoRTOSkernelAVR.c - Kernel for AVR processors

	2017.03.03 10:00	Initial release

	(c) 2017 Nicholas O. Lindan,
	Released under GPL V3 https://www.gnu.org/licenses/gpl.html */

/*	For AVR processors - change for different processors.
	The only thing needed from this file is the definition
	of the stack pointer register as "SP" */
#include <avr\io.h>
#include <avr\interrupt.h>

#include "CoRTOScomdefs.h"
#include "CoRTOStask.h"
#include "CoRTOSasmmacrosAVR.h"
#include "CoRTOSkernel.h"

/**********************************************************************
*
*	Public variables
*
**********************************************************************/

uint8_t current_task;

/**********************************************************************
*
*	The local OS Kernel variables
*
**********************************************************************/

/*	The system saves the stack pointer for each task when it relinquishes
	control, and then restores it when the task gets control again. */
static uint16_t sp_save [number_of_tasks];

/*	Top of the stack for each task - used when a task is started from
	its entry point.  May need to change this to accomodate different
	stack pointer sizes. */
static uint16_t starting_stack [number_of_tasks];

/*	Indicates a task is to start from its entry point, normal for all
	tasks when they first get control.  The flag is set again if a task
	executes a return.  Also see the reset/restart functions. */
static boolean start_from_beginning [number_of_tasks];

/*	Reset if a task executes a return.  Would need to be reset by an
	external agency for the task to execute again.  Manipulated
	by the suspend/resume functions. */
static boolean suspended [number_of_tasks];

/************************************************************************
*
*	relinquish () - the whole of CoRTOS in one function
*
************************************************************************/

void relinquish (void)
{
	pushall();
	/*	Save the currently executing task, as it will be replaced by
		a new current task. */
	sp_save[current_task] = SP;

	while (1)
	{
		do
		{
			/*	Go to the next task, the if statement is preferable to
				the % mod operator. */
			if (++current_task == number_of_tasks)
				current_task = 0;
		/*	Skip the task and go on to the next one if the new one is
			inactive. */
		} while (suspended[current_task] != false);

		if (start_from_beginning[current_task] == true)
		{
			/*	The start from beginning flag is set at power-up and when a
				task is reset/restarted or has returned.  Reset the flag so
				the task resumes from where it left off in the next go-round. */
			start_from_beginning[current_task] = false;
			/*	Since it is a virgin start, the task gets a virgin stack.
				cli/sei are needed for the AVR where the SP is set by two
				8-bit register-register transfers.  An interrupt in the middle
				would be a small disaster. */
			cli ();
			SP = starting_stack[current_task];
			sei ();
			/*	Give control to the task. */
			start_addresses[current_task] ();
			/*	If the task executes a return then it is out of the system.
				If restarted it will begin at its entry point. */
			suspended[current_task] = true;
			start_from_beginning[current_task] = true;
		}
		else
		{
			/*	Load the stack pointer for the new current task, current_task
				was incremented above.  Enable/disable only needed for 8 bit
				machines like the AVR. */
			cli ();
			SP = sp_save[current_task];
			sei ();
			/*	Go back to where this task called relinquish (). */
			popall();
			return;
		}
	}
}

/************************************************************************
*
*	suspend/resume/restart ...
*
************************************************************************/

/*	A suspended task will be skipped over by relinquish(). */
void suspend (void)
{
	suspended[current_task] = true;
}

/*	A resumed task will again be given control by relinquish(). */
void resume_task (uint8_t tn)
{
	suspended[tn] = false;
}

/*	CAUTION:

	Be very careful when calling these two functions - be sure the
	task is not the owner of, or qued to, a resource, or has signals
	or messages pending.  These issues need to be dealt with
	seperately. */

/*	A restarted task will be given control by relinquish() starting
	from its entry point. */
void restart_task (uint8_t tn)
{
	start_from_beginning[tn] = true;
	suspended[tn] = false;
}

/*	A reset task will be skipped over by relinquish(), if it is
	made active again by a call to resume_task() then it will
	begin from its entry point. */
void reset_task (uint8_t tn)
{
	start_from_beginning[tn] = true;
	suspended[tn] = true;
}

/************************************************************************
*
*	start_CoRTOS
*
************************************************************************/

/*	This function is called from main() to start the OS.  It initializes
	the variables used by relinquish() for controling task execution.

	Control isn't given to the OS but to the first task, which then
	gives control to the OS on its first call to relinquish().

	This 'backwards into the future' method of starting a real time OS
	is rather common.  The tasks and the OS are co-routines that start
	each other in something resembling a dance. */

void start_CoRTOS (void)
{
	static uint8_t tn;
	static uint16_t spv;

	spv = SP;
	for (tn = 0; tn < number_of_tasks; tn++)
	{
		starting_stack[tn] = spv;
		spv -= task_stack_size[tn];
		start_from_beginning[tn] = true;
		suspended[tn] = false;
	}

	/*	The start_from_beginning flag is cleared as we start the task
		from its entry point here, relinquish() will start this task
		again from the point where it surrenders control to relinquish() */
	start_from_beginning[0] = false;
	current_task = 0;
	start_addresses[0] ();

	/*	If the first task, with the task number 0, executes a return
		then control will pass to this point and CoRTOS will cease
		execution.

		Control will then pass to the code that called start_CoRTOS() */
}

