#include "main.h"

#include <stdarg.h>
#include <stdio.h>

#include "InitializeBoard.h"

//////////////////////////////////////////
// Defines
//////////////////////////////////////////
// HI 16MHz clock to ACLK 6 - 14 KHz range ratio limits
#define CLOCK_RATIO_6KHZ				2666
#define CLOCK_RATIO_12KHZ				1142

//////////////////////////////////////////
// Members
//////////////////////////////////////////
UINT16		g_nLastHFToLFRate;					// For debug

//////////////////////////////////////////
// Forward
//////////////////////////////////////////
static BOOL InitializeClock();

//////////////////////////////////////////
// Code
//////////////////////////////////////////
// Initial initialization requires flag polling etc, so disable compiler warnings temporary
#pragma diag_push

#pragma diag_suppress 1528 // Detected flag polling using SFRIFG1. Recommend using an interrupt combined with enter LPMx and ISR
#pragma diag_suppress 1543 // Detected for loop with repetitive data transfer. Recommend using DMA
#pragma diag_suppress 1544 // Detected loop counting up. Recommend loops count down as detecting zeros is easier


// Initialize board for main operation
// RAM is lost here regardless of bAfterReset
#pragma FUNCTION_OPTIONS(InitializeBoard, "--opt_level=off")
BOOL InitializeBoard()
{
	// Configure reset logic
	SFRIE1 = 0;

	// Configure Reset Pin
	SFRRPCR = (BIT4 | SYSRSTRE | SYSRSTUP);				// PIN with RESET function + Pullup

	// TODO: Configure unused pins as I/O function, output direction, and left unconnected on the PC board.

	// The application should make sure that port PJ is configured properly to prevent a floating input.
	// 		JTAG TEST pin is pulled to ground internally and not configurable, so nothing to do here. TODO: Re-check this
	SYSCTL = 0x0;

	// Order of port initialization must not be changed
	// 1. Initialize Ports: PxDIR, PxREN, PxOUT
//	PxSEL0 = PxSEL1 ... = 0x00;	// Set by reset TODO: Init anyway!
	// TODO: Preserve pump IO & operation
	P1DIR = P2DIR = P3DIR = P4DIR = P5DIR = P6DIR = P7DIR = P8DIR = PJDIR_L = PJDIR_H = 0x0;	// All hi impedance inputs
	P1REN = P2REN = P3REN = P4REN = P5REN = P6REN = P7REN = P8REN = PJREN_L = PJREN_H = 0x0;	// All weak pullups/pulldowns - off
	P1OUT = P2OUT = P3OUT = P4OUT = P5OUT = P6OUT = P7OUT = P8OUT = PJOUT_L = PJOUT_H = 0x0;	// All outputs to 0
	P1IE = P2IE = P3IE = P4IE = /* P5IE = P9IE = PJIE = */ 0x0;		// Disable all pin interrupts

	LED_OPERATIONS_PORT &= ~LED_OPERATIONS_MASK;                     // to off
	LED_OPERATIONS_PORT_DIR |= LED_OPERATIONS_MASK;  		        // to output direction

    LED_ERROR_PORT &= ~LED_ERROR_MASK;                     // to off
    LED_ERROR_PORT_DIR |= LED_ERROR_MASK;                 // to output direction

    // 2. Apply IO port configuration to pins
    // Any reset pins are hi-impedance till this is set (incl. some wakeup will set this bit and disable port configuration effect on pins)
	PM5CTL0 &= ~LOCKLPM5;

	// Init clock system for max performance
	// InitializeClock will setup pins direction & pins function, so do it after port initialization
	// If you wish early clock init, you can move this above port initialization, but change port init not to touch clock pins
	BOOL bInitOK = InitializeClock();

	// 3. TODO: If not wake-up from LPMx.5: clear all PxIFGs to avoid erroneous port interrupts
	// PxIFG flags can't be cleared before LOCKLPM5 is cleared
	P1IFG = P2IFG = P3IFG = P4IFG = /* P5IFG = P9IFG = PJIFG = */ 0x0;

	// 4. Enable port interrupts in PxIE

	return bInitOK;
}

// Measure SMCLK vs ACLK ratio
// Used to check if clocks are working properly (since there is no HW indication on internal clocks)
// Uses T3
static UINT16 MeasureClocksRatio()
{
	UINT16 nMeasuredFreq_CURR, nMeasuredFreq_PREV;		// SMCLK ticks per single ACLK cycle

	// Init clock for capture, run on SMCLK, capture on ACLK, to measure their proportions
	TA3CTL = 0;							// Disable clock
	TA3CCTL0 = TA3CCTL1 = 0;
	TA3EX0 = TAIDEX_0;
	TA3CTL = TASSEL__SMCLK | ID__1 | MC__CONTINUOUS | TACLR;
	TA3CCTL0 = CM_1 | CCIS_1 | CAP | SCS;

	// Capture clocks
	TA3R = 0;							// Clear because we also monitor overflow
	CLR_BIT_ATOMIC_W(TA3CTL, TACLR);	// Clear TA3 - overflow is monitored ~4ms if ACLK is not ticking
	CLR_BIT_ATOMIC_W(TA3CTL, TAIFG);	// Clear TA3 - overflow flag
	CLR_BIT_ATOMIC_W(TA3CCTL0, CCIFG);	// Clear capture flag
	int i = 2;
	while(i > 0)
	{
		if(TA3CCTL0 & CCIFG)					// Check if capture occured
		{
			nMeasuredFreq_PREV = nMeasuredFreq_CURR;
			nMeasuredFreq_CURR = TA3CCR0;
			CLR_BIT_ATOMIC_W(TA3CCTL0, CCIFG);	// Clear capture flag
			i--;
		}

		if(TA3CTL & TAIFG)						// Check if overflow while waiting for ACLK
		{
			nMeasuredFreq_CURR = nMeasuredFreq_PREV = 0;
			break;
		}
	}

	// Stop clock
	TA3CTL = TA3CCTL0 = 0;

	return nMeasuredFreq_CURR - nMeasuredFreq_PREV;
}

// Clocks
// HFXTCLK = 16 MHz (external)
// LFXTCLK = Not connected (external)
// VLOCLK = 10 KHz (internal)
// DCOCLK = 1 MHz - 24 MHz software selectable (internal)
// MODOSC = 5 MHz (internal) - along with LFMODCLK, used as a backup for lots of modules MCLK, SMCLK, WDT...
// LFMODCLK = 39 KHz - MODCLK / 128 (internal) - along with MODOSC, used as a backup for lots of modules MCLK, SMCLK, WDT...

// Available for use (All have dividers : 1/2/4/8/16/32):
// MCLK: Master clock 				[LFXTCLK, VLOCLK, LFMODCLK, DCOCLK, MODCLK, HFXTCLK] - used by the CPU and system
// SMCLK: Subsystem master clock 	[LFXTCLK, VLOCLK, LFMODCLK, DCOCLK, MODCLK, HFXTCLK] - software selectable by individual peripheral modules
// MODCLK: Module clock 			[MODOSC]	- software selectable by individual peripheral modules
// VLOCLK: VLO clock 				[VLO]		- software selectable by individual peripheral modules
// ACLK: Auxiliary clock 			[LFXTCLK, VLOCLK, LFMODCLK] - software selectable by individual peripheral modules

// WDT : After PUC 32-ms with SMCLK
// If SMCLK or ACLK fails as the WDT_A clock source, LFMODCLK clock is automatically selected (if configured as disabled, it will be enabled) as the WDT clock source.

// Initialize clock for max freq
// This function will try to :
// x. Configure clock system for HI freq. operation on external oscillator, with backup internal DCO
// x. Configure LO freq. ACLK clock system operation on VLO clk (used for WDT, drive external pumps, use in lo-power modes), with backup LFMODCLK
// x. Try not to disturb any driving external circuity driven by ACLK during sleep (before this call)
//		Because of this, we can't switch from LFMODCLK back to VLO (for lower power during sleep)
//		TODO: Try to check & use VLO instead of LFMODCLK somehow, without disturbing driving of external hardware, or switch to HI osc drive, then try to switch to VLO!
#pragma FUNCTION_OPTIONS(InitializeClock, "--opt_level=off")
static BOOL InitializeClock()
{
// After PUC :
//	LFXTCLK (not connected) -> LFXTCLK - ACLK [/1]
//	DCOCLK @ 8 MHz / 8 -> MCLK + SMCLK = 1MHz
//	LFXTCLK - disabled
//	HFXTCLK - disabled

	///////////////////////////////////////////////////////////////////////////////////////
	// Setup FRAM Wait states before switch to HI Freq (FRAM Max freq = 8 MHz)
	///////////////////////////////////////////////////////////////////////////////////////
	FRCTL0_H = 0xA5;						// Unlock FRAM module registers for writing

	FRCTL0_L = NWAITS_1_L | ACCTEIE_L;		// 1 Wait state, enable access time violation NMI
	GCCTL0 = UBDRSTEN | FRPWR;				// Enable PUC on bit flip, FRAM power-ON, FRAM auto power up after LPM
	GCCTL1 = 0;								// Various error flags

	FRCTL0_H = (BYTE) ~0xA5;				// Lock FRAM module registers for writing

	///////////////////////////////////////////////////////////////////////////////////////
	// Configure clock system
	// Registers CSCTL0 to CSCTL6 configure the CS module
	///////////////////////////////////////////////////////////////////////////////////////
	// Configure HI Freq OSC, without disturbing ACLK, since if we wakeup from low power mode, we may drive something with ACLK (as the only available clock in low power mode)
	CSCTL0_H = CSKEY_H;						// Unlock clock module registers for writing

	// If ACLK source = LFXTCLK, we can't clear SFRIFG1.OFIFG below (do not disturb other settings, since we know they work somehow, we reach this code somehow)
	if((CSCTL2 & (SELA0 | SELA1 | SELA2)) == SELA__LFXTCLK)
		CSCTL2 = ((CSCTL2 & ~(SELA0 | SELA1 | SELA2)) | SELA__VLOCLK);

	CSCTL6 = MODCLKREQEN | SMCLKREQEN | MCLKREQEN | ACLKREQEN;	// Enable, so we can choose operational clock
											// If = 0, this does not disable fail-safe features of MCLK, SMCLK and WDT (other?)
											// Also these are ignored in some PM states, check "3.2.7 Operation From Low-Power Modes, Requested by Peripheral Modules" in MSP UserGuide

	// Configure external HFXT OSC (if we are running currently on this, following code will just override same settings so it will no have any effect on current execution)
	// CSCTL4.HFXTOFF = 1 will not switch off osc if we currently are running MCLK on it, try not to override temporary with invalid settings
	PJSEL0_L |= BIT6;						// Configure port function - will set pins IO direction
	PJSEL1_L &= ~BIT6;

	CSCTL4 = LFXTOFF | HFFREQ1 | (HFXTDRIVE0 | HFXTDRIVE0);
	CSCTL5 = ENSTFCNT2;						// Enable start counter for HFXT when available - clear external OSC fault bits

	// Select HFXTCLK for SMCLK
	CSCTL2 = ((CSCTL2 & ~(SELS0 | SELS1 | SELS2)) | SELS__HFXTCLK);

	// Select clock divider = 1 for SMCLK, MCLK (ACLK may drive something)
	CSCTL3 = ((CSCTL3 & (DIVA0 | DIVA1 | DIVA2)) | DIVS__1 | DIVM__1);	// May disrupt current clock

	// Wait to see if HF clock will be stable
	UINT16 nCntr = 0;
	while(SFRIFG1 & OFIFG)
	{
		 if(nCntr++ > 16000)				// ~4 ms timeout @ 16MHz (calc with 4 clock cycles)
            break;

		CLR_BIT_ATOMIC_W(CSCTL5, HFXTOFFG);	// Will stay clear if OSC is ok
		CLR_BIT_ATOMIC_W(CSCTL5, LFXTOFFG);	// Should be clear also to be able to clear OFIFG

		CLR_BIT_ATOMIC_W(SFRIFG1, OFIFG);		// Can't be cleared if HFXTOFFG is set, so no chance of missing interrupt

		CLR_WDT();
	}

//	SET_BIT_ATOMIC_W(SFRIFG1, OFIFG); // for tests

	// If fault is persistent, use DCO
	if(SFRIFG1 & OFIFG)
	{
		// Select DCO for main HF clock
		// ERRATTA : CS12
		// operation : CSCTL1 = DCORSEL | DCOFSEL_4;			// DCOCLK Clock configuration - DCORSEL = 1, DCOFSEL = 4 = 16 MHz
		UINT16 nOrigCSCTL3 = CSCTL3;
		CSCTL3 = CSCTL3 & (~(0x77)) | DIVS__4 | DIVM__4;
		CSCTL1 = DCORSEL | DCOFSEL_4;							// DCOCLK Clock configuration - DCORSEL = 1, DCOFSEL = 4 = 16 MHz
		__delay_cycles(60);										// Delay by ~10us to let DCO settle. 60 cycles = 20 cycles buffer + (10us / (1 / 4MHz))
		CSCTL3 = nOrigCSCTL3;

		CSCTL2 = ((CSCTL2 & ~(SELS0 | SELS1 | SELS2)) | SELS__DCOCLK);		// Use DCO for SMCLK
		CSCTL2 = ((CSCTL2 & ~(SELM0 | SELM1 | SELM2)) | SELM__DCOCLK);		// Use DCO for MCLK
	}
	else
	{
		// Select HFXTCLK for main HF clock
//		CSCTL1 = 0;								// DCOCLK Clock configuration - Not used for now @ 1MHz
		// - Do not touch it, because it may overshoot and affect FRAM (see ERRATA)

		CSCTL2 = ((CSCTL2 & ~(SELS0 | SELS1 | SELS2)) | SELS__HFXTCLK);	// Use HFXTCLK for SMCLK
		CSCTL2 = ((CSCTL2 & ~(SELM0 | SELM1 | SELM2)) | SELM__HFXTCLK);	// Use HFXTCLK for MCLK

		SFRIE1 |= OFIE;			// Enable interrupt for external OSC fail
	}

	/////////////////////////////////////////////////////////////
	// Configure ACLK & WDT
	/////////////////////////////////////////////////////////////
	BOOL bSwitchToBackup = TRUE;
	WDTCTL = (WDTPW | WDTSSEL__LFMODCLK | WDTIS0 | WDTIS1 | WDTCNTCL);				// Temporary @ 13 sec.

	// Be careful not to disturb clock, because it may drive something
	// If ACLK drive OSC is different from SELA__LFMODCLK, it should be configured as SELA__VLOCLK
	if((CSCTL2 & (SELA0 | SELA1 | SELA2)) != SELA__LFMODCLK)
	{
		// Reconfigure as VLOCLK, if running already, nothing will happen and external HW will continue to be clocked
		CSCTL3 = ((CSCTL3 & ~(DIVA0 | DIVA1 | DIVA2)) | DIVA__1);
		CSCTL2 = ((CSCTL2 & ~(SELA0 | SELA1 | SELA2)) | SELA__VLOCLK);

		WDTCTL = (WDTPW | WDTSSEL__VLO | WDTIS0 | WDTIS1 | WDTCNTCL);				// 43 - 105 sec with VLOCLK

		// Check if relation between clocks are in range (this will check also if master clock is ok)
		g_nLastHFToLFRate = MeasureClocksRatio();
		bSwitchToBackup = ((g_nLastHFToLFRate < CLOCK_RATIO_12KHZ) || (g_nLastHFToLFRate > CLOCK_RATIO_6KHZ));
	}

// bSwitchToBackup = TRUE;		// For tests

	// If VLCLK failed, try using LFMODCLK @ 39 KHz as ACLK @ 10 KHz
	if(bSwitchToBackup)
	{
		// Switch ACLK to LFMODCLK
		CSCTL3 = ((CSCTL3 & ~(DIVA0 | DIVA1 | DIVA2)) | DIVA__4);
		CSCTL2 = ((CSCTL2 & ~(SELA0 | SELA1 | SELA2)) | SELA__LFMODCLK);

		WDTCTL = (WDTPW | WDTSSEL__LFMODCLK | WDTIS0 | WDTIS1 | WDTCNTCL);				// LFMODCLK @ 39KHz - 13 sec timeout

		// Check if relation between clocks are in range (this will check also if master clock is ok)
		g_nLastHFToLFRate = MeasureClocksRatio();
		bSwitchToBackup = ((g_nLastHFToLFRate < CLOCK_RATIO_12KHZ) || (g_nLastHFToLFRate > CLOCK_RATIO_6KHZ));
	}

	// Permanent error configuring ACLK (means device is totally useless)
	if(bSwitchToBackup)
	{
		// Try with VLO clk when next reset (by main code if initialization is not successfull)
		CSCTL3 = ((CSCTL3 & ~(DIVA0 | DIVA1 | DIVA2)) | DIVA__1);
		CSCTL2 = ((CSCTL2 & ~(SELA0 | SELA1 | SELA2)) | SELA__VLOCLK);
	}

	CSCTL0_H = (BYTE) ~CSKEY_H;				// Disable clock module registers write

	CLR_WDT();

	return !bSwitchToBackup;
}

void Halt()
{
	DISABLE_INTS();
	while(TRUE)
	{
		CLR_WDT();
	}
}

#pragma diag_pop
