/*
 * pinout.h
 *
 *  Created on: 17.11.2024
 *      Author: 
 */

#include <msp430.h>
#include "pinout.h"
#include "ports.h"

void configureClocks() {
    // Set DCO frequency to 1 MHz
    CSCTL0_H = CSKEY >> 8; // Unlock CS registers
    CSCTL1 = DCOFSEL_3;    // Set DCO range select to 1 MHz
    CSCTL2 = SELA__VLOCLK | SELS__DCOCLK | SELM__DCOCLK; // Set ACLK = VLO, SMCLK = DCO, MCLK = DCO
    CSCTL3 = DIVA__1 | DIVS__1 | DIVM__1; // Set all dividers to divide by 1
    CSCTL0_H = 0;          // Lock CS registers
}

void initUART() {
    PDIR_UART_TX |= BIT_UART_TX;    //Set as output
    POUT_UART_TX &= ~BIT_UART_TX;   //Set 0
    PSEL0_UART_TX |= BIT_UART_TX;   //Set UART mode
    PSEL1_UART_TX &= ~BIT_UART_TX;

    PDIR_UART_RX &= ~BIT_UART_RX;    //Set as input
    PREN_UART_RX &= ~BIT_UART_RX;   //Disable pullup
    PSEL0_UART_RX |= BIT_UART_RX;   //Set UART mode
    PSEL1_UART_RX &= ~BIT_UART_RX;

    UCA3CTL1 |= UCSWRST;
    UCA3CTL1 |= UCSSEL_2;
    UCA3BR0 = 0x08;
    UCA3BR1 = 0x00;     //115200 baud
    UCA3CTL1 &= ~UCSWRST;
}

void initSPI() {
    PSEL1_SPI_CLOCK1 &= ~BIT_SPI_CLOCK1;
    PSEL0_SPI_CLOCK1 |= BIT_SPI_CLOCK1;
    PSEL1_SPI_SOMI1 &= ~BIT_SPI_SOMI1;
    PSEL0_SPI_SOMI1 |= BIT_SPI_SOMI1;
    PSEL1_SPI_SIMO1 &= ~BIT_SPI_SIMO1;
    PSEL0_SPI_SIMO1 |= BIT_SPI_SIMO1;
    UCA0CTL1 |= UCSWRST;
    UCA0CTL0 |= UCCKPH + UCCKPL + UCMSB + UCMST + UCSYNC + UCSSEL__SMCLK; // 3-pin, 8-bit SPI master
    UCA0CTL1 |= UCSSEL_2; // SMCLK
    UCA0BR0 |= 0x08; // /2
    UCA0BR1 = 0; //
    UCA0CTL1 &= ~UCSWRST; // **Initialize USCI state machine**

    PSEL1_SPI_CLOCK2 &= ~BIT_SPI_CLOCK2;
    PSEL0_SPI_CLOCK2 |= BIT_SPI_CLOCK2;
    PSEL1_SPI_SOMI2 &= ~BIT_SPI_SOMI2;
    PSEL0_SPI_SOMI2 |= BIT_SPI_SOMI2;
    PSEL1_SPI_SIMO2 &= ~BIT_SPI_SIMO2;
    PSEL0_SPI_SIMO2 |= BIT_SPI_SIMO2;
    UCA1CTL1 |= UCSWRST;
    UCA1CTL0 |= UCCKPH + UCCKPL + UCMSB + UCMST + UCSYNC + UCSSEL__SMCLK;
    UCA1CTL1 |= UCSSEL_2;
    UCA1BR0 |= 0x08;
    UCA1BR1 = 0;
    UCA1CTL1 &= ~UCSWRST;

    PSEL1_SPI_CLOCK3 &= ~BIT_SPI_CLOCK3;
    PSEL0_SPI_CLOCK3 |= BIT_SPI_CLOCK3;
    PSEL1_SPI_SOMI3 &= ~BIT_SPI_SOMI3;
    PSEL0_SPI_SOMI3 |= BIT_SPI_SOMI3;
    PSEL1_SPI_SIMO3 &= ~BIT_SPI_SIMO3;
    PSEL0_SPI_SIMO3 |= BIT_SPI_SIMO3;
    UCA2CTL1 |= UCSWRST;
    UCA2CTL0 |= UCCKPH + UCCKPL + UCMSB + UCMST + UCSYNC + UCSSEL__SMCLK;
    UCA2CTL1 |= UCSSEL_2;
    UCA2BR0 |= 0x08;
    UCA2BR1 = 0;
    UCA2CTL1 &= ~UCSWRST;

    PSEL1_SPI_CLOCK4 &= ~BIT_SPI_CLOCK4;
    PSEL0_SPI_CLOCK4 |= BIT_SPI_CLOCK4;
    PSEL1_SPI_SOMI4 &= ~BIT_SPI_SOMI4;
    PSEL0_SPI_SOMI4 |= BIT_SPI_SOMI4;
    PSEL1_SPI_SIMO4 &= ~BIT_SPI_SIMO4;
    PSEL0_SPI_SIMO4 |= BIT_SPI_SIMO4;
    UCB2CTL1 |= UCSWRST;
    UCB2CTL0 |= UCCKPH + UCCKPL + UCMSB + UCMST + UCSYNC + UCSSEL__SMCLK;
    UCB2CTL1 |= UCSSEL_2;
    UCB2BR0 |= 0x08;
    UCB2BR1 = 0;
    UCB2CTL1 &= ~UCSWRST;
}

// Sends one Byte over SPI bus s
void SendUCA0Data(char val)
{
    while (UCA0STATW & UCBUSY);           // USCI_A1 TX buffer ready?
    UCA0TXBUF = val;
}

void SendUCA1Data(char val)
{
    while (UCA1STATW & UCBUSY);
    UCA1TXBUF = val;
}

void SendUCA2Data(char val)
{
    while (UCA2STATW & UCBUSY);
    UCA2TXBUF = val;
}

void SendUCB2Data(char val)
{
    while (UCB2STATW & UCBUSY);
    UCB2TXBUF = val;
}


// Testing for power-on
void testV2A() {
    WDTCTL = WDTPW | WDTHOLD;               // Stop watchdog timer
    setAllHighImpedance();
    PM5CTL0 &= ~LOCKLPM5;               // Disable the GPIO power-on default high-impedance mode to activate previously configured port settings

    initAllPE();
    initGPIOAsOutput(BIT_PE_COMMON3V3, POUT_PE_COMMON3V3, PDIR_PE_COMMON3V3, PSEL0_PE_COMMON3V3, PSEL1_PE_COMMON3V3, 1);
    setAllPE(1);
}

// Testing for power-off
void testV2B() {
    WDTCTL = WDTPW | WDTHOLD;
    setAllHighImpedance();
    PM5CTL0 &= ~LOCKLPM5;

    initAllPE();
    initGPIOAsOutput(BIT_PE_COMMON3V3, POUT_PE_COMMON3V3, PDIR_PE_COMMON3V3, PSEL0_PE_COMMON3V3, PSEL1_PE_COMMON3V3, 1);
    setAllPE(0);
}

// Testing for UART
void testV2C() {
    WDTCTL = WDTPW | WDTHOLD;               // Stop watchdog timer
    setAllHighImpedance();
    PM5CTL0 &= ~LOCKLPM5;               // Disable the GPIO power-on default high-impedance mode to activate previously configured port settings

    initAllPE();
    initGPIOAsOutput(BIT_PE_COMMON3V3, POUT_PE_COMMON3V3, PDIR_PE_COMMON3V3, PSEL0_PE_COMMON3V3, PSEL1_PE_COMMON3V3, 1);
    setAllPE(1);

    configureClocks();
    initUART();

    long rxCnt = 0;
    uchar in = 0;
    while(1) {
        UCA3TXBUF = 'z';
        in = UCA3RXBUF;
        UCA3RXBUF = 0x00;
        if (in != 0)
            rxCnt++;
    }
}

// Testing for SPI functionality,  reading SSIB
void testV2D() {
    WDTCTL = WDTPW | WDTHOLD;               // Stop watchdog timer
        setAllHighImpedance();
        PM5CTL0 &= ~LOCKLPM5;               // Disable the GPIO power-on default high-impedance mode to activate previously configured port settings

        initAllPE();
        initGPIOAsOutput(BIT_PE_COMMON3V3, POUT_PE_COMMON3V3, PDIR_PE_COMMON3V3, PSEL0_PE_COMMON3V3, PSEL1_PE_COMMON3V3, 1);
        setAllPE(1);

        configureClocks();
        initSPI();
        while(1) {
            SendUCA0Data('z');
            SendUCA1Data('z');
            SendUCA2Data('z');
            SendUCB2Data('z');
        }
}

// Test for reading HMC
void testV2E() {

}


void main(void) {
     testV2D();
}





