/* --COPYRIGHT--,BSD
 * Copyright (c) 2017, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//******************************************************************************
//   MSPBoot: MSP430G2553 Host using I2C with MSP430G2553 Target
//
//   Description: Host follows BSL protocol to program a MSP430G2553 target
//   via I2C using MSPBoot. Please refer to SLAA600 for more information
//
//   ***NOTE: REMOVE LED2(P1.6) JUMPER FOR PROPER I2C COMMUNICATION
//   
//   Instructions:
//   1. Connect host and target as seen below
//   2. At startup, LED2 of the host will be ON
//   3. Press S2 on the Host to start transferring the first application image
//      a. LED1 on the host will blink at start of transfer, then remain on when 
//         complete
//      b. LED1 on the target will periodically blink when the first application 
//         has been programmed
//   4. Press S2 on the target to force it to enter MSPBoot code	
//      a. LED1 will stop blinking
//   5. Press S2 on the host to start transferring the second application image
//      a. LED1 on the host will blink at start of transfer, then remain off when 
//         complete
//      b. LED1 on the target will blink once when the image is successfully 
//         programmed
//   6. Press S2 on the target and P2.0 will toggle high and low
//   7. Press S2 on the host to repeat the process starting at step 1
//
//             MSP430G2553 (HOST)                  MSP430G2553 (TARGET)
//             ------------------                  ------------------
//      /|\   |                  |/|\          /|\|                  |   /|\
//       --o--|P1.3/S2           | |            | |           P1.3/S2|--o--
//      \|/   |            S1/RST|--            --|RST/S1            |   \|/
//            |                  |                |                  |
//            |                  |                |                  |   
//            |                  |   /|\   /|\    |                  |
//            |                  |   10k   10k    |                  | 
//            |                  |    |     |     |                  |
//     LED1<--|P1.0  P1.6/UCB0SCL|----|-----+---->|P1.6/UCB0SCL  P1.0|-->LED1
//            |                  |    |           |                  |
//  LED2<-  \-|P1.6  P1.7/UCB0SDA|<---+---------->|P1.7/UCB0SDA  P1.6|-/  ->LED2
//                                       
//	 Caleb Overbay
//   Texas Instruments Inc.
//   June 2017
//******************************************************************************

#include <msp430.h> 
#include <stdint.h>
#include <stdbool.h>
#include <stdio.h>
#include "bsl.h"
#include "Crc.h"

//
// Target Application files
// Contain the sample applications sent to the target
//
//TODO: Update the file name's if they were changed
#if (defined(SINGLE_IMAGE))
//#include "TargetApps\App1_G2553_I2C_Single_Image.c"
#include "TargetApps\App_Yunding.c"
#elif (defined(DUAL_IMAGE))
#include "TargetApps\App1_G2553_I2C_Dual_Image.c"
#include "TargetApps\App2_G2553_I2C_Dual_Image.c"
#else
#error Define a valid target
#endif

/* Statics */
static bool bPerformBSL;
static uint16_t CRC_Val1;

/*
* TODO: Update these values to match the values defined
* in your target device's linker command file
*/
static uint16_t CRC_Addr = 0xE300;            // (_Appl_Checksum in target's linker command file)
static uint16_t App_StartAddress = 0xE304;    // (_Appl_Start_Memory in target's linker command file)
static uint16_t App_EndAddress = 0xF8FF;      // (_Appl_End in in target's linker command file)


/* Error Checking Macro */
#define CHECK_RESPONSE()  if(res != BSL_OK_RES)             \
{                                             \
    break;                                    \
}

static uint16_t Calc_App_CRC(uint16_t * Addr_array, uint16_t *Size_array, uint8_t ** DataPtr_Array, uint8_t num_arrays);
void I2C_SendBSLEntrance(void);

uint8_t res2=0;
int main(void)
{
    uint8_t res;
    uint8_t section;

    /* Stop the watchdog timer */
    WDTCTL = WDTPW | WDTHOLD;

    if ((CALBC1_8MHZ == 0xFF) || (CALDCO_8MHZ == 0xFF))
    {
        while (1) ;                     /* Trap if DCO constants are not available */
    }
    BCSCTL1 = CALBC1_8MHZ;              /* Set DCO to default of 8Mhz               */
    DCOCTL = CALDCO_8MHZ;

    /* Calculate CRC for both applications */
    CRC_Val1 = Calc_App_CRC((uint16_t *)&App1_Addr[0],
                             (uint16_t *)&App1_Size[0],
                             (uint8_t **)&App1_Ptr[0],
                             sizeof(App1_Addr)/ sizeof (App1_Addr[0]));
    /* Resetting our event flag */
    bPerformBSL = false;

    BSL_Init();

    /* Start P1.3 (S2 button) as interrupt with pull-up */
    P1OUT |= (BIT3);
    P1OUT &= ~BIT0;
    P1DIR |= BIT0|BIT1;
    P1REN |= BIT3;
    P1IES |= BIT3;
    P1IFG &= ~BIT3;
    P1IE |= BIT3;

  //  P1OUT = 0;
    //set P1.0 LED to show we are ready
    P1OUT |= BIT0;

    while (1)
    {
        P1IFG &= ~BIT3; // Clear button flag before going to sleep
        P1OUT |= (BIT0);
        __bis_SR_register(LPM0_bits + GIE);
        __delay_cycles(25);
        __disable_interrupt();

        while (bPerformBSL == true)
        {
//            I2C_SendBSLEntrance();

            /* Blinking LED to signal BSL start */
            P1OUT ^= (BIT0);
            __delay_cycles(2500000);
            P1OUT ^= (BIT0);
            __delay_cycles(2500000);
            P1OUT ^= (BIT0);
            __delay_cycles(2500000);
            P1OUT ^= (BIT0);

            I2C_SendBSLEntrance();
            __delay_cycles(2500000);

            if (!BSL_slavePresent())
                break;
            /* Sending the BSL Entry command to user app on target */
            BSL_sendSingleByte(VBOOT_ENTRY_CMD);
            __delay_cycles(500000);

            BSL_flush();

            res2=0;
            __delay_cycles(50);
            /* Sending the version command */
            res2 = BSL_sendCommand(BSL_VERSION_CMD);

            /* Making sure the version matches (only check upper nibble) */
            if ((res2&0xF0) == (VBOOT_VERSION&0xF0))
            {
                __delay_cycles(50);
                res=0;
                __delay_cycles(50);
                /* Erasing the user application */
                res = BSL_sendCommand(BSL_ERASE_APP_CMD);
                __delay_cycles(50);
                CHECK_RESPONSE();


                for (section = 0; section < (sizeof(App1_Addr)/ sizeof (App1_Addr[0])) ; section++)
                {
                    /* Sending the segments*/
                    res = BSL_programMemorySegment((uint32_t)App1_Addr[section],
                                                   App1_Ptr[section], (uint32_t)App1_Size[section]);
                    CHECK_RESPONSE();
                }
                /* Sending the CRC */
                res = BSL_programMemorySegment(CRC_Addr,
                        (uint8_t *) &CRC_Val1, 2);
                CHECK_RESPONSE();

                /* Jumping to user code */
                res = BSL_sendCommand(BSL_JMP_APP_CMD);
                // End of cycle completed OK
                P1OUT ^= (BIT0);
                __delay_cycles(2500000);
                P1OUT ^= (BIT0);
                __delay_cycles(2500000);
                P1OUT ^= (BIT0);
                __delay_cycles(2500000);
                P1OUT ^= (BIT0);
            }

            bPerformBSL = false;
       //     sentBSLFlipFlop = !sentBSLFlipFlop;
        }
    }
}

/* Sends single I2C Byte (start and stop included) */
void I2C_SendBSLEntrance(void)
{
    UCB0CTL1 |= UCTR + UCTXSTT;             // I2C TX, start condition
    UCB0TXBUF = 0x90;
    while (UCB0CTL1 & UCTXSTT);
    UCB0CTL1 |= UCTXSTP;                    // I2C stop condition after 1st TX

    __delay_cycles(2500000);
    UCB0CTL1 |= UCTR + UCTXSTT;             // I2C TX, start condition
    UCB0TXBUF = 0x90;
    while (UCB0CTL1 & UCTXSTT);

    while (!(IFG2 & UCB0TXIFG));
    UCB0TXBUF = 0x5A;
    while (!(IFG2 & UCB0TXIFG));
    UCB0CTL1 |= UCTXSTP;                    // I2C stop condition after 1st TX
    while (UCB0CTL1 & UCTXSTP);
}

/* Push button that will invoke BSL send */
#pragma vector=PORT1_VECTOR
__interrupt void Port_1(void)
{
    P1IFG &= ~BIT3;
    bPerformBSL = true;
    __bic_SR_register_on_exit(LPM0_bits);
}

/* Calculate the CRC of the application*/
static uint16_t Calc_App_CRC(uint16_t * Addr_array, uint16_t *Size_array, uint8_t ** DataPtr_Array, uint8_t num_arrays)
{
    uint16_t addr;
    uint8_t i;
    uint16_t CRC_result = 0xFFFF;

    // Calculate CRC for the whole Application address range
    for (addr=App_StartAddress; addr <= App_EndAddress; addr++)
    {
        for (i = 0; i < num_arrays; i ++)
        {
            // Check if address is defined by application image
            if ( (addr >= Addr_array[i]) &&
                 (addr < (Addr_array[i] + Size_array[i])) )
            {
                // If address is defined, add it to the CRC
                crc16AddByte(DataPtr_Array[i][addr-Addr_array[i]],&CRC_result);
                break;
            }
        }
        if (i==num_arrays)
        {
            // If not, simply add 0xFF
            crc16AddByte(0xFF,&CRC_result);
        }
        if(addr == 0xf9e2)
        {
            __no_operation();
        }
    }

    return CRC_result;
}
