#ifndef LEASC_SUPPORT_H__
#define LEASC_SUPPORT_H__

//*****************************************************************************
//  Texas Instruments, Inc
//  Dec 2020
//******************************************************************************

#include <stdint.h>

//******************************************************************************
//
//! \addtogroup leasc_support LEASC Support
//! @{
//
//******************************************************************************

//******************************************************************************
//
// If building with a C++ compiler, make all of the definitions in this header
// have a C binding.
//
//******************************************************************************
#ifdef __cplusplus
extern "C"
{
#endif

//******************************************************************************
//
//! \brief  Saturate to number range defined by a minimum and maximum.
//
//******************************************************************************
#define __saturate(a, min, max) (((a)>(max))?(max):(((a)<(min))?(min):(a)))

//******************************************************************************
//
//! \brief  Add two 16-bit integers with saturation.
//!
//! \param  a   16-bit integer input.
//! \param  b   16-bit integer input.
//! \return     Saturated 16-bit result.
//
//******************************************************************************
static inline int16_t __adds(int16_t a, int16_t b)
{
    int32_t res = (int32_t)a + (int32_t)b;
    return __saturate(res, INT16_MIN, INT16_MAX);
}

//******************************************************************************
//
//! \brief  Subtract two 16-bit integers with saturation.
//!
//! \param  a   16-bit integer input.
//! \param  b   16-bit integer input.
//! \return     Saturated 16-bit result.
//
//******************************************************************************
static inline int16_t __subs(int16_t a, int16_t b)
{
    int32_t res = (int32_t)a - (int32_t)b;
    return __saturate(res, INT16_MIN, INT16_MAX);
}

//******************************************************************************
//
//! \brief  Add two 32-bit integers with saturation.
//!
//! \param  a   32-bit integer input.
//! \param  b   32-bit integer input.
//! \return     Saturated 32-bit result.
//
//******************************************************************************
static inline int32_t __addl(int32_t a, int32_t b)
{
    int64_t res = (int64_t)a + (int64_t)b;
    return __saturate(res, INT32_MIN, INT32_MAX);
}

//******************************************************************************
//
//! \brief  Subtract two 32-bit integers with saturation.
//!
//! \param  a   32-bit integer input.
//! \param  b   32-bit integer input.
//! \return     Saturated 32-bit result.
//
//******************************************************************************
static inline int32_t __subl(int32_t a, int32_t b)
{
    int64_t res = (int64_t)a - (int64_t)b;
    return __saturate(res, INT32_MIN, INT32_MAX);
}

//******************************************************************************
//
//! \brief  Multiply two 16-bit fixed point inputs with saturation and 32-bit
//!         result.
//!
//! \param  a   16-bit input in fixed point format.
//! \param  b   16-bit input in fixed point format.
//! \return     Saturated 32-bit result.
//
//******************************************************************************
static inline int32_t __mpysf(int16_t a, int16_t b)
{
    if (a == INT16_MIN && b == INT16_MIN) {
        return INT32_MAX;
    }
    else {
        return (((int32_t)a * (int32_t)b) << 1);
    }
}

//******************************************************************************
//
//! \brief  Multiply two 32-bit fixed point inputs with saturation.
//!
//! LEA can only perform 16-bit multiplication with 32-bit results. Due to this
//! limitation 32-bit multiply is implemented using three 16-bit multiply
//! operations. The lower 32-bits of the result are not accumulated and the
//! fourth multiply operation of the lower 16-bits of each input is skipped. The
//! result of this optimized operation is accurate to 30.5 bits.
//!
//! \param  a   32-bit input in fixed point format.
//! \param  b   32-bit input in fixed point format.
//! \return     Saturated 32-bit result.
//
//******************************************************************************
static inline int32_t __mpylf(int32_t a, int32_t b)
{
    int32_t res = __mpysf(a >> 16, b >> 16);
    res += ((a & 0xffff) * (b >> 16)) >> 15;
    res += ((a >> 16) * (b & 0xffff)) >> 15;
    return res;
}

//******************************************************************************
//
//! \brief  Circular buffer increment of a pointer.
//!
//! \param  ptr     Pointer to increment
//! \param  incr    Increment size
//! \param  mask    32-bit circular buffer mask
//! \return         Pointer after incrementing and applying circular buffer
//!                 mask
//
//******************************************************************************
static inline int16_t *__circular_s(int16_t *ptr, int16_t incr, uint16_t mask)
{
    mask = ((mask+1)*sizeof(int32_t))-1;
    uintptr_t base = (intptr_t)ptr & ~mask;
    uintptr_t addr = (intptr_t)(ptr + incr);
    return (int16_t *)(base + (addr & mask));
}

//******************************************************************************
//
//! \brief  Circular buffer increment of a pointer.
//!
//! \param  ptr     Pointer to increment
//! \param  incr    Increment size
//! \param  mask    32-bit circular buffer mask
//! \return         Pointer after incrementing and applying circular buffer
//!                 mask
//
//******************************************************************************
static inline int32_t *__circular_l(int32_t *ptr, int16_t incr, uint16_t mask)
{
    mask = ((mask+1)*sizeof(int32_t))-1;
    uintptr_t base = (intptr_t)ptr & ~mask;
    uintptr_t addr = (intptr_t)(ptr + incr);
    return (int32_t *)(base + (addr & mask));
}

//******************************************************************************
//
// Mark the end of the C bindings section for C++ compilers.
//
//******************************************************************************
#ifdef __cplusplus
}
#endif

//******************************************************************************
//
// Close the Doxygen group.
//! @}
//
//******************************************************************************

#endif /* LEASC_SUPPORT_H__ */
