#include "msp430fr5994.h"

#include "lea_params.h"
#include "reference/reference_c.h"

#include <math.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>


// Convert MSP430 address to internal LEA address mapping.
#define LEA_CONVERT_ADDRESS(x)    ((uint16_t)(((uintptr_t)(x) >> 2) & 0xffff))

// Convert LEA address to MSP430 address mapping.
#define LEA_REVERT_ADDRESS(x)     ((uintptr_t)(((uint32_t)(x) << 2)))

// Test Size
#define VECTOR_SIZE       16

// Test Data
const int16_t initInputA[VECTOR_SIZE] = {
    1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16
};
const int16_t initInputB[VECTOR_SIZE] = {
    0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15
};

#pragma DATA_SECTION(inputA, ".leaRAM");
#pragma DATA_ALIGN(inputA, 4);
int16_t inputA[VECTOR_SIZE];

#pragma DATA_SECTION(inputB, ".leaRAM");
#pragma DATA_ALIGN(inputB, 4);
int16_t inputB[VECTOR_SIZE];

#pragma DATA_SECTION(output, ".leaRAM");
#pragma DATA_ALIGN(output, 4);
int16_t output[VECTOR_SIZE];

#pragma DATA_SECTION(params, ".leaRAM");
#pragma DATA_ALIGN(params, 4);
LEA_raddmsParams params;

int16_t outputGolden[VECTOR_SIZE];

volatile bool success = false;
volatile uint16_t benchmark;

void main(void)
{

    WDTCTL = WDTPW | WDTHOLD;       // Stop watchdog timer
    PM5CTL0 &= ~LOCKLPM5;           // Clear lock bit

    CSCTL0_H = CSKEY_H;
    CSCTL3 = 0;
    CSCTL0_H = 0;

    /* Initialize LEASC. */
    LEACNF0 = LEALPR | LEAILPM;
    LEASCCNF1 = 0;
    LEASCCNF2 = 0;
    LEASCCNF2 |= LEASCMT >> 2;
    LEASCPMS1 = 0;
    LEASCPMS0 = 0;
    LEASCPMDST = 0;
    LEASCPMCTL |= LEASCCMDEN;  // Enable LEA by setting command enable
    LEASCIFG = LEASCPMCMDIFG | LEASCSDIIFG | LEASCOORIFG | LEASCCOVLIFG; // Clear all interrupt flags
    LEASCIE |= LEASCPMCMDIE; // Enable the command complete interrupt

    /* Load input data */
    memcpy(inputA,initInputA,sizeof(initInputA));
    memcpy(inputB,initInputB,sizeof(initInputB));

    /* Initialize golden using reference C code */
    lea_raddms(inputA, inputB, outputGolden, VECTOR_SIZE, 1, 1, 1);

    /* Set parameters */
    params.vectorSize = VECTOR_SIZE;
    params.input2 = LEA_CONVERT_ADDRESS(inputB);
    params.output = LEA_CONVERT_ADDRESS(output);
    params.input1Offset = 1;
    params.input2Offset = 1;
    params.outputOffset = 1;

    /* Load source arguments to LEA. */
    LEASCPMS0 = LEA_CONVERT_ADDRESS(inputA);
    LEASCPMS1 = LEA_CONVERT_ADDRESS(&params);

    /* Benchmark start */
    TA0CTL = TACLR;
    TA0CTL = TASSEL_2 + MC_2;

    /* Invoke RAM command with interrupts. */
    LEAPMCB = LEACMD__ADDMATRIX | LEAITFLG1;

    /* Enable Interrupt & enter LPM0 mode. */
    __bis_SR_register(GIE + LPM0_bits);

    /* Benchmark stop */
    benchmark = TA0R;
    TA0CTL = MC_0;

    /* Compare the result calculated by reference c and LEA */
    success = (memcmp(&output, &outputGolden, sizeof(&outputGolden)) == 0);

    /* BREAK HERE, RAM version */
    __no_operation();

    while(1);
}

// LEA interrupt routine
#pragma vector=LEA_SC_VECTOR
__interrupt void LEA_ISR(void)
{
    uint16_t flags = LEAIFG;
    LEAIFG |= flags;
    __bic_SR_register_on_exit(LPM0_bits);
}
