/*
 * Copyright (c) 2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,

 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 *  ======== main_tirtos.c ========
 */
#include <stdint.h>

/* POSIX Header files */
#include <pthread.h>

/* RTOS header files */
#include <ti/sysbios/BIOS.h>
#include <ti/drivers/dpl/HwiP.h>
#include <ti/sysbios/knl/Clock.h>

/* Example/Board Header files */
#include "Board.h"
#include <ti/devices/msp432p4xx/driverlib/driverlib.h>

extern void *mainThread(void *arg0);

void RTC_C_Init(void);
void RTC_C_IRQHandler(uintptr_t arg);

/* Stack size in bytes */
#define THREADSTACKSIZE    1024

/*
 *  ======== main ========
 */
int main(void)
{
    pthread_t           thread;
    pthread_attr_t      pAttrs;
    struct sched_param  priParam;
    int                 retc;
    int                 detachState;

    /* Call driver init functions */
    Board_initGeneral();

    /* Set priority and stack size attributes */
    pthread_attr_init(&pAttrs);
    priParam.sched_priority = 1;

    detachState = PTHREAD_CREATE_DETACHED;
    retc = pthread_attr_setdetachstate(&pAttrs, detachState);
    if (retc != 0) {
        /* pthread_attr_setdetachstate() failed */
        while (1);
    }

    pthread_attr_setschedparam(&pAttrs, &priParam);

    retc |= pthread_attr_setstacksize(&pAttrs, THREADSTACKSIZE);
    if (retc != 0) {
        /* pthread_attr_setstacksize() failed */
        while (1);
    }

    retc = pthread_create(&thread, &pAttrs, mainThread, NULL);
    if (retc != 0) {
        /* pthread_create() failed */
        while (1);
    }

    /* Initialize and configure RTC. RTC will be used as Clock_tick,
       this also needs to be enabled in the cfg file */
    RTC_C_Init();

    BIOS_start();

    return (0);
}

void RTC_C_Init(void)
{
    RTC_C_Calendar calendarTime;
    HwiP_Params hwiParams;

    /* Enable BCLK and configure REFCLK to 32KHz */
    MAP_CS_setReferenceOscillatorFrequency(CS_REFO_128KHZ);
    MAP_CS_initClockSignal(CS_BCLK, CS_REFOCLK_SELECT, CS_CLOCK_DIVIDER_4);

    /* Configure RTC to 1ms interval timer*/
    MAP_RTC_C_initCalendar(&calendarTime, RTC_C_FORMAT_BCD);
//    MAP_RTC_C_definePrescaleEvent(RTC_C_PRESCALE_0, RTC_C_PSEVENTDIVIDER_32); //1ms
    MAP_RTC_C_definePrescaleEvent(RTC_C_PRESCALE_0, RTC_C_PSEVENTDIVIDER_64); //2ms
//    MAP_RTC_C_definePrescaleEvent(RTC_C_PRESCALE_0, RTC_C_PSEVENTDIVIDER_128); //4ms
//    MAP_RTC_C_definePrescaleEvent(RTC_C_PRESCALE_0, RTC_C_PSEVENTDIVIDER_256); //8ms

    /* Enable RTC interrupt */
    MAP_RTC_C_enableInterrupt(RTC_C_PRESCALE_TIMER0_INTERRUPT);
    MAP_Interrupt_enableInterrupt(INT_RTC_C);

    /* Create RTOS hwi */
    HwiP_Params_init(&hwiParams);
    hwiParams.arg = 5;
    hwiParams.priority = 5;
    HwiP_create(45, RTC_C_IRQHandler, &hwiParams);

    /* Start the RTC */
    MAP_RTC_C_startClock();
}

void RTC_C_IRQHandler(uintptr_t arg)
{
    uint32_t status;

    status = MAP_RTC_C_getEnabledInterruptStatus();
    MAP_RTC_C_clearInterruptFlag(status);

    /* Toggling the output on the LED */
    if(status & RTC_C_PRESCALE_TIMER0_INTERRUPT)
    {
//        P1OUT ^= BIT0;
        Clock_tick();
    }
}
