/*
 * Copyright (c) 2014, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *******************************************************************************
 *
 *                       MSP430 CODE EXAMPLE DISCLAIMER
 *
 * MSP430 code examples are self-contained low-level programs that typically
 * demonstrate a single peripheral function or device feature in a highly
 * concise manner. For this the code may rely on the device's power-on default
 * register values and settings such as the clock configuration and care must
 * be taken when combining code from several examples to avoid potential side
 * effects. Also see www.ti.com/grace for a GUI- and www.ti.com/msp430ware
 * for an API functional library-approach to peripheral configuration.
 * --/COPYRIGHT--*/


//                MSP430FR2311
//             -----------------
//         /|\|                 |
//          | |                 |
//          --|RST              |
//      /|\   |     P1.2/UCB0SDA|----
//       --o--|P1.3             |   |
//      \|/   |     P1.3/UCB0SCL|----
//            |                 |
//            |            P1.0 |--> LED
//            |            P2.0 |--> LED


//******************************************************************************
//#define DEBUG   //Define it if use debug mode
#include <msp430.h>
#include "bsl_app.h"


void Software_Trim();                       // Software Trim to get the best DCOFTRIM value
#define MCLK_FREQ_MHZ 1                     // MCLK = 1MHz
unsigned char rx_data;

void main(void)
{   unsigned short i,j;
    WDTCTL = WDTPW | WDTHOLD;             // Stop watchdog timer
    SYSCFG0 = FRWPPW;
    *(unsigned char *) _jump_key = 0xAA;                     // Write key value
    SYSCFG0 = FRWPPW | PFWP ;

    for(i=RAM_Interrupt_Start,j=FRAM_Interrupt_Start;i<RAM_Interrupt_End;i++,j++)
        *(unsigned char *) i =*(unsigned char *) j;// Write the vector

  SYSCTL = SYSRIVECT;
  P1OUT &= ~BIT0;                         // Clear P1.0 output latch for a defined power-on state
  P1DIR |= BIT0;                          // Set P1.0 to output direction
  // Configure UART pins
  P1SEL0 |= BIT2 | BIT3;                    // I2C pins
  PM5CTL0 &= ~LOCKLPM5;                     // Disable the GPIO power-on default high-impedance mode

  __bis_SR_register(SCG0);                // Disable FLL
//  CSCTL3 = SELREF__XT1CLK;                // Set XT1 as FLL reference source
  CSCTL1 = DCOFTRIMEN_1 | DCOFTRIM0 | DCOFTRIM1 | DCORSEL_0;// DCOFTRIM=3, DCO Range = 1MHz
  CSCTL2 = FLLD_0 + 30;                   // DCODIV = 1MHz
  __delay_cycles(3);
  __bic_SR_register(SCG0);                // Enable FLL
  Software_Trim();                        // Software Trim to get the best DCOFTRIM value

  // Configure USCI_B0 for I2C mode
  UCB0CTLW0 = UCSWRST;                      // Software reset enabled
  UCB0CTLW0 |= UCMODE_3 | UCSYNC;           // I2C mode, sync mode
  UCB0I2COA0 = 0x48 | UCOAEN;               // own address is 0x48 + enable
  UCB0CTLW0 &= ~UCSWRST;                    // clear reset register
  UCB0IE |= UCRXIE0 | UCTXIE0 | UCSTPIE;              // transmit,stop interrupt enable


  __bis_SR_register(GIE);

  while(1)
  {
      P1OUT ^= BIT0;                      // Toggle P1.0 using exclusive-OR
      __delay_cycles(300000);             // Delay for 100000*(1/MCLK)=0.1s

  }

}

#pragma vector = USCI_B0_VECTOR
__interrupt void USCIB0_ISR(void)
{

  switch(__even_in_range(UCB0IV, USCI_I2C_UCBIT9IFG))
  {
    case USCI_NONE: break;                  // Vector 0: No interrupts
    case USCI_I2C_UCALIFG: break;           // Vector 2: ALIFG
    case USCI_I2C_UCNACKIFG: break;         // Vector 4: NACKIFG
    case USCI_I2C_UCSTTIFG: break;          // Vector 6: STTIFG
    case USCI_I2C_UCSTPIFG:                 // Vector 8: STPIFG
      UCB0IFG &= ~UCSTPIFG;                 // Clear stop condition int flag
      break;
    case USCI_I2C_UCRXIFG3: break;          // Vector 10: RXIFG3
    case USCI_I2C_UCTXIFG3: break;          // Vector 14: TXIFG3
    case USCI_I2C_UCRXIFG2: break;          // Vector 16: RXIFG2
    case USCI_I2C_UCTXIFG2: break;          // Vector 18: TXIFG2
    case USCI_I2C_UCRXIFG1: break;          // Vector 20: RXIFG1
    case USCI_I2C_UCTXIFG1: break;          // Vector 22: TXIFG1
    case USCI_I2C_UCRXIFG0:
        rx_data = UCB0RXBUF;
        #ifndef DEBUG
        if(rx_data==0xAA)                    //Jump to BSL mode
        {
            __disable_interrupt();              // disable interrupts
            *(unsigned char *)_jump_key = 0xFF;// Write the jump key
            UCB0IE=0;
            ((void ( * )())0x1000)();    // jump to BSL
        }
        #endif
        break;         // Vector 22: RXIFG0
        case USCI_I2C_UCTXIFG0:                 // Vector 24: TXIFG0
        if(rx_data==0xAA)                    //Jump to BSL mode
        {
        #ifdef DEBUG
            UCB0TXBUF =  0x11;           //Response to host target in application mode
        #endif
        }else
        {
            if(rx_data==0xBB)            //Host check target status
                UCB0TXBUF =  0x22;       //Response to host target in application mode
        }
       break;                               // Vector 26: TXIFG0
    case USCI_I2C_UCBCNTIFG: break;         // Vector 28: BCNTIFG
    case USCI_I2C_UCCLTOIFG: break;         // Vector 30: clock low timeout
    case USCI_I2C_UCBIT9IFG: break;         // Vector 32: 9th bit
    default: break;
  }
}





void Software_Trim()
{
    unsigned int oldDcoTap = 0xffff;
    unsigned int newDcoTap = 0xffff;
    unsigned int newDcoDelta = 0xffff;
    unsigned int bestDcoDelta = 0xffff;
    unsigned int csCtl0Copy = 0;
    unsigned int csCtl1Copy = 0;
    unsigned int csCtl0Read = 0;
    unsigned int csCtl1Read = 0;
    unsigned int dcoFreqTrim = 3;
    unsigned char endLoop = 0;

    do
    {
        CSCTL0 = 0x100;                         // DCO Tap = 256
        do
        {
            CSCTL7 &= ~DCOFFG;                  // Clear DCO fault flag
        }while (CSCTL7 & DCOFFG);               // Test DCO fault flag

        __delay_cycles((unsigned int)3000 * MCLK_FREQ_MHZ);// Wait FLL lock status (FLLUNLOCK) to be stable
                                                           // Suggest to wait 24 cycles of divided FLL reference clock
        while((CSCTL7 & (FLLUNLOCK0 | FLLUNLOCK1)) && ((CSCTL7 & DCOFFG) == 0));

        csCtl0Read = CSCTL0;                   // Read CSCTL0
        csCtl1Read = CSCTL1;                   // Read CSCTL1

        oldDcoTap = newDcoTap;                 // Record DCOTAP value of last time
        newDcoTap = csCtl0Read & 0x01ff;       // Get DCOTAP value of this time
        dcoFreqTrim = (csCtl1Read & 0x0070)>>4;// Get DCOFTRIM value

        if(newDcoTap < 256)                    // DCOTAP < 256
        {
            newDcoDelta = 256 - newDcoTap;     // Delta value between DCPTAP and 256
            if((oldDcoTap != 0xffff) && (oldDcoTap >= 256)) // DCOTAP cross 256
                endLoop = 1;                   // Stop while loop
            else
            {
                dcoFreqTrim--;
                CSCTL1 = (csCtl1Read & (~DCOFTRIM)) | (dcoFreqTrim<<4);
            }
        }
        else                                   // DCOTAP >= 256
        {
            newDcoDelta = newDcoTap - 256;     // Delta value between DCPTAP and 256
            if(oldDcoTap < 256)                // DCOTAP cross 256
                endLoop = 1;                   // Stop while loop
            else
            {
                dcoFreqTrim++;
                CSCTL1 = (csCtl1Read & (~DCOFTRIM)) | (dcoFreqTrim<<4);
            }
        }

        if(newDcoDelta < bestDcoDelta)         // Record DCOTAP closest to 256
        {
            csCtl0Copy = csCtl0Read;
            csCtl1Copy = csCtl1Read;
            bestDcoDelta = newDcoDelta;
        }

    }while(endLoop == 0);                      // Poll until endLoop == 1

    CSCTL0 = csCtl0Copy;                       // Reload locked DCOTAP
    CSCTL1 = csCtl1Copy;                       // Reload locked DCOFTRIM
    while(CSCTL7 & (FLLUNLOCK0 | FLLUNLOCK1)); // Poll until FLL is locked
}



