/*
 * Copyright (c) 2014, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *******************************************************************************
 *
 *                       MSP430 CODE EXAMPLE DISCLAIMER
 *
 * MSP430 code examples are self-contained low-level programs that typically
 * demonstrate a single peripheral function or device feature in a highly
 * concise manner. For this the code may rely on the device's power-on default
 * register values and settings such as the clock configuration and care must
 * be taken when combining code from several examples to avoid potential side
 * effects. Also see www.ti.com/grace for a GUI- and www.ti.com/msp430ware
 * for an API functional library-approach to peripheral configuration.
 *
 * --/COPYRIGHT--*/


//                MSP430FR2311
//             -----------------
//         /|\|                 |
//          | |                 |
//          --|RST              |
//      /|\   |     P1.7/UCA0TXD|----
//       --o--|P1.3             |   |
//      \|/   |     P1.6/UCA0RXD|----
//            |                 |
//            |            P1.0 |--> LED
//            |            P2.0 |--> LED


//******************************************************************************
//#define DEBUG   //Define it if use debug mode
#include <msp430.h>
#include "bsl_app.h"


void Software_Trim();                       // Software Trim to get the best DCOFTRIM value
#define MCLK_FREQ_MHZ 1                     // MCLK = 1MHz

void main(void)
{   unsigned short i,j;
    WDTCTL = WDTPW | WDTHOLD;             // Stop watchdog timer
    SYSCFG0 = FRWPPW;
    *(unsigned char *) _jump_key = 0xAA;                     // Write key value
    SYSCFG0 = FRWPPW | PFWP ;

    for(i=RAM_Interrupt_Start,j=FRAM_Interrupt_Start;i<RAM_Interrupt_End;i++,j++)
        *(unsigned char *) i =*(unsigned char *) j;// Write the vector

  SYSCTL = SYSRIVECT;
  P1OUT &= ~BIT0;                         // Clear P1.0 output latch for a defined power-on state
  P1DIR |= BIT0;                          // Set P1.0 to output direction
  P2OUT &= ~BIT0;                         // Clear P2.0 output latch for a defined power-on state
  P2DIR |= BIT0;                          // Set P2.0 to output direction
  // Configure UART pins
  P1SEL0 |= BIT6 | BIT7;                    // set 2-UART pin as second function
  PM5CTL0 &= ~LOCKLPM5;                     // Disable the GPIO power-on default high-impedance mode

  __bis_SR_register(SCG0);                // Disable FLL
//  CSCTL3 = SELREF__XT1CLK;                // Set XT1 as FLL reference source
  CSCTL1 = DCOFTRIMEN_1 | DCOFTRIM0 | DCOFTRIM1 | DCORSEL_0;// DCOFTRIM=3, DCO Range = 1MHz
  CSCTL2 = FLLD_0 + 30;                   // DCODIV = 1MHz
  __delay_cycles(3);
  __bic_SR_register(SCG0);                // Enable FLL
  Software_Trim();                        // Software Trim to get the best DCOFTRIM value

  // to activate 1previously configured port settings
  // Configure UART
  UCA0CTLW0 = UCSWRST;
  UCA0CTLW0 |= UCSSEL__SMCLK| UCPEN | UCPAR;                    // set ACLK as BRCLK
  UCA0ABCTL=0;
  UCA0STATW=0;
  // Baud Rate calculation. Referred to UG 17.3.10
  // (1) N=1000000/9600/16=6.827
  // (2) OS16=1, UCBRx=INT(N)=8
  // (4) Fractional portion = 0.827. Refered to UG Table 17-4, UCBRSx=0x20.
  UCA0BR0 = 0x6;                              // INT(1000000/9600/16=6)
  UCA0BR1=0;
  UCA0MCTLW = UCOS16  | (0x20 << 1) |  UCBRF_8;
  UCA0IFG=0;
  UCA0IE=0;
  UCA0CTLW0 &= ~UCSWRST;                    // Initialize eUSCI
  UCA0IE |= UCRXIE;                         // Enable USCI_A0 RX interrupt
  __bis_SR_register(GIE);

  while(1)
  {
      P1OUT ^= BIT0;                      // Toggle P1.0 using exclusive-OR
      __delay_cycles(300000);             // Delay for 100000*(1/MCLK)=0.1s

  }

}

#pragma vector=USCI_A0_VECTOR
__interrupt void USCI_A0_ISR(void)

{

    unsigned char rx_data;
  switch(__even_in_range(UCA0IV,USCI_UART_UCTXCPTIFG))
  {

    case USCI_NONE: break;
    case USCI_UART_UCRXIFG:
      P2OUT ^= BIT0;
      while(!(UCA0IFG&UCTXIFG));
      rx_data=UCA0RXBUF;
      if(rx_data==0xAA)                    //Jump to BSL mode
      {
    #ifndef DEBUG
          __disable_interrupt();              // disable interrupts
          SYSCFG0 = FRWPPW;
          *(unsigned char *)_jump_key = 0xFF;// Write the jump key
          SYSCFG0 = FRWPPW | PFWP ;
          UCA0CTLW0 = UCSWRST;
          UCA0MCTLW = 0;
          UCA0IE=0;
          ((void ( * )())0x1000)();    // jump to BSL
    #else
          UCA0TXBUF =  0x11;           //Response to host target in application mode
    #endif
      }else
      {
          if(rx_data==0xBB)            //Host check target status
              UCA0TXBUF =  0x22;       //Response to host target in application mode
      }
      __no_operation();
      break;
    case USCI_UART_UCTXIFG: break;
    case USCI_UART_UCSTTIFG: break;
    case USCI_UART_UCTXCPTIFG: break;
    default: break;
  }
}



void Software_Trim()
{
    unsigned int oldDcoTap = 0xffff;
    unsigned int newDcoTap = 0xffff;
    unsigned int newDcoDelta = 0xffff;
    unsigned int bestDcoDelta = 0xffff;
    unsigned int csCtl0Copy = 0;
    unsigned int csCtl1Copy = 0;
    unsigned int csCtl0Read = 0;
    unsigned int csCtl1Read = 0;
    unsigned int dcoFreqTrim = 3;
    unsigned char endLoop = 0;

    do
    {
        CSCTL0 = 0x100;                         // DCO Tap = 256
        do
        {
            CSCTL7 &= ~DCOFFG;                  // Clear DCO fault flag
        }while (CSCTL7 & DCOFFG);               // Test DCO fault flag

        __delay_cycles((unsigned int)3000 * MCLK_FREQ_MHZ);// Wait FLL lock status (FLLUNLOCK) to be stable
                                                           // Suggest to wait 24 cycles of divided FLL reference clock
        while((CSCTL7 & (FLLUNLOCK0 | FLLUNLOCK1)) && ((CSCTL7 & DCOFFG) == 0));

        csCtl0Read = CSCTL0;                   // Read CSCTL0
        csCtl1Read = CSCTL1;                   // Read CSCTL1

        oldDcoTap = newDcoTap;                 // Record DCOTAP value of last time
        newDcoTap = csCtl0Read & 0x01ff;       // Get DCOTAP value of this time
        dcoFreqTrim = (csCtl1Read & 0x0070)>>4;// Get DCOFTRIM value

        if(newDcoTap < 256)                    // DCOTAP < 256
        {
            newDcoDelta = 256 - newDcoTap;     // Delta value between DCPTAP and 256
            if((oldDcoTap != 0xffff) && (oldDcoTap >= 256)) // DCOTAP cross 256
                endLoop = 1;                   // Stop while loop
            else
            {
                dcoFreqTrim--;
                CSCTL1 = (csCtl1Read & (~DCOFTRIM)) | (dcoFreqTrim<<4);
            }
        }
        else                                   // DCOTAP >= 256
        {
            newDcoDelta = newDcoTap - 256;     // Delta value between DCPTAP and 256
            if(oldDcoTap < 256)                // DCOTAP cross 256
                endLoop = 1;                   // Stop while loop
            else
            {
                dcoFreqTrim++;
                CSCTL1 = (csCtl1Read & (~DCOFTRIM)) | (dcoFreqTrim<<4);
            }
        }

        if(newDcoDelta < bestDcoDelta)         // Record DCOTAP closest to 256
        {
            csCtl0Copy = csCtl0Read;
            csCtl1Copy = csCtl1Read;
            bestDcoDelta = newDcoDelta;
        }

    }while(endLoop == 0);                      // Poll until endLoop == 1

    CSCTL0 = csCtl0Copy;                       // Reload locked DCOTAP
    CSCTL1 = csCtl1Copy;                       // Reload locked DCOFTRIM
    while(CSCTL7 & (FLLUNLOCK0 | FLLUNLOCK1)); // Poll until FLL is locked
}



