/* --COPYRIGHT--,BSD_EX
 * Copyright (c) 2013, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *******************************************************************************
 * 
 *                       MSP430 CODE EXAMPLE DISCLAIMER
 *
 * MSP430 code examples are self-contained low-level programs that typically
 * demonstrate a single peripheral function or device feature in a highly
 * concise manner. For this the code may rely on the device's power-on default
 * register values and settings such as the clock configuration and care must
 * be taken when combining code from several examples to avoid potential side
 * effects. Also see www.ti.com/grace for a GUI- and www.ti.com/msp430ware
 * for an API functional library-approach to peripheral configuration.
 *
 * --/COPYRIGHT--*/
//******************************************************************************
//  MSP430i20xx test demo, loading calibration data to INFO memory and preserving TLV data.
//
//
//
//  ACLK = 32kHz, MCLK = SMCLK = Calibrated DCO = 16.384MHz
//  * Ensure low_level_init.c is included when building/running this example *
//
//               MSP430i20xx
//             -----------------
//         /|\|                |
//          | |                |
//          --|RST             |
//            |                |
//            |            P1.4|-->LED
//            |                |
//
//  JD Crutchfield
//  Texas Instruments, Inc
//  July 2020
//  Built with Code Composer Studio v9.3
//******************************************************************************
#include "msp430.h"




/*
 * ************** Calibration Data **********************
 */
//Board 4 Data
const long CalibrationThermopileOffset = -99297;
const long CalibrationThermopileScalingFactor = 141;
const long CalibrationThermistorOffset = 1081231;

#define LUT_LENGTH      51
const long ThermopileLUT[LUT_LENGTH] = // Gain 200
{   -1955893 ,
    -1864379 ,
    -1776400 ,
    -1687297 ,
    -1597071 ,
    -1505723 ,
    -1413251 ,
    -1319656 ,
    -1224938 ,
    -1129097 ,
    -1028655 ,
     -934046 ,
     -834836 ,
     -734502 ,
     -633046 ,
     -530467 ,
     -426764 ,
     -321939 ,
     -215991 ,
     -108919 ,
           0 ,
      108593 ,
      219034 ,
      330598 ,
      443284 ,
      557094 ,
      672027 ,
      788083 ,
      905262 ,
     1023564 ,
     1130071 ,
     1263537 ,
     1385209 ,
     1508003 ,
     1631920 ,
     1756961 ,
     1883124 ,
     2010410 ,
     2138820 ,
     2268352 ,
     2390536 ,
     2530787 ,
     2663689 ,
     2797713 ,
     2932861 ,
     3069132 ,
     3206526 ,
     3345043 ,
     3484683 ,
     3625446 ,
     3766905
};

static const long ThermistorLUT[LUT_LENGTH] =
{     7594308   ,
      7481260 ,
      7367079 ,
      7251875 ,
      7135765 ,
      7018867 ,
      6901305 ,
      6783205 ,
      6664693 ,
      6545899 ,
      6426953 ,
      6307983 ,
      6189119 ,
      6070485 ,
      5952208 ,
      5834407 ,
      5717198 ,
      5600695 ,
      5485004 ,
      5370226 ,
      5256744 ,
      5143799 ,
      5031926 ,
      4921691 ,
      4813153 ,
      4705779 ,
      4599506 ,
      4494846 ,
      4391825 ,
      4290482 ,
      4190839 ,
      4092097 ,
      3995112 ,
      3899934 ,
      3806491 ,
      3714912 ,
      3624605 ,
      3536185 ,
      3449521 ,
      3364702 ,
      3281729 ,
      3199885 ,
      3119873 ,
      3041665 ,
      2965220 ,
      2890576 ,
      2817674 ,
      2746988 ,
      2678083 ,
      2611197 ,
      2546410

};

/****************************************/

void SaveMSP430TLVdata(void);
void EraseInfoMemory(void);
void WriteCalibrationDataToInfoMemory(void);

#define TLV_ADDRESS     0x13C0
unsigned int TLV_Data[(TLV_END - TLV_START)/2 + 1] = {0};
unsigned int TLVlength = (TLV_END - TLV_START)/2 + 1;



void main(void) {
    WDTCTL = WDTPW | WDTHOLD;     // Stop Watchdog Timer

    __delay_cycles(32000000);

    SaveMSP430TLVdata();

    EraseInfoMemory();

    WriteCalibrationDataToInfoMemory();

    while(1) {
        __no_operation();
        LPM0;
    }
}




void SaveMSP430TLVdata(void) {  //Read from Flash
    int *ptrRead;
    int x;

    ptrRead = (int *)TLV_ADDRESS; //Point to beginning of information memory

    for(x=0; x<TLVlength; x++){
        TLV_Data[x] = *ptrRead++;
    }
}


void EraseInfoMemory(void) { //Erase Information Memory Segment

    int *ptrErase;

    ptrErase =(int *) 0x1000; //pointer to beginning of InfoFlash Segment

    FCTL2 = FWKEY + FSSEL1 + 37;//Divide SMCLK(16.3MHz) by 38
    FCTL1 = FWKEY + ERASE; //Put into erase segment mode
    if(FCTL3 & LOCKSEG){  //Info mem is locked, need to unlock
        FCTL3 = FWKEY + LOCKSEG;  // Unlock info mem
    }
    else{
        FCTL3 = FWKEY; //unlock flash
    }

    *ptrErase = 0; //dummy value to erase Flash

    FCTL3 = FWKEY + LOCKSEG + LOCK; //lock flash
}



void WriteCalibrationDataToInfoMemory(void) {
    int x;
    int *ptrTLVFlash;
    long *calibrationFlash;

    ptrTLVFlash =(int *) TLV_ADDRESS;

    FCTL2 = FWKEY + FSSEL1 + 25; //Divide SMCLK
    if(FCTL3 & LOCKSEG){  //Info mem is locked, need to unlock
        FCTL3 = FWKEY + LOCKSEG;  // Unlock info mem
    }
    else{
        FCTL3 = FWKEY; //unlock flash
    }

    FCTL1 = FWKEY + WRT; //Set Write

    //Restore TLV after erase
    for(x=0; x<TLVlength; x++){  *ptrTLVFlash++ = TLV_Data[x]; }

    //Write Calibration data
    calibrationFlash =(long *) 0x1000;

    *calibrationFlash++ = CalibrationThermopileOffset;
    *calibrationFlash++ = CalibrationThermopileScalingFactor;
    *calibrationFlash++ = CalibrationThermistorOffset;
    for(x=0; x<LUT_LENGTH; x++){ *calibrationFlash++ = ThermopileLUT[x]; }
    for(x=0; x<LUT_LENGTH; x++){ *calibrationFlash++ = ThermistorLUT[x]; }



    FCTL1 = FWKEY; //Clear Write
    FCTL3 = FWKEY + LOCKSEG + LOCK; //Lock Flash
}







