/* --COPYRIGHT--,BSD
 * Copyright (c) 2012, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/

//
// Include files
//
#include "deviceSpecifics.h"

#include "i2cBslHost.h"
#include "i2cMaster.h"
#include "MSP-EXP430F5438 HAL\hal_MSP-EXP430F5438.h"
#include "i2cbslUserInterface.h"

//
// Project Definitions
//


//
// Target Application files
// Contain the sample applications sent to the target
//
#ifdef TARGET_G2553
#include "TargetApps\App1_G2553_I2C_Simple.c"
#include "TargetApps\App1_G2553_I2C_BSLBased.c"
#include "TargetApps\App1_G2553_I2C_SMBus.c"
#include "TargetApps\App1_G2553_I2C_BSLBased_DualImg.c"
#include "TargetApps\App2_G2553_I2C_Simple.c"
#include "TargetApps\App2_G2553_I2C_BSLBased.c"
#include "TargetApps\App2_G2553_I2C_SMBus.c"
#include "TargetApps\App2_G2553_I2C_BSLBased_DualImg.c"
#elif (defined(TARGET_G2452))
#include "TargetApps\App1_G2452_I2C_Simple.c"
#include "TargetApps\App1_G2452_I2C_BSLBased.c"
#include "TargetApps\App1_G2452_I2C_SMBus.c"
#include "TargetApps\App1_G2452_I2C_BSLBased_DualImg.c"
#include "TargetApps\App2_G2452_I2C_Simple.c"
#include "TargetApps\App2_G2452_I2C_BSLBased.c"
#include "TargetApps\App2_G2452_I2C_SMBus.c"
#include "TargetApps\App2_G2452_I2C_BSLBased_DualImg.c"
#elif (defined(TARGET_FR5739))
#include "TargetApps\App1_FR5739_I2C_Simple.c"
#include "TargetApps\App1_FR5739_I2C_BSLBased.c"
#include "TargetApps\App1_FR5739_I2C_SMBus.c"
#include "TargetApps\App1_FR5739_I2C_BSLBased_DualImg.c"
#include "TargetApps\App2_FR5739_I2C_Simple.c"
#include "TargetApps\App2_FR5739_I2C_BSLBased.c"
#include "TargetApps\App2_FR5739_I2C_SMBus.c"
#include "TargetApps\App2_FR5739_I2C_BSLBased_DualImg.c"
#else
#error Define a valid target processor 
/*! Define the Target processor: TARGET_G2553 or TARGET_G2452 */
// IAR/CCS projects have Target configuration in project settings depending on 
// target configuration
#endif

/*! Assigns a number to the corresponding menu item */
#define SUBMENU_ITEM(x,y)               (x*7+y)

//
//  MAIN MENU
/*! Main Menu Options */
static char mainmenuText[]={
#ifdef TARGET_G2553
  "BSL host G2x3\0"
#elif (defined(TARGET_G2452))
  "BSL host G2x2\0"
#elif (defined(TARGET_FR5739))
  "BSL host FR5739\0"
#endif
  " 1. Simple   \0"
  " 2. BSL-based\0"
  " 3. SMBus    \0"
  " 4. DualImg  \0"
  " 5. App Cmds \0"
  " 6.          \0"
  " 7.          \0"
  " 8.          \0"
};

//
// SIMPLE MENU
/*! Downloads App1 using Simple mode */
#define BOOTPROCEDURE_DOWNLOAD_APP1_SIMPLE  SUBMENU_ITEM(0,0)
/*! Downloads App2 using Simple mode */
#define BOOTPROCEDURE_DOWNLOAD_APP2_SIMPLE  SUBMENU_ITEM(0,1)
/*! Simple menu Options */
static char SimpleMenuText[]={
  " Simple Cmds \0"
  " 1. APP1     \0"\
  " 2. APP2     \0"\
  " 3.          \0"
  " 4.          \0"\
  " 5.          \0"
  " 6.          \0"
  " 7.          \0"
  " 8.          \0"
};

//
// BSL-BASED MENU
/*! Requests version  */
#define BOOTPROCEDURE_GET_VERSION       SUBMENU_ITEM(1,0)
/*! Downloads App1 using BSL-based mode */
#define BOOTPROCEDURE_DOWNLOAD_APP1_BSL SUBMENU_ITEM(1,1)
/*! Downloads App2 using BSL-based mode */
#define BOOTPROCEDURE_DOWNLOAD_APP2_BSL SUBMENU_ITEM(1,2)
/*! Erases the Application Area */
#define BOOTPROCEDURE_APP_ERASE         SUBMENU_ITEM(1,3)
/*! Test command with incorrect CRC */
#define BOOTPROCEDURE_INCORRECTCHKSUM   SUBMENU_ITEM(1,4)

/*! BSL-based menu Options */
static char BSLMenuText[]={
  " BSL Cmds    \0"
  " 1. Version  \0"\
  " 2. APP1     \0"\
  " 3. APP2     \0"\
  " 4. App Erase\0"\
  " 5. BadChks  \0"
  " 6.          \0"
  " 7.          \0"
  " 8.          \0"
};

//
// SMBus-BASED MENU
/*! Requests version  */
#define BOOTPROCEDURE_SMBUS_VERSION     SUBMENU_ITEM(2,0)
/*! Downloads App1 using SMBus-based mode */
#define BOOTPROCEDURE_SMBUS_APP1        SUBMENU_ITEM(2,1)
/*! Downloads App2 using SMBus-based mode */
#define BOOTPROCEDURE_SMBUS_APP2        SUBMENU_ITEM(2,2)
/*! Erases the Application Area */
#define BOOTPROCEDURE_SMBUS_APPERASE    SUBMENU_ITEM(2,3)
/*! Test command with incorrect PEC */
#define BOOTPROCEDURE_SMBUS_BADAPPERASE SUBMENU_ITEM(2,4)
/*! Test command with more data than expected */
#define BOOTPROCEDURE_SMBUS_OVERFLOW    SUBMENU_ITEM(2,5)
/*! Test Timeout detection */
#define BOOTPROCEDURE_SMBUS_TIMEOUT     SUBMENU_ITEM(2,6)
/*! SMBus-based menu Options */
static char SMBusMenuText[]={
  " SMBus Cmds  \0"
  " 1. Version  \0"\
  " 2. APP1     \0"\
  " 3. APP2     \0"
  " 4. App Erase\0"\
  " 5. Bad Erase\0"
  " 6. Overflow \0"
  " 7. Timeout  \0"
  " 8.          \0"
};

//
// DUAL Image BSL-based MENU
/*! Requests version  */
#define BOOTPROCEDURE_GET_VERSION_DUALIMG       SUBMENU_ITEM(3,0)
/*! Downloads App1 using Dual Image mode */
#define BOOTPROCEDURE_DOWNLOAD_APP1_DUALIMG     SUBMENU_ITEM(3,1)
/*! Downloads App2 using Dual Image mode */
#define BOOTPROCEDURE_DOWNLOAD_APP2_DUALIMG     SUBMENU_ITEM(3,2)
/*! Erases the Virtual Application area (download area) */
#define BOOTPROCEDURE_APP_ERASE_DUALIMG         SUBMENU_ITEM(3,3)
/*! Sends a command with incorrect CRC */
#define BOOTPROCEDURE_INCORRECTCHKSUM_DUALIMG   SUBMENU_ITEM(3,4)
/*! Erases a segment of the virtual application area (download area) */
#define BOOTPROCEDURE_ERASESEG1_DUALIMG         SUBMENU_ITEM(3,5)
/*! Erases a segment of the actual download area (protected) */
#define BOOTPROCEDURE_ERASESEG2_DUALIMG         SUBMENU_ITEM(3,6)
/*! DUAL Image BSL-based menu Options */
static char DIMenuText[]={
  " DualImg Cmd \0"
  " 1. Version  \0"\
  " 2. APP1     \0"\
  " 3. APP2     \0"\
  " 4. App Erase\0"\
  " 5. BadChks  \0"
#ifdef TARGET_G2553
  " 6. EraseE000\0"
  " 7. EraseC000\0"
#elif (defined(TARGET_G2452))
  " 6. EraseEE00\0"
  " 7. EraseE000\0"
#elif (defined(TARGET_FR5739))
  " 6. EraseDE00\0"
  " 7. EraseC200\0"
#endif      
  " 8.          \0"
};

//
// Application requests (only in APP2 when CI PHY-DL is shared)
/*! Toggles LED1 */
#define BOOTPROCEDURE_APP_LED1          SUBMENU_ITEM(4,0)
/*! Toggles LED2 */
#define BOOTPROCEDURE_APP_LED2          SUBMENU_ITEM(4,1)
/*! Toggles Both LEDS */
#define BOOTPROCEDURE_APP_LEDBOTH       SUBMENU_ITEM(4,2)
/*! Force BSL mode */
#define BOOTPROCEDURE_APP_FORCEBSL      SUBMENU_ITEM(4,3)
/*! Application mode menu Options */
static char AppMenuText[]={
  " App Cmds    \0"
  " 1. LED1     \0"
  " 2. LED2     \0"
  " 3. LED BOTH \0"
  " 4. Force BSL\0"
  " 5.          \0"
  " 6.          \0"
  " 7.          \0"
  " 8.          \0"
};


/*! List of submenus */
submenu_list_t submenu_list[] =
{
      {&SimpleMenuText[0], 2},
      {&BSLMenuText[0], 5},
      {&SMBusMenuText[0], 7},
      {&DIMenuText[0], 7},
      {&AppMenuText[0], 4},
};

//
// Local function Prototypes
//
static void PrintResponse (I2cBslResponse * Response);
static void PrintSMBResponse (I2cSMBCmd * Response);

//
// Local variables
//
static char outputText[14]="          \0\0\0\0";

/******************************************************************************
 * @brief   Prints response from BSL command to LCD
 *
 * @param Response Pointer to structure with BSL response
 *
 * @return none
 *****************************************************************************/
static void PrintResponse (I2cBslResponse * Response){
    uint8_t i;

    halLcdPrintLine("  BSL Response:", 3, OVERWRITE_TEXT);
    for (i=0; i < 13; i++)
        outputText[i] = ' ';
    hex2text(Response->i2cBslPiMessage,outputText);
    halLcdPrintLine(outputText,4, OVERWRITE_TEXT);
}

/******************************************************************************
 * @brief   Prints response from SMBus command to LCD
 *
 * @param Response Pointer to structure with BSL response
 *
 * @return none
 *****************************************************************************/
static void PrintSMBResponse (I2cSMBCmd * Response){
    uint8_t  i;

    halLcdPrintLine("  BSL Response:", 3, OVERWRITE_TEXT);
    for (i=0; i < 13; i++)
        outputText[i] = ' ';

    hex2text(Response->buf[0],outputText);
    halLcdPrintLine(outputText,4, OVERWRITE_TEXT);
}

/******************************************************************************
 * @brief   Main function
 *  Initializes MCU, user interface, waits for a menu entry and executes
 *   the appropiate action
 *
 * @return none
 *****************************************************************************/
void main(void)
{
  uint8_t procedureSelected = 0;
  I2cBslCmd mainI2cBslCmd;
  I2cBslResponse mainI2cBslResponse;
  I2cSMBCmd SMB_Cmd, SMB_Resp;
  int ret;

  WDTCTL = WDTPW+WDTHOLD;   // Stop watchdog timer
  // LCD contrast and backlight can be adjusted in initUserInterface
  initUserInterface();      // Initialize LCD, buttons

  i2cInit();                // Initialize I2C as Bus Master

  while (1)
  {
    // Show the Main menu and wait for user selection  
    procedureSelected = menuSelection(mainmenuText, &submenu_list[0]);
    mainI2cBslResponse.i2cBslPiMessage = 0;
    switch (procedureSelected)
    {
        /****** PROCEDURES for SIMPLE Protocol *************/
        case BOOTPROCEDURE_DOWNLOAD_APP1_SIMPLE:
        // Application 1 is sent to the target using Simple Protocol
            halLcdPrintLine("Downloading..\0",0, OVERWRITE_TEXT);
            ret = i2CSimple_DownloadtoTarget(App1_Simple_0, sizeof(App1_Simple_0));
            if (ret < 0 )
                halLcdPrintLine("Error\0",2, OVERWRITE_TEXT);
            else
                halLcdPrintLine("APP1 Download OK\0",2, OVERWRITE_TEXT);
            __delay_cycles(1500*DELAY_MS);
        break;

        case BOOTPROCEDURE_DOWNLOAD_APP2_SIMPLE:
        // Application 2 is sent to the target using Simple Protocol
            halLcdPrintLine("Downloading..\0",0, OVERWRITE_TEXT);
            ret = i2CSimple_DownloadtoTarget(App2_Simple_0, sizeof(App2_Simple_0));
            if (ret < 0 )
                halLcdPrintLine("Error\0",2, OVERWRITE_TEXT);
            else
                halLcdPrintLine("APP2 Download OK\0",2, OVERWRITE_TEXT);
            __delay_cycles(1500*DELAY_MS);
        break;
        
        /****** PROCEDURES for BSL-BASED and DUAL Image Protocol *************/
        case BOOTPROCEDURE_GET_VERSION:
        case BOOTPROCEDURE_GET_VERSION_DUALIMG:
            // Version is requested from the target
            i2cBsl_CmdVersion(&mainI2cBslCmd);      							
            halLcdPrintLine("BSL Version\0",0, OVERWRITE_TEXT);
            if (i2cBsl_ProcessCmd(&mainI2cBslCmd, &mainI2cBslResponse) < 0)
                halLcdPrintLine("Error in PI\0",2, OVERWRITE_TEXT);
            else
                PrintResponse(&mainI2cBslResponse);
            __delay_cycles(750*DELAY_MS);
        break;
        case BOOTPROCEDURE_APP_ERASE:
        case BOOTPROCEDURE_APP_ERASE_DUALIMG:
            // The application area is erased from target
            i2cBsl_CmdMassErase(&mainI2cBslCmd);
            halLcdPrintLine("Mass Erase\0",0, OVERWRITE_TEXT);
            if (i2cBsl_ProcessCmd(&mainI2cBslCmd, &mainI2cBslResponse) < 0)
                halLcdPrintLine("Error in PI\0",2, OVERWRITE_TEXT);
            else
                PrintResponse(&mainI2cBslResponse);
            __delay_cycles(750*DELAY_MS);
        break;
        case BOOTPROCEDURE_INCORRECTCHKSUM:
        case BOOTPROCEDURE_INCORRECTCHKSUM_DUALIMG:
            // Sends a command with incorrect CRC (error is expected)
            i2cBsl_CmdBadVersion(&mainI2cBslCmd);      							
            halLcdPrintLine("BSL Version\0",0, OVERWRITE_TEXT);
            if (i2cBsl_ProcessCmd(&mainI2cBslCmd, &mainI2cBslResponse) < 0)
                halLcdPrintLine("Error in PI\0",2, OVERWRITE_TEXT);
            else
                PrintResponse(&mainI2cBslResponse);
            __delay_cycles(750*DELAY_MS);
        break;
        
        /****** PROCEDURES for BSL-BASED Protocol *************/
        case BOOTPROCEDURE_DOWNLOAD_APP1_BSL:
        // Sends APP1 to the target using BSL protocol
            halLcdPrintLine("Downloading..\0",0, OVERWRITE_TEXT);
            ret = i2cBsl_DownloadtoTarget(App1_BSLBased_0, App1_BSLBased_Addr[0], sizeof(App1_BSLBased_0));
            if (ret < 0 )
                halLcdPrintLine("Error\0",2, OVERWRITE_TEXT);
            else
                halLcdPrintLine("APP1 Download OK\0",2, OVERWRITE_TEXT);
            __delay_cycles(1500*DELAY_MS);
        break;
        case BOOTPROCEDURE_DOWNLOAD_APP2_BSL:
        // Sends APP2 to the target using BSL protocol
            halLcdPrintLine("Downloading..\0",0, OVERWRITE_TEXT);
            ret = i2cBsl_DownloadtoTarget(App2_BSLBased_0, App2_BSLBased_Addr[0], sizeof(App2_BSLBased_0));
            if (ret < 0 )
                halLcdPrintLine("Error\0",2, OVERWRITE_TEXT);
            else
                halLcdPrintLine("APP2 Download OK\0",2, OVERWRITE_TEXT);
            __delay_cycles(1500*DELAY_MS);
        break;

        /****** PROCEDURES for DualImage BSL-based Protocol *************/
        case BOOTPROCEDURE_DOWNLOAD_APP1_DUALIMG:
        // Sends APP1 to the target using BSL protocol in Dual-Image mode
            halLcdPrintLine("Downloading..\0",0, OVERWRITE_TEXT);
            ret = i2cBsl_DownloadtoTarget(App1_BSLBased_DualImg_0, App1_BSLBased_DualImg_Addr[0], sizeof(App1_BSLBased_DualImg_0));
            if (ret < 0 )
                halLcdPrintLine("Error\0",2, OVERWRITE_TEXT);
            else
                halLcdPrintLine("APP1 Download OK\0",2, OVERWRITE_TEXT);
            __delay_cycles(1500*DELAY_MS);
        break;
        case BOOTPROCEDURE_DOWNLOAD_APP2_DUALIMG:
        // Sends APP2 to the target using BSL protocol in Dual-Image mode
            halLcdPrintLine("Downloading..\0",0, OVERWRITE_TEXT);
            ret = i2cBsl_DownloadtoTarget(App2_BSLBased_DualImg_0, App2_BSLBased_DualImg_Addr[0], sizeof(App2_BSLBased_DualImg_0));
            if (ret < 0 )
                halLcdPrintLine("Error\0",2, OVERWRITE_TEXT);
            else
                halLcdPrintLine("APP2 Download OK\0",2, OVERWRITE_TEXT);
            __delay_cycles(1500*DELAY_MS);
        break;
        case BOOTPROCEDURE_ERASESEG1_DUALIMG:
        // Erases a segment in the virtual application area (will erase Download area instead)
#ifdef TARGET_G2553
            i2cBsl_CmdEraseSeg(&mainI2cBslCmd, 0xE000);
#elif (defined(TARGET_G2452))
            i2cBsl_CmdEraseSeg(&mainI2cBslCmd, 0xEE00);
#elif (defined(TARGET_FR5739))
            i2cBsl_CmdEraseSeg(&mainI2cBslCmd, 0xDE00);
#endif          
            halLcdPrintLine("Seg Erase\0",0, OVERWRITE_TEXT);
            if (i2cBsl_ProcessCmd(&mainI2cBslCmd, &mainI2cBslResponse) < 0)
                halLcdPrintLine("Error in PI\0",2, OVERWRITE_TEXT);
            else
                PrintResponse(&mainI2cBslResponse);
            __delay_cycles(750*DELAY_MS);
        break;
        case BOOTPROCEDURE_ERASESEG2_DUALIMG:
        // Erases a segment in what is physically the Download area (error expected)
#ifdef TARGET_G2553
            i2cBsl_CmdEraseSeg(&mainI2cBslCmd, 0xC000);
#elif (defined(TARGET_G2452))
            i2cBsl_CmdEraseSeg(&mainI2cBslCmd, 0xE000);
#elif (defined(TARGET_FR5739))
            i2cBsl_CmdEraseSeg(&mainI2cBslCmd, 0xC200);
#endif          
            halLcdPrintLine("Seg Erase\0",0, OVERWRITE_TEXT);
            if (i2cBsl_ProcessCmd(&mainI2cBslCmd, &mainI2cBslResponse) < 0)
                halLcdPrintLine("Error in PI\0",2, OVERWRITE_TEXT);
            else
                PrintResponse(&mainI2cBslResponse);
            __delay_cycles(750*DELAY_MS);
        break;
        
        
        /****** PROCEDURES for SMBus based Protocol *************/
        case BOOTPROCEDURE_SMBUS_VERSION:
        // Requests bootloader version
           i2cSMB_CmdVersion(&SMB_Cmd);      							
            halLcdPrintLine("Boot Version\0",0, OVERWRITE_TEXT);
            if (i2cSMB_ProcessCmd(&SMB_Cmd, &SMB_Resp) < 0)
                halLcdPrintLine("Error in PI\0",2, OVERWRITE_TEXT);
            else
                PrintSMBResponse(&SMB_Resp);
            __delay_cycles(750*DELAY_MS);
        break;
        case BOOTPROCEDURE_SMBUS_APPERASE:
        // Erases the application area in SMBus based protocol
            i2cSMB_CmdAppErasePEC(&SMB_Cmd);      							
            halLcdPrintLine("App Erase\0",0, OVERWRITE_TEXT);
            if (i2cSMB_ProcessCmd(&SMB_Cmd, &SMB_Resp) < 0)
                halLcdPrintLine("Error in PI\0",2, OVERWRITE_TEXT);
            else
                PrintSMBResponse(&SMB_Resp);
            __delay_cycles(750*DELAY_MS);
        break;
        case BOOTPROCEDURE_SMBUS_BADAPPERASE:
        // Sends a command with incorrect PEC (Error expected)
           i2cSMB_CmdBadAppErasePEC(&SMB_Cmd);      							
            halLcdPrintLine("App Erase\0",0, OVERWRITE_TEXT);
            if (i2cSMB_ProcessCmd(&SMB_Cmd, &SMB_Resp) < 0)
                halLcdPrintLine("Error in PI\0",2, OVERWRITE_TEXT);
            else
                PrintSMBResponse(&SMB_Resp);
            __delay_cycles(750*DELAY_MS);
        break;
        case BOOTPROCEDURE_SMBUS_APP1:
        // Download APP1 using SMBus protocol
            halLcdPrintLine("Downloading..\0",0, OVERWRITE_TEXT);
            ret = i2CSMB_DownloadtoTarget(App1_SMBus_0, App1_SMBus_Addr[0], sizeof(App1_SMBus_0));
            if (ret < 0 )
                halLcdPrintLine("Error\0",2, OVERWRITE_TEXT);
            else
                halLcdPrintLine("APP1 Download OK\0",2, OVERWRITE_TEXT);
            __delay_cycles(1500*DELAY_MS);
        break;
        case BOOTPROCEDURE_SMBUS_APP2:
        // Download APP2 using SMBus protocol
            halLcdPrintLine("Downloading..\0",0, OVERWRITE_TEXT);
            ret = i2CSMB_DownloadtoTarget(App2_SMBus_0, App2_SMBus_Addr[0], sizeof(App2_SMBus_0));
            if (ret < 0 )
                halLcdPrintLine("Error\0",2, OVERWRITE_TEXT);
            else
                halLcdPrintLine("APP2 Download OK\0",2, OVERWRITE_TEXT);
            __delay_cycles(1500*DELAY_MS);
        break;
        case BOOTPROCEDURE_SMBUS_OVERFLOW:
        // Send command with more data than expected
            halLcdPrintLine("Downloading..\0",0, OVERWRITE_TEXT);
            i2cSMB_CmdRxDataBlock(&SMB_Cmd, App1_SMBus_Addr[0], 31, (unsigned char *) &App1_SMBus_0[0]);
            if (i2cSMB_ProcessCmd(&SMB_Cmd, &SMB_Resp) < 0)
                halLcdPrintLine("Error in PI\0",2, OVERWRITE_TEXT);
            else
                PrintSMBResponse(&SMB_Resp);
            __delay_cycles(750*DELAY_MS);
            
        break;
        case BOOTPROCEDURE_SMBUS_TIMEOUT:
            halLcdPrintLine("Timeout\0",0, OVERWRITE_TEXT);
            if (i2cForceTimeout() < 0)
                halLcdPrintLine("Error in PI\0",2, OVERWRITE_TEXT);
            else
                halLcdPrintLine("OK\0",2, OVERWRITE_TEXT);
            __delay_cycles(750*DELAY_MS);
        break;
        
        /****** PROCEDURES for Application *************/
        case BOOTPROCEDURE_APP_LED1:
        // Toggle LED 1
            if (i2CApp_LED1() < 0)
                halLcdPrintLine("Error in PI\0",2, OVERWRITE_TEXT);
        break;
        case BOOTPROCEDURE_APP_LED2:
        // Toggle LED 2
            if (i2CApp_LED2() < 0)
                halLcdPrintLine("Error in PI\0",2, OVERWRITE_TEXT);
        break;
        case BOOTPROCEDURE_APP_LEDBOTH:
        // Toggle both LEDs
            if (i2CApp_LEDBoth() < 0)
                halLcdPrintLine("Error in PI\0",2, OVERWRITE_TEXT);
        break;
        case BOOTPROCEDURE_APP_FORCEBSL:
        // Force target to jump to BSL 
            if (i2CApp_ForceBSL() < 0)
                halLcdPrintLine("Error in PI\0",2, OVERWRITE_TEXT);
        break;

    } // switch
  } // while(1)
}// main


/******************************************************************************
 * @brief   Low level function
 *  Stops watchdog before data segment initialization
 *
 * @return 1 : Initializes the data segment
 *****************************************************************************/
#ifdef __IAR_SYSTEMS_ICC__
int __low_level_init(void)
#elif defined (__TI_COMPILER_VERSION__)
int _system_pre_init(void)
#endif
{
  WDTCTL = WDTPW+WDTHOLD;
  return 1; //1 - Perform data segment initialization.
}
