/* --COPYRIGHT--,BSD
 * Copyright (c) 2012, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
#include <stdint.h>
#include <msp430.h>

#include "bsl.h"

void BSL_Comm_Init(void)
{
      /* Initializing I2C Module */
    P1SEL |= BIT6 + BIT7;                 // Assign I2C pins to USCI_B0
    P1SEL2 |= BIT6 + BIT7;                 // Assign I2C pins to USCI_B0
    UCB0CTL1 |= UCSWRST;                   // Enable SW reset
    UCB0CTL0 = UCMST + UCMODE_3 + UCSYNC;      // I2C Master, synchronous mode
    UCB0CTL1 = UCSSEL_2 + UCSWRST;           // Use SMCLK, keep SW reset
    UCB0BR0 = 80;                          // fSCL = SMCLK/80 = ~100kHz
    UCB0BR1 = 0;
    UCB0I2CSA = 0x40;                      // Set slave address
    UCB0CTL1 &= ~UCSWRST;                  // Clear SW reset, resume operation

}


/* Reads one byte from BSL target (start and stop included)  */
uint8_t BSL_getResponse(void)
{
    uint8_t ret;
    UCB0CTL1 &= ~UCTR;                      // I2C RX
    UCB0CTL1 |= UCTXSTT;                    // I2C start condition

    while (UCB0CTL1 & UCTXSTT);
    UCB0CTL1 |= UCTXSTP;

    while (!(IFG2 & UCB0RXIFG));
    ret = UCB0RXBUF;

    while (UCB0CTL1 & UCTXSTP);

    return ret;
}

/* Checks if a slave is responding */
uint8_t BSL_slavePresent(void)
{
    UCB0STAT = 0x00;                        // Clear all flags
    UCB0CTL1 |= UCTR + UCTXSTT + UCTXSTP;   // I2C TX, start condition
    while (UCB0CTL1 & UCTXSTP);

    if (UCB0STAT & UCNACKIFG)
        return 0;                           // If flag set, slave is not responding
    else
        return 1;                           // If flag clear, slave ACK'ed

}

/* Sends single I2C Byte (start and stop included) */
void BSL_sendSingleByte(uint8_t ui8Byte)
{
    UCB0CTL1 |= UCTR + UCTXSTT;             // I2C TX, start condition
    UCB0TXBUF = ui8Byte;

    while (UCB0CTL1 & UCTXSTT);

    UCB0CTL1 |= UCTXSTP;                    // I2C stop condition after 1st TX

    while (UCB0CTL1 & UCTXSTP);

}

/* Sends application image to BSL target (start and stop included)
 * This is a polling function and is blocking. */
void BSL_sendPacket(tBSLPacket tPacket)
{
    uint16_t ii;
    volatile uint8_t crch, crcl;

    UCB0CTL1 |= UCTR + UCTXSTT;             // I2C TX, start condition
    UCB0TXBUF = 0x80;
    while (UCB0CTL1 & UCTXSTT);             // Loop until I2C STT is sent

    while (!(IFG2 & UCB0TXIFG));
    UCB0TXBUF = tPacket.ui8Length;

    while (!(IFG2 & UCB0TXIFG));
    UCB0TXBUF = tPacket.tPayload.ui8Command;

    if(tPacket.ui8Length > 1)
    {
        while (!(IFG2 & UCB0TXIFG));
        UCB0TXBUF = tPacket.tPayload.ui8Addr_L;

        while (!(IFG2 & UCB0TXIFG));
        UCB0TXBUF = tPacket.tPayload.ui8Addr_H;

        for (ii=0; ii < (tPacket.ui8Length-3); ii++)
        {
            while (!(IFG2 & UCB0TXIFG));
            UCB0TXBUF = tPacket.tPayload.ui8pData[ii];
        }
    }

    while (!(IFG2 & UCB0TXIFG));
    crcl = (uint8_t)(tPacket.ui16Checksum & 0xFF);
    UCB0TXBUF = crcl;

    while (!(IFG2 & UCB0TXIFG));
    crch = (uint8_t)(tPacket.ui16Checksum >> 8);
    UCB0TXBUF = crch;

    while (!(IFG2 & UCB0TXIFG));
    UCB0CTL1 |= UCTXSTP;       // I2C stop condition before the last TX

    while (UCB0CTL1 & UCTXSTP);
}


void BSL_flush(void)
{
  IFG2 &= ~(UCB0RXIFG);
}
