/*
 * \file   TI_MSPBoot_AppMgr.c
 *
 * \brief  Application Manager. Handles App validation, decides if device
 *         should jump to App or stay in bootloader
 */
/* --COPYRIGHT--,BSD
 * Copyright (c) 2017, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/

//
// Include files
//
#include <stdbool.h>
#include "msp430.h"
#include "TI_MSPBoot_Common.h"
#include "TI_MSPBoot_AppMgr.h"
#include "crc.h"

//
//  Global variables
//
/*! Password sent by Application to force boot mode. This variable is in a fixed
    location and should keep same functionality and location in Boot and App */
extern uint16_t  PassWd;

/*! Status and Control byte. This variable is in a fixed
 location and should keep same functionality and location in Boot and App */
extern uint8_t  StatCtrl;

//
//  Local function prototypes
//
static bool TI_MSPBoot_AppMgr_BootisForced(void);
static void TI_MSPBoot_Restore_DefaultClockSettings(void);

/******************************************************************************
 *
 * @brief   Checks if an Application is valid
 *  Depending on CONFIG_APPMGR_APP_VALIDATE, this function can validate app by:
 *  CONFIG_APPMGR_APP_VALIDATE  |    Function
 *          2                   | Check if reset vector is different from 0xFFFF
 *          other               | Application is expected to be valid
 *
 *
 * @return  true if application is valid,
 *          false if applicaiton is invalid
 *****************************************************************************/
#if (CONFIG_APPMGR_APP_VALIDATE == 2)
static bool TI_MSPBoot_AppMgr_AppisValid(void)
{
    // Check if Application Reset vector exists
    if (*(volatile uint16_t *)(&_Appl_Reset_Vector) != 0xFFFF)
    {
        return true;
    }
    else
    {
        return false;
    }
}
#else
// Always assume that Application is valid
#warning "Application is not validated"
#define TI_MSPBoot_AppMgr_AppisValid()   true
#endif


/******************************************************************************
 *
 * @brief   Decides whether to stay in MSPBoot or if device should jump to App
 *  MSPBoot:  Boot mode is forced by a call from App, OR
 *          Boot mode is forced by an external event (button pressed), OR
 *          Application is invalid
 *  App:    Boot mode is not forced, AND
 *          Application is valid
 *
 * @return  true if application is valid and should be executed
 *          false if we must stay in Boot mode
 *****************************************************************************/
bool TI_MSPBoot_AppMgr_ValidateApp(void)
{
    if ((TI_MSPBoot_AppMgr_BootisForced() == false) && 
        (TI_MSPBoot_AppMgr_AppisValid() == true))
    {
		TI_MSPBoot_Restore_DefaultClockSettings();
        return true;  // Boot is not forced and App is valid
    }
    else
    {
        return false; // Boot is forced or App is valid
    }
}


/******************************************************************************
 *
 * @brief   Jump to Appplication 
 *          A Reset is forced in order to validate Application after Reset
 *          Software BOR is used on devices supporting this feature,
 *          other devices use a PUC by an invalid watchdog write
 *          Check HW_RESET_BOR 
 *
 * @return  None
 *****************************************************************************/
void TI_MSPBoot_AppMgr_JumpToApp(void)
{
#if defined (HW_RESET_BOR)
    // Force a Software BOR
    PMMCTL0 = PMMPW | PMMSWBOR;
#else
    // Force a PUC reset by writing incorrect Watchdog password
    WDTCTL = 0xDEAD;
#endif
}


/******************************************************************************
 *
 * @brief   Checks if Boot mode is forced
 *          Boot mode is forced by an application call sending a request and 
 *          password, or by an external event such as a button press
 *
 * @return  true Boot mode is forced
 *          false Boot mode is not forced
 *****************************************************************************/
static bool TI_MSPBoot_AppMgr_BootisForced(void)
{
    bool ret = false;

    // Check if application is requesting Boot mode and password is correct
    if (((StatCtrl & BOOT_APP_REQ) != 0) && (PassWd == BSL_PASSWORD))
    {
        ret = true;
    }

    // Check for an external event such as button on Launchpad
    __delay_cycles(10000);   // Wait for pull-up to go high
    //If button is pressed, force BSL
    if (HW_ENTRY_CONDITION)
    {
        ret = true;
    }

    // Clear Status and Control byte and password
    PassWd = 0;
    StatCtrl = 0;
    return ret;
}

/******************************************************************************
 *
 * @brief   Restores the clock system to default settings
 *
 * @return  none
 *****************************************************************************/
static void TI_MSPBoot_Restore_DefaultClockSettings(void)
{
	//TODO: Restore clock system to default settings
}
