//******************************************************************************
//  MSP430F22x4 Demo - Executing code from RAM
//
//  Description: This program demonstrates how a parts of a program can be
//  executed from FLASH and RAM. First, the main program copies the function
//  from Flash to RAM during runtime. The main program executes from Flash and
//  it calls the RAM function and executes in RAM and finally jumps back to
//  Flash. The function first erases flash seg C, then it increments all values
//  in seg C, then it erases seg D, then copies seg C to seg D.
//
//  Important Notes:
//
//  1. Copy a new linker file (lnk_custom_F2274.xcl) before modifying it. See
//     the zip file for the modified linker file. 2 lines were added and 1 line
//     was changed. Goto Project Options --> Linker --> Config. Overide the
//     linker command file and insert the new .xcl file.
//
//     -QRAMCODE=FLASHCODE                         // Needed to tell compiler that user will copy flash code to ram code
//     -Z(DATA)RAMCODE=0400-05FF                   // Place this in RAM location
//     //-Z(CODE)FLASHCODE=8000-80FA               // or you can add FLASH segment separately
//
//     -Z(CODE)FLASHCODE,CSTART,ISR_CODE=8000-FFDD // Add FLASHCODE to Flash location
//     -P(CODE)CODE=8000-FFDD
//
//  2. Erasing and Writing to flash running from RAM requires a few checks of
//     the BUSY flag. See the user guide for the flow diagram.
//
//     while(FCTL3&BUSY);
//
//  ACLK = n/a, MCLK = SMCLK = CALxxx_1MHZ = 1MHz
//  //* Set Breakpoint on NOP in the Mainloop to avoid Stressing Flash *//
//
//
//  W. Goh & M. Morales & A. Dannenberg
//  Texas Instruments Inc.
//  December 2008
//  Built with IAR Embedded Workbench Version: 4.11B
//******************************************************************************

#include "msp430x22x4.h"
#include "string.h"

// Function prototypes
void write_SegC (char value);
void copy_C2D (void);

char value;                                 // 8-bit value to write to seg C

#pragma segment="FLASHCODE"                 // Define flash segment code
#pragma segment="RAMCODE"

// This function is in flash to be copied to RAM
#pragma location="RAMCODE"                  
void write_SegC (char value)
{
  char *Flash_ptr;                          // Flash pointer
  unsigned int i;

  Flash_ptr = (char *) 0x1040;              // Initialize Flash pointer
  while(FCTL3&BUSY);                        // Check BUSY flag; Important if running this code from RAM
  FCTL1 = FWKEY + ERASE;                    // Set Erase bit
  FCTL3 = FWKEY;                            // Clear Lock bit
  
  *Flash_ptr = 0;                           // Dummy write to erase Flash segment
  while(FCTL3&BUSY);                        // Check BUSY flag; Important if running this code from RAM
  FCTL1 = FWKEY + WRT;                      // Set WRT bit for write operation

  for (i=0; i<64; i++)
  {
    while(FCTL3&BUSY);                        // Check BUSY flag; Important if running this code from RAM
    *Flash_ptr++ = value;                   // Write value to flash
  }

  while(FCTL3&BUSY);                        // Check BUSY flag; Important if running this code from RAM
  FCTL1 = FWKEY;                            // Clear WRT bit
  FCTL3 = FWKEY + LOCK;                     // Set LOCK bit
}

// This function is in flash to be copied to RAM
#pragma location="RAMCODE"                  
void copy_C2D (void)
{
  char *Flash_ptrC;                         // Segment C pointer
  char *Flash_ptrD;                         // Segment D pointer
  unsigned int i;

  Flash_ptrC = (char *) 0x1040;             // Initialize Flash segment C pointer
  Flash_ptrD = (char *) 0x1000;             // Initialize Flash segment D pointer
  while(FCTL3&BUSY);                        // Check BUSY flag; Important if running this code from RAM
  FCTL1 = FWKEY + ERASE;                    // Set Erase bit
  FCTL3 = FWKEY;                            // Clear Lock bit
  *Flash_ptrD = 0;                          // Dummy write to erase Flash segment D
  while(FCTL3&BUSY);                        // Check BUSY flag; Important if running this code from RAM
  FCTL1 = FWKEY + WRT;                      // Set WRT bit for write operation

  for (i=0; i<64; i++)
  {
    while(FCTL3&BUSY);                        // Check BUSY flag; Important if running this code from RAM
    *Flash_ptrD++ = *Flash_ptrC++;          // copy value segment C to segment D
  }
  
  while(FCTL3&BUSY);                        // Check BUSY flag; Important if running this code from RAM
  FCTL1 = FWKEY;                            // Clear WRT bit
  FCTL3 = FWKEY + LOCK;                     // Set LOCK bit
}


//#pragma language=default                    // This is not required. Can be removed.
                                              // This is to inform compiler to put main code in Flash
                                              // that is not copied to RAM

//Copy function from Flash to RAM
void copy_flash_to_RAM(void)
{
  
  unsigned char *flash_start_ptr;           // Initialize pointers
  unsigned char *flash_end_ptr;
  unsigned char *RAM_start_ptr;
  
  //Initialize flash and ram start and end address
  flash_start_ptr = __segment_begin("FLASHCODE");
  flash_end_ptr = __segment_end("FLASHCODE");
  RAM_start_ptr = __segment_begin("RAMCODE");
  
  //calculate function size
  unsigned long function_size = (unsigned long)(flash_end_ptr) - (unsigned long)(flash_start_ptr);
 
  // Copy flash function to RAM 
  memcpy(RAM_start_ptr,flash_start_ptr,function_size);
}


void main(void)
{
  WDTCTL = WDTPW + WDTHOLD;                 // Stop WatchDog timer
  if (CALBC1_1MHZ ==0xFF || CALDCO_1MHZ == 0xFF)                                     
  {  
    while(1);                               // If calibration constants erased
                                            // do not load, trap CPU!!
  }   
  BCSCTL1 = CALBC1_1MHZ;                    // Set DCO to 1MHz
  DCOCTL = CALDCO_1MHZ;
  FCTL2 = FWKEY + FSSEL_1 + FN1;            // MCLK/3 for Flash Timing Generator
  P1DIR |= 0x01;
  P1OUT |= 0x00;
  value = 0;
  
  copy_flash_to_RAM();                      // call function to copy from flash to RAM
  
  // Parts of this while loop executes from Flash. The write_SegC and copy_C2D
  // is executed from RAM. The program basically will execute from FLASH until
  // it calls write_SegC function which will jump the program to RAM.
  // After executing write_SegC in RAM, it will jump back to flash to continue
  // executing. When copy_C2D is called, it jumps to RAM and back to Flash when
  // done.
  while(1)
  {
      P1OUT ^= 0x01;
      for(int i=0;i<20000;i++);
      write_SegC(value++);                  // This portion of the code is executed from RAM
      copy_C2D();                           // This portion of the code is executed from RAM
      __no_operation();                     // Set breakpoint here
  }
}



