// Standard C Includes
#include <stdio.h>
#include <stdint.h>
#include <stdbool.h>
#include <stdarg.h>
#include <string.h>


// Standard SYSBIOS Includes
#include <xdc/std.h>              //mandatory - have to include first, for BIOS types
#include <xdc/cfg/global.h>        //header file for statically defined objects/handles
#include <xdc/runtime/Log.h>
#include <xdc/runtime/Error.h>
#include <xdc/runtime/System.h>
#include <xdc/runtime/Types.h>
#include <ti/drivers/dpl/DebugP.h>
#include <ti/sysbios/BIOS.h>        //mandatory - if you call APIs like BIOS_start()

//#include <ti/sysbios/hal/Hwi.h>
#include <ti/sysbios/family/arm/m3/Hwi.h>
#include <ti/sysbios/knl/Swi.h>
#include <ti/sysbios/knl/Task.h>

#include <ti/sysbios/knl/Clock.h>
#include <ti/sysbios/family/arm/msp432/Seconds.h>

#include <ti/sysbios/family/arm/msp432/Timer.h>
#include <ti/sysbios/family/arm/msp432/ClockFreqs.h>
//#include <ti/sysbios/hal/Timer.h>
#include <ti/sysbios/knl/Semaphore.h>
#include <ti/sysbios/knl/Queue.h>
#include <ti/sysbios/knl/Mailbox.h>

// RTOS Includes
#include <ti/drivers/Power.h>
#include <ti/drivers/power/PowerMSP432.h>
#include <ti/drivers/GPIO.h>
#include <ti/drivers/gpio/GPIOMSP432.h>
#include <ti/drivers/dma/UDMAMSP432.h>
#include <ti/drivers/UART.h>
#include <ti/drivers/uart/UARTMSP432.h>
#include <ti/drivers/SPI.h>
#include <ti/drivers/spi/SPIMSP432DMA.h>


// Simplelink SDK (Driverlib) includes
#include <ti/devices/msp432p4xx/driverlib/driverlib.h>
#include <ti/devices/msp432p4xx/driverlib/debug.h>



// Clock module always uses TIMER ID 0
#define TESTTIMER_ID   1

// Calculate interrupt IDs from timer ID
#define TESTTIMER_INTERRUPT_ID  (INT_TA0_0 + (2*TESTTIMER_ID))


Timer_Handle g_testTimer_hndl = NULL;
Timer_Params g_testTimer_params;
Hwi_Params   g_testTime_HWI_params;

Error_Block g_eb;

extern PowerMSP432_PerfLevel PowerMSP432_perfLevels[];

void __error__(char *pcFilename, unsigned long line);
void testTimer_HWI(UArg arg0);
Timer_Handle TestTimerInit(void);

/*
 *  =============================== GPIO ===============================
 */
/* Place into subsections to allow the TI linker to remove items properly */
#if defined(__TI_COMPILER_VERSION__)
#pragma DATA_SECTION(GPIOMSP432_config, ".const:GPIOMSP432_config")
#endif

typedef enum LRR_MicroBoard_GPIONames
{
  LED,
  TPa,
  TPb
}LRR_MicroBoard_GPIONames;


/*
 * Array of Pin configurations
 * NOTE: The order of the pin configurations must coincide with what was
 *       defined in micro_revb.h
 * NOTE: Pins not used for interrupts should be placed at the end of the
 *       array.  Callback entries can be omitted from callbacks array to
 *       reduce memory usage.
 */
GPIO_PinConfig gpioPinConfigs[] =
{
    /* LED 1.0 */
    GPIOMSP432_P1_0   | GPIO_CFG_OUT_STD | GPIO_CFG_OUT_STR_HIGH | GPIO_CFG_OUT_LOW,
    /* TPa 3.2 */
    GPIOMSP432_P3_2   | GPIO_CFG_OUT_STD | GPIO_CFG_OUT_STR_HIGH | GPIO_CFG_OUT_LOW,
    /* TBb 6.5 */
    GPIOMSP432_P6_5   | GPIO_CFG_OUT_STD | GPIO_CFG_OUT_STR_HIGH | GPIO_CFG_OUT_LOW,
};

/*
 * Array of callback function pointers
 * NOTE: The order of the pin configurations must coincide with what was
 *       defined in MSP_EXP432P401R.h
 * NOTE: Pins not used for interrupts can be omitted from callbacks array to
 *       reduce memory usage (if placed at end of gpioPinConfigs array).
 */
GPIO_CallbackFxn gpioCallbackFunctions[] =
{
};

const GPIOMSP432_Config GPIOMSP432_config =
{
    .pinConfigs = (GPIO_PinConfig *) gpioPinConfigs,
    .callbacks = (GPIO_CallbackFxn *) gpioCallbackFunctions,
    .numberOfPinConfigs = sizeof(gpioPinConfigs) / sizeof(GPIO_PinConfig),
    .numberOfCallbacks = sizeof(gpioCallbackFunctions) / sizeof(GPIO_CallbackFxn),
    .intPriority = (~0) };


/*
 *  =============================== Power ===============================
 */
const PowerMSP432_ConfigV1 PowerMSP432_config =
{
    .policyInitFxn = &PowerMSP432_initPolicy,
    .policyFxn = &PowerMSP432_sleepPolicy,
    .initialPerfLevel = 3,
    .enablePolicy = false,
    .enablePerf = true,
    .enableParking = true
};

//--------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------
void __error__(char *pcFilename, unsigned long line)
{
  System_printf("ASSERT: File: %s, Line: %d\n", pcFilename, line);
  System_flush();
}


//--------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------
void testTimer_HWI(UArg arg0)
{
  GPIO_toggle(TPa);
}


//--------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------
Timer_Handle TestTimerInit(void)
{
  uint32_t     smclk_rate_hz_u32;

  // Setup the timer HWI
  // I do this explicitly so that I can control the HWIs priority.
  Hwi_Params_init(&g_testTime_HWI_params);

  g_testTime_HWI_params.instance->name = "TimerHWI";
  g_testTime_HWI_params.arg            = NULL;
  g_testTime_HWI_params.priority       = 0x40;  // Highest dispatched priority.  I have the two higher priority
                                                // levels set to be zero latency interrupts (i.e. non-dispatched)

  // Now setup the timer itself
  smclk_rate_hz_u32 = PowerMSP432_perfLevels[Power_getPerformanceLevel()].SMCLK;

  Timer_Params_init(&g_testTimer_params);

  g_testTimer_params.instance->name  = "TestTimer";

  g_testTimer_params.period          = 1666;

  g_testTimer_params.periodType      = ti_sysbios_interfaces_ITimer_PeriodType_MICROSECS;
  g_testTimer_params.runMode         = ti_sysbios_interfaces_ITimer_RunMode_CONTINUOUS;
  g_testTimer_params.startMode       = ti_sysbios_interfaces_ITimer_StartMode_AUTO;

  g_testTimer_params.clockSource     = ti_sysbios_family_arm_msp432_Timer_Source_SMCLK;
  g_testTimer_params.synchronous     = true;  // SMCLK is synchronous to MCLK, timer reads can be done directly

  g_testTimer_params.extFreq.hi      = 0;
  g_testTimer_params.extFreq.lo      = smclk_rate_hz_u32;

  // Set dividers such that
  //   * the desired period (per .period) can be reached in <= 65536 counts (TimerA's are 16 bit)
  //   * the desired period (per .period) is reached in a whole number of TimerA counts (for maximum accuracy)
  // I'm setting for uSec counts
  if (smclk_rate_hz_u32 == 12000000)
  {
    g_testTimer_params.inputDivider    = ti_sysbios_family_arm_msp432_Timer_ID_2;
    g_testTimer_params.inputDividerExp = ti_sysbios_family_arm_msp432_Timer_IDEX_6;
  }
  else if (smclk_rate_hz_u32 == 24000000)
  {
    g_testTimer_params.inputDivider    = ti_sysbios_family_arm_msp432_Timer_ID_4;
    g_testTimer_params.inputDividerExp = ti_sysbios_family_arm_msp432_Timer_IDEX_6;
  }
  else if (smclk_rate_hz_u32 == 6000000)
    {
      g_testTimer_params.inputDivider    = ti_sysbios_family_arm_msp432_Timer_ID_1;
      g_testTimer_params.inputDividerExp = ti_sysbios_family_arm_msp432_Timer_IDEX_6;
    }
  else if (smclk_rate_hz_u32 == 3000000)
    {
      g_testTimer_params.inputDivider    = ti_sysbios_family_arm_msp432_Timer_ID_1;
      g_testTimer_params.inputDividerExp = ti_sysbios_family_arm_msp432_Timer_IDEX_3;
    }
  else
  {
    // Don't know how to setup prescalers for currently selected SMCLK rate
    ASSERT(0);
  }

  g_testTimer_params.nesting         = false;
  g_testTimer_params.controlRegInit  = g_testTimer_params.clockSource;
  g_testTimer_params.hwiParams       = &g_testTime_HWI_params;

  g_testTimer_hndl = Timer_create(TESTTIMER_ID, testTimer_HWI, &g_testTimer_params, &g_eb);
  if (g_testTimer_hndl == NULL)
  {
    Log_error0("Timer Init Error");
    ASSERT(0);
    return(NULL);
  }
  return(g_testTimer_hndl);
}


//--------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------
int main(void)
{
  System_printf("TimerA Problem Example\n");
  System_flush();

  MAP_WDT_A_holdTimer();              // Turn off watchdog
  Power_init();
  GPIO_init();
  TestTimerInit();
  MAP_Interrupt_enableMaster();       // Enabled Master Interrupts

  /* Start BIOS */
  BIOS_start();

  return (0);
}
