/* --COPYRIGHT--,BSD
 * Copyright (c) 2017, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//*****************************************************************************
// Development main.c for MSP430FR2522 and MSP430FR2512 CapTIvate devices.
//
// This starter application initializes the CapTIvate touch library
// for the touch panel specified by CAPT_UserConfig.c/.h via a call to
// CAPT_appStart(), which initializes and calibrates all sensors in the
// application, and starts the CapTIvate interval timer.
//
// Then, the capacitive touch interface is driven by calling the CapTIvate
// application handler, CAPT_appHandler().  The application handler manages
// whether the user interface (UI) is running in full active scan mode, or
// in a low-power wake-on-proximity mode.
//
// \version 1.60.00.03
// Released on November 22, 2017
//
//*****************************************************************************

#include <msp430.h>                      // Generic MSP430 Device Include
#include "driverlib.h"                   // MSPWare Driver Library
#include "captivate.h"                   // CapTIvate Touch Software Library
#include "CAPT_App.h"                    // CapTIvate Application Code
#include "CAPT_BSP.h"                    // CapTIvate EVM Board Support Package
#include "UART.h"

#define UART_FC_H       P2OUT |= 0x40;
#define UART_FC_L       P2OUT &= ~0x40;

#define BL_ON           P2OUT |= 0x10;
#define BL_OFF          P2OUT &= ~0x10;

#define RED_ON           P2OUT |= 0x04;
#define RED_OFF          P2OUT &= ~0x04;

#define BLUE_ON           P2OUT |= 0x08;
#define BLUE_OFF          P2OUT &= ~0x08;

//P2.2 -- red led
//P2.3 -- blue led
//P2.4 -- bl
void initLed(void)
{
    //  P2DIR |= 0x1C;
    //  P2OUT |= 0x1C;

    P2DIR |= 0x5C;
    P2OUT |= 0x5C;


    P2DIR       |= BIT6;
    P2REN       |= BIT6;
    P2OUT       |= BIT6;

    PM5CTL0 &= ~LOCKLPM5;

    BL_OFF;
    //UART_FC_H;
    UART_FC_L;
}

void delayms(unsigned int ms)
{
    unsigned int i;
    for(i = 0; i < 5; i ++)
    {
        while(ms--);
    }
}

#define Test_Pakage_Size (3)
static unsigned char Touched_Button[Test_Pakage_Size];

void my_button_callback(tSensor* pSensor)
{
    Touched_Button[0] = 0xAA;
    Touched_Button[1] = 0x55;
    Touched_Button[2] = 0xDD;
    if((pSensor->bSensorTouch == true)&&(pSensor->bSensorPrevTouch == false))
    {
        UART_FC_L;
        delayms(1);
        Touched_Button[1] = (uint8_t)(((tButtonSensorParams*)(pSensor->pSensorParams))->ui16DominantElement) + 1;
        UART_transmitBuffer(Touched_Button, 3);
        UART_FC_H;
    }
}


//
// RECEIVE_BUFFER_SIZE defines the size of the UART receive buffer.
//
#define RECEIVE_BUFFER_SIZE        64

//
// g_ui8ReceiveBuffer is the UART receive buffer. It is filled with
// incoming characters via the UART receive event handler.
//
uint8_t g_ui8ReceiveBuffer[RECEIVE_BUFFER_SIZE];

//
// g_ui8ReceiveIndex is the UART receive buffer index. It defines the
// buffer index that the next received byte is placed into.
//
uint8_t g_ui8ReceiveIndex;

//
// g_bGotNewLine is a flag used by the UART receive event handler to
// signal the background loop (the main application thread) that a full
// ASCII line has been received in the buffer.
//
bool g_bGotNewLine = false;

//
// The UART receive handler is called by the UART port driver whenever a new
// character is received.
//
bool receiveHandler(uint8_t ui8Data)
{
    if (g_bGotNewLine == false)
    {
        if (g_ui8ReceiveIndex<RECEIVE_BUFFER_SIZE)
        {
            //
            // It is safe to receive the new character into the buffer.
            // If the data is a ASCII return, signal the background app
            // and exit awake.
            //
            g_ui8ReceiveBuffer[g_ui8ReceiveIndex++] = ui8Data;
            if (ui8Data=='\r')
            {
                g_bGotNewLine = true;
                return true;
            }
            else
            {
                UART_transmitByteImmediately(ui8Data);
            }
        }
        else
        {
            //
            // The buffer is full, return new line to print what
            // we have right now
            //
            g_bGotNewLine = true;
            return true;
        }
    }
    return false;
}

const tUARTPort g_myUartPort =
{
 .pbReceiveCallback = &receiveHandler,
 .pbErrorCallback = 0,
 .peripheralParameters.selectClockSource = EUSCI_A_UART_CLOCKSOURCE_SMCLK,
 .peripheralParameters.clockPrescalar = 26,
 .peripheralParameters.firstModReg = 0,
 .peripheralParameters.secondModReg = 0xB6,
 .peripheralParameters.parity = EUSCI_A_UART_NO_PARITY,
 .peripheralParameters.msborLsbFirst = EUSCI_A_UART_LSB_FIRST,
 .peripheralParameters.numberofStopBits = EUSCI_A_UART_ONE_STOP_BIT,
 .peripheralParameters.uartMode = EUSCI_A_UART_MODE,
 .peripheralParameters.overSampling = EUSCI_A_UART_OVERSAMPLING_BAUDRATE_GENERATION
};

void main(void)
{
    //
    // Initialize the MCU
    // BSP_configureMCU() sets up the device IO and clocking
    // The global interrupt enable is set to allow peripherals
    // to wake the MCU.
    //
    WDTCTL = WDTPW | WDTHOLD;
    BSP_configureMCU();

    MAP_CAPT_registerCallback(&BTN00, &my_button_callback);

    __bis_SR_register(GIE);

    UART_openPort(&g_myUartPort);

    //
    // Start the CapTIvate application
    //
    CAPT_appStart();

    //
    // Background Loop
    //
    while(1)
    {
        //
        // Run the captivate application handler.
        //
        CAPT_appHandler();

        //
        // This is a great place to add in any
        // background application code.
        //
        if (g_bGotNewLine == true)
        {
            UART_transmitBuffer("\n\rReceived: ", sizeof("\n\rReceived: "));
            UART_transmitBuffer(g_ui8ReceiveBuffer, g_ui8ReceiveIndex);
            UART_transmitBuffer("\n\r", sizeof("\n\r"));
            UART_transmitBuffer("Waiting for a new line: ", sizeof("Waiting for a new line: "));
            g_ui8ReceiveIndex = 0;
            g_bGotNewLine = false;
        }

        //
        // End of background loop iteration
        // Go to sleep if there is nothing left to do
        //
        CAPT_appSleep();

    } // End background loop
} // End main()



