/*
 * Copyright (c) 2020, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/*!****************************************************************************
 *  @file       dl_timerh.h
 *  @brief      General Purpose Timer High Resolution (TIMH) Peripheral
 *              Interface
 *  @defgroup   TIMH General Purpose Timer High Resolution (TIMH)
 *
 *
 *  @anchor ti_dl_dl_timerh__Overview
 *  # Overview
 *
 *  The TimerH Driver Library allows you to configure the High Resolution
 *  Counter Timer (TIMH) module in output compare, input capture, PWM output,
 *  one-shot and periodic modes. For detailed TIMH features please refer to the
 *   Device Technical Reference Manual (TRM)
 *
 *  <hr>
 ******************************************************************************
 */
/** @addtogroup TIMH
 * @{
 */
#ifndef ti_dl_dl_timerh__include
#define ti_dl_dl_timerh__include

#include <stdbool.h>
#include <stdint.h>

#include <ti/driverlib/dl_timer.h>

#ifdef __MSPM0_HAS_TIMER_H__

#ifdef __cplusplus
extern "C" {
#endif

/* clang-format off */


/** @addtogroup DL_TIMERH_CAPTURE_COMPARE_INDEX
 *  @{
 */
/**
 * @brief  Index associated to Capture Compare 0
 */
#define DL_TIMERH_CAPTURE_COMPARE_0_INDEX                    DL_TIMER_CC_0_INDEX

/**
 * @brief  Index associated to Capture Compare 1
 */
#define DL_TIMERH_CAPTURE_COMPARE_1_INDEX                    DL_TIMER_CC_1_INDEX

/** @}*/

/** @addtogroup DL_TIMERH_INTERRUPT
 *  @{
 */

/*!
 * @brief Redirects to common @ref DL_TIMER_INTERRUPT_ZERO_EVENT
 */
#define DL_TIMERH_INTERRUPT_ZERO_EVENT           (DL_TIMER_INTERRUPT_ZERO_EVENT)

/*!
* @brief Redirects to common @ref DL_TIMER_INTERRUPT_LOAD_EVENT
*/
#define DL_TIMERH_INTERRUPT_LOAD_EVENT           (DL_TIMER_INTERRUPT_LOAD_EVENT)

/*!
* @brief Redirects to common @ref DL_TIMER_INTERRUPT_CC0_DN_EVENT
*/
#define DL_TIMERH_INTERRUPT_CC0_DN_EVENT       (DL_TIMER_INTERRUPT_CC0_DN_EVENT)

/*!
* @brief Redirects to common @ref DL_TIMER_INTERRUPT_CC1_DN_EVENT
*/
#define DL_TIMERH_INTERRUPT_CC1_DN_EVENT       (DL_TIMER_INTERRUPT_CC1_DN_EVENT)

/*!
* @brief Redirects to common @ref DL_TIMER_INTERRUPT_CC0_UP_EVENT
*/
#define DL_TIMERH_INTERRUPT_CC0_UP_EVENT       (DL_TIMER_INTERRUPT_CC0_UP_EVENT)

/*!
* @brief Redirects to common @ref DL_TIMER_INTERRUPT_CC1_UP_EVENT
*/
#define DL_TIMERH_INTERRUPT_CC1_UP_EVENT       (DL_TIMER_INTERRUPT_CC1_UP_EVENT)

/*!
* @brief Redirects to common @ref DL_TIMER_INTERRUPT_OVERFLOW_EVENT
*/
#define DL_TIMERH_INTERRUPT_OVERFLOW_EVENT   (DL_TIMER_INTERRUPT_OVERFLOW_EVENT)

/** @}*/

/** @addtogroup DL_TIMERH_EVENT
 *  @{
 */
/*!
 * @brief Redirects to common @ref DL_TIMER_EVENT_ZERO_EVENT
 */
#define DL_TIMERH_EVENT_ZERO_EVENT                   (DL_TIMER_EVENT_ZERO_EVENT)

/*!
* @brief Redirects to common @ref DL_TIMER_EVENT_LOAD_EVENT
*/
#define DL_TIMERH_EVENT_LOAD_EVENT                   (DL_TIMER_EVENT_LOAD_EVENT)

/*!
* @brief Redirects to common @ref DL_TIMER_EVENT_CC0_DN_EVENT
*/
#define DL_TIMERH_EVENT_CC0_DN_EVENT               (DL_TIMER_EVENT_CC0_DN_EVENT)

/*!
* @brief Redirects to common @ref DL_TIMER_EVENT_CC1_DN_EVENT
*/
#define DL_TIMERH_EVENT_CC1_DN_EVENT               (DL_TIMER_EVENT_CC1_DN_EVENT)

/*!
* @brief Redirects to common @ref DL_TIMER_EVENT_CC0_UP_EVENT
*/
#define DL_TIMERH_EVENT_CC0_UP_EVENT               (DL_TIMER_EVENT_CC0_UP_EVENT)

/*!
* @brief Redirects to common @ref DL_TIMER_EVENT_CC1_UP_EVENT
*/
#define DL_TIMERH_EVENT_CC1_UP_EVENT               (DL_TIMER_EVENT_CC1_UP_EVENT)

/*!
* @brief Redirects to common @ref DL_TIMER_EVENT_OVERFLOW_EVENT
*/
#define DL_TIMERH_EVENT_OVERFLOW_EVENT           (DL_TIMER_EVENT_OVERFLOW_EVENT)

/*!
* @brief Redirects to common @ref DL_TIMER_EVENT_DC_EVENT
*/
#define DL_TIMERH_EVENT_DC_EVENT                       (DL_TIMER_EVENT_DC_EVENT)

/** @}*/

/** @addtogroup DL_TIMERH_IIDX
 *  @{
 */

/**
 * @brief Redirects to common @ref DL_TIMER_IIDX_ZERO
 */
#define DL_TIMERH_IIDX_ZERO                                   DL_TIMER_IIDX_ZERO

/**
 * @brief Redirects to common @ref DL_TIMER_IIDX_ZERO
 */
#define DL_TIMERH_IIDX_LOAD                                   DL_TIMER_IIDX_LOAD

/**
 * @brief Redirects to common @ref DL_TIMER_IIDX_ZERO
 */
#define DL_TIMERH_IIDX_CC0_DN                               DL_TIMER_IIDX_CC0_DN

/**
 * @brief Redirects to common @ref DL_TIMER_IIDX_ZERO
 */
#define DL_TIMERH_IIDX_CC1_DN                               DL_TIMER_IIDX_CC1_DN

/**
 * @brief Redirects to common @ref DL_TIMER_IIDX_ZERO
 */
#define DL_TIMERH_IIDX_CC0_UP                               DL_TIMER_IIDX_CC0_UP

/**
 * @brief Redirects to common @ref DL_TIMER_IIDX_ZERO
 */
#define DL_TIMERH_IIDX_CC1_UP                               DL_TIMER_IIDX_CC1_UP

/**
 * @brief Redirects to common @ref DL_TIMER_IIDX_ZERO
 */
#define DL_TIMERH_IIDX_OVERFLOW                           DL_TIMER_IIDX_OVERFLOW

/** @}*/

/** @addtogroup DL_TIMERH_PUBLISHER_INDEX
 *  @{
 */

/**
 * @brief Redirects to common @ref DL_TIMER_PUBLISHER_INDEX_0
 */
#define DL_TIMERH_PUBLISHER_INDEX_0                   DL_TIMER_PUBLISHER_INDEX_0

/**
 * @brief Redirects to common @ref DL_TIMER_PUBLISHER_INDEX_1
 */
#define DL_TIMERH_PUBLISHER_INDEX_1                   DL_TIMER_PUBLISHER_INDEX_1

/** @}*/

/** @addtogroup DL_TIMERH_SUBSCRIBER_INDEX
 *  @{
 */

/**
 * @brief Redirects to common @ref DL_TIMER_SUBSCRIBER_INDEX_0
 */
#define DL_TIMERH_SUBSCRIBER_INDEX_0                 DL_TIMER_SUBSCRIBER_INDEX_0
/**
 * @brief Redirects to common @ref DL_TIMER_SUBSCRIBER_INDEX_1
 */
#define DL_TIMERH_SUBSCRIBER_INDEX_1                 DL_TIMER_SUBSCRIBER_INDEX_1

/** @}*/

/** @addtogroup DL_TIMERH_EVENT_ROUTE
 *  @{
 */


/**
 * @brief Redirects to common @ref DL_TIMER_EVENT_ROUTE_1
 */
#define DL_TIMERH_EVENT_ROUTE_1                           DL_TIMER_EVENT_ROUTE_1

/**
 * @brief Redirects to common @ref DL_TIMER_EVENT_ROUTE_2
 */
#define DL_TIMERH_EVENT_ROUTE_2                           DL_TIMER_EVENT_ROUTE_2

/** @}*/


/**
 * @brief Redirects to common @ref DL_Timer_enablePower
 */
#define DL_TimerH_enablePower                               DL_Timer_enablePower

/**
 * @brief Redirects to common @ref DL_Timer_disablePower
 */
#define DL_TimerH_disablePower                             DL_Timer_disablePower

/**
 * @brief Redirects to common @ref DL_Timer_isPowerEnabled
 */
#define DL_TimerH_isPowerEnabled                         DL_Timer_isPowerEnabled

/**
 * @brief Redirects to common @ref DL_Timer_reset
 */
#define DL_TimerH_reset                                           DL_Timer_reset

/**
 * @brief Redirects to common @ref DL_Timer_isReset
 */
#define DL_TimerH_isReset                                       DL_Timer_isReset

/*!
 * @brief Redirects to common @ref DL_Timer_setCCPDirection
 */
#define DL_TimerH_setCCPDirection                       DL_Timer_setCCPDirection

/*!
 * @brief Redirects to common @ref DL_Timer_getCCPDirection
 */
#define DL_TimerH_getCCPDirection                       DL_Timer_getCCPDirection

/*!
 * @brief Redirects to common @ref DL_Timer_setCCPOutputDisabled
 */
#define DL_TimerH_setCCPOutputDisabled             DL_Timer_setCCPOutputDisabled

/*!
 * @brief Redirects to common @ref DL_Timer_getClockConfig
 */
#define DL_TimerH_enableClock                               DL_Timer_enableClock

/*!
 * @brief Redirects to common @ref DL_Timer_getClockConfig
 */
#define DL_TimerH_disableClock                             DL_Timer_disableClock

/*!
 * @brief Redirects to common @ref DL_Timer_isClockEnabled
 */
#define DL_TimerH_isClockEnabled                         DL_Timer_isClockEnabled

/*!
 * @brief Redirects to common @ref DL_Timer_configCrossTrigger
 */
#define DL_TimerH_configCrossTrigger                 DL_Timer_configCrossTrigger

/*!
 * @brief Redirects to common @ref DL_Timer_configCrossTriggerSubscriber
 */
#define DL_TimerH_configCrossTriggerSubscriber \
                                           DL_Timer_configCrossTriggerSubscriber

/*!
 * @brief Redirects to common @ref DL_Timer_configCrossTriggerInputCond
 */
#define DL_TimerH_configCrossTriggerInputCond \
                                            DL_Timer_configCrossTriggerInputCond

/*!
 * @brief Redirects to common @ref DL_Timer_configCrossTriggerEnable
 */
#define DL_TimerH_configCrossTriggerEnable     DL_Timer_configCrossTriggerEnable

/*!
 * @brief Redirects to common @ref DL_Timer_getCrossTriggerConfig
 */
#define DL_TimerH_getCrossTriggerConfig           DL_Timer_getCrossTriggerConfig

/*!
 * @brief Redirects to common @ref DL_Timer_getCrossTriggerSubscriber
 */
#define DL_TimerH_getCrossTriggerSubscriber   DL_Timer_getCrossTriggerSubscriber

/*!
 * @brief Redirects to common @ref DL_Timer_getCrossTriggerInputCond
 */
#define DL_TimerH_getCrossTriggerInputCond     DL_Timer_getCrossTriggerInputCond

/*!
 * @brief Redirects to common @ref DL_Timer_getCrossTriggerEnable
 */
#define DL_TimerH_getCrossTriggerEnable           DL_Timer_getCrossTriggerEnable

/*!
 * @brief Redirects to common @ref DL_Timer_getCrossTriggerInstance
 */
#define DL_TimerH_getCrossTriggerInstance       DL_Timer_getCrossTriggerInstance

/*!
 * @brief Redirects to common @ref DL_Timer_setCrossTriggerInstance
 */
#define DL_TimerH_setCrossTriggerInstance       DL_Timer_setCrossTriggerInstance

/*!
 * @brief Redirects to common @ref DL_Timer_generateCrossTrigger
 */
#define DL_TimerH_generateCrossTrigger             DL_Timer_generateCrossTrigger

/*!
 * @brief Redirects to common @ref DL_Timer_resetCounterMode
 */
#define DL_TimerH_resetCounterMode                     DL_Timer_resetCounterMode

/*!
 * @brief Redirects to common @ref DL_Timer_setDebugReleaseBehavior
 */
#define DL_TimerH_setDebugReleaseBehavior       DL_Timer_setDebugReleaseBehavior

/*!
 * @brief Redirects to common @ref DL_Timer_getDebugReleaseBehavior
 */
#define DL_TimerH_getDebugReleaseBehavior       DL_Timer_getDebugReleaseBehavior

/*!
 * @brief Redirects to common @ref DL_Timer_setCounterControl
 */
#define DL_TimerH_setCounterControl                   DL_Timer_setCounterControl

/*!
 * @brief Redirects to common @ref DL_Timer_getCounterZeroControl
 */
#define DL_TimerH_getCounterZeroControl           DL_Timer_getCounterZeroControl

/*!
 * @brief Redirects to common @ref DL_Timer_getCounterAdvanceControl
 */
#define DL_TimerH_getCounterAdvanceControl     DL_Timer_getCounterAdvanceControl

/*!
 * @brief Redirects to common @ref DL_Timer_getCounterLoadControl
 */
#define DL_TimerH_getCounterLoadControl           DL_Timer_getCounterLoadControl

/*!
 * @brief Redirects to common @ref DL_Timer_setCounterValueAfterEnable
 */
#define DL_TimerH_setCounterValueAfterEnable DL_Timer_setCounterValueAfterEnable

/*!
 * @brief Redirects to common @ref DL_Timer_getCounterValueAfterEnable
 */
#define DL_TimerH_getCounterValueAfterEnable DL_Timer_getCounterValueAfterEnable

/*!
 * @brief Redirects to common @ref DL_Timer_setCounterRepeatMode
 */
#define DL_TimerH_setCounterRepeatMode             DL_Timer_setCounterRepeatMode

/*!
 * @brief Redirects to common @ref DL_Timer_getCounterRepeatMode
 */
#define DL_TimerH_getCounterRepeatMode             DL_Timer_getCounterRepeatMode

/*!
 * @brief Redirects to common @ref DL_Timer_setCaptureCompareValuePipeline
 */
#define DL_TimerH_setCaptureCompareValuePipeline \
                                         DL_Timer_setCaptureCompareValuePipeline

/*!
 * @brief Redirects to common @ref DL_Timer_getCaptureCompareValue
 */
#define DL_TimerH_getCaptureCompareValue         DL_Timer_getCaptureCompareValue

/*!
 * @brief Redirects to common @ref DL_Timer_setCaptureCompareCtl
 */
#define DL_TimerH_setCaptureCompareCtl             DL_Timer_setCaptureCompareCtl

/*!
 * @brief Redirects to common @ref DL_Timer_getCaptureCompareCtl
 */
#define DL_TimerH_getCaptureCompareCtl             DL_Timer_getCaptureCompareCtl

/*!
 * @brief Redirects to common @ref DL_Timer_setCaptureCompareOutCtl
 */
#define DL_TimerH_setCaptureCompareOutCtl       DL_Timer_setCaptureCompareOutCtl

/*!
 * @brief Redirects to common @ref DL_Timer_getCaptureCompareOutCtl
 */
#define DL_TimerH_getCaptureCompareOutCtl       DL_Timer_getCaptureCompareOutCtl

/*!
 * @brief Redirects to common @ref DL_Timer_setCaptureCompareAction
 */
#define DL_TimerH_setCaptureCompareAction       DL_Timer_setCaptureCompareAction

/*!
 * @brief Redirects to common @ref DL_Timer_getCaptureCompareAction
 */
#define DL_TimerH_getCaptureCompareAction       DL_Timer_getCaptureCompareAction

/*!
 * @brief Redirects to common @ref DL_Timer_setCaptureCompareInput
 */
#define DL_TimerH_setCaptureCompareInput \
                                           DL_Timer_setCaptureCompareInput

/*!
 * @brief Redirects to common @ref DL_Timer_getCaptureCompareInput
 */
#define DL_TimerH_getCaptureCompareInput \
                                           DL_Timer_getCaptureCompareInput

/*!
 * @brief Redirects to common @ref DL_Timer_setCaptureCompareInputFilter
 */
#define DL_TimerH_setCaptureCompareInputFilter \
                                           DL_Timer_setCaptureCompareInputFilter

/*!
 * @brief Redirects to common @ref DL_Timer_getCaptureCompareInputFilter
 */
#define DL_TimerH_getCaptureCompareInputFilter \
                                           DL_Timer_getCaptureCompareInputFilter

/*!
 * @brief Redirects to common @ref DL_Timer_enableCaptureCompareInputFilter
 */
#define DL_TimerH_enableCaptureCompareInputFilter \
                                        DL_Timer_enableCaptureCompareInputFilter

/*!
 * @brief Redirects to common @ref DL_Timer_disableCaptureCompareInputFilter
 */
#define DL_TimerH_disableCaptureCompareInputFilter \
                                       DL_Timer_disableCaptureCompareInputFilter

/*!
 * @brief Redirects to common @ref DL_Timer_isCaptureCompareInputFilterEnabled
 */
#define DL_TimerH_isCaptureCompareInputFilterEnabled \
                                     DL_Timer_isCaptureCompareInputFilterEnabled


/*!
 * @brief Redirects to common @ref DL_Timer_setExternalTriggerEvent
 */
#define DL_TimerH_setExternalTriggerEvent       DL_Timer_setExternalTriggerEvent

/*!
 * @brief Redirects to common @ref DL_Timer_getExternalTriggerEvent
 */
#define DL_TimerH_getExternalTriggerEvent       DL_Timer_getExternalTriggerEvent

/*!
 * @brief Redirects to common @ref DL_Timer_enableExternalTrigger
 */
#define DL_TimerH_enableExternalTrigger           DL_Timer_enableExternalTrigger

/*!
 * @brief Redirects to common @ref DL_Timer_disableExternalTrigger
 */
#define DL_TimerH_disableExternalTrigger         DL_Timer_disableExternalTrigger

/*!
 * @brief Redirects to common @ref DL_Timer_isExternalTriggerEnabled
 */
#define DL_TimerH_isExternalTriggerEnabled     DL_Timer_isExternalTriggerEnabled

/*!
 * @brief Redirects to common @ref DL_Timer_startCounter
 */
#define DL_TimerH_startCounter                             DL_Timer_startCounter

/*!
 * @brief Redirects to common @ref DL_Timer_stopCounter
 */
#define DL_TimerH_stopCounter                               DL_Timer_stopCounter

/*!
 * @brief Redirects to common @ref DL_Timer_setTimerCount
 */
#define DL_TimerH_setTimerCount                           DL_Timer_setTimerCount

/*!
 * @brief Redirects to common @ref DL_Timer_isRunning
 */
#define DL_TimerH_isRunning                                   DL_Timer_isRunning

/*!
 * @brief Redirects to common @ref DL_Timer_enableInterrupt
 */
#define DL_TimerH_enableInterrupt                       DL_Timer_enableInterrupt

/*!
 * @brief Redirects to common @ref DL_Timer_disableInterrupt
 */
#define DL_TimerH_disableInterrupt                     DL_Timer_disableInterrupt

/*!
 * @brief Redirects to common @ref DL_Timer_getEnabledInterrupts
 */
#define DL_TimerH_getEnabledInterrupts             DL_Timer_getEnabledInterrupts

/*!
 * @brief Redirects to common @ref DL_Timer_getEnabledInterruptStatus
 */
#define DL_TimerH_getEnabledInterruptStatus   DL_Timer_getEnabledInterruptStatus

/*!
 * @brief Redirects to common @ref DL_Timer_getRawInterruptStatus
 */
#define DL_TimerH_getRawInterruptStatus           DL_Timer_getRawInterruptStatus

/*!
 * @brief Redirects to common @ref DL_Timer_getPendingInterrupt
 */
#define DL_TimerH_getPendingInterrupt               DL_Timer_getPendingInterrupt

/*!
 * @brief Redirects to common @ref DL_Timer_clearInterruptStatus
 */
#define DL_TimerH_clearInterruptStatus             DL_Timer_clearInterruptStatus

/*!
 * @brief Redirects to common @ref DL_Timer_configPipelineUpdate
 */
#define DL_TimerH_configPipelineUpdate             DL_Timer_configPipelineUpdate

/*!
 * @brief Redirects to common @ref DL_Timer_getPipelineUpdateMode
 */
#define DL_TimerH_getPipelinUpdateMode            DL_Timer_getPipelineUpdateMode

/*!
 * @brief Redirects to common @ref DL_Timer_setPublisherChanID
 */
#define DL_TimerH_setPublisherChanID                 DL_Timer_setPublisherChanID

/*!
 * @brief Redirects to common @ref DL_Timer_getPublisherChanID
 */
#define DL_TimerH_getPublisherChanID                 DL_Timer_getPublisherChanID

/*!
 * @brief Redirects to common @ref DL_Timer_setSubscriberChanID
 */
#define DL_TimerH_setSubscriberChanID               DL_Timer_setSubscriberChanID

/*!
 * @brief Redirects to common @ref DL_Timer_getSubscriberChanID
 */
#define DL_TimerH_getSubscriberChanID               DL_Timer_getSubscriberChanID

/*!
 * @brief Redirects to common @ref DL_Timer_enableEvent
 */
#define DL_TimerH_enableEvent                               DL_Timer_enableEvent

/*!
 * @brief Redirects to common @ref DL_Timer_disableEvent
 */
#define DL_TimerH_disableEvent                             DL_Timer_disableEvent

/*!
 * @brief Redirects to common @ref DL_Timer_getEnabledEvents
 */
#define DL_TimerH_getEnabledEvents                     DL_Timer_getEnabledEvents

/*!
 * @brief Redirects to common @ref DL_Timer_getEnabledEventStatus
 */
#define DL_TimerH_getEnabledEventStatus           DL_Timer_getEnabledEventStatus

/*!
 * @brief Redirects to common @ref DL_Timer_getRawEventsStatus
 */
#define DL_TimerH_getRawEventsStatus                 DL_Timer_getRawEventsStatus

/*!
 * @brief Redirects to common @ref DL_Timer_clearInterruptStatus
 */
#define DL_TimerH_clearInterruptStatus             DL_Timer_clearInterruptStatus

/*!
 * @brief Redirects to common @ref DL_Timer_getLoadValue
 */
#define DL_TimerH_getLoadValue                             DL_Timer_getLoadValue

/*!
 * @brief Redirects to common @ref DL_Timer_getTimerCount
 */
#define DL_TimerH_getTimerCount                           DL_Timer_getTimerCount

/*!
 * @brief Redirects to common @ref DL_Timer_saveConfiguration
 */
#define DL_TimerH_saveConfiguration                   DL_Timer_saveConfiguration

/*!
 * @brief Redirects to common @ref DL_Timer_restoreConfiguration
 */
#define DL_TimerH_restoreConfiguration             DL_Timer_restoreConfiguration

/* clang-format on */

/**
 * @brief Redirects to common @ref DL_Timer_backupConfig
 *
 */
typedef DL_Timer_backupConfig DL_TimerH_backupConfig;

/*!
 *  @brief  Configuration struct for @ref DL_TimerH_setClockConfig.
 */
typedef struct {
    /*! Selects timer module clock source @ref DL_TIMER_CLOCK*/
    DL_TIMER_CLOCK clockSel;
    /*! Selects the timer module clock divide ratio
    @ref DL_TIMER_CLOCK_DIVIDE */
    DL_TIMER_CLOCK_DIVIDE divideRatio;
} DL_TimerH_ClockConfig;

/*!
 *  @brief  Configuration struct for @ref DL_TimerH_initTimerMode.
 */
typedef struct {
    /*! One shot or Periodic mode configuration. One of
     * @ref DL_TIMER_TIMER_MODE */
    DL_TIMER_TIMER_MODE timerMode;
    /*! Actual period will be period_actual=(period +1)T_TIMCLK
     * where T_TIMCLK is the period of the timer source clock. */
    uint32_t period;
    /*! Start timer after configuration @ref DL_TIMER */
    DL_TIMER startTimer;
    /*! Generate intermediate counter interrupt
     * @ref DL_TIMER_INTERM_INT*/
    DL_TIMER_INTERM_INT genIntermInt;
    /*! Counter value when intermediate interrupt should be generated. This
     * member must be set to 0 when
     * @ref genIntermInt == DL_TIMER_INTERM_INT_DISABLED */
    uint32_t counterVal;
} DL_TimerH_TimerConfig;

/*!
 * @brief Configuration struct for @ref DL_TimerH_initCaptureMode.
 */
typedef struct {
    /*! Selects capture mode. One of @ref DL_TIMER_CAPTURE_MODE */
    DL_TIMER_CAPTURE_MODE captureMode;
    /*! Specifies the capture period. period_actual=(period +1)T_TIMCLK
     * where T_TIMCLK is the period of the timer source clock. */
    uint32_t period;
    /*! Start timer after configuration @ref DL_TIMER */
    DL_TIMER startTimer;
    /*! In edge count mode this parameter specified the time of edge detection
     *  mode. One of @ref DL_TIMER_CAPTURE_EDGE_DETECTION_MODE*/
    DL_TIMER_CAPTURE_EDGE_DETECTION_MODE edgeCaptMode;
    /*! Selects input channel used for capture mode */
    DL_TIMER_INPUT_CHAN inputChan;
    /*! In edge count mode this parameter specified the time of edge detection
     *  mode. One of @ref DL_TIMER_CC_INPUT_INV */
    uint32_t inputInvMode;
} DL_TimerH_CaptureConfig;

/*!
 * @brief Configuration struct for @ref DL_TimerH_initCaptureTriggerMode.
 */
typedef struct {
    /*! Selects capture mode. One of @ref DL_TIMER_CAPTURE_MODE */
    DL_TIMER_CAPTURE_MODE captureMode;
    /*! Specifies the capture period. period_actual=(period +1)T_TIMCLK
     * where T_TIMCLK is the period of the timer source clock. */
    uint32_t period;
    /*! Start timer after configuration @ref DL_TIMER */
    DL_TIMER startTimer;
} DL_TimerH_CaptureTriggerConfig;

/*!
 * @brief Configuration struct for @ref DL_TimerH_initCaptureCombinedMode.
 */
typedef struct {
    /*! Specifies the capture period. period_actual=(period +1)T_TIMCLK
     * where T_TIMCLK is the period of the timer source clock. */
    uint32_t period;
    /*! Start timer after configuration @ref DL_TIMER */
    DL_TIMER startTimer;
    /*! Specifies the edge detection mode. One of
        @ref DL_TIMER_CAPTURE_EDGE_DETECTION_MODE*/
    DL_TIMER_CAPTURE_EDGE_DETECTION_MODE edgeCaptMode;
    /*! Selects input channel used for capture mode */
    DL_TIMER_INPUT_CHAN inputChan;
    /*! In edge count mode this parameter specified the time of edge detection
     *  mode. One of @ref DL_TIMER_CC_INPUT_INV */
    uint32_t inputInvMode;
} DL_TimerH_CaptureCombinedConfig;

/*!
 * @brief Configuration struct for @ref DL_TimerH_initCompareMode.
 */
typedef struct {
    /*! Selects capture mode. One of @ref DL_TIMER_COMPARE_MODE */
    DL_TIMER_COMPARE_MODE compareMode;
    /*! Specifies the intial count value. In edge count mode this value will
     *  decrease everytime the configured edge detection mode is detected.
     *  The counter will be reset to this value after the reaching zero.
     *  Valid range [0-4294967295]. */
    uint32_t count;
    /*! Specifies the edge detection mode. One of
        @ref DL_TIMER_CAPTURE_EDGE_DETECTION_MODE*/
    DL_TIMER_COMPARE_EDGE_DETECTION_MODE edgeDetectMode;
    /*! Selects input channel used for compare mode */
    DL_TIMER_INPUT_CHAN inputChan;
    /*! Specifies if input should be inverted. One of
     *  @ref DL_TIMER_CC_INPUT_INV */
    uint32_t inputInvMode;
    /*! Start timer after configuration @ref DL_TIMER */
    DL_TIMER startTimer;
} DL_TimerH_CompareConfig;

/*!
 * @brief Configuration struct for @ref DL_TimerH_initCompareTriggerMode.
 */
typedef struct {
    /*! Selects capture mode. One of @ref DL_TIMER_COMPARE_MODE */
    DL_TIMER_COMPARE_MODE compareMode;
    /*! Specifies the intial count value. In edge count mode this value will
     *  decrease everytime the configured edge detection mode is detected.
     *  The counter will be reset to this value after the reaching zero.
     *  Valid range [0-4294967295]. */
    uint32_t count;
    /*! Specifies the edge detection mode. One of
        @ref DL_TIMER_CAPTURE_EDGE_DETECTION_MODE*/
    DL_TIMER_COMPARE_EDGE_DETECTION_MODE edgeDetectMode;
    /*! Start timer after configuration @ref DL_TIMER */
    DL_TIMER startTimer;
} DL_TimerH_CompareTriggerConfig;

/**
 * @brief Configuration struct for @ref DL_TimerH_initPWMMode.
 */
typedef struct {
    /*! Specifies the pwm period.*/
    uint32_t period;
    /*! Specifies the PWM Mode. One of @ref DL_TIMER_PWM_MODE */
    DL_TIMER_PWM_MODE pwmMode;
    /*! Start timer after configuration @ref DL_TIMER */
    DL_TIMER startTimer;
} DL_TimerH_PWMConfig;

/**
 *  @brief      Configure timer H source clock
 *  @param[in]  gptimer        Pointer to the register overlay for the
 *                             peripheral
 *  @param[in]  config         Pointer to the clock configuration struct
 *                             @ref DL_TimerH_ClockConfig.
 */
void DL_TimerH_setClockConfig(
    GPTIMER_Regs *gptimer, DL_TimerH_ClockConfig *config);

/**
 *  @brief      Get  timer H source clock configuration
 *  @param[in]  gptimer        Pointer to the register overlay for the
 *                             peripheral
 *  @param[in]  config         Pointer to the clock configuration struct
 *                             @ref DL_TimerH_ClockConfig.
 */
void DL_TimerH_getClockConfig(
    GPTIMER_Regs *gptimer, DL_TimerH_ClockConfig *config);

/**
 *  @brief       Sets timer load register value
 *
 *  @param[in]  gptimer      Pointer to the register overlay for the
 *                           peripheral
 *  @param[in]  value        Value to be loaded to timer counter
 *                           register. Valid Range 0-1048575
 *
 */
__STATIC_INLINE void DL_TimerH_setLoadValue(
    GPTIMER_Regs *gptimer, uint32_t value)
{
    gptimer->COUNTERREGS.LOAD = (value & GPTIMER_LOAD_LD_MAXIMUM);
}

/**
 *  @brief      Sets Timer Capture Compare Value
 *
 *  @param[in]  gptimer        Pointer to the register overlay for the
 *                             peripheral
 *  @param[in]  value          Value to write to capture compare register
 *  @param[in]  ccIndex        Index associated to capture compare register
 *                             @ref DL_TIMER_CC_INDEX.
 *
 */
void DL_TimerH_setCaptureCompareValue(
    GPTIMER_Regs *gptimer, uint32_t value, DL_TIMER_CC_INDEX ccIndex);

/**
 *  @brief      Configure timer in one shot or periodic timer mode
 *
 *  @param[in]  gptimer        Pointer to the register overlay for the
 *                             peripheral
 *  @param[in]  config         Pointer to the mode configuration struct
 *                             @ref DL_TimerH_TimerConfig.
 *
 */
void DL_TimerH_initTimerMode(
    GPTIMER_Regs *gptimer, DL_TimerH_TimerConfig *config);

/**
 *  @brief      Configure timer in edge count, period capture, edge time or
 *              pulse-width capture mode
 *
 *  @param[in]  gptimer        Pointer to the register overlay for the
 *                             peripheral
 *  @param[in]  config         Pointer to the mode configuration struct
 *                             @ref DL_TimerH_CaptureConfig.
 *
 */
void DL_TimerH_initCaptureMode(
    GPTIMER_Regs *gptimer, DL_TimerH_CaptureConfig *config);

/**
 *  @brief      Configure timer in edge count, period capture, edge time or
 *              pulse-width capture mode using the trigger as input source
 *
 *  @param[in]  gptimer        Pointer to the register overlay for the
 *                             peripheral
 *  @param[in]  config         Pointer to the mode configuration struct
 *                             @ref DL_TimerH_CaptureTriggerConfig.
 *
 */
void DL_TimerH_initCaptureTriggerMode(
    GPTIMER_Regs *gptimer, DL_TimerH_CaptureTriggerConfig *config);

/**
 *  @brief      Configure timer in combined pulse-width and period capture
 *
 *  @param[in]  gptimer        Pointer to the register overlay for the
 *                             peripheral
 *  @param[in]  config         Pointer to the mode configuration struct
 *                             @ref DL_TimerH_CaptureCombinedConfig.
 *
 */
void DL_TimerH_initCaptureCombinedMode(
    GPTIMER_Regs *gptimer, DL_TimerH_CaptureCombinedConfig *config);

/**
 *  @brief      Configure timer in edge count compare mode
 *
 *  @param[in]  gptimer        Pointer to the register overlay for the
 *                             peripheral
 *  @param[in]  config         Pointer to the mode configuration struct
 *                             @ref DL_TimerH_CompareConfig.
 *
 */
void DL_TimerH_initCompareMode(
    GPTIMER_Regs *gptimer, DL_TimerH_CompareConfig *config);

/**
 *  @brief      Configure timer in edge count compare mode using the trigger as
 *              input source
 *
 *  @param[in]  gptimer        Pointer to the register overlay for the
 *                             peripheral
 *  @param[in]  config         Pointer to the mode configuration struct
 *                             @ref DL_TimerH_CompareTriggerConfig.
 *
 */
void DL_TimerH_initCompareTriggerMode(
    GPTIMER_Regs *gptimer, DL_TimerH_CompareTriggerConfig *config);

/**
 * @brief Configure timer in Pulse Width Modulation Mode
 *
 * @param gptimer              Pointer to the register overlay for the
 *                             peripheral
 * @param config               Pointer to the mode configuration struct
 *                             @ref DL_TimerH_PWMConfig.
 */
void DL_TimerH_initPWMMode(GPTIMER_Regs *gptimer, DL_TimerH_PWMConfig *config);

#ifdef __cplusplus
}
#endif

#endif /* __MSPM0_HAS_TIMER_H__ */

#endif /* ti_dl_dl_timerh__include */
/** @}*/
