/*
 * -------------------------------------------
 *    MSP432 DriverLib - v4_00_00_11 
 * -------------------------------------------
 *
 * --COPYRIGHT--,BSD,BSD
 * Copyright (c) 2017, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/

/* DriverLib Includes */
#include <ti/devices/msp432p4xx/driverlib/driverlib.h>

/* Custom includes */
#include <System/cs.h>
#include <System/gpio.h>
#include <System/helpers.h>
#include <stdint.h>
#include <stdbool.h>

void CY15B104Q_read_StatusRegister(unsigned char* readData);
unsigned char* CY15B104Q_writeAndReadBytes(int txData, unsigned char txBytesNumber, unsigned char rxBytesNumber);
void CY15B104Q_writeInt(int data, unsigned char bytesNumber);
unsigned char* CY15B104Q_readBytes(unsigned char bytesNumber);

#define CY15B104Q_ASSERT_CS()     (P2OUT &= ~BIT0)  // !CS = 0
#define CY15B104Q_DEASSERT_CS()   (P2OUT |= BIT0)   // !CS = 1
#define CY15B104Q_ASSERT_WP()     (P2OUT &= ~BIT4)  // !FRAM_WP = 0
#define CY15B104Q_DEASSERT_WP()   (P2OUT |= BIT4)   // !FRAM_WP = 1
#define CY15B104Q_POWER_ON()      (P2OUT &= ~BIT6)  // !FRAM_EN = 0
#define CY15B104Q_POWER_OFF()     (P2OUT |= BIT6)   // !FRAM_EN = 1

#define CY15B104Q_MAX_RX_BYTES  10

const eUSCI_SPI_MasterConfig framSpiConfig =
{
        EUSCI_A_SPI_CLOCKSOURCE_SMCLK,             // SMCLK clock source
        24000000,                                  // SMCLK = 24 MHz
        1000000,                                   // SPICLK = 1 MHz (maximum 25 MHz)
        EUSCI_A_SPI_MSB_FIRST,                     // MSB First
        EUSCI_A_SPI_PHASE_DATA_CHANGED_ONFIRST_CAPTURED_ON_NEXT, // Phase polarity (CPHA = 0); SPI mode 0
        EUSCI_B_SPI_CLOCKPOLARITY_INACTIVITY_HIGH,  // Low polarity (CPOL = 0); SPI mode 0
        EUSCI_A_SPI_3PIN                           // 3 wire SPI mode
};
unsigned char rxData[10];
volatile bool framTxFlag = false, framRxFlag = false;

int main(void)
{
    // Enabling the FPU with stacking enabled for floating point calculations
    MAP_FPU_enableModule();
    MAP_FPU_enableLazyStacking();

    GPIO_init();
    Clock_System_init();

    MAP_Interrupt_enableSleepOnIsrExit();
    MAP_Interrupt_enableMaster(); // Enabling MASTER interrupts

    // Configuring GPIO
    MAP_GPIO_setAsOutputPin(GPIO_PORT_P2, GPIO_PIN6); // !FRAM_EN
    MAP_GPIO_setAsOutputPin(GPIO_PORT_P2, GPIO_PIN4); // !FRAM_WP
    MAP_GPIO_setAsOutputPin(GPIO_PORT_P2, GPIO_PIN5); // !FRAM_HOLD
    MAP_GPIO_setAsOutputPin(GPIO_PORT_P2, GPIO_PIN0); // !SPI_A1_STE

    // Setting GPIO states
    MAP_GPIO_setOutputHighOnPin(GPIO_PORT_P2, GPIO_PIN0); // !SPI_A1_STE = 1
    MAP_GPIO_setOutputHighOnPin(GPIO_PORT_P2, GPIO_PIN6); // !FRAM_EN = 1; deactivate FRAM
    MAP_GPIO_setOutputLowOnPin(GPIO_PORT_P2, GPIO_PIN4); // !FRAM_WP = 0; activate WP
    MAP_GPIO_setOutputHighOnPin(GPIO_PORT_P2, GPIO_PIN5); // !FRAM_HOLD = 1; deactivate HOLD

    // Configuring SPI pins
    MAP_GPIO_setAsPeripheralModuleFunctionOutputPin(GPIO_PORT_P2, GPIO_PIN1, GPIO_PRIMARY_MODULE_FUNCTION); // SPI_A1_CLK
    MAP_GPIO_setAsPeripheralModuleFunctionOutputPin(GPIO_PORT_P2, GPIO_PIN3, GPIO_PRIMARY_MODULE_FUNCTION); // SPI_A1_MOSI
    MAP_GPIO_setAsPeripheralModuleFunctionOutputPin(GPIO_PORT_P2, GPIO_PIN2, GPIO_PRIMARY_MODULE_FUNCTION); // SPI_A1_MISO

    // Configuring SPI peripheral
    MAP_SPI_initMaster(EUSCI_A1_BASE, &framSpiConfig);
    MAP_SPI_enableModule(EUSCI_A1_BASE);
    MAP_SPI_enableInterrupt(EUSCI_A1_BASE, EUSCI_A_SPI_TRANSMIT_INTERRUPT);
    MAP_SPI_enableInterrupt(EUSCI_A1_BASE, EUSCI_A_SPI_RECEIVE_INTERRUPT);
    MAP_Interrupt_enableInterrupt(INT_EUSCIA1);

    CY15B104Q_POWER_ON();

    unsigned char value = 0;
    CY15B104Q_read_StatusRegister(&value);

    CY15B104Q_POWER_OFF();

    while(1)
    {
        MAP_PCM_gotoLPM0InterruptSafe();
        MAP_Interrupt_enableMaster();
    }
}

void CY15B104Q_read_StatusRegister(unsigned char* readData)
{
    unsigned char *rxData;
    rxData = CY15B104Q_writeAndReadBytes(0x05, 1, 1);

    *readData = *rxData;
}

unsigned char* CY15B104Q_writeAndReadBytes(int txData, unsigned char txBytesNumber, unsigned char rxBytesNumber)
{
    CY15B104Q_ASSERT_CS(); // !SPI_A1_STE = 0

    CY15B104Q_writeInt(txData, txBytesNumber);

    unsigned char *rx;
    rx = CY15B104Q_readBytes(rxBytesNumber);

    CY15B104Q_DEASSERT_CS(); // !SPI_A1_STE = 1

    return rxData;
}

void CY15B104Q_writeInt(int data, unsigned char bytesNumber)
{
    int byte = 0;
    for(byte = 0; byte < bytesNumber; byte++)
    {
        // Polling to see if the TX buffer is ready
        //while (!(MAP_SPI_getInterruptStatus(EUSCI_A1_BASE,EUSCI_A_SPI_TRANSMIT_INTERRUPT)));

        while (framTxFlag == false)
        {
            MAP_PCM_gotoLPM0InterruptSafe();
            MAP_Interrupt_enableMaster();
        }
        framTxFlag = false;

        // Transmitting data to slave
        MAP_SPI_transmitData(EUSCI_A1_BASE, GetByte(data, byte));
    }
}

unsigned char* CY15B104Q_readBytes(unsigned char bytesNumber)
{
    int byte = 0, count = bytesNumber;
    if (bytesNumber > CY15B104Q_MAX_RX_BYTES) count = CY15B104Q_MAX_RX_BYTES;
    for(byte = 0; byte < count; byte++)
    {
        while (framTxFlag == false)
        {
           MAP_PCM_gotoLPM0InterruptSafe();
           MAP_Interrupt_enableMaster();
        }
        framTxFlag = false;

        // Transmitting data to slave
        MAP_SPI_transmitData(EUSCI_A1_BASE, 0x00); // Dummy write to clock out data; genereaza clock-ul necesar pentru citirea datelor

        while (framRxFlag == false)
        {
            MAP_PCM_gotoLPM0InterruptSafe();
            MAP_Interrupt_enableMaster();
        }
        framRxFlag = false;

        int jj;
        for (jj=0;jj<100;jj++);
        rxData[byte] = MAP_SPI_receiveData(EUSCI_A1_BASE);
    }
    return rxData;
}
