//******************************************************************************
// FILENAME: lcdDriver.h
//
// DESCRIPTION: lcd class defines,
//
// AUTHOR: Eric Lovegren
//
// CREATION DATE: 9/19/2013
//
// Copyright (c) 2013 by NewGate Instrument
//
//*****************************************************************************
#ifndef LCD_DRIVER_H
#define LCD_DRIVER_H

// annunciator bits
#define ANNUNC_ERROR         0x01
#define ANNUNC_SQRT          0x02
#define ANNUNC_PRCNT         0x04
#define ANNUNC_ZERO_PRCNT    0x08
#define ANNUNC_FIFTY_PRCNT   0x10
#define ANNUNC_HUNDRED_PRCNT 0x20
#define ANNUNC_MINUS         0x40
#define ANNUNC_RESERVED2     0x80
#define ANNUNC_PRCNT_BIT2    0x04

// decimal bits
#define DEC_ONE_BIT0         0x01  // The first decimal point
#define DEC_TWO_BIT1         0x02  // starting from the right of
#define DEC_THREE_BIT2       0x04  // the display is number ONE
#define DEC_FOUR_BIT3        0x08
#define DEC_FIVE_BIT4        0x10
#define DEC_SIX_BIT5         0x20
#define DEC_NEG_SIGN_BIT6    0x40
#define DEC_RESERVED_BIT7    0x80

// local definitions
#define NUM_TOP_LINE_CHARS      7
#define NUM_BOTTOM_LINE_CHARS   6
#define NUM_NUMERIC_LINE_CHARS  7
#define LCD_DRIVER_RAM_SIZE     40 // 3 line display RAM buffer size
#define PCF8576_BUFF_SIZE       LCD_DRIVER_RAM_SIZE >> 1
#define NUM_USER_DEF_LCD        9

// Character positions
enum
{
  CHAR1 = 0, // 0
  CHAR2, // 1
  CHAR3, // 2
  CHAR4, // 3
  CHAR5, // 4
  CHAR6, // 5
  CHAR7, // 6
};

enum
{
  NUM1 = 0, // 0
  NUM2, // 1
  NUM3, // 2
  NUM4, // 3
  NUM5, // 4
  NUM6, // 5
  NUM7, // 6
};

// state of sending data to two PCF8576 LCD driver
enum
{
  CH1_STOP,           //0
  CH2_STOP,           //1
  CH1_SET_DATA,       //2
  CH2_SET_DATA,       //3
  CH1_MODE_CHG,       //4
  CH2_MODE_CHG,       //5
  LCD_HOLD_TIME_ON,   //6
};
// Misc
#define FIVE_PERCENT           (tU8)5    // 5% increments for bar graph
#define BG_INDEX_MULTIPLIER    (tU8)8    // Offset into bar graph table

// Control Bytes for IIC Drivers
#define IIC_SLAVE_ADRS          0x70 >> 1 // use MSP standard I2C function
#define IIC_MODE_DISABLE_OUTPUT 0xC0
#define IIC_MODE_SET_14         0xC0        //disable output...to cause a blinking effect
#define IIC_MODE_SET_14E        0xC8    //re-enable output...to cause a blinking effect
#define IIC_MODE_SET_13         0xDB  //0xCB        //re-enable output...to cause a blinking effect
//unused #define IIC_MODE_SET_12         0xCE
#define IIC_DATA_PTR            0x80
#define IIC_SELECT_DEV0         0xE0
#define IIC_SELECT_DEV1         0xE1
#define IIC_BANK_SELECT         0xF8
#define IIC_BLINK               0x70

#define IIC_SDA                 BIT1
#define IIC_SCL                 BIT2
#define IIC_PILSE_WIDTH         __delay_cycles(25)
#define IIC_START_PILSE         __delay_cycles(2)
#define I2C_SET_DATA_RATE_30KBPS             30000
#define I2C_SET_DATA_RATE_21KBPS             21000
#define I2C_SET_DATA_RATE_1KBPS               1000

typedef struct
{
   tU8 topLine[NUM_TOP_LINE_CHARS];
   tU8 bottomLine[NUM_BOTTOM_LINE_CHARS];
   tU8 numLine[NUM_NUMERIC_LINE_CHARS];
   tU8 annunc;
   tU8 decimals;
   tU8 bargraph;
   tU8 u8LcdIdx;
   tU8 u8IntDate;  // put the " " after the dicimal point, when it is 1
   tU8 u8evenAddr;  //unused
} tDisplayData;

#define EMV_NUM_CHARS   42
#define EMV_NUM_NUMERICAL 14
#define NUMBER_MONTH_ANNUAL 12

#define EACH_MONTH_CHARS    NUM_TOP_LINE_CHARS
#define EMV_LOG_SIZE        NUM_TOP_LINE_CHARS
#define TIME_TITLE_SIZE     NUM_BOTTOM_LINE_CHARS

typedef tReturnStatus (*lcdFunc)(tDisplayData *tLcdData);

typedef struct
{
   tU8  u8TopChars[NUM_TOP_LINE_CHARS];
   tU8  u8BottomChars[NUM_BOTTOM_LINE_CHARS];
   lcdFunc LcdNumerical;
   tU32 u32LcdCtrl;
   void *u32SnsVerAddr;     // display sensor variabe RAM data address
   tU8  u8DataType;         // u8, u16, u32, f32, etc
} tLcdShowData;
#define UNUSED_SNS_VAR_ADDR  0  //RAM location 0 

#if 0
  DISPAY_SNS1 = 1,
  DISPAY_SNS2 = 2,
  DISPAY_PT   = 4,
  DISPAY_COMM = 8,
  DISPAY_PWR_ON = 16,     // display at power on once
  DISPAY_OFF  = 32,
  DISPAY_ALL_CASE = 0xFF,
#endif
#define  DISPAY_OFF     0x80000000
#define  DISPAY_PWR_ON  0x40000000
#define  LCD_ADDR       0x20000000
#define  LCD_FR_REV     0x00000001
#define  LCD_485BD      0x00000002
#define  LCD_232BD      0x00000004
#define  LCD_DPV_S1     0x00000008
#define  LCD_SPV_S1     0x00000010
#define  LCD_ESTV_S1    0x00000020
#define  LCD_RTD_V      0x00000040
#define  LCD_USER1      0x00000080
#define  LCD_USER2      0x00000100
#define  LCD_USER3      0x00000200  
#define  LCD_USER4      0x00000400  
#define  LCD_USER5      0x00000800  
#define  LCD_USER6      0x00001000  
#define  LCD_USER7      0x00002000  
#define  LCD_USER8      0x00004000  
#define  LCD_USER9      0x00008000  
#define  LCD_BD_TEMP    0x00010000  
#define  LCD_VCC_V      0x00020000  
#define  LCD_SOLAR_V    0x00040000  

#define ONCE_LOOP_EVERY 3
// for data type, u8, u16, u32. f32. etc
enum
{
  VER_U8DATA  = 0,
  VER_U16DATA,   //1
  VER_S32DATA,   //2
  VER_F32DATA,   //3
  DATA_TYPE_UNUSED,  
  RESERVED_DATA_TYPE = 0xff   //0xff
};

typedef struct
{
  tU8 u8DevModeSet;
  tU8 u8DataPte;
  tU8 u8SelectDev;
  tU8 u8BankCMD;
  tU8 u8BlinkCMD;
} tCmdPcf8576;

typedef void (*writedata)(void);

typedef struct
{
  tU8 u8DevModeSet;
  tU8 u8DataPte;
  tU8 u8SelectDev;
  tU8 u8BankCMD;
  tU8 u8BlinkCMD;
  tU8 u8ChxData[PCF8576_BUFF_SIZE];
} tI2CDataFrame;

#define SEND_LCD_FRAME_SIZE  sizeof (tI2CDataFrame)

typedef struct
{
  tU8 u8I2cState;
  tU8 u8EvenAddr;             // unused
  writedata sendDataFunc;
} tI2cCtrl;
typedef struct
{
  tU8 u8SysCode;                        // the system code or current code
  tU8 u8EvenAddr;                       // unused
  tU8 u8LcdTxt[NUM_BOTTOM_LINE_CHARS];  // LCD lower line text
}code2LcdText;
// holding timer control -> u8LcdCtrl
#define SET_LCD_HOLD_TIMER    BIT0
#define START_LCD_HOLD_TIMER  BIT1

// LCD process function prototypes definitions:
static void BcdToLcdChar(tU8 *u8BcdPtr, tU8 *u8LcdPtr, tU8 u8Num);

static tReturnStatus updateDigitalClock(tDisplayData *tLcdData);
static tReturnStatus updateDate(tDisplayData *tLcdData);
static tReturnStatus hwSerialLCD(tDisplayData *tLcdData);
static tReturnStatus fw0ReleaseNum(tDisplayData *tLcdData);
static tReturnStatus fw1ReleaseNum(tDisplayData *tLcdData);
static tReturnStatus mdbsAddrLcd(tDisplayData *tLcdData);

static tReturnStatus rs485BaudLcd(tDisplayData *tLcdData);
static tReturnStatus rs232BaudLcd(tDisplayData *tLcdData);
static tReturnStatus tempFeatureBd(tDisplayData *tLcdData);
static tReturnStatus vcc0FeatureBd(tDisplayData *tLcdData);
static tReturnStatus vcc1FeatureBd(tDisplayData *tLcdData);
static void clearLcdSupports(tDisplayData *tLcdData);
static tReturnStatus setAllLcdSupports(tDisplayData *tLcdData);

static tReturnStatus pvPtValue(tDisplayData *tLcdData);

static tReturnStatus snsVarToLcd(tDisplayData *tLcdData);

static tReturnStatus sensor1Id(tDisplayData *tLcdData);
static tReturnStatus sensor2Id(tDisplayData *tLcdData);
// following remove static, it is to remove compile warning
tReturnStatus notImplement(tDisplayData *tLcdData);
static tReturnStatus presUnit2Txt(tDisplayData *tLcdData, tU8 u8PressUnit);
static tReturnStatus ptUnit2Txt(tDisplayData *tLcdData, tU8 u8PressUnit);

// user defined LCD
static tReturnStatus userDefLcd(tDisplayData *tLcdData);

#endif //LCD_DRIVER_H
