#ifndef CTRL_MSP_UART
#define CTRL_MSP_UART
//=============================================================================
// Header: Master Harder
// Date Time: 4/8/2010 10:31 AM
// Author: Xuedong.Liu
//
// Description: Define UART message structure for communication receive and transmitte
// message.  It lock and unlock communication betweem locl and network port, and
// betweem MODBUS, BASP and HART protocols.
//
//=============================================================================


/***************************************
*
*      Serial/UART register map
*
***************************************/
#define UCTL    U1CTL       /* UART 1 Control */
#define UTCTL   U1TCTL      /* UART 1 Transmit Control */
#define URCTL   U1RCTL      /* UART 1 Receive Control */
#define UMCTL   U1MCTL      /* UART 1 Modulation Control */
#define UBR0    U1BR0       /* UART 1 Baud Rate 0 */
#define UBR1    U1BR1       /* UART 1 Baud Rate 1 */
#define RXBUF   U1RXBUF     /* UART 1 Receive Buffer */
#define TXBUF   U1TXBUF     /* UART 1 Transmit Buffer */

#define IE      IE2         /* Interrupt Enable 2   */
#define URXIE   URXIE1
#define UTXIE   UTXIE1

#define ME      ME2         /* Module Enable 2      */
#define URXE    URXE1
#define UTXE    UTXE1

/***************************************
*
*      Serial/UART Port Parameters
*
**************************************/
#define	N_8_1	CHAR
#define	E_7_1	PENA|PEV
#define	Odd_8_1	CHAR|PENA

/***************************************
*
*      Serial/UART Interrupt Handler 
*
***************************************/
#define	RX_BUF_SIZE		    64    // enlarged because of longer Dp calculations

/***************************************
*
*      Serial/UART BSAP Driver
*
***************************************/
#define	MODBUS_ASCII_MODE	    (tU8)0
#define	MODBUS_RTU_MODE		    1
#define MODBUS_DEFAULT_ADDRESS  1
#define	BSAP_MODE		        2
#define	EBSAP_MODE		        3
#define USB_LOOK_CHAR_LOOPS   10  // look take 9bits time, and receive 8bits time, the smallest common factor is 72, which is 9 chars

/*************************************************
*
*      Serial/UART BSAP/MODBUS Driver Transmit States
*
* Note: these 'states' are organized to allow
* the state variable to be incremented whenever
* a 'state transition' is required.
*
*************************************************/
#define	TS_DONE          0xff       // Transmit Driver Done (common for BSAP and Modbus)
#define	TS_START         0xfe       // Transmit Driver Start (common for BSAP and Modbus)

enum
{
  TS_SEND_DLE1 = 0,
  TS_SEND_STX,
  TS_SEND_DATA,
  TS_SEND_DLE2,
  TS_SEND_ETX,
  TS_SEND_CRC1,
  TS_SEND_CRC2,
  TS_SEND_DONE,
  TX_AA_3051S,
  TX_OFF_RX_ON,
};
enum
{
  TS_MODBUS_HIGH = 0,
  TS_MODBUS_LOW,
  TS_MODBUS_CR,
  TS_MODBUS_LF,
};

/***************************************
*
*      Serial/UART Baud Rate Codes
*
***************************************/
enum  BaudrateUSBPort
{
  NOT_SUPPORT = 0,
  BAUD1200,
  BAUD2400,
  BAUD4800,
  BAUD9600,               // RADIO port communication default
  BAUD19200,
  BAUD38400,
  BAUD57600,
  BAUD115200,
};
#define BAUD_SIZE       9      // defined baudrate size, it includes not_support entry

#define U16BD1200_CHAR_TIMEOUT   7684 // the counts a char at   1200 baud, timer CLK is 922000Hz,
#define U16BD2400_CHAR_TIMEOUT   3842 // the counts a char at   2400 baud, timer CLK is 922000Hz,
#define U16BD4800_CHAR_TIMEOUT   1921 // the counts a char at   4800 baud, timer CLK is 922000Hz,
#define U16BD9600_CHAR_TIMEOUT    961 // the counts a char at   9600 baud, timer CLK is 922000Hz,
#define U16BD19200_CHAR_TIMEOUT   481 // the counts a char at  19200 baud, timer CLK is 922000Hz,
#define U16BD38400_CHAR_TIMEOUT   241 // the counts a char at  38400 baud, timer CLK is 922000Hz,
#define U16BD57600_CHAR_TIMEOUT   161 // the counts a char at  57600 baud, timer CLK is 922000Hz,
#define U16BD115200_CHAR_TIMEOUT   81 // the counts a char at 115200 baud, timer CLK is 922000Hz,

#define PORT_TIMEOUT_xCHARS        1 // multiple 2, 7684 = one expect char, one margin

#define F32BD1200_ACHAR_MSEC        8.334f // a char timer at   1200 baud is this msec when CKL is 922000Hz
#define F32BD2400_ACHAR_TIME        4.167f // a char timer at   2400 baud is this msec when CKL is 922000Hz
#define F32BD4800_ACHAR_TIME        2.084f // a char timer at   4800 baud is this msec when CKL is 922000Hz
#define F32BD9600_ACHAR_TIME        1.042f // a char timer at   9600 baud is this msec when CKL is 922000Hz
#define F32BD19200_ACHAR_TIME       0.522f // a char timer at  19200 baud is this msec when CKL is 922000Hz
#define F32BD38400_ACHAR_TIME       0.261f // a char timer at  38400 baud is this msec when CKL is 922000Hz
#define F32BD57600_ACHAR_TIME       0.175f // a char timer at  57600 baud is this msec when CKL is 922000Hz
#define F32BD115200_ACHAR_TIME      0.088f // a char timer at 115200 baud is this msec when CKL is 922000Hz

#define CLEAR_CLK_SELEC     (tU8)~(BIT6 | BIT7)  // bit 6 and 7 is clock selection
#define CLEAR_MODULA_REG    0                // Bit1,2,3 are BRS bits and bit4,5,6,7 are BRF bits.

// u8RxCharTimeOutStatus bits defines
#define STARTS_CHAR_TIMEOUT BIT0    // lest one driver starts inter char timeout

// UART interrupt source
#define VECTOR_TXIFG       4       // Vector 4 - TXIFG
#define VECTOR_RXIFG       2       // Vector 2 - RXIFG
#define VECTOR_NOINT       0       // Vector 0 - no interrupt

// DMA interrupt source
#define VECTOR_DMA0        2       // DMA2IFG  = DMA Channel 0
#define VECTOR_DMA1        4       // DMA4IFG  = DMA Channel 1
#define VECTOR_DMA2        6       // DMA6IFG  = DMA Channel 2
#define VECTOR_DMA3        8       // DMA8IFG  = DMA Channel 3
#define VECTOR_DMA4        10      // DMA10IFG = DMA Channel 4
#define VECTOR_DMA5        12      // DMA12IFG = DMA Channel 5
#define VECTOR_DMA6        14      // DMA14IFG = DMA Channel 6
#define VECTOR_DMA7        16      // DMA16IFG = DMA Channel 7

typedef struct setUsbLineBaud
{
  tU8 u8SelectClk;
  tU8 u8BaudLow;
  tU8 u8BaudHigh;
  tU8 u8CtrlModule;
} SETCOMMBAUD;

/***************************************
*
*      Serial/UART I/O Types
*
***************************************/
#define	DEV_UNUSED		0
#define	DEV_SLAVE		1
#define	DEV_CUST		11

#define	MS_NODERR		-2		// INVALID SLAVE NODE NUMBER
#define	MS_REQSENDERR	-14		// REQUEST SEND COMMUNICATION ERROR
#define	MS_TMOERR		-15		// COMMUNICATION TIMEOUT ERROR
#define	MS_COMFAIL		-16		// SLAVE COMMUNICATIONS FAILURE
#define	MS_ALLOCFAIL	-21		// COULDN'T ALLOC. A COM BUFFER

#define	ONE_SECOND				16		// Device Driver timer ticks in a second

typedef	struct {
  tU8       u8RxState;       // internal - see check_rx_status()
  tU8       u8Index;         // internal - see check_rx_status()
  tU8       u8RxCount;       // number of bytes in transport layer
  tU8       u8PreambleCnt;   // rx how many priamble before good information
  tU8       u8LrcChk;
  tU8       u8MyAddr;
  MSG_CRC16 calc;
}RxMsgStruc;

enum
{
  INIT_TX_CHANNEL_CTRL = 0,
  TX_CHANNEL_CTRL_HART,
  TX_CHANNEL_CTRL_BASP,
  TX_CHANNEL_CTRL_MDBUS,
};
#define u8CLEAR_CRC_CHAR    (tU8)0xff
#define u8CLEAR_LRC_CHAR    (tU8)0

/**************************************************
*
*      Serial/UART Data Services prototypes
*
**************************************************/
extern  void    RS232_tS16(void);
extern	void	setBaud(tF32 f32baud);
extern  tS16     validate_baud_rate(tS16 baudrate, tU8 *destination);
extern	void	enable_rx_tS16(void);
extern	void	raise_rts(void);
extern	void	lower_rts(void);

extern	tU8	checkTXdone(void);
extern	void    initTxChCtrl(void);
extern	void	TxStartMsg(void);
extern  void    TxStartHART_rsp(void);
extern  void	TxHART_init(tU8 *msg, tU8 count, tU16 RTSdelay);
extern  tU8    get_HART_TX_state(void);
extern  void tx_HART_response(RxMsgStruc *rxs, tU8 total_bytes);

/**************************************************
*
*      Serial/UART Protocol Driver prototypes
*
**************************************************/
extern  void    initBSAP(void);
extern  void    BSAPdriver(void);

/***************************************
*
*      Serial/SPI register map
*
***************************************/
//MSP430FG4618

// Serial EEPROM Commanand Code Definitions
#define EEPRDSR     0x05    // read status register
#define EEPWRSR     0x01    // write status register
#define EEPREAD     0x03    // read memory data, 13 bits address, 1 to n data
#define EEPWRITE    0x02    // write memory data, 13 bits address, 1 to n data
#define EEPWREN     0x06    // set write enable latch
#define EEPWRDI     0x04    // write disable
#define EEPSECTOR_E 0x20    // sector erase command 4K in size
#define EEPBLOCK_E  0xD8    // block erease command 64K in size
#define EEPWDN      0xAB    // was B9
#define EEPNORMAL   0xAB   

// Serial EEPROM Read Status Register Bit Definitions
#define EEPWPEN BIT7        //
#define EEPBP1 BIT3
#define EEPBP0 BIT2
#define EEPWEN BIT1
#define EEPWIP BIT0

// port setting
enum
{
  PORT_SET2_8N1,   // =0 8 data bits, non parity bit, 1 stop bit, total = 10 bits/byte
  PORT_SET2_8O1,   // =1 8 data bits, odd parity bit, 1 stop bit, total = 11 bits/byte
};

#define ERASE_START    BIT0

#endif // CTRL_MSP_UART
