#ifndef SYSTEMs_CROSS_REFERENCE
#define SYSTEMs_CROSS_REFERENCE
//=============================================================================
// Header: JT808 between systems veriable cross reference define 
// DateTime: 8/28/2013 1:50:32 PM
// Author: Xuedong Liu
//
// Copyright (c) 2013  NewGate Instrument  
//
//=============================================================================
// type definitions
typedef struct {
  tU8 u8SysCode;                 // the system code or current code
  tU8 u83095Code;                // Map to 3095 code
} CODE3095_MAPPING;

// more generic define  
#define RMT_UNKNOW     252
#define RMT_SPECIAL    253

typedef struct {
  tU8               *u8LookValue;     // look this system code
  const CODE3095_MAPPING  *lkTable;   // look code in this table
  tU8               u8TableSize;      // table size
  ACCESS_VER_INFO   *inOutInfo;       // read/write value
} DO_CROSS_REF;

// constant definitions
static const CODE3095_MAPPING flangeMtl3095System[] =  // map between system and 3095
{
  {0,                    0}, // flange material -> carbon steel
  {2,                    2}, // flange material -> 316 stainless steel
  {3,                    3}, // flange material -> Hastelioy C
  {4,                    4}, // flange material -> Monel
  {24,                  24}, // flange material -> Kyner
  {30,         RMT_SPECIAL}, // flange material -> Hastelloy C276, 3095 new 
  {239,        RMT_SPECIAL}, // flange material -> Monel 400, 3095 new
  {RMT_UNKNOW, RMT_UNKNOW},  // flange material -> Unknown
  {RMT_SPECIAL,RMT_SPECIAL}, // flange material -> Special
};
#define FLANGE_MTL_SIZE lengthof(flangeMtl3095System)

static const CODE3095_MAPPING flangeType3095System[] =   // map between system and 3095
{
  {12,  12},                  //Flange Type ->Conventional (Traditional)
  {13,  13},                  //Flange Type ->Coplanar
  {14,  14},                  //Flange Type ->Remote Seal
  {15,  15},                  //Flange Type ->Level; 3 in, 150 lb.
  {16,  16},                  //Flange Type ->Level; 4in, 150 lb.
  {17,  17},                  //Flange Type ->Level; 3 in, 300lb.
  {18,  18},                  //Flange Type ->Level; 4in, 300 lb.
  {19,  19},                  //Flange Type ->Level; DN 80, PN 40
  {20,  20},                  //Flange Type ->Level; DN 100, PN 40
  {21,  21},                  //Flange Type ->Level; DN 100, PN 10/16
  {22,  22},                  //Flange Type ->Level; 2 in, 150 lb.
  {23,  23},                  //Flange Type ->Level; 2in, 300 lb.
  {24,  24},                  //Flange Type ->Level; DN 50, PN 6
  {25,  25},                  //Flange Type ->Level; DN 50, PN 40
  {44,  RMT_SPECIAL},         //Flange Type ->1/2in, 14NPT, new to 3095
  {45,  RMT_SPECIAL},         //Flange Type ->DIN16288G 1/2 A male, new to 3095
  {46,  RMT_SPECIAL},         //Flange Type ->1/4in,18NPT, new to 3095
  {240, RMT_SPECIAL},         //Flange Type ->Auto Clave F-250-C, new to 3095
  {241, RMT_SPECIAL},         //Flange Type ->Tri-Clamp, new to 3095
  {242, RMT_SPECIAL},         //Flange Type ->Fractional Line Fit, new to 3095
  {243, RMT_SPECIAL},         //Flange Type ->1/8in, NPTF, new to 3095
  {244, RMT_SPECIAL},         //Flange Type ->VCR, new to 3095
  {245, RMT_SPECIAL},         //Flange Type ->PMC, new to 3095
  {246, RMT_SPECIAL},         //Flange Type ->RC 1/4, new to 3095
  {247, RMT_SPECIAL},         //Flange Type ->RC 1/2, new to 3095
  {250, 250},                 //Flange Type ->Reserved
  {251, RMT_SPECIAL},         //Flange Type ->Reserved, new to 3095
  {252, 252},                 //Flange Type ->Unknown
  {RMT_SPECIAL,RMT_SPECIAL},  //Flange Type ->Special
  {254, RMT_SPECIAL},         //Flange Type ->Reserved, new to 3095
  {255, RMT_SPECIAL},         //Flange Type ->Reserved, new to 3095
};
#define FLANGE_TYPE_SIZE lengthof(flangeType3095System)

static const CODE3095_MAPPING drainVent3095System[] = // map between system and 3095
{
  {0,   RMT_SPECIAL},   //Drain Vent Type Code ->Carbon Steel, new to 3095
  {2,   2},             //Drain Vent Type Code ->316 Stainless Steel
  {3,   3},             //Drain Vent Type Code ->Hastelloy C
  {4,   4},             //Drain Vent Type Code ->Monel
  {30,  RMT_SPECIAL},   //Drain Vent Type Code ->Hastelloy C276, new to 3095
  {239, RMT_SPECIAL},   //Drain Vent Type Code ->Monel 400, new to 3095
  {251, 251},           //Drain Vent Type Code ->None
  {252, 252},           //Drain Vent Type Code ->Unknown
  {RMT_SPECIAL, RMT_SPECIAL},//Drain Vent Type Code ->Special
};
#define DRAINVENT_TYPE_SIZE lengthof(drainVent3095System)

static const CODE3095_MAPPING oRingMtl3095System[] =  // map between system and 3095
{
  {10,  10},                  //O-Ring Material ->PTFE (TeflonR)
  {11,  11},                  //O-Ring Material ->Viton
  {12,  12},                  //O-Ring Material ->Buna-N
  {13,  13},                  //O-Ring Material ->Ethyl-Prop
  {36,  RMT_SPECIAL},         //O-Ring Material ->PTFE Glass, new to 3095
  {37,  RMT_SPECIAL},         //O-Ring Material ->PTFE Graphite, new to 3095
  {251, RMT_SPECIAL},         //O-Ring Material ->None, new to 3095
  {252, 252},                 //O-Ring Material ->Unknown
  {RMT_SPECIAL, RMT_SPECIAL}, //O-Ring Material ->Special
};
#define ORING_MTL_SIZE lengthof(oRingMtl3095System)

static const CODE3095_MAPPING remoteSealT3095System[] =  // map between system and 3095
{
  {1,   RMT_SPECIAL},         //Remote Seal Type ->Reserved
  {2,   2},                   //Remote Seal Type ->CTW
  {3,   3},                   //Remote Seal Type ->EFW
  {4,   4},                   //Remote Seal Type ->PFW
  {5,   5},                   //Remote Seal Type ->RFW
  {6,   6},                   //Remote Seal Type ->RTW
  {7,   7},                   //Remote Seal Type ->SCW
  {8,   8},                   //Remote Seal Type ->SSW
  {9,   9},                   //Remote Seal Type ->High Temperature
  {10,  10},                  //Remote Seal Type ->FFW
  {11,  11},                  //Remote Seal Type ->UCW
  {12,  12},                  //Remote Seal Type ->TSW
  {13,  RMT_SPECIAL},         //Remote Seal Type ->NWSP, new to 3095
  {14,  RMT_SPECIAL},         //Remote Seal Type ->SSAP, new to 3095
  {15,  RMT_SPECIAL},         //Remote Seal Type ->SSHP, new to 3095
  {16,  RMT_SPECIAL},         //Remote Seal Type ->TFS, new to 3095
  {251, 250},                 //Remote Seal Type ->None
  {252, 252},                 //Remote Seal Type ->Unknown
  {RMT_SPECIAL, RMT_SPECIAL}, //Remote Seal Type ->Special
};
#define REMOTE_SEAL_SIZE lengthof(remoteSealT3095System)

static const CODE3095_MAPPING rmtSealFuildT3095System[] =  // map between system and 3095
{
  {1,RMT_SPECIAL},            //Remote Seal Fill Fluid ->Reserved
  {2,   2},                   //Remote Seal Fill Fluid ->Silicone Oil
  {3,   3},                   //Remote Seal Fill Fluid ->Syltherm 800
  {4,   4},                   //Remote Seal Fill Fluid ->Inert
  {5,   5},                   //Remote Seal Fill Fluid ->Glycerin H2O
  {6,   6},                   //Remote Seal Fill Fluid ->Prop Gly/H2O
  {7,   7},                   //Remote Seal Fill Fluid ->Neobee-M20
  {8,   RMT_SPECIAL},         //Remote Seal Fill Fluid ->Syltherm XLT, new to 3095
  {9,   RMT_SPECIAL},         //Remote Seal Fill Fluid ->Dioctyl phthalate, new to 3095
  {10,  RMT_SPECIAL},         //Remote Seal Fill Fluid ->D. C. 704, new to 3095
  {11,  RMT_SPECIAL},         //Remote Seal Fill Fluid ->Therminol 66, new to 3095
  {12,  RMT_SPECIAL},         //Remote Seal Fill Fluid ->D. C. 210H, new to 3095
  {13,  RMT_SPECIAL},         //Remote Seal Fill Fluid ->Distilled water, new to 3095
  {14,  RMT_SPECIAL},         //Remote Seal Fill Fluid ->D. C. 200, new to 3095
  {15,  RMT_SPECIAL},         //Remote Seal Fill Fluid ->D. C. 705, new to 3095
  {251, 251},                 //Remote Seal Fill Fluid ->None
  {252, 250},                 //Remote Seal Fill Fluid ->Unknown
  {RMT_SPECIAL, RMT_SPECIAL}, //Remote Seal Fill Fluid ->Special
};
#define REMOTE_SEALFILL_FUILD_SIZE lengthof(rmtSealFuildT3095System)

static const CODE3095_MAPPING rmtSealIsoMtl3095System[] =  // map between system and 3095
{
  {2,   2},                 //Remote Seal Isolate Material ->316 Stainless Steel
  {3,   3},                 //Remote Seal Isolate Material ->Hastelloy C
  {5,   5},                 //Remote Seal Isolate Material ->Tantalum
  {9,   9},                 //Remote Seal Isolate Material ->Co-Cr-Ni
  {34,  RMT_SPECIAL},       //Remote Seal Isolate Material ->PTFE Coated 316L SST, new to 3095
  {239, RMT_SPECIAL},       //Remote Seal Isolate Material ->Monel 400, new to 3095
  {240, RMT_SPECIAL},       //Remote Seal Isolate Material ->Nickel 201, new to 3095
  {251, 251},               //Remote Seal Isolate Material ->None
  {252, 252},               //Remote Seal Isolate Material ->Unknown
  {RMT_SPECIAL,RMT_SPECIAL},//Remote Seal Isolate Material ->Special
};
#define RMT_SEALISO_MTL_SIZE lengthof(rmtSealIsoMtl3095System)

#define DEFAULT95_DP_UNIT     1
static const CODE3095_MAPPING dpUnit3095System[] =  // map DP unit between system and 3095
{
  {U_inH2OAt68F,      6},                 // map inH2O@68F between system and 3095
  {U_inHgAt0C,        RMT_UNKNOW},        // map inHg@0C between system and 3095
  {U_ftH20At68F,      RMT_UNKNOW},        // map ftH2O@68F between system and 3095
  {U_mmH20At68F,      RMT_UNKNOW},        // map mmH2O@68F between system and 3095
  {U_mmHgAt0C,        RMT_UNKNOW},        // map mmHg@0C between system and 3095
  {U_psi,             5},                 // map psi between system and 3095
  {U_bars,            RMT_UNKNOW},        // map bar between system and 3095
  {U_milliBars,       RMT_UNKNOW},        // map mbar between system and 3095
  {U_gramPerSqCm,     RMT_UNKNOW},        // map g/Cm2 between system and 3095
  {U_kiloGramPerSqCm, RMT_UNKNOW},        // map Kg/Cm2 between system and 3095
  {U_pascals,         2},                 // map Pa between system and 3095
  {U_kiloPascals,     3},                 // map KPa between system and 3095
  {U_torr,            RMT_UNKNOW},        // map torr between system and 3095
  {U_atm,             RMT_UNKNOW},        // map atmospheres between system and 3095
  {U_megaPascals,     5},                 // map MPa between system and 3095
  {U_inH20At4C,       RMT_UNKNOW},        // map inH2O@4C between system and 3095
  {U_mmH20At4C,       RMT_UNKNOW},        // map mmH2O@4C between system and 3095
  {U_inH2OAt60F,      DEFAULT95_DP_UNIT}, // map inH2O@60F between system and 3095, this is 3095 default
};
#define DP_UNIT_SIZE   lengthof(dpUnit3095System)

static const CODE3095_MAPPING ptUnit3095System[] =  // map DP unit between system and 3095
{
  {U_celsius,     20},           // U_celsius between system and 3095
  {U_fahrenheit,  21},           // U_fahrenheit between system and 3095
  {U_rankine,     RMT_UNKNOW},   // U_rankine between system and 3095
  {U_kelvin,      RMT_UNKNOW},   // U_kelvin between system and 3095
  {255,           RMT_UNKNOW},   // no unit between system and 3095
};
#define PT_UNIT_SIZE   lengthof(ptUnit3095System)

#define PRESS_TYPE_UNIT_MAP           1
#define TEMPERATURE_TYPE_UNIT_MAP     2

// function prototype
tU8 comm95RdCrossRef(DO_CROSS_REF *do3095Ref);
tU8 comm95WtCrossRef(DO_CROSS_REF *do3095Ref);

#endif // #ifndef SYSTEMs_CROSS_REFERENCE
