//=============================================================================
// Header: this is field update firmware API functions
// Date Time: 1/12/2013 10:48:56 AM
// Author: Xuedong.Liu
//
// Description: This is field update firmware API functions file. It is
//     copied from TI demo file and modified to fit our application.
//==============================================================

// constant definitions, type definitions
#include "masterHeader.h"

// global volatile variables
#include "externGlobalVar.h"

//global function prototypes definitions
#include "externFunc.h"

#define FAST_WRITE
// function prototype.
void BSL430_API_init(void);
tReturnStatus BSL430_lock_BSL(void);
tReturnStatus BSL430_unlock_flash(void);
tReturnStatus BSL430_openMemory(void);
tReturnStatus BSL430_closeMemory(void);
tReturnStatus BSL430_callAddress(unsigned long addr);
tReturnStatus BSL430_writeMemory( tU32 startAddr, tU8* data, tU8 u8Size );
tReturnStatus BSL430_writeByte(unsigned long addr, char data );
#ifndef FAST_WRITE
tReturnStatus BSL430_writeWord(unsigned long addr, int data );
#else
tReturnStatus BSL430_writeWord(unsigned long addr, tU32 data );
#endif
tReturnStatus BSL430_eraseSection( char block_erase, unsigned long addr );
tReturnStatus BSL430_massErase(void);

// global veriables
tU8 LockedStatus, u8LastSecNum, u8MakeEvenAddr;
tS16 FwRamKey;

/*******************************************************************************
*Function:    BSL430_API_init
*Description: sets the key for writing to flash,  sets device state
*******************************************************************************/
void BSL430_API_init(void)
{
  LockedStatus = DEFAULT_STATE;
  FwRamKey = FWKEY;
}

/*******************************************************************************
*Function:    BSL430_lock_BSL
*Description: Locks the BSL
*Returns:
*             SUCCESSFUL_OPERATION  BSL Locked
*******************************************************************************/
tReturnStatus BSL430_lock_BSL(void)
{
  LockedStatus = LOCKED;
  return SUCCESSFUL_OPERATION;
}

tReturnStatus BSL430_unlock_flash(void)
{
  LockedStatus = UNLOCKED;
  return SUCCESSFUL_OPERATION;
}

/*******************************************************************************
*Function:    BSL430_openMemory
*Description: Unlocks the Flash for writing
*Returns:
*             SUCCESSFUL_OPERATION  Flash is now open for writing.
*             BSL_LOCKED            Correct Password has not yet been given
*******************************************************************************/
tReturnStatus BSL430_openMemory(void)
{
  tReturnStatus exceptions = BSL_LOCKED;
  if( LockedStatus == UNLOCKED )
  {
    exceptions = SUCCESSFUL_OPERATION;
    FCTL3 = FwRamKey;                       // Clear Lock bit
    FCTL1 = FwRamKey + WRT;                 // Set write bit
  }
  return exceptions;
}
/*******************************************************************************
*Function:    BSL430_closeMemory
*Description: Locks the Flash against writing
*Returns:
*             SUCCESSFUL_OPERATION  Flash is now locked.
*******************************************************************************/
tReturnStatus BSL430_closeMemory(void)
{
  FCTL1 = FwRamKey;                         // Clear WRT bit
  FCTL3 = FwRamKey + LOCK;                  // Set LOCK bit

  return SUCCESSFUL_OPERATION;
}

/*******************************************************************************
*Function:    BSL430_callAddress
*Description: Loads the Program Counter with the supplied address
*Parameters:
*           unsigned long addr    The address to which the function call should go
*Returns:
*           SUCCESSFUL_OPERATION  Called location has returned
*           BSL_LOCKED            Correct Password has not yet been given
*Note:
*           If successful, this function does not return.
*******************************************************************************/
tReturnStatus BSL430_callAddress(unsigned long addr)
{
  if( LockedStatus == UNLOCKED )
  {
    ((void (*)(void))addr)();                   // type cast addr to function ptr, call
    return SUCCESSFUL_OPERATION;
  }
  return BSL_LOCKED;                        // can only be reached if BSL is locked
}
/*******************************************************************************
*Function:    BSL430_writeMemory
*Description: Writes a byte array starting at a given address.
*             Note: the function will write in word mode if possible
*             (when start address is even)
*Parameters:
*           unsigned long startAddr        The address to which the write should begin
*           int size                       The number of bytes to write
*           char* data                     The array of bytes to write (must be even aligned)
*Returns:
*           SUCCESSFUL_OPERATION           Bytes written successfully
*           MEMORY_WRITE_CHECK_FAILED      A byte in data location post-write does not match data parameter
*                                          Note: write stops immediatly after a byte check fails
*           BSL_LOCKED                     Correct Password has not yet been given
*           VOLTAGE_CHANGE_DURING_PROGRAM  Voltage changed during write (of a single byte/word)
*******************************************************************************/
tReturnStatus BSL430_writeMemory( tU32 startAddr, tU8* data, tU8 u8Size )
{
  tU32 i, temp, leftChar;
  tReturnStatus exceptions = SUCCESSFUL_OPERATION;
  temp = startAddr + (tU32)u8Size;
  for( i = startAddr; i < temp; i++ )
  {
#ifdef FAST_WRITE
    leftChar = temp - i;
    if ((leftChar >= 4) && (((tU32)data&0x00000001) == 0))
    {
      //even address start and has at least 4 bytes data to write
      exceptions = BSL430_writeWord( i, *(tU32 *)data);
      // data point to 5 bytes
      data+=4;
      // i increase 3 in for loop increase once more
      i+=3;
    }
    else
    {
      exceptions = BSL430_writeByte( i, *data );
      data+=1;
    }
#else
    exceptions = BSL430_writeByte( i, *data );
    data+=1;
#endif
    if( exceptions != SUCCESSFUL_OPERATION )
    {
      return exceptions;
    } // if
  } // for
  return exceptions;
}
/*******************************************************************************
*Function:    BSL430_writeByte
*Description: Writes a byte at a given address
*Parameters:
*           unsigned long addr             The address to which the byte should be written
*           char data                      The byte to write
*Returns:
*           SUCCESSFUL_OPERATION           Byte written successfully
*           MEMORY_WRITE_CHECK_FAILED      Byte in data location post-write does not match data parameter
*           VOLTAGE_CHANGE_DURING_PROGRAM  Voltage changed during write
*           BSL_LOCKED                     Correct Password has not yet been given
*******************************************************************************/
tReturnStatus BSL430_writeByte(unsigned long addr, char data )
{
  tReturnStatus exceptions;
  if( LockedStatus == UNLOCKED )
  {
    exceptions = SUCCESSFUL_OPERATION;
    while( FCTL3 & BUSY )
    {
      _NOP();
    }
    __data20_write_char( addr, data );
    while( FCTL3 & BUSY )
    {
      _NOP();
    }
    if( data != __data20_read_char( addr ))
    {
      exceptions = MEMORY_WRITE_CHECK_FAILED;
    }
  }
  else
  {
    exceptions = BSL_LOCKED;
  }
  return exceptions;
}
/*******************************************************************************
*Function:    BSL430_writeWord
*Description: Writes a word at a given address
*Parameters:
*           unsigned long addr             The address to which the word should be written
*           int data                       The byte to write
*Returns:
*           SUCCESSFUL_OPERATION           Word written successfully
*           MEMORY_WRITE_CHECK_FAILED      Word in data location post-write does not match data parameter
*           VOLTAGE_CHANGE_DURING_PROGRAM  Voltage changed during write
*           BSL_LOCKED                     Correct Password has not yet been given
*******************************************************************************/
#ifndef FAST_WRITE
tReturnStatus BSL430_writeWord(unsigned long addr, int data )
#else
tReturnStatus BSL430_writeWord(unsigned long addr, tU32 data )
#endif
{
  tReturnStatus exceptions;
  if( LockedStatus == UNLOCKED )
  {
    exceptions = SUCCESSFUL_OPERATION;
    while( FCTL3 & BUSY )
    {
      _NOP();
    }
#ifndef FAST_WRITE
    __data20_write_short( addr, data );
#else
    __data20_write_long( addr, data );
#endif
    while( FCTL3 & BUSY )
    {
      _NOP();
    }
#ifndef FAST_WRITE
    if( data != __data20_read_short( addr ))
#else
    if( data != __data20_read_long( addr ))
#endif
    {
      exceptions = MEMORY_WRITE_CHECK_FAILED;
    }
    if( FCTL4 & VPE )
    {
      exceptions = VOLTAGE_CHANGE_DURING_PROGRAM;
    }
  }
  else
  {
    exceptions = BSL_LOCKED;
  }
  return exceptions;
}
/*******************************************************************************
*Function:    BSL430_eraseLocation
*Description: Erases a memory segment which contains a given address
*Parameters:
*           char block_erase               currently unused 1: erase block 0: erase segment
*           unsigned long addr             An address which is within the segment to be erased
*Returns:
*           SUCCESSFUL_OPERATION           Segment erased
*           BSL_LOCKED                     Correct Password has not yet been given
*******************************************************************************/
tReturnStatus BSL430_eraseSection( char block_erase, unsigned long addr )
{
  tReturnStatus exceptions;
  if( LockedStatus == UNLOCKED )
  {
    exceptions = SUCCESSFUL_OPERATION;
    while( FCTL3 & BUSY )
    {
      _NOP();
    }
    FCTL3 = FwRamKey;                       // Clear Lock bit
    FCTL1 = FwRamKey + MERAS;               // Set MERAS bit
    __data20_write_char( addr, 0 );         // Dummy write to erase Flash seg
//    startFlashEraseWaiteTimer(); //startFlashWaiteTimer();
  }
  else
  {
    exceptions = BSL_LOCKED;
  }
  return exceptions;
}



/*******************************************************************************
*Function:    BSL430_massErase
*Description: Mass erases the entire MSP430 device
*Returns:
*           SUCCESSFUL_OPERATION           Flash erased
*           BSL_LOCKED                     Correct Password has not yet been given
*******************************************************************************/
tReturnStatus BSL430_massErase(void)
{
  tReturnStatus exceptions = SUCCESSFUL_OPERATION;
  volatile char *Flash_ptr;                 // Flash pointer

  while( FCTL3 & BUSY )
  {
    _NOP();
  }
  FCTL3 = FwRamKey;
  while( FCTL3 & BUSY )
  {
    _NOP();
  }
  Flash_ptr = (char *)INTERRUPT_VECTOR_START;// Initialize Flash pointer
  FCTL1 = FwRamKey + MERAS + ERASE;          // Set Mass Erase bit
  *Flash_ptr = 0;                            // Dummy write to erase main flash
  while( FCTL3 & BUSY )
  {
    _NOP();
  }
  FCTL3 = FwRamKey + LOCK;                  // Set LOCK bit

  return exceptions;
}//=============================================================================
// end file by xuedong
//=============================================================================


//=============================================================================
// end file by xuedong
//=============================================================================
