//=============================================================================
// Header: JTI808 feature board MSP430F5438A init
// Date Time: 1/19/2013 9:53 AM
// Author: Xuedong.Liu
//
// Description: init system clock, ports, USCIs, RTC, ADC.
//
//=============================================================================

// constant definitions, type definitions
#include "masterHeader.h"

// global volatile variables
#include "externGlobalVar.h"

//global function prototypes definitions
#include "externFunc.h"

// local static variables
static void chargeOn(void);
static void chargeOff(void);
static tF32 BatFactor, solarFacor; 
static tF32 //f32OffsetBatV,              // battery offset
       SCurrentVolts,                   // battery volt
       f32SolarVolts,                   // Solar Volt
       SCurrentBTemp;                   // MSP temperature in C
       //f32OffsetSolarV;                 // Solar Volt offset

static tU16 u16BatAdcCnt[2];
static tF32 f32BatApplyV[2];
static tU8  u8CalStatus;

static struct s_TLV_ADC_Cal_Data * pADCCAL;
static tU8 bADCCAL_bytes;
static tF32 f32AdcSlop, f32AdcOffset;
static void Get_TLV_Info(tU8 tag, tU8 instance, tU8 *length, tU16 **data_address);

/*******************************************************************************
 * void ReadADCCAL(void)
 * 
 * This function will search the TLV block for the ADC CAL tag and store the 
 * address of the first data in the block in the reference pointer passed as an 
 * argument. The data is then saved to local variables.
 ******************************************************************************/
static void Get_TLV_Info(uint8_t tag, uint8_t instance, uint8_t *length, uint16_t **data_address)
{
  char *TLV_address = (char *)TLV_START;         // TLV Structure Start Address

  while((TLV_address < (char *)TLV_END)
        && ((*TLV_address != tag) || instance)   // check for tag and instance
        && (*TLV_address != TLV_TAGEND))         // do range check first
  {
    if (*TLV_address == tag) instance--;         // repeat till requested instance is reached
    TLV_address += *(TLV_address + 1) + 2;       // add (Current TAG address + LENGTH) + 2
  }
  
  if (*TLV_address == tag)                       // Check if Tag match happened..
  {
    *length = *(TLV_address + 1);                  // Return length = Address + 1
    *data_address = (uint16_t *)(TLV_address + 2); // Return address of first data/value info = Address + 2
  }
  else                                           // If there was no tag match and the end of TLV structure was reached..
  {
    *length = 0;                                 // Return 0 for TAG not found
    *data_address = 0;                           // Return 0 for TAG not found
  }
}

void ReadADCCAL(void)
{

  // Read ADC12 Temperature Sensor Calibration Values
  Get_TLV_Info(TLV_ADCCAL, 0, &bADCCAL_bytes, (unsigned int **)&pADCCAL);
  
  //ADC_15V_Ref_30C = pADCCAL->adc_ref15_30_temp;
  //ADC_15V_Ref_85C = pADCCAL->adc_ref15_85_temp;
  // reference is 1500mV 2.25mV/count internal build, MSP calibrated at 30C and 85C
  //
  //RTG: Feb2016, see pg. 82 in SLAU208OJune 2008Revised May 2015, for the following
  //
  f32AdcSlop = (85.0f - 30.0f) / (pADCCAL->adc_ref15_85_temp - pADCCAL->adc_ref15_30_temp);      //RTG: Feb2016

  f32AdcOffset = pADCCAL->adc_ref15_30_temp;                                                     //RTG: Feb2016
  
}

// global & local function implementation
void startMspAdcSeq(void)
{
    // clear the flag which it makes the control comeing in here
    u8Layer1TaskScheduler &= (tU8)~START_MSP_ADC;
    sysRamData.diag.u16XmtStatus[3] |= ADC5638_START_ON;
    
    //Select internal ref = 1.5V
    REF_setReferenceVoltage(__MSP430_BASEADDRESS_REF__,
        REF_VREF1_5V);
      
    //Internal Reference ON
    REF_enableReferenceVoltage(__MSP430_BASEADDRESS_REF__);

    //Enable memory buffer 4 interrupt
    ADC12_enableInterrupt(__MSP430_BASEADDRESS_ADC12_PLUS__,
        ADC12IE4);  

    waste_time(100);   // was 30 make 2st ADC right    !!!!RTG

    //Enable/Start first sampling and conversion cycle
    // MSP430F5638, faster conversion
    //  Use the repeated sequence of channels
    ADC12_startConversion(__MSP430_BASEADDRESS_ADC12_PLUS__,
        ADC12_MEMORY_0,
        ADC12_SEQOFCHANNELS); //*/ ADC12_REPEATED_SEQOFCHANNELS);   RTG: Feb2016, replaced REPEATED_SEQ...

#if 0
    // board resistor are different
    if((info5638D.u8HardwareId > 1) && (info5638D.u8HardwareId != 0x80))
    {
      BatFactor  = REF3BD_VCNTS_BATT_V;       
    }
    else
    {
      BatFactor  = VCNTS_BATT_V;  
    }
#else
    BatFactor  = REF3BD_VCNTS_BATT_V;       
    if(!((info5638D.u8HardwareId > 1) && (info5638D.u8HardwareId != 0x80)))
    {
      __no_operation();
    }
#endif
    solarFacor = VCNTS_SOLAR_V;   

}

static void chargeOn(void)
{ 
  GPIO_setOutputLowOnPin(
      __MSP430_BASEADDRESS_PORT6_R__,  //battery charge on
      GPIO_PORT_P6,
      P6_0_SP_SHNT_OUT);  
}
      
static void chargeOff(void)
{
// MSP temperatuer conversion not right, when it is right add this in
  GPIO_setOutputHighOnPin(             //battery charge off
      __MSP430_BASEADDRESS_PORT6_R__,
      GPIO_PORT_P6,
      P6_0_SP_SHNT_OUT); 

}

void initChargeAndTime(void)
{
  chargeOn();
  SCurrentBTemp = 0.0f;
}

//=============================================================================
//  Function: batteryChargeOnOff()
//_____________________________________________________________________________
//
//   Description: This routine handles the ADC input state machine.
//           It converts Battery volts, three AI's, does AO voltage
//           validation and Battery measurement.
//   Note:  FIX ME the battery and solar volts, connect to MSP430F5638
//          this function should move to there.
//
//=============================================================================
void batteryChargeOnOff(void)
{
    tF32 fptemp, fptempA;

    // clear the flag
    u8Layer2TaskScheduler &= (tU8)~CHARGE_BAT_HS_AI;
    
    fptemp = (tF32)u16MspAdcCnt[1]; 		//debug test, raw ADC counts
    
    fptemp = (tF32)u16MspAdcCnt[0]; 		//raw ADC counts
	
#if 1
    //RTG: Feb2016, see pg. 82 in SLAU208OJune 2008Revised May 2015, for the following
    //
    //Temperature unit is in Celsius           					//RTG
    fptempA = fptemp = ((fptemp - f32AdcOffset) * f32AdcSlop) + 30.0f;   	//RTG

#else
    //RTG: Feb2016, from ADC12 Device Library example code (not veru accurate)
    fptempA = fptemp = ((fptemp - 1855.00) * 667.00) / 4096.0;
	
#endif
    __no_operation();

    // prfect zero, just start, or less than 15C in 1 minutes, change the control viarable
    SCurrentBTemp = fptempA;
    sysRam56D.exchg6Pv.f32FbdTemp = SCurrentBTemp - info5638B.f32MspBdTempOffset;

    // else it could be error, 1min can not change 15C in the air 
    __no_operation();
    sysRamData.diag.u16XmtStatus[3] &= ~ADC5638_START_ON;
                
    // reference is 3.3V from 
    // apply resistor network values   
    // no battery, only solar power, 
    // solar power measure from outside is 4.49V, ref is 3.3
    // ADC converter readings solarV = 3.79V, batV = 4.23V
    // when ref is 3.5, solarV = 4.018V Bat = 4.489V
    // 
    fptemp = (tF32)u16MspAdcCnt[2];
    f32SolarVolts = fptemp/solarFacor - info5638B.f32OffsetSolarV;
    //new value = calculated + offset    
    sysRam56D.exchg6Pv.f32SolarV = f32SolarVolts;
    
    fptemp = (tF32)u16MspAdcCnt[3];
    if(u8CalStatus == CAL_BAT_OFFSET)
    {
      u16BatAdcCnt[0] = u16MspAdcCnt[3];
      u8CalStatus = DONE_SET_CAL_NUM;
    }
    else if(u8CalStatus == CAL_BAT_SLOP)
    {
      u16BatAdcCnt[1] = u16MspAdcCnt[3];      
      // calculate slop
      fptempA = (f32BatApplyV[1] - f32BatApplyV[0]);
      fptempA /= (tF32)(u16BatAdcCnt[1] - u16BatAdcCnt[0]);
      info5638B.f32SlopBatV = fptempA ;
      // calculate offset     
      info5638B.f32OffsetBatV = 
        f32BatApplyV[0] - fptempA * (tF32)u16BatAdcCnt[0];
      infoD.xmt.u8KeptStatus |= BAT_CAL_DONE;
      
      addOneTypeExchg(TYPE_INFOD);

      u8CalStatus = DONE_SET_CAL_NUM;
      u8OtherTaskScheduler |= DYNAMIC_WT_IMFOB;  // do EEP section A write process
    }
    else
    {
      // do nothing extra      
    }
    
    if((infoD.xmt.u8KeptStatus & BAT_CAL_DONE) == BAT_CAL_DONE)
    {
      // do nothing extra
      fptemp *= BatFactor;
    }
    else
    {
      // do nothing extra      
    }
    
    //new value = calculated + offset
    fptemp = fptemp/BatFactor;
    fptemp *= info5638B.f32SlopBatV;
    fptemp += info5638B.f32OffsetBatV;

    // apply damping, in case measurement jumps, or has error  
    //dampedValue = (B * inInfo->f32NewValue) + ((1.0f - B) * (inInfo->f32OldValue));    
    //fptemp *= 0.25;
    //SCurrentVolts = sysRam56D.exchg6Pv.f32BatV =  0.75f * SCurrentVolts;
    SCurrentVolts = fptemp;
    sysRam56D.exchg6Pv.f32BatV = SCurrentVolts;
    
    if(sysRam56D.exchg6Pv.f32BatV > DEAL_BATTERY)
    {
      setBatInstallStatus(ON);
      batInCtrlOutDefault();

      __no_operation();        

      // compute T effect in V, test charge on/off
      // threshold decreases with heat
      fptemp = SCurrentBTemp;
      fptemp = 5.128824f - (0.011378f * fptemp) + (0.000075333f * fptemp * fptemp); //charge limit
      fptemp *= 1.1f;  // make 10% limit band
      
      if (SCurrentBTemp < -20.0f)
      {
        // HW requirement:"if the temperature is less then -20C disable the regulator
        chargeOff();
      }
      else if (SCurrentVolts > fptemp)
      {
        // in a day and night, battery volts > limit, turn off charge not over charge 
        chargeOff();
      }
      else
      {
        chargeOn();
        __no_operation();  
      }
    }
    else
    {
      setBatInstallStatus(OFF);
      batoutCtrlOutLow();
      chargeOff();      
    }

    addOneTypeExchg(TYPE_EX6_PVS);  // 5638 use it
}

tU8 calBatVolt(ACCESS_VER_INFO *rwVarInfo)
{ 
  // assumption is ADC linery. default offset is 0.
  // calibration get new offset = input - calculated.
  if(sysRam56D.exchg6Pv.f32BatV < 3.0f)
  {
    u8CalStatus = CAL_BAT_OFFSET;
    f32BatApplyV[0] = sysRam56D.exchg6Pv.f32BatV;
    //info5638B.f32OffsetBatV = SCurrentVolts - sysRam56D.exchg6Pv.f32BatV;
  }
  else
  {
    u8CalStatus = CAL_BAT_SLOP;
    f32BatApplyV[1] = sysRam56D.exchg6Pv.f32BatV; 
    //info5638B.f32SlopBatV = (SCurrentVolts - info5638B.f32OffsetBatV)/sysRam56D.exchg6Pv.f32BatV;
  }
  // save calibration result
  //u8OtherTaskScheduler |= DYNAMIC_WT_IMFOB;  // do EEP section A write process
  return rwVarInfo->u8RspBufPtr;
}

tU8 calSolarVolt(ACCESS_VER_INFO *rwVarInfo)
{
  // assumption is ADC linery. default offset is 0.
  // calibration get new offset = input - calculated
  info5638B.f32OffsetSolarV = f32SolarVolts - sysRam56D.exchg6Pv.f32SolarV;
  // save calibration result
  u8OtherTaskScheduler |= DYNAMIC_WT_IMFOB;  // do EEP section A write process
  return rwVarInfo->u8RspBufPtr;
}

tU8 calMspTempInC(ACCESS_VER_INFO *rwVarInfo)
{
  // assumption is ADC linery. default offset is 0.
  // calibration get new offset = input - calculated.
  info5638B.f32MspBdTempOffset = SCurrentBTemp - sysRam56D.exchg6Pv.f32FbdTemp;
  // save calibration result
  u8OtherTaskScheduler |= DYNAMIC_WT_IMFOB;  // do EEP section A write process
  return rwVarInfo->u8RspBufPtr;
}


//=============================================================================
// end file by xuedong
//=============================================================================

