//=============================================================================
// Header: MSP430F5438A Timer control and allocation
// Date Time: 1/25/2013 12:58 PM
// Author: Xuedong.Liu
//
// Description: JT808 MSP430F5438A firmware use timer in RTD sampling,
//              Read/Write/ereas FLash, flash program, char gap on
//              USB ports, RTS on USB,
//              schedule task to run, this can be controled by RTC too.
//
//=============================================================================

// constant definitions, type definitions
#include "masterHeader.h"

// global volatile variables
#include "externGlobalVar.h"

// global function prototypes definitions:
#include "externFunc.h"

// static variables
static tU8 u8ReTxType, u8RadioTaskFlage;


// local function prototypes definitions:
// NA

// global & local function implementation
//=============================================================================
// FUNCTION:    initTimer
//
// TA0     1. TA0.0 is free
//         2. TA0.1 is free
//         3. TA0.2 is free
//         4. TA0.3 is free
//         5. TA0.4 is free
//         5.1 overflow is free
// TA1     1. TA1.0 is free
//         2. TA1.1 is free
//         3. TA1.2 is free
// TB0     1. TB0.0 is free
//         2. TB0.1 is free
//         3. TB0.2 is free
//         4. TB0.3 is free
//         5. TB0.4 is free
//         6. TB0.5 is free
//         7. TB0.6 is free
// TA2     1. TA2.0 is free
//         2. TA2.1 is free
//         3. TA2.2 is free
//=============================================================================
void initTimer(void)
{
    //======= start of timer TA0.A0A5 init ==============
    // TA0 use ACLK = 32768Hz, continuous mode count upto 0x0FFFFh,
    TimerB_selectCounterLength(
        __MSP430_BASEADDRESS_T0B7__,
        TIMERB_COUNTER_16BIT);              // countinuous mode from 0 to 0xFFFF

    TimerB_startContinuousMode(             // init timer B , 0 to 7
        __MSP430_BASEADDRESS_T0B7__,
        TIMERB_CLOCKSOURCE_ACLK, 
        TIMERB_CLOCKSOURCE_DIVIDER_1,
        TIMERB_TBIE_INTERRUPT_DISABLE,//TIMERB_TBIE_INTERRUPT_DISABLE, TIMERB_TBIE_INTERRUPT_ENABLE     // overflow interrupt disabled
        TIMERB_DO_CLEAR);
    
    TimerA_startContinuousMode(
        __MSP430_BASEADDRESS_T1A3__,            // on MSP430F5638 HSC
        TIMERA_CLOCKSOURCE_EXTERNAL_TXCLK,      // HSC input, read TA1R, at any time
        TIMERA_CLOCKSOURCE_DIVIDER_1,
        TIMERA_TAIE_INTERRUPT_DISABLE,          // overflow interrupt disabled
        TIMERA_DO_CLEAR);  
 
    TimerA_startContinuousMode(
        __MSP430_BASEADDRESS_T2A3__,            // on MSP430F5638 HSC
        TIMERA_CLOCKSOURCE_EXTERNAL_TXCLK,      // HSC input, read TA1R, at any time
        TIMERA_CLOCKSOURCE_DIVIDER_1,
        TIMERA_TAIE_INTERRUPT_DISABLE,          // overflow interrupt disabled
        TIMERA_DO_CLEAR);  
 
    TimerA_startCounter(
        __MSP430_BASEADDRESS_T1A3__,
        TIMERA_CONTINUOUS_MODE);
   
    startTaskScheduler();
}

void startTaskScheduler(void)
{
    TimerB_setCompareValue(
        __MSP430_BASEADDRESS_T0B7__,
        TIMERB_CAPTURECOMPARE_REGISTER_6,   
        TB0R + 0x0FFF);        

    TimerB_enableCaptureCompareInterrupt(
        __MSP430_BASEADDRESS_T0B7__,
        TIMERB_CAPTURECOMPARE_REGISTER_6);
}

void startExchgSleepTimer(void)
{
    TimerB_setCompareValue(
        __MSP430_BASEADDRESS_T0B7__,
        TIMERB_CAPTURECOMPARE_REGISTER_4,   
        TB0R + 16384);        // ACLK = 32768  16384/32768 = 500mSec

    TimerB_enableCaptureCompareInterrupt(
        __MSP430_BASEADDRESS_T0B7__,
        TIMERB_CAPTURECOMPARE_REGISTER_4);  
}

void stopExchgSleepTimer(void)
{
    TimerB_disableCaptureCompareInterrupt(
        __MSP430_BASEADDRESS_T0B7__,
        TIMERB_CAPTURECOMPARE_REGISTER_4);    
}

void exchgSleepTimerExpired(void)
{
  // idle state
  setExchgState(IDLE);
  // stop timer
  stopExchgSleepTimer();
}

void startWaitOtherReadTimer(void)
{   
    TimerB_setCompareValue(
        __MSP430_BASEADDRESS_T0B7__,
        TIMERB_CAPTURECOMPARE_REGISTER_5,   
        TB0R + 1024);        // ACLK = 32768  1024/32768 = 32mSec

    TimerB_enableCaptureCompareInterrupt(
        __MSP430_BASEADDRESS_T0B7__,
        TIMERB_CAPTURECOMPARE_REGISTER_5);
}

void stopWaitOtherReadTimer(void)
{
    TimerB_disableCaptureCompareInterrupt(
        __MSP430_BASEADDRESS_T0B7__,
        TIMERB_CAPTURECOMPARE_REGISTER_5);  
}

void setReTxType(tU8 u8TxType)
{
  u8ReTxType = u8TxType;
}

void otherMspNoReadReTx(void)
{
  // idle state
  setExchgState(IDLE);
  // stop timer
  stopWaitOtherReadTimer();
  // other did not get this type info, do again  
  addOneTypeExchg(u8ReTxType);  // 5638 use it
}
#if 1
void radioTimerStart(tU16 u16TimerV, tU8 u8SetFlag)
{
  u8RadioTaskFlage = u8SetFlag;
  TimerB_setCompareValue(
      __MSP430_BASEADDRESS_T0B7__,
      TIMERB_CAPTURECOMPARE_REGISTER_3,   
      TB0R + u16TimerV);        // ACLK = 32768  32768/32768 = 1Sec, 100msec = 3000/32768 = 100mSec
                                // 50/32768 = 1.5mSec, 360/32768 = 10.98mSec
  TimerB_enableCaptureCompareInterrupt(
      __MSP430_BASEADDRESS_T0B7__,
      TIMERB_CAPTURECOMPARE_REGISTER_3);  
}

void radioTimerStop(void)
{
    TimerB_disableCaptureCompareInterrupt(
        __MSP430_BASEADDRESS_T0B7__,
        TIMERB_CAPTURECOMPARE_REGISTER_3);
    if(u8RadioTaskFlage == TX_DMA_START)
    {
       UART_transmitData (__MSP430_BASEADDRESS_USCI_A0__,
                  0x00); //RADIO_DELIMITER_CHAR);
    }
    else
    {
      u8Layer2TaskScheduler |= DONE_TX_RADIO_MSG;
    }
}
#endif
//=============================================================================
// end file by xuedong
//=============================================================================

