//=============================================================================
// Header: intrefaceLcd.c
// Date Time: 1/24/2013 10:59 AM
// Author: Xuedong.Liu
//
// Description: it has the two methods that the MSP communicates with I2C display.
//     one of them is set MSP I/O pins to function I2C. but it isn't working. HW
//     is working on solution. The other way is set to I/O pins in generic I/O.
//     software taggle the data line and clocl line.  it works but it use more
//     process time and power.  When use generic I/O pins to control I2C, set MY_IIC
//     in C compiler.
//
//     MSP coneect to LCD as following, use two PCF8576C. address E0 and E1
//
//            LCD PCF8576C                MSP430F5638
//                slave                     master
//            -----------------          -----------------
//           |         SDA (10)|<------>|P8.5/UCB2SDA  XIN|-
//           |                 |        |                 |
//           |                 |        |             XOUT|-
//           |         SCL (11)|<------>|P8.6/UCB2SCL     |
//           |                 |        |                 |
//           |                 |<-------|P8.7/LCD_PWR
//
//  I2C master transmitter sending multiple bytes using the USCI_B2 TX interrupt.
//  reference to MSP430x54x_uscib0_i2c_08.c SMCLK = ~1.045MHz
//
//=============================================================================
//constant definitions, type definitions
#include "masterHeader.h"

//global volatile variables
#include "externGlobalVar.h"

//global function prototypes definitions
#include "externFunc.h"

// constant definitions
// Numeric Conversion Tables for 3 Line Display
// 7 segment characters a-g
// 12 different characters
// 7 different tables to map character value to LCD Driver ram.
/* Numerical Pattern 1
**  byte1
** --------
** abcd fgex  -> x is decimal point
** xxxx xxxx
*/
__flash  tU8 numTable1[13] =             /* DIGIT 1 to 7 */
{
  0xFA, 0x60, 0xd6, /* 0, 1, 2 */
  0xF4, 0x6C, 0xBC, /* 3, 4, 5 */
  0xBE, 0xE0, 0xFE, /* 6, 7, 8 */
  0xEC, 0x04, 0x9E, /* 9, -, E */
  0x00,             /* ' ' */
};

// (Numerical0) pointer to correct numerical table for the current character
const tU8 *pNumTables[] =
{
  (tU8 *) &numTable1[0],
  (tU8 *) &numTable1[0],
  (tU8 *) &numTable1[0],
  (tU8 *) &numTable1[0],
  (tU8 *) &numTable1[0],
  (tU8 *) &numTable1[0],
  (tU8 *) &numTable1[0],
};

// (Numerical0) offset into display ram for characters
const tU8 numRamOffset[] = {14, 15, 16, 17, 18, 19, 20};

// (Decimals0) decimal table
const tU8 decimalTable[] = {0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01};


//*
// Text Conversion Tables For 3 Line Display
// 14 segment characters a-n
// 41 different characters
// 4 different tables to map character value to LCD Driver ram.
//
/*   Top Text Pattern
**  byte1     byte2
** --g-----  -g------
** xe1f,lmjh n2ka,dcbx
** 0xxx,xxxx xxxx,xxx0
*/
const tU8 textTable1[42][2] =
{
  {0x50, 0x1E}, {0x00, 0x06}, {0x60, 0x5A}, /* 0,1,2 */
  {0x20, 0x5E}, {0x30, 0x46}, {0x30, 0x5C}, /* 3,4,5 */
  {0x70, 0x5C}, {0x00, 0x16}, {0x70, 0x5E}, /* 6,7,8 */
  {0x30, 0x56}, {0x70, 0x56}, {0x06, 0x5E}, /* 9,A,B */
  {0x50, 0x18}, {0x06, 0x1E}, {0x70, 0x58}, /* C,D,E */
  {0x70, 0x10}, {0x50, 0x5C}, {0x70, 0x46}, /* F,G,H */
  {0x06, 0x18}, {0x80, 0x0E}, {0x70, 0xA0}, /* I,J,K */
  {0x50, 0x08}, {0x51, 0x26}, {0x51, 0x86}, /* L,M,N */
  {0x50, 0x1E}, {0x70, 0x52}, {0x50, 0x9E}, /* O,P,Q */
  {0x70, 0xD2}, {0x30, 0x5C}, {0x06, 0x10}, /* R,S,T */
  {0x50, 0x0E}, {0x58, 0x20}, {0x58, 0x86}, /* U,V,W */
  {0x09, 0xA0}, {0x05, 0x20}, {0x08, 0x38}, /* X,Y,Z */
  {0x2F, 0xE0}, {0x08, 0x20}, {0x39, 0xE4}, /* *,/,% */
  {0x20, 0x40}, {0x00, 0x00}, {0xff, 0xff}, /* -, ,all on*/
};

/* Top line text character */
const tU8 *pText0Tables[] =
{
  (tU8 *) textTable1,   /* char1 */
  (tU8 *) textTable1,   /* char2 */
  (tU8 *) textTable1,   /* char3 */
  (tU8 *) textTable1,   /* char4 */
  (tU8 *) textTable1,   /* char5 */
  (tU8 *) textTable1,   /* char6 */
  (tU8 *) textTable1    /* char7 */
};

// (Text0) offset into display ram for characters
const tU8 text0RamOffset[] = {0, 2, 4, 6, 8, 10, 12};

/*  Bottom Text Pattern
**  byte1     byte2
** --g-----  --g-----
** xe1f,lmjh dn2k,xcba
** 0xxx,xxxx xxxx,0xxx
*/
const tU8 textTable2[69][2] =
{
  {0x50, 0x87}, {0x00, 0x06}, {0x60, 0xA3}, /* 0,1,2 -> line 1-> last 3*/
  {0x20, 0xA7}, {0x30, 0x26}, {0x30, 0xA5}, /* 3,4,5 -> 2-> 6*/
  {0x70, 0xA5}, {0x00, 0x07}, {0x70, 0xA7}, /* 6,7,8 -> 3->9*/
  {0x30, 0x27}, {0x70, 0x27}, {0x06, 0xA7}, /* 9,A,B -> 4->12*/
  {0x50, 0x81}, {0x06, 0x87}, {0x70, 0xA1}, /* C,D,E -> 5->15*/
  {0x70, 0x01}, {0x50, 0xA5}, {0x70, 0x26}, /* F,G,H -> 6->18*/
  {0x06, 0x81}, {0x40, 0x86}, {0x70, 0x50}, /* I,J,K -> 7->21*/
  {0x50, 0x80}, {0x51, 0x16}, {0x51, 0x46}, /* L,M,N -> 8->24*/  
  {0x50, 0x87}, {0x70, 0x23}, {0x50, 0xC7}, /* O,P,Q -> 9->27*/
  {0x70, 0x63}, {0x30, 0xA5}, {0x06, 0x01}, /* R,S,T -> 10->30*/
  {0x50, 0x86}, {0x58, 0x10}, {0x18, 0x46}, /* U,V,W -> 11->33*/
  {0x09, 0x50}, {0x05, 0x10}, {0x08, 0x91}, /* X,Y,Z -> 12->36*/
  {0x2F, 0x70}, {0x08, 0x10}, {0x39, 0x74}, /* *,/,% -> 13->39*/
  {0x20, 0x20}, {0x00, 0x00}, {0x08, 0xA4}, /* -, ,a -> 14->42*/ // this is index 41 -> a  
  {0x70, 0xC0}, {0x60, 0xA0}, {0x00, 0x00}, /* b,c,d -> 15->45*/
  {0x00, 0x00}, {0x00, 0x00}, {0x00, 0xB6}, /* e,f,g -> 16->48*/
  {0x00, 0x00}, {0x04, 0x00}, {0x00, 0x00}, /* h,i,j -> 17->51*/
  {0x06, 0x50}, {0x00, 0x00}, {0x64, 0x24}, /* k,l,m -> 18->54*/
  {0x64, 0x00}, {0x60, 0xA4}, {0x00, 0x00}, /* n,o,p -> 19->57*/  
  {0x00, 0x00}, {0x60, 0x00}, {0x00, 0x00}, /* q,r,s -> 20->60*/
  {0x70, 0x80}, {0x00, 0x00}, {0x00, 0x00}, /* t,u,v -> 21->63*/
  {0x00, 0x00}, {0x00, 0x00}, {0x00, 0x00}, /* w,x,y -> 22->66*/
  {0x00, 0x00}, {0x7f, 0xf7},               /*  ,all on ->line 23-> 68*/
};

/* (Text1) pointer to correct text table for the current character */
const tU8 *pText1Tables[] =
{
  (tU8 *) textTable2,   /* char15 */
  (tU8 *) textTable2,   /* char16 */
  (tU8 *) textTable2,   /* char17 */
  (tU8 *) textTable2,   /* char18 */
  (tU8 *) textTable2,   /* char19 */
  (tU8 *) textTable2    /* char20 */
};

/* (Text1) offset into display ram for characters */
const tU8 text1RamOffset[] = {21, 23, 25, 27, 29, 31};

//*
// Anunnciator Conversion Table
// and display ram offset table
//
/*                               -      & */
const  tU8 annunciatorTable[] = {0x01, 0x01};
/*                             -  %  */
const  tU8 annunRamOffset[] = {20, 13};

// type definitions

// global variables

// static variables
static tU8 *PTxData;                     // Pointer to TX data
static tU8 TXByteCtr;
static tU8 I2cCmdBuff[5];
static tU8 txBuffer[LCD_DRIVER_RAM_SIZE];
static tU8 j, n, m;                      // for diagnostice LCD
static tI2CDataFrame tI2CFrameChx;

// function prototype definitions:
static void diagSetGlassData(tDisplayData *lcdData);
static void setAllSegments(void);
static void clearAllSegments(void);
static void writeCh1Stop(void);
static void writeCh2Stop(void);
static void writeCh1Data(void);
static void writeCh2Data(void);
static void writeCh1Mode(void);
static void writeCh2Mode(void);
static void initStartDMALcdTx(tU8 *u8CommTxBuf, tU16 u16TxSize);

const tI2cCtrl I2cStateFunc[] =
{
  {CH1_STOP, 0, writeCh1Stop},
  {CH2_STOP, 0, writeCh2Stop},
  {CH1_SET_DATA, 0, writeCh1Data},
  {CH2_SET_DATA, 0, writeCh2Data},
  {CH1_MODE_CHG, 0, writeCh1Mode},
  {CH2_MODE_CHG, 0, writeCh2Mode},
};

static void writeCh1Stop(void)
{
  I2cCmdBuff[0] = IIC_MODE_DISABLE_OUTPUT;
  I2cCmdBuff[1] = IIC_SELECT_DEV0;
  PTxData = I2cCmdBuff;      // TX array start address
  TXByteCtr = 2;
}

static void writeCh2Stop(void)
{
  I2cCmdBuff[0] = IIC_MODE_DISABLE_OUTPUT;
  I2cCmdBuff[1] = IIC_SELECT_DEV1;
  PTxData = I2cCmdBuff;      // TX array start address
  TXByteCtr = 2;
}

static void writeCh1Data(void)
{
  tI2CFrameChx.u8DevModeSet = IIC_MODE_SET_14;
  tI2CFrameChx.u8DataPte = IIC_DATA_PTR;
  tI2CFrameChx.u8SelectDev = IIC_SELECT_DEV0;
  tI2CFrameChx.u8BankCMD = IIC_BANK_SELECT;
  tI2CFrameChx.u8BlinkCMD = IIC_BLINK;//IIC_BANK_SELECT;
  memmove(tI2CFrameChx.u8ChxData, txBuffer, PCF8576_BUFF_SIZE);
  PTxData = (tU8 *)&tI2CFrameChx;      // TX array start address
  TXByteCtr = SEND_LCD_FRAME_SIZE+1;   // for DMA transmit data last char, may not send right.
}

static void writeCh2Data(void)
{
  tI2CFrameChx.u8DevModeSet = IIC_MODE_SET_14;
  tI2CFrameChx.u8DataPte = IIC_DATA_PTR;
  tI2CFrameChx.u8SelectDev = IIC_SELECT_DEV1;
  tI2CFrameChx.u8BankCMD = IIC_BANK_SELECT;
  tI2CFrameChx.u8BlinkCMD = IIC_BLINK;//IIC_BANK_SELECT;
  memmove(tI2CFrameChx.u8ChxData, &txBuffer[PCF8576_BUFF_SIZE], PCF8576_BUFF_SIZE);
  PTxData = (tU8 *)&tI2CFrameChx;      // TX array start address
  TXByteCtr = SEND_LCD_FRAME_SIZE+1;   // for DMA transmit data last char, may not send right.
}

static void writeCh1Mode(void)
{
  I2cCmdBuff[0] = IIC_MODE_SET_14E;
  I2cCmdBuff[1] = IIC_SELECT_DEV0;
  PTxData = I2cCmdBuff;      // TX array start address
  TXByteCtr = 2;
}

static void writeCh2Mode(void)
{
  I2cCmdBuff[0] = IIC_MODE_SET_14E;
  I2cCmdBuff[1] = IIC_SELECT_DEV1;
  PTxData = I2cCmdBuff;      // TX array start address
  TXByteCtr = 2;
}

//*******************************************************************************
// Function: clearAllSegments
//
// Description: Clears all the segments on the display glass.
//
// Assumptions: none
//
// Parameters: none
//
// Returns: none
//
//*******************************************************************************
void clearAllSegments(void)
{
  // Clear the data bytes to clear all segments.
  memset(txBuffer, 0x00, LCD_DRIVER_RAM_SIZE);
}

//*******************************************************************************
// Function: setAllSegments
//
// Description: Sets all the segments on the display glass.
//
// Assumptions: none
//
// Parameters: none
//
//
// Returns: none
//
//*******************************************************************************
static void setAllSegments(void)
{
  // Set the data bytes to set all segments.
  memset(txBuffer, 0xFF, LCD_DRIVER_RAM_SIZE);
}

static void diagSetGlassData(tDisplayData *lcdData)
{
  tU8 i;
  // test top line
  for (i = CHAR1; i <= CHAR5; i++)
  {
    lcdData->topLine[i] = u8LcdChar[m + i];
  }
  m += CHAR5;
  if (m >= EMV_NUM_CHARS)
  {
    m = 0;
  }
  // test bottom line
  for (i = CHAR1; i < CHAR6; i++)
  {
    lcdData->bottomLine[i] = lcdData->bottomLine[i + 1];
  }
  lcdData->bottomLine[CHAR6] = u8LcdChar[j];

  // test numerical
  for (i = NUM1; i < NUM7; i++ )
  {
    lcdData->numLine[i] = lcdData->numLine[i + 1];
  }
  lcdData->numLine[NUM7] = u8LcdNum[n];
  j++;
  if (j == EMV_NUM_CHARS)
  {
    j = 0;
  }
  n++;
  if (n == EMV_NUM_NUMERICAL)
  {
    n = 0;
  }
  // annunc has 7 of them. bit 7 not use, set to 1, what will happens
  if (lcdData->annunc == 0 )
  {
    lcdData->annunc = 1;
  }
  else
  {
    lcdData->annunc = (tU8)lcdData->annunc << 1;
  }

  // decimals in 6 positions, bit 6, 7 not use, set them to 1, what will happen
  if (lcdData->decimals == 0 )
  {
    lcdData->decimals = 1;
  }
  else
  {
    lcdData->decimals = (tU8)lcdData->decimals << 1;
  }

  // bar is in the unit of % from 0% to 100%, 20 bars: 100/20 = 5%/each
  if (lcdData->bargraph >= 100 )
  {
    lcdData->bargraph = 0;
  }
  else
  {
    lcdData->bargraph++;
  }
}

//==============================================================================
//
// FUNCTION:        diagInitLcd
//
// DESCRIPTION:     This function does LCD control pin init and holding time init.
//                  it is a demo how to control each element of LCD.
//
// PARAMETERS:      NAME        TYPE       DESCRIPTION
//                  --------- ---------- ------------------------------------------
//                  bDiagnOn    tBoolean  when it is ON, MSP is in foevere diagnostice loop.
//
// RETURNS:         None
//
// MODIFIES:        the structure tDisplayData to make display as following.
//                  1st text 5 chars a group from u8LcdChar
//                  2nd text walk the chars from u8LcdChar, starting from Char6 to char1
//                  numerical walk the digis from u8LcdNum, starting from Num7 to Num1
//                  annunc, decimals, amd bargraph are fied on a place.
//
// NOTES:
//
//==============================================================================
void diagInitLcd(tBoolean bDiagnOn)
{
  tDisplayData lcdData;

  u8NumTimeInt = 0;
  // set LCD sending data state to 1st state
  I2cStateReg = CH1_STOP;
  initLcdRowNum();

  // use MSP standard I2C function
  while (bDiagnOn == ON)
  {
    // call sending state functions
    I2cStateFunc[I2cStateReg].sendDataFunc();

    // DAM channel 2 trasnmitter buffer data to LCD on I2C protocol
    initStartDMALcdTx(PTxData, TXByteCtr);        // set buffer/size to DAM2 register

    I2C_masterMultiByteSendStart(__MSP430_BASEADDRESS_USCI_B1__,
         *PTxData++);

    // enable DMA
    DMA_enableTransfers (__MSP430_BASEADDRESS_DMAX_6__,
        DMA_CHANNEL_3);

   // lowe power mode, sleep MSP, waite for sending all the data ending interrupt
    __no_operation();                       // Remain in LPM0 until all data

    while (I2C_masterIsSTOPSent (__MSP430_BASEADDRESS_USCI_B1__)
           != I2C_SENDING_STOP)
    {
      _NOP();
    }

    I2cStateReg++;
    // after all the sending state go through in this if
    if (I2cStateReg >= LCD_HOLD_TIME_ON)
    {
        while (I2C_masterIsSTOPSent (__MSP430_BASEADDRESS_USCI_B1__)
                != I2C_SENDING_STOP)
        {
            _NOP();
        }

        // sleep waite for time interrup
        __no_operation();                    // Remain in LPM0 until all data

        // for next time sending the LCD data state, starting from the 1st state.
        I2cStateReg = CH1_STOP;
        // reflash LCD data once
        diagSetGlassData(&lcdData);
        // covert from the chars of top line, buttom line, digi line and etc to the LCD know data
        if (txBuffer[0] == 0xFF)
        {
            clearAllSegments();
        }
        else
        {
            setAllSegments();
        }

        I2C_masterSendStart(__MSP430_BASEADDRESS_USCI_B1__);

        // enable DMA
        DMA_enableTransfers (__MSP430_BASEADDRESS_DMAX_6__,
            DMA_CHANNEL_3);

        // lowe power mode, sleep MSP, waite for sending all the data ending interrupt
        __no_operation();                       // Remain in LPM0 until all data

    }
  }
}

//==============================================================================
//
// FUNCTION:        initDMALcdTx
//
// DESCRIPTION:     DMA is used to transfer a block to I2C port
//                  LCD data. USCIBxTXIFG will trigger DMA.
//==============================================================================
static void initStartDMALcdTx(tU8 *u8CommTxBuf, tU16 u16TxSize)
{
    // trig by UCB1TxIFG = 23

    //Initialize and Setup DMA Channel 3 -> LCD Tx
    // Base Address of the DMA Module
    // Configure DMA channel 3
    // Configure channel for single transfer
    // DMA transfers will be disabled and interrupt flag will be set after every
    //   size transfer
    // Use DMA Trigger Source 23 (UCB1TXIFG), reference data sheet Table 15, DMA trigger assigments
    // Transfer Byte-to-byte
    // Trigger transfer on signal held high
    DMA_init(__MSP430_BASEADDRESS_DMAX_6__,
        DMA_CHANNEL_3,
        DMA_TRANSFER_SINGLE,
        u16TxSize,
        DMA_TRIGGERSOURCE_23,                   // UCB1 Tx IFG
        DMA_SIZE_SRCBYTE_DSTBYTE,               // DMASBDB
        DMA_TRIGGER_HIGH);                      // DMALEVEL

    // Base Address of the DMA Module
    // Configure DMA channel 3
    // Use TxString as source
    // Increment source address after every transfe
    DMA_setSrcAddress(__MSP430_BASEADDRESS_DMAX_6__,
        DMA_CHANNEL_3,
        (unsigned long)u8CommTxBuf,
        DMA_DIRECTION_INCREMENT);               // DMASRCINCR_3

     // Base Address of the DMA Module
     // Configure DMA channel 3
     // Use UCSI B1 Tx Buffer as destination
     // Don't move the destination address after every transfer
    DMA_setDstAddress(__MSP430_BASEADDRESS_DMAX_6__,
        DMA_CHANNEL_3,
        I2C_getTransmitBufferAddressForDMA(__MSP430_BASEADDRESS_USCI_B1__),
        DMA_DIRECTION_UNCHANGED);

    DMA_enableInterrupt(__MSP430_BASEADDRESS_DMAX_6__,
        DMA_CHANNEL_3);                         // DMAIE
}

//===========================================================================
// Function: updateGlass
//
// Description: Takes passed in data, converts to segment data, and
//              sends out the data.
//
// Assumptions: none
//
// Parameters: pDisplayData -- pointer to data to be put on glass.
//
// Modifies: driverData array.
//
// Returns: none
//
//=======================================================================
void updateGlass(tDisplayData *pDisplayData)
{
  tU8 i;
  tU8 tableOffset;
  tU8 localChar;
  tU8 const *pTable;
  tU8 ramOffset;
  tU8 decimalMask;
  tU8 annunMask;

  // Clear display ram
  (void) memset(txBuffer, 0, LCD_DRIVER_RAM_SIZE);


  // Update TEXT0 string
  for ( i = CHAR1; i <= CHAR7; i++)
  {
    localChar = pDisplayData->topLine[i];

    // Check limits and calculate offset into table
    if ((localChar >= 'A') && (localChar <= 'Z'))
    {
      tableOffset = localChar - 55;
    }
    else if ((localChar >= '0') && (localChar <= '9'))
    {
      tableOffset = localChar - 48;
    }
    else if (localChar == ' ')
    {
      tableOffset = 40;
    }
    else if (localChar == '/')
    {
      tableOffset = 37;
    }
    else if (localChar == '%')
    {
      tableOffset = 38;
    }
    else if (localChar == '-')
    {
      tableOffset = 39;
    }
    else
    {
      tableOffset = 41; //36; // Use '*' for any invalid characters.
    }

    pTable = pText0Tables[i];
    ramOffset = text0RamOffset[i];

    /* OR the first byte of each character since it may overlap with
    ** the next character. */
    txBuffer[ramOffset] = pTable[tableOffset * 2];
    txBuffer[ramOffset + 1] = pTable[(tableOffset * 2)+ 1];
  }// End update TEXT0 string

  // Update TEXT1 string
  for ( i = CHAR1; i <= CHAR6; i++)
  {
    localChar = pDisplayData->bottomLine[i];
    // Check limits and calculate offset into table
    if ((localChar >= 'A') && (localChar <= 'Z'))
    {
      tableOffset = localChar - 55;
    }    
    else if ((localChar >= 'a') && (localChar <= 'z'))
    {
      // get index 
      tableOffset = localChar - 56;  //start from a -> index 41;
    }
    else if ((localChar >= '0') && (localChar <= '9'))
    {
      tableOffset = localChar - 48;
    }
    else if (localChar == ' ')
    {
      tableOffset = 40;
    }
    else if (localChar == '/')
    {
      tableOffset = 37;
    }
    else if (localChar == '%')
    {
      tableOffset = 38;
    }
    else if (localChar == '-')
    {
      tableOffset = 39;
    }
    else
    {
      tableOffset = 67; //36; // Use '*' for any invalid characters.
    }

    pTable = pText1Tables[i];
    ramOffset = text1RamOffset[i];

    /* OR the first byte of each character since it may overlap with
    ** the next character. */
    txBuffer[ramOffset] |= pTable[tableOffset * 2];
    txBuffer[ramOffset + 1] = pTable[(tableOffset * 2)+ 1];
  }// End update TEXT1 string

  // Update NUMERICAL0 string
  // Get decimal point and negative sign
  decimalMask = 0x01;

  for ( i = 0; i < 7; i++)
  {
    localChar = pDisplayData->numLine[i];

    // Check numerical0 limits and calculate offset into table
    if ((localChar >= '0') && (localChar <= '9'))
    {
      tableOffset = localChar - 48;
    }
    else if (localChar == '-')
    {
      tableOffset = 10;
    }
    else if (localChar == 'E')
    {
      tableOffset = 11;
    }
    else
    {
      tableOffset = 12; // Out of limits character equals space
    }

    pTable = pNumTables[i];
    ramOffset = numRamOffset[i];

    /* OR both bytes */
    txBuffer[ramOffset] |= pTable[tableOffset];

    // Check decimal point and negative sign
    if (pDisplayData->decimals & decimalMask)
    {
      txBuffer[ramOffset] |= decimalTable[i];
      //txBuffer[ramOffset] |= jt808MapDecimal;
    }

    // Shift the decimal bit mask for the next decimal to the left.
    decimalMask <<= 1;
  }

   // Update ANNUNCIATORS
   annunMask = 1;

   for ( i = 0; i < 6; i++)
   {
     ramOffset = annunRamOffset[i];
     // If annunciator bit is true set the bit in the display ram
     if (pDisplayData->annunc & annunMask)
     {
       txBuffer[ramOffset] |= annunciatorTable[i];
     }
     annunMask <<= 1;
   }
}

void LcdLayer1Driver(void)
{
  if (I2cStateReg >= LCD_HOLD_TIME_ON)
  {
    // for next time sending the LCD data state, starting from the 1st state.
  }
  else
  {
    // has more data to the glass, to make the complete screen
    // call sending LCD data state function, set data that will be send out
    I2cStateFunc[I2cStateReg].sendDataFunc();

    initStartDMALcdTx(PTxData, TXByteCtr);          // set buffer/size to DAM2 register

    // I2C TX, start condition
    I2C_masterMultiByteSendStart(__MSP430_BASEADDRESS_USCI_B1__,
         *PTxData++);

    // enable DMA
    DMA_enableTransfers (__MSP430_BASEADDRESS_DMAX_6__,
        DMA_CHANNEL_3);
    
    I2C_masterSendStart(__MSP430_BASEADDRESS_USCI_B1__);
    
    I2cStateReg++;       // move the LCD data state
  }

  // clear last interrupt handler set the flag
  u8Layer1TaskScheduler &= (tU8)~(INTERF_SEND_LCD_CTRL1);// clear the send flag, the flag is set by interrupt handler
}
//=============================================================================
// end file by xuedong
//=============================================================================
