//=============================================================================
// Header: interfaceModbusStates.c
// Date Time: 1/15/2013 1:59 PM
// Author: Xuedong.Liu
//
// Description: The feature board communicates with Modbus Master.
//
//=============================================================================

//constant definitions, type definitions
#include "masterHeader.h"

//global volatile variables
#include "externGlobalVar.h"

//global function prototypes definitions
#include "externFunc.h"

// constant definitions
// NA

// type definitions
// note for FW update demo host use BASP CRC, 3095, 205, 3808 may use MdCRC algorithm
#define CRC16  MdCRC16 //u16BaspCCIT// MdCRC16

// global variables
//NA

// static variables
tS16   s16CntBytes;         // how many bytes in data field or bytes after FC
tU8    u8AsciiState;        // when modbus is in ASCII mode, state tracking
tU8    u8TempAsciiBuf[2];   // temporary storage for ASCII tU8acters
static COMM_ENV *portEnv;
static RxMsgStruc rxStrucPtrMD[EMV_NUM_COMM];             // Input Buffer and Message Control Structure for Modbus
//static tU8 u8RadioHdB;      // radio header bytes, expected 6 bytes when count for lenght LSB
//static tU8 u8RadioCheckSum; // radio checksum add all byts after legth, at end 0xff - checksum, or + RxCkSum = 0xff
//static tU8 u8RadioEscapeOn; // char need to escaped

// function prototypes definitions:
void initMBrxMsg(RxMsgStruc *rxStrucPtr);
void systemInitMd(COMM_ENV *ptEnv);
tU8 u8CheckParity(tU8 ch);
tU8 u8AsciiToHex(tU8 val);
tU8 u8AsciiStringToHex(tU8 *src);
tU8 checkMBrxStatus(RxMsgStruc *rxStrucPtr, COMM_ENV *ptEnv);

// ASCII mode, state function prototyping
tU8 u8BeginMDAsciiSate(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr);
tU8 u8ReceivedAsciiHiNibble(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr);
tU8 u8ReceivedAsciiLowNibble(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr);
tU8 u8ReceivedAsciiLrcLowNibble(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr);
tU8 u8ReceivedMDAsciiCR(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr);
tU8 u8receivedMDAsciiLF(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr);
// RTU mode, state function prototyping
tU8 u8RxRadioDelimiter(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr);
tU8 u8RxRadioMsbLength(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr);
tU8 u8RxRadioLsbLength(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr);
tU8 radioDeEscapedCheckSum(tU8 *u8RxDate);
tU8 u8ReceivedMDRtuAddress(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr);
tU8 u8ReceivedMDRtuFuncCode(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr);
tU8 u8ReceivedMDRtuRequestData(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr);
tU8 u8ReceivedMDRtuRegCount(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr);
tU8 u8ReceivedMDRtuHiCRC(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr);
tU8 u8ReceivedMDRtuLowCRC(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr);
tU8 u8ReceivedMDRtuByteCount(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr);

//state process table, note: sort ascending by 1st field
const comm_receive_porcess MDasciiStateFuncStruc[] =
{
  {u8ASCII_STATE_BEGIN,       u8BeginMDAsciiSate},          // state = 0
  {u8ASCII_STATE_HI_NIBBLE,   u8ReceivedAsciiHiNibble},     // state = 1
  {u8ASCII_STATE_LOW_NIBBLE,  u8ReceivedAsciiLowNibble},    // state = 2
  {u8ASCII_STATE_LRC1,        u8ReceivedAsciiHiNibble},     // state = 3
  {u8ASCII_STATE_LRC2,        u8ReceivedAsciiLrcLowNibble}, // state = 4
  {u8ASCII_STATE_CR,          u8ReceivedMDAsciiCR},         // state = 5
  {u8ASCII_STATE_LF,          u8receivedMDAsciiLF},         // state = 6
};

//state process table, note: sort ascending by 1st field
const comm_receive_porcess MDrtuStateFuncStruc[] =
{
#if 0
  {u8RTU_STATE_DELIMTER, u8RxRadioDelimiter},        // state = 0
  {u8RTU_STATE_MSB_LEN,  u8RxRadioMsbLength},        // state = 1
  {u8RTU_STATE_LSB_LEN,  u8RxRadioLsbLength},        // state = 2
#endif
  {u8RTU_STATE_ID,    u8ReceivedMDRtuAddress},       // state = 0+3
  {u8RTU_STATE_FC,    u8ReceivedMDRtuFuncCode},      // state = 1+3
  {u8RTU_STATE_LDATA, u8ReceivedMDRtuRequestData},   // state = 2+3
  {u8RTU_STATE_SDATA, u8ReceivedMDRtuRegCount},      // state = 3+3
  {u8RTU_STATE_CRC1,  u8ReceivedMDRtuLowCRC},        // state = 4+3
  {u8RTU_STATE_CRC2,  u8ReceivedMDRtuHiCRC},         // state = 5+3
  {u8RTU_STATE_BCNT,  u8ReceivedMDRtuByteCount},     // state = 6+3
};

//==============================================================================
//
// FUNCTION:        u8CheckParity
//
// DESCRIPTION:     heck a characters parity.
//
// PARAMETERS:      NAME    TYPE       DESCRIPTION
//                  ----- --------  ------------------------------------------
//                  ch      tU8     character need to check
//
// RETURNS:         tU8 - True  = Even Parity,
//                        False = Odd Parity,
//
// PRE-CONDITIONS:  NA
//
// POST-CONDITIONS: NA
//
// MODIFIES:        NA
//
// NOTES:           NA
//
//==============================================================================
tU8 u8CheckParity(tU8 ch)
{
  tU8 n = (tU8)0x00, temp;
  if (ch)
  {
    do
    {
      n++;
      temp = ch-(tU8)0x01;
      ch &= temp;
    } while (ch);
  }
  return ((n & (tU8)0x01) ? FALSE : TRUE);
}

//==============================================================================
//
// FUNCTION:        MdCRC16
//
// DESCRIPTION:     MdCRC16()  does one tU8 CRC-16 calculation messages.
//                  Polynomial is x^16 + x^15 + x^2 + 1.
//                  This routine was adapted from an article in IEEE Micro
//                  magazine, "A Tutorial on CRC Computations", pg 62-76,
//                  August 1988.
//
// PARAMETERS:      NAME        TYPE       DESCRIPTION
//                  -----     --------    ------------------------------------------
//                  *msg        tU8         one byte u8Data comes in
//                  *check      MSG_CRC16   before calculate this byte crc.  it can be
//                                          message other bytes before this coming byte
//                                          or, it can be initial CRC, 0xFFFF or 0x00FF
//                  tS16length  tS16        unused to match with BASP CRC function
// RETURNS:         tU16 - CRC, with coming in byte and input CRC
//
// PRE-CONDITIONS:  NA
//
// POST-CONDITIONS: NA
//
// MODIFIES:        CRC input structure and CRC low and high bytes
//
// NOTES:           This CRC is from 3808 Modbus. it is different then BASP CRC
//                  example: BASP CRC for 0x01, 0x65, 0x01, 0x80 is 0x8F, 0x8B
//                  this one is 0x11, 0xF7.
//                  3095, MVS205, 3808 may use this CRC. need test to determin
//                  For FW update changed to BASP CRC.
//==============================================================================
tU16 MdCRC16(tU8 *msg, tS16 tS16length, MSG_CRC16 *check)
{
  // received tU8acter
  check->crc_l ^=  *msg;
  check->crc.byte[0] = (u8CheckParity(check->crc_l) ? (tU8)0 : (tU8)7);
  check->crc.byte[1] = check->crc_l;

  //==============================================================
  // Rotate crc working area right 1-bit, ^ with last crc low byte
  //==============================================================
  if (check->crc.word & (tU16)0x0001)
  {
    check->crc.word = (check->crc.word >> 1) | (tU16)0x8000;
  }
  else
  {
    check->crc.word = (check->crc.word >> 1);
  }
  check->crc.byte[1] ^= check->crc_l;

  //==============================================================
  //Rotate crc working area right 1-bit, ^ with last crc high byte
  //==============================================================
  if (check->crc.word & (tU16)0x0001)
  {
    check->crc.word = (check->crc.word >> 1) | (tU16)0x8000;
  }
  else
  {
    check->crc.word = (check->crc.word >> 1);
  }
  check->crc.byte[0] ^= check->crc_h;

  // Save New CRC
  check->crc_l = check->crc.byte[0];
  check->crc_h = check->crc.byte[1];

  return(check->crc.word);
}// MdCRC16()

//==============================================================================
//
// FUNCTION:        u8AsciiToHex
//
// DESCRIPTION:     local routine to convert a singl ASCII hex digit to its
//                  binary value.
//
// PARAMETERS:      NAME    TYPE   DESCRIPTION
//                  ----   ------  ------------------------------------------
//                  val    tU8     ASCII digit to be converted is in 0 to 9
//                                 or A to F
//
// RETURNS:         tU8 binary value
//
// PRE-CONDITIONS:  NA
//
// POST-CONDITIONS: NA
//
// MODIFIES:        NA
//
// NOTES:           NA
//
//==============================================================================
tU8 u8AsciiToHex(tU8 val)
{
  if ((val >= '0') && (val <= '9'))
  {
    val -= '0';
  }
  else
  {
    val = (val - 'A') + (tU8)10;
  }
  return val;
} // u8AsciiToHex()

//==============================================================================
//
// FUNCTION:        u8AsciiStringToHex
//
// DESCRIPTION:     Public routine to convert a 2 byte ASCII hex string to a
//                  binary number.  ASCII string is in 0 to 9 or A to F
//
// PARAMETERS:      NAME    TYPE   DESCRIPTION
//                  ------- -----  ------------------------------------------
//                  *src    tU8    pointers to source string
//
// RETURNS:         tU8 hex u8Data
//
// PRE-CONDITIONS:  NA
//
// POST-CONDITIONS: NA
//
// MODIFIES:        NA
//
// NOTES:           NA
//
//==============================================================================
tU8 u8AsciiStringToHex(tU8 *src)
{
  tU8 digit, val;

  //Error[Pm152]: array indexing shall only be applied to objects defined as an array type (MISRA C 2004 rule 17.4)
  digit = u8AsciiToHex(src[0]);
  val = digit << 4;
  //Error[Pm152]: array indexing shall only be applied to objects defined as an array type (MISRA C 2004 rule 17.4)
  digit = u8AsciiToHex(src[1]);
  digit = digit + val;
  return digit;
} // u8AsciiStringToHex

//==============================================================================
//
// FUNCTION:       initMBrxMsg
//
// DESCRIPTION:    prepares to start monitoring the request for a Modbus message.
//                 A call to this function will reset character
//                 pointers, and state variables used by the checkMBrxStatus()
//                 funtion.  All characters already processed by prior calls to
//                 checkMBrxStatus() will be discarded, and the comm variables
//                 associated with receiving a message will be made ready to
//                 start looking for the beginning of the next Modbus message.                                    *
//
// PARAMETERS:      NAME        TYPE          DESCRIPTION
//                  ----------- -----------  ------------------------------------------
//                  *rxStrucPtr RxMsgStruc    Receive message structure pointer
//
// RETURNS:         tU8 hex u8Data
//
// PRE-CONDITIONS:  NA
//
// POST-CONDITIONS: NA
//
// MODIFIES:        NA
//
// NOTES:            This routine is reentrant and may be called from any Modbus
//                   task, or all at the same time without fear of interference.
//                   do we need to reentry???
//==============================================================================
void systemInitMd(COMM_ENV *ptEnv)
{
  initMBrxMsg(&rxStrucPtrMD[ptEnv->u8PortID]);   // initiate a receive
}

void initMBrxMsg(RxMsgStruc *rxStrucPtr)
{
  rxStrucPtr->u8RxState = u8RTU_STATE_ID; //u8RTU_STATE_DELIMTER;
#if 0
  u8RadioCheckSum = 0;    // clear the checkSum
  u8RadioEscapeOn = OFF;  // start with the char is not escaped
#endif
  u8AsciiState               = u8ASCII_STATE_BEGIN;
  rxStrucPtr->u8Index        = u8INIT_CHAR_LOCATION;

  // clear CRC for calculation
  rxStrucPtr->calc.crc_l = rxStrucPtr->calc.crc.byte[0] = u8CLEAR_CRC_CHAR;	  //rxStrucPtr->calc.crc_l =
  rxStrucPtr->calc.crc_h = rxStrucPtr->calc.crc.byte[1] = u8CLEAR_CRC_CHAR;          //rxStrucPtr->calc.crc_h =
  rxStrucPtr->u8LrcChk   = u8CLEAR_LRC_CHAR;
} // initMBrxMsg()

//==============================================================================
//
// FUNCTION:        u8BeginMDAsciiSate
//
// DESCRIPTION:     waiting on ':' start of Modbus ASCII request string
//
// PARAMETERS:      NAME        TYPE       DESCRIPTION
//                  ----------- ---------- ------------------------------------------
//                  *rxStrucPtr RxMsgStruc communication environments structure
//                  *u8DataPtr  tU8        received and look this character
//
// RETURNS:         tU8 receiving status
//
// PRE-CONDITIONS:  NA
//
// POST-CONDITIONS: NA
//
// MODIFIES:        Modbus receiving state veriable
//
// NOTES:           receiving state can move on or in this state
//
//==============================================================================
tU8 u8BeginMDAsciiSate(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr)
{
  if (*u8DataPtr == ASCII_COLON)
  {
    u8AsciiState = u8ASCII_STATE_HI_NIBBLE;
  }
  else
  {
    u8AsciiState = u8ASCII_STATE_BEGIN;
  }
  return RXMSG_ACTIVE;
}

//==============================================================================
//
// FUNCTION:        u8ReceivedAsciiHiNibble
//
// DESCRIPTION:     get high nibble, or 1st LRC byte
//
// PARAMETERS:      NAME        TYPE       DESCRIPTION
//                  ----------- ---------- ------------------------------------------
//                  *rxStrucPtr RxMsgStruc communication environments structure
//                  *u8DataPtr  tU8        received and look this character
//
// RETURNS:         tU8 receiving status
//
// PRE-CONDITIONS:  NA
//
// POST-CONDITIONS: NA
//
// MODIFIES:        Modbus receiving state veriable
//
// NOTES:           receiving state can move on
//
//==============================================================================
tU8 u8ReceivedAsciiHiNibble(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr)
{
  u8TempAsciiBuf[0] = *u8DataPtr;
  if (u8AsciiState == u8ASCII_STATE_LRC1)
  {
    u8AsciiState = u8ASCII_STATE_LRC2;
  }
  else
  {
    u8AsciiState = u8ASCII_STATE_LOW_NIBBLE;
  }
  return RXMSG_ACTIVE;
}

//==============================================================================
//
// FUNCTION:        u8ReceivedAsciiLowNibble
//
// DESCRIPTION:     get low nibble
//
// PARAMETERS:      NAME        TYPE       DESCRIPTION
//                  ----------- ---------- ------------------------------------------
//                  *rxStrucPtr RxMsgStruc communication environments structure
//                  *u8DataPtr  tU8        received and look this character
//
// RETURNS:         tU8 receiving status
//
// PRE-CONDITIONS:  NA
//
// POST-CONDITIONS: NA
//
// MODIFIES:        Modbus receiving state veriable
//
// NOTES:           receiving state goes to where, ASCII_STATE_LRC1, LRC2, or Hi_Nibble
//
//==============================================================================
tU8 u8ReceivedAsciiLowNibble(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr)
{
  u8TempAsciiBuf[1] = *u8DataPtr;
  // u8Data set to single hex tU8
  *u8DataPtr = u8AsciiStringToHex(u8TempAsciiBuf);
  // state, when it is RTU mode
  rxStrucPtr->u8RxState = MDrtuStateFuncStruc[rxStrucPtr->u8RxState].MDStateFuncPtr(rxStrucPtr, u8DataPtr);

  // base on HEX tU8 or RTU mode states, to jump to right ASCII state, or record u8Data, calculate LRC
  if (rxStrucPtr->u8RxState == u8RTU_STATE_CRC1)
  {
    u8AsciiState = u8ASCII_STATE_LRC1;
  }
  else
  {
    // waite to receive next ASCII
    u8AsciiState = u8ASCII_STATE_HI_NIBBLE;
    // record u8Data, record in RTU, not redo it again
    // u8MDRequest[rxStrucPtr->u8Index++] = *u8DataPtr;
    //calculation LRC
    rxStrucPtr->u8LrcChk ^= *u8DataPtr;
  }
  return RXMSG_ACTIVE;
}

//==============================================================================
//
// FUNCTION:        u8ReceivedAsciiLrcLowNibble
//
// DESCRIPTION:     received LRC 2nd byte in ASCII format, convert it to Hex and
//                  compare with calculate LRC. if they match, process message.
//                  if not match, reinit receiving environments, wat for request.
//
// PARAMETERS:      NAME        TYPE       DESCRIPTION
//                  ----------- ---------- ------------------------------------------
//                  *rxStrucPtr RxMsgStruc communication environments structure
//                  *u8DataPtr  tU8        received and look this character
//
// RETURNS:         tU8 receiving status code
//
// PRE-CONDITIONS:  NA
//
// POST-CONDITIONS: NA
//
// MODIFIES:        Modbus receiving state veriable, or environments structure
//
// NOTES:
//
//==============================================================================
tU8 u8ReceivedAsciiLrcLowNibble(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr)
{
  u8TempAsciiBuf[1] = *u8DataPtr;
  *u8DataPtr = u8AsciiStringToHex(u8TempAsciiBuf);
  // compare ASCII LRC received and calculate
  if (*u8DataPtr == rxStrucPtr->u8LrcChk)
  {
    // good CRC move on
    u8AsciiState = u8ASCII_STATE_CR;
  }
  else
  {
    // bad LRC start over, set system dignostic
    initMBrxMsg(&rxStrucPtrMD[portEnv->u8PortID]);   // start all over
    sysRamData.diag.u16CrcErrCnt++;
  }
  return RXMSG_ACTIVE;
}

//==============================================================================
//
// FUNCTION:        u8ReceivedMDAsciiCR, u8receivedMDAsciiLF
//
// DESCRIPTION:     Modbus standard ASCII frame, start with ';' stop with ASCII
//                  'CR' and 'LF'.  if checksum is right to receive CR and LF.
//                  if checksum not right, ignore these two characters.
//
// PARAMETERS:      NAME         TYPE       DESCRIPTION
//                  ------------ ---------- ------------------------------------------
//                  *rxStrucPtr  RxMsgStruc communication environments structure
//                  *u8DataPtr   tU8        received and look this character
//
// RETURNS:         tU8 receiving status
//
// PRE-CONDITIONS:  NA
//
// POST-CONDITIONS: NA
//
// MODIFIES:        Modbus receiving state veriable, or environments structure
//
// NOTES:          go to next state or restart from beginning
//
//==============================================================================
tU8 u8ReceivedMDAsciiCR(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr)
{
  if (*u8DataPtr == ASCII_CR)
  {
    u8AsciiState = u8ASCII_STATE_LF;
  }
  else
  {
    initMBrxMsg(&rxStrucPtrMD[portEnv->u8PortID]);   // start all over
  }
  return RXMSG_ACTIVE;
}
tU8 u8receivedMDAsciiLF(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr)
{
  tU8 u8Status = RXMSG_ACTIVE;
  if (*u8DataPtr == ASCII_LF)
  {
    u8Status = RXMSG_SUCCESS;
  }

  initMBrxMsg(&rxStrucPtrMD[portEnv->u8PortID]);   // start all over
  return u8Status;
}
#if 0
// Note: !!!!!!!!!
// radio length is not used, and radio check sum is not used
// but de-escaped characters is improtent to Rx the request
// escaped char is importent to Tx the response
tU8 radioDeEscapedCheckSum(tU8 *u8RxDate)
{
  tU8 temp = *u8RxDate;
  if ((temp == RADIO_EXCAPED_MARK) && (u8RadioEscapeOn == OFF))
  {
     u8RadioEscapeOn = ON;
  }
  else if (u8RadioEscapeOn == ON)
  {
    temp = temp ^ RADIO_CHAR_XOR_TO;
    u8RadioCheckSum += temp;
    u8RadioEscapeOn = OFF;
    *u8RxDate = temp;
  }
  else
  {
    // it is not escaped mark char, or escaped = off, do nothing.
    // or normal char, do check sum,
    u8RadioCheckSum += temp;
    // return char unchanged
    *u8RxDate = temp;
    // escaped switch is off
    u8RadioEscapeOn = OFF;
  }
  return u8RadioEscapeOn;
}

tU8 u8RxRadioDelimiter(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr)
{
  if (*u8DataPtr == RADIO_DELIMITER_CHAR)
  {
    // this is start delimiter, go to next state
    rxStrucPtr->u8RxState = u8RTU_STATE_MSB_LEN;
    // record u8Data
    u8MDRequest[portEnv->u8PortID][rxStrucPtr->u8Index++] = *u8DataPtr;
    // Radio not do checkSum, modbus protocol not start yet
  }
  else
  {
    initMBrxMsg(&rxStrucPtrMD[portEnv->u8PortID]);   // initiate a receive
  }
  return RXMSG_ACTIVE;
}

tU8 u8RxRadioMsbLength(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr)
{
  // next is radio packet length MSB, go to next state
  rxStrucPtr->u8RxState = u8RTU_STATE_LSB_LEN;
  u8RadioHdB = 0;
  // record u8Data
  u8MDRequest[portEnv->u8PortID][rxStrucPtr->u8Index++] = *u8DataPtr;
  // Radio not do checkSum, modbus protocol not start yet
  return RXMSG_ACTIVE;
}

tU8 u8RxRadioLsbLength(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr)
{
  union
  {
    tU8 u8LenB[2];
    tU16 u16LenInt;
  } radioMsgLen;
  if(u8RadioHdB >= 1)
  {
    // do radio checkSum
    u8RadioEscapeOn = radioDeEscapedCheckSum(u8DataPtr);
    if (u8RadioEscapeOn == OFF)
    {
        // next is radio packet length MSB, go to next state
        if (u8RadioHdB >= RADIO_HEADER_BYTES)
        {
          // exit from this state, get complete header bytes, move to next state
          rxStrucPtr->u8RxState = u8RTU_STATE_ID;
        }
        // record u8Data, no need modbus protocal, the modebus char not in yet
        u8MDRequest[portEnv->u8PortID][rxStrucPtr->u8Index++] = *u8DataPtr;
    }
  }
  else
  {
     // this byte is LSB size of Radio RF package
     radioMsgLen.u8LenB[1] = u8MDRequest[portEnv->u8PortID][1];
     radioMsgLen.u8LenB[0] = *u8DataPtr;

     // record u8Data, no need modbus protocal, the modebus char not in yet
     u8MDRequest[portEnv->u8PortID][rxStrucPtr->u8Index++] = *u8DataPtr;

     if (radioMsgLen.u16LenInt < RADIO_HEADER_LENGTH)
     {
       // do not care this small msg
        initMBrxMsg(&rxStrucPtrMD[portEnv->u8PortID]);   // initiate a receive
     }
  }
  u8RadioHdB++;
  return RXMSG_ACTIVE;
}
#endif
//==============================================================================
//
// FUNCTION:        u8ReceivedMDRtuAddress
//
// DESCRIPTION:     look received char, it is my address or it is preamble.
//                  if it is preamble waite for address field.
//                  if it is my address go to next state, receive FC
//                  if it is not my address/preamble. in this state, look
//                  preamble or my address
//
// PARAMETERS:      NAME        TYPE       DESCRIPTION
//                  ----------- ---------- ------------------------------------------
//                  *rxStrucPtr RxMsgStruc communication environments structure
//                  *u8DataPtr  tU8        received and look this character
//
// RETURNS:         tU8 receiving status
//
// PRE-CONDITIONS:  NA
//
// POST-CONDITIONS: NA
//
// MODIFIES:        Modbus receiving state veriable, or environments structure
//
// NOTES:          go to next state or restart from beginning
//
//==============================================================================
tU8 u8ReceivedMDRtuAddress(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr)
{
#if 0 //def __JT808_MSP430F5638__
  // do radio checkSum
  u8RadioEscapeOn = radioDeEscapedCheckSum(u8DataPtr);
  if (u8RadioEscapeOn == OFF)
#endif
  {
      if ((*u8DataPtr == 1 /*infoA.comm.u8ModbusAddr*/) ||
          (*u8DataPtr == MD_3095BROADCAST_ADDR) ||
          (*u8DataPtr == MD_205BROADCAST_ADDR))
      {
        // the reauest is for me, go to next state, and set inter char timeout enable
        rxStrucPtr->u8RxState = u8RTU_STATE_FC;
        // only the message if for this device, start to care the inter char error.
        // this is a way, driver goes back normal when the host deaded in the middle sending request
        //startRxCharTimeOut();
        // record u8Data
        u8MDRequest[portEnv->u8PortID][rxStrucPtr->u8Index++] = *u8DataPtr;
        // calculate check sum
        rxStrucPtr->calc.crc.word = CRC16(u8DataPtr, 1, &rxStrucPtr->calc);
      }
      else
      {
        initMBrxMsg(&rxStrucPtrMD[portEnv->u8PortID]);   // initiate a receive
      }
  }
  return RXMSG_ACTIVE;
}
//==============================================================================
//
// FUNCTION:        u8ReceivedMDRtuFuncCode
//
// DESCRIPTION:     after received Trasnmitter Modbus Address is match with internal
//                  address veraible, come in here top get Modbus Function Code
//
// PARAMETERS:      NAME        TYPE        DESCRIPTION
//                  -------     ----------  ------------------------------------------
//                  *rxStrucPtr RxMsgStruc  communication environments structure
//                  *u8DataPtr  tU8         received and look this character
//
// RETURNS:         tU8 receiving status code
//
// PRE-CONDITIONS:  NA
//
// POST-CONDITIONS: NA
//
// MODIFIES:        Modbus receiving state veriable, or environments structure,
//                  or calculate checksum fro incoming request.
//
// NOTES:          Function code support, 01, 02, 03, 04, 05, 08, 15, 16, 69, 70
//
//==============================================================================
tU8 u8ReceivedMDRtuFuncCode(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr)
{
#if 0 //def __JT808_MSP430F5638__
  // do radio checkSum
  u8RadioEscapeOn = radioDeEscapedCheckSum(u8DataPtr);
  if (u8RadioEscapeOn == OFF)
#endif
  {
      // record u8Data
      u8MDRequest[portEnv->u8PortID][rxStrucPtr->u8Index++] = *u8DataPtr;
      // calculate crc for u8Data
      rxStrucPtr->calc.crc.word = CRC16(u8DataPtr, 1, &rxStrucPtr->calc);

      if (((*u8DataPtr <= MD_CMD_FC05) && (*u8DataPtr >= MD_CMD_FC01)) ||
          (*u8DataPtr == MD_CMD_FC69) || (*u8DataPtr == MD_CMD_FC08))
      {
        // cmd supported, and request has 4 bytes u8Data after FC + 2 bytes CRC
        s16CntBytes = SHORT_MD_CMD_CNT;
        // next state is to receive short request
        rxStrucPtr->u8RxState = u8RTU_STATE_SDATA;
      }
      else if ((*u8DataPtr == MD_CMD_FC15) || (*u8DataPtr == MD_CMD_FC16) ||
          (*u8DataPtr == MD_CMD_FC70))
      {
        // cmd supported, and request has 5 bytes u8Data after FC, 5th byte is count for u8Data field + 2 bytes CRC
        s16CntBytes = LONG_MD_CMD_CNT;
        // next state is to receive long request, the 5th byte is u8Data field longth
        rxStrucPtr->u8RxState = u8RTU_STATE_LDATA;
      }
      else if (*u8DataPtr == MD_CMD_FC06)
      {
        // cmd supported, and request has 6 bytes u8Data after FC + 2 bytes CRC
        s16CntBytes = LONG_MD_CMD_CNT + 1;
        // next state is to receive short request
        rxStrucPtr->u8RxState = u8RTU_STATE_SDATA;
      }
      else if (*u8DataPtr == MD_CMD_FC65)
      {
        // next state is to receive field update firmware request byte count
        rxStrucPtr->u8RxState = u8RTU_STATE_BCNT;
      }
      else
      {
        //u8RxDataPtr = rxStrucPtr->msg_start;	            // start over where is looked position
        initMBrxMsg(&rxStrucPtrMD[portEnv->u8PortID]);   // initiate a receive
      }
  }
  return RXMSG_ACTIVE;
}
//==============================================================================
//
// FUNCTION:        u8ReceivedMDRtuRequestData
//
// DESCRIPTION:     received request has the format as:
//
//                  addr, FC, 2bytes start register, 2bytes quantity of registers,
//                  count u8Data, u8Datafiled, 2bytes crc
//
//                  after JT808 received address, Function code.  the request
//                  has u8Data field, it come in here receiving u8Datafields....
//
// PARAMETERS:      NAME         TYPE       DESCRIPTION
//                  ------------ ---------- ------------------------------------------
//                  *rxStrucPtr  RxMsgStruc communication environments structure
//                  *u8DataPtr   tU8        received and look this character
//
// RETURNS:         tU8 receiving status code
//
// PRE-CONDITIONS:  NA
//
// POST-CONDITIONS: NA
//
// MODIFIES:        Modbus receiving state veriable,
//                  or calculate checksum fro incoming request.
//
// NOTES:          when u8Data field coounter is zero, go to receiving crc state
//
//==============================================================================
tU8 u8ReceivedMDRtuRequestData(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr)
{
#if 0 //def __JT808_MSP430F5638__
  // do radio checkSum
  u8RadioEscapeOn = radioDeEscapedCheckSum(u8DataPtr);
  if (u8RadioEscapeOn == OFF)
#endif
  {
      // record u8Data
      u8MDRequest[portEnv->u8PortID][rxStrucPtr->u8Index++] = *u8DataPtr;
      // calculate crc for u8Data
      rxStrucPtr->calc.crc.word = CRC16(u8DataPtr, 1, &rxStrucPtr->calc);

      --s16CntBytes;
      if (s16CntBytes == 0)	                    // determine remaining bytes
      {
        // last byte is u8Data field count.
        s16CntBytes = (tS16)*u8DataPtr;
        rxStrucPtr->u8RxState = u8RTU_STATE_SDATA;
      }
  }
  return RXMSG_ACTIVE;
}

//==============================================================================
//
// FUNCTION:        u8ReceivedMDRtuRegCount
//
// DESCRIPTION:     received request has the format as:
//
//                  addr, FC, 2bytes start register, 2bytes quantity of registers,
//                  count u8Data, u8Datafiled, 2bytes crc
//
//                  after JT808 received address, Function code.  the request
//                  no u8Data field, it come in here.  after received last byte,
//                  jump to received crc high byte.
//
// PARAMETERS:      NAME        TYPE       DESCRIPTION
//                  ----------- ---------- ------------------------------------------
//                  *rxStrucPtr RxMsgStruc communication environments structure
//                  *u8DataPtr  tU8        received and look this character
//
// RETURNS:         tU8 receiving status code
//
// PRE-CONDITIONS:  NA
//
// POST-CONDITIONS: NA
//
// MODIFIES:        Modbus receiving state veriable,
//                  or calculate checksum for incoming request.
//
// NOTES:           request start register and quantity of register fields.
//
//==============================================================================
tU8 u8ReceivedMDRtuRegCount(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr)
{
#if 0 //def __JT808_MSP430F5638__
  // do radio checkSum
  u8RadioEscapeOn = radioDeEscapedCheckSum(u8DataPtr);
  if (u8RadioEscapeOn == OFF)
#endif
  {
      // record u8Data
      u8MDRequest[portEnv->u8PortID][rxStrucPtr->u8Index++] = *u8DataPtr;
      // calculate crc for u8Data
      rxStrucPtr->calc.crc.word = CRC16(u8DataPtr, 1, &rxStrucPtr->calc);

      --s16CntBytes;
      if (s16CntBytes == 0)
      {
        rxStrucPtr->u8RxState = u8RTU_STATE_CRC1;
      }
  }
  return RXMSG_ACTIVE;
}
//==============================================================================
//
// FUNCTION:        u8ReceivedMDRtuLowCRC, u8ReceivedMDRtuHiCRC
//
// DESCRIPTION:     received request has the format as:
//
//                  addr, FC, 2bytes start register, 2bytes quantity of registers,
//                  count u8Data, u8Datafiled, 2bytes crc
//
//                  received crc low byte because 3095, 205 send low byte 1st,
//                  jump to received crc high byte.
//                  if recevied crc = calculate crc, good message porcess it.
//                  otherwise, the request is bad message, state starts all over.
//
// PARAMETERS:      NAME         TYPE       DESCRIPTION
//                  ------------ ---------- ------------------------------------------
//                  *rxStrucPtr  RxMsgStruc communication environments structure
//                  *u8DataPtr   tU8        received and look this character
//
// RETURNS:         tU8 receiving status
//
// PRE-CONDITIONS:  NA
//
// POST-CONDITIONS: NA
//
// MODIFIES:        Modbus receiving state veriable,
//                  or calculate checksum fro incoming request.
//
// NOTES:           request start register and quantity of register fields.
//
//==============================================================================
tU8 u8ReceivedMDRtuLowCRC(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr)
{
#if 0 //def __JT808_MSP430F5638__
  // do radio checkSum
  u8RadioEscapeOn = radioDeEscapedCheckSum(u8DataPtr);
  if (u8RadioEscapeOn == OFF)
#endif
  {
      // record u8Data
      u8MDRequest[portEnv->u8PortID][rxStrucPtr->u8Index++] = *u8DataPtr;
      rxStrucPtr->u8RxState = u8RTU_STATE_CRC2;
  }
  return RXMSG_ACTIVE;
}
tU8 u8ReceivedMDRtuHiCRC(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr)
{
  union
  {
    tU16    word;
    tU8     byte[2];
  }rec_crc;
  tU8 u8Status;
#if 0 //def __JT808_MSP430F5638__
  // do radio checkSum
  u8RadioEscapeOn = radioDeEscapedCheckSum(u8DataPtr);
  if (u8RadioEscapeOn == OFF)
#endif
  {
      rec_crc.byte[0] = u8MDRequest[portEnv->u8PortID][rxStrucPtr->u8Index - (tU8)1];
      rec_crc.byte[1] = *u8DataPtr;

      // The CRC for whole frame
      if (rec_crc.word == rxStrucPtr->calc.crc.word)
      {
        // least 2 for crc
        u8MDRequest[portEnv->u8PortID][rxStrucPtr->u8Index++] = *u8DataPtr;
        // when it is 205 mode, check the register limits, rec_crc not use any more
        rec_crc.byte[1] = u8MDRequest[portEnv->u8PortID][MD_IDX_MS_START_REG];
        rec_crc.byte[0] = u8MDRequest[portEnv->u8PortID][MD_IDX_LS_START_REG];
        // where did we stop look received byte
        u8Status = RXMSG_SUCCESS;
#if 0 //def __JT808_MSP430F5638__
        // take out the Radio RF header
        memmove(&u8MDRequest[portEnv->u8PortID][0],
                &u8MDRequest[portEnv->u8PortID][8],
                rxStrucPtr->u8Index);
#endif
      }
      else
      {
        //==========================================================
        // bad CRC, start over, set system diagnostic bit
        //==========================================================
        initMBrxMsg(&rxStrucPtrMD[portEnv->u8PortID]);   // initiate a receive
        //u8Status = RXMSG_CHECKSUM_ERRR;
        u8Status = RXMSG_ACTIVE;
        sysRamData.diag.u16CrcErrCnt++;
      }
  }
  return u8Status;
}
//==============================================================================
//
// FUNCTION:        u8ReceivedMDRtuByteCount
//
// DESCRIPTION:     this is BSL field update firmware message
//                  BSL core CMD start with message length, one byte long
//                  this function is waiting for this byte.
//
// PARAMETERS:      NAME         TYPE       DESCRIPTION
//                  ------------ ---------- ------------------------------------------
//                  *rxStrucPtr  RxMsgStruc communication environments structure
//                  *u8DataPtr   tU8        received and look this character
//
// RETURNS:         tU8 receiving status
//
// PRE-CONDITIONS:  NA
//
// POST-CONDITIONS: NA
//
//==============================================================================
tU8 u8ReceivedMDRtuByteCount(RxMsgStruc *rxStrucPtr, tU8 *u8DataPtr)
{
#if 0 //def __JT808_MSP430F5638__
  // do radio checkSum
  u8RadioEscapeOn = radioDeEscapedCheckSum(u8DataPtr);
  if (u8RadioEscapeOn == OFF)
#endif
  {
      // record u8Data
      u8MDRequest[portEnv->u8PortID][rxStrucPtr->u8Index++] = *u8DataPtr;
      // cmd supported, and request has 6 bytes u8Data after FC + 2 bytes CRC
      s16CntBytes = *u8DataPtr;
      // next state is to receive short request
      rxStrucPtr->u8RxState = u8RTU_STATE_SDATA;
      // calculate crc for u8Data
      rxStrucPtr->calc.crc.word = CRC16(u8DataPtr, 1, &rxStrucPtr->calc);
  }
  return RXMSG_ACTIVE;
}
//==============================================================================
//
// FUNCTION:        checkMBrxStatus
//
// DESCRIPTION:     it parses a MODBUS message into the u8MDRequest[] array.
//                  All verifications of proper message length, framing and
//                  CRC/LRCs are done here. If a good messge is received to
//                  the proper address, this function returns the number of
//                  binary bytes in the message from the address field to the
//                  end of u8Data inclusive.  If no message is received, and
//                  the communication port is inactive, it returns FALSE.
//
// PARAMETERS:      NAME         TYPE         DESCRIPTION
//                  -------      ----------   -------------------------------------
//                  *rxStrucPtr  RxMsgStruc   communication environments structure
//
// RETURNS:         tU8 u8Status = RXMSG_nnnnnn u8Status code
//
// PRE-CONDITIONS:  NA
//
// POST-CONDITIONS: NA
//
// MODIFIES:        NA
//
// NOTES:           NA
//
//==============================================================================
tU8 checkMBrxStatus(RxMsgStruc *rxStrucPtr, COMM_ENV *ptEnv)
{
    tU8 u8Data = u8CommInbuf[ptEnv->u8PortID][ptEnv->u8LookCharPtr];
    tU8 u8Status = RXMSG_ACTIVE;

    // Modbus is in RTU mode
    u8Status = MDrtuStateFuncStruc[rxStrucPtr->u8RxState].MDStateFuncPtr(rxStrucPtr, &u8Data);
    if (rxStrucPtr->u8RxState > u8RTU_STATE_ID) //u8RTU_STATE_DELIMTER)
    {
        if(ptEnv->u8PortID == RADIO_PORTID)
        {
            // radio only Tx message, should not get in here
        }
        // USB prot driver take are the inter char time error up to 64K
    }
    // inter chars timout check in controlUART file
    return u8Status;
} // checkMBrxStatus()

//==============================================================================
//
// FUNCTION:      ModbusLayer1Driver
//
// DESCRIPTION:   MODBUS slave task.  As long as the carrier detect line
//                remains active, this task will monitor the communication Port
//                if address is this unit, it is good Modbus request, it is
//                support Modbus function.  the respond will be send out to the
//                host within timeout or turn-around time.
//
// PARAMETERS:      NAME    TYPE       DESCRIPTION
//                  ----- ---------- ------------------------------------------
//
// RETURNS:         tU16 modbus timer, current counts
//
// PRE-CONDITIONS:  NA
//
// POST-CONDITIONS: NA
//
// MODIFIES:        NA
//
// NOTES:           NA
//
//==============================================================================
void ModbusLayer1Driver(COMM_ENV *ptEnv)
{
  tU8 u8CommRxStatus;
  portEnv = ptEnv;
  // check for incomming tU8acters
  u8CommRxStatus = checkMBrxStatus(&rxStrucPtrMD[ptEnv->u8PortID], ptEnv);

  if (u8CommRxStatus == RXMSG_SUCCESS)
  {
    // request is modbus message
    ptEnv->u8DriverStatus |= MSG_MEET_MD_STD;
    if(ptEnv->u8PortID == RADIO_PORTID)
    {
        // disable inter char interrupts
        ptEnv->u8RxCharTimeOutStatus &= (tU8)~STARTS_CHAR_TIMEOUT;
    }

    // restart RX
    initMBrxMsg(&rxStrucPtrMD[ptEnv->u8PortID]);
  } // end else
}// ModbusLayer1Driver()

//=============================================================================
// end file by xuedong
//=============================================================================
