//=============================================================================
// Header: JT808 Utilities
// Date Time: 1/25/2013 12:58 PM
// Author: Xuedong.Liu
//
// Description: JT808 use following function to modify the varibale in layer3
//
//=============================================================================
//constant definitions, type definitions
#include "masterHeader.h"

//global volatile variables
#include "externGlobalVar.h"

//global function prototypes definitions
#include "externFunc.h"


// constant definitions
// 4 PVs type = 0 -> DP; 1 -> SP; 2 -> PT; 3 -> ST;
// 10 samples
#define TOTAL_SAMPLES  4 //max 16
#define PV_TYPES       4
tF32 f32AverageOver[17] = // TOTAL_SAMPLES max is 16
  {0.0f, 1.0f, 2.0f, 3.0f, 4.0f, 5.0f, 6.0f, 7.0f, 8.0f, 9.0f,
  10.0f, 11.0f, 12.0f, 13.0f, 14.0f, 15.0f, 16.0f};

tF32 f32Samples[PV_TYPES][TOTAL_SAMPLES]; // data buffer
tU8  u8SmpIdx[PV_TYPES];                  // sample points
tU8  u8BufIdx[PV_TYPES];                  // buffer index

// type definitions

// global variables

// static variables

// function prototype definitions:

void packAscii(tU8 *asciiSrc, tU8 *packedDest, tU8 asciiLength);
tU8 unpackAscii(tU8 *packedSrc, tU8 *asciiDest, tU8 packedLength);
void cmdFbMspReset(void);
tF32 applyDamping(CAL_PV_STRUC *inInfo);
tF32 applyTrims(CAL_PV_STRUC *inInfo);

/*************************************************************************
*  Function:  applyDamping() from 3808 /w modification Date: 01-Sep-2002 *
*________________________________________________________________________*
*                                                                        *
*   Description: The minimum damping is 0.125sec, Maxmum Damping is 30sec*
*                B deamping value = 0.125/inInfo->fpDampValue,           *
*                damping on dampled y(n) = B*x(n) + (1-B)*y(n-1),        *
*                damping off y(n) = x(n)                                 *
*                                                                        *
*   Parameters: float inInfo->fpDampValue = B -> damping Time value      *
*               float inInfo->fpOldValue = y(n-1) -> damped value        *
*               float inInfo->fpNewValue = x(n) -> current sample value  *
*               float inInfo->f32SampingTime = PV sampling time          *
*               tU8   inInfo->u8OnState                                  *
*   Returns:     The damped value y(n) for the indicated variable.       *
*                                                                        *
**************************************************************************/
tF32 applyDamping(CAL_PV_STRUC *inInfo)
{
  tF32 B, dampedValue = inInfo->f32NewValue;

  //==========================================================
  // calculate B, this could put in changing sample/damping times so that function runtime is shorter
  //==========================================================
  B = (inInfo->f32SampingTime)/(inInfo->f32DampTime + inInfo->f32SampingTime);

  //==========================================================
  // calculate damped value
  //==========================================================
  dampedValue = (B * inInfo->f32NewValue) + ((1.0f - B) * (inInfo->f32OldValue));

  return dampedValue;
}

tF32 applyTrims(CAL_PV_STRUC *inInfo)
{
  inInfo->f32NewValue *= inInfo->f32TrimSlop;
  inInfo->f32NewValue += inInfo->f32TrimOffset;
  return inInfo->f32NewValue;
}

tF32 applyScale(CAL_PV_STRUC *inInfo)
{
  // scale to count, y
  inInfo->f32NewValue = (inInfo->u16FullScaleY2 / inInfo->f32PvUrl) * inInfo->f32NewValue;
  // x = (y - b)/m
  inInfo->f32NewValue -= inInfo->u16ScaleOffset;
  inInfo->f32NewValue /= inInfo->u16ScaleSlope;
  return inInfo->f32NewValue;
}

tF32 convertPresUnitsStd(CAL_PV_STRUC *inInfo)
{
  tU8 i;
  tF32 f32InFactor, f32OutFactor;
  // default is PSI
  f32InFactor = f32OutFactor = 1.0f;
  // in unit and out unit is same no action.
  if (inInfo->u8InUnit != inInfo->u8OutUnit)
  {
    // if in unit isn't out unit, convert f32NewValue to/from standard
    if (inInfo->u8InUnit != U_psi)
    {
      // in unit isn't PSI convert live to PSI
      for (i = 0; i < P_CONV_TABLE_SIZE; i++)
      {
        if (inInfo->u8InUnit == presConvTable[i].u8LookUnit)
        {
          f32InFactor = presConvTable[i].f32Factor2Std;
          break;
        }
      }
      if(i >= P_CONV_TABLE_SIZE)
      {
        // didn't find the unit code
        f32InFactor = 1.0f;
      }
    }
    if(inInfo->u8OutUnit != U_psi)
    {
      // out unit isn't PSI convert live from PSI
      for (i = 0; i < P_CONV_TABLE_SIZE; i++)
      {
        if (inInfo->u8OutUnit == presConvTable[i].u8LookUnit)
        {
          f32OutFactor = presConvTable[i].f32Factor2Std;
          break;
        }
      }
      if(i >= P_CONV_TABLE_SIZE)
      {
        // didn't find the unit code
        f32OutFactor = 1.0f;
      }
    }
    inInfo->f32UnitsCnvtF = f32OutFactor/f32InFactor;
    f32InFactor = inInfo->f32NewValue * inInfo->f32UnitsCnvtF;
    inInfo->f32NewValue = f32InFactor;
  }

  return inInfo->f32NewValue;
}

tF32 convertPtUnitsStd(CAL_PV_STRUC *inInfo)
{
  tU8 i;
  // in unit and out unit is same no action.
  if (inInfo->u8InUnit != inInfo->u8OutUnit)
  {
    // if in unit isn't out unit, convert f32NewValue to/from standard
    if (inInfo->u8InUnit != U_celsius)
    {
      // in unit isn't Celsius convert live to Celsius
      for (i = 0; i < PT_CONV_TABLE_SIZE; i++)
      {
        if (inInfo->u8InUnit == ptConvTable[i].u8LookUnit)
        {
          inInfo->f32NewValue -= ptConvTable[i].f32Offset2Std;
          inInfo->f32NewValue *= ptConvTable[i].f32Numer2Std;
          inInfo->f32NewValue /= ptConvTable[i].f32Denomin2Std;
          break;
        }
      }
      if(i >= PT_CONV_TABLE_SIZE)
      {
        // didn't find the unit code, no change made, set status
      }
    }

    if (inInfo->u8OutUnit != U_celsius)
    {
      // out unit isn't Celsius convert live from Celsius
      for (i = 0; i < PT_CONV_TABLE_SIZE; i++)
      {
        if (inInfo->u8OutUnit == ptConvTable[i].u8LookUnit)
        {
          inInfo->f32NewValue *= ptConvTable[i].f32Denomin2Std;
          inInfo->f32NewValue /= ptConvTable[i].f32Numer2Std;
          inInfo->f32NewValue += ptConvTable[i].f32Offset2Std;
          break;
        }
      }
      if(i >= PT_CONV_TABLE_SIZE)
      {
        // didn't find the unit code, no change made, set status
      }
    }
  }

  return inInfo->f32NewValue;
}

tU32 calcLive2RawCnt(CAL_PV_STRUC *inInfo)
{
  // note: 1. f32PvUrl and f32PvLrl use sensor reported Saturation Limit
  // note: 2. f32NewValue is before trim, damping, and unit conversion
  tU32 u32OutRawCnt;
  // coounts per process variable value.
  tF32 f32CntPerValue = (CNT3095_HI_LIMIT - CNT3095_LO_LIMIT)/(inInfo->f32PvUrl - inInfo->f32PvLrl);
  // current value has this many counts
  f32CntPerValue = ((inInfo->f32NewValue - inInfo->f32PvLrl) * f32CntPerValue);

  u32OutRawCnt = (tU32)f32CntPerValue;
  return u32OutRawCnt;
}

void calcDpUnitchange (void)
{
  tF32 *chgDpUnit[LINK_DP_UNIT_CHANGE] =
  {
    &sysRamData.dp[0].f32DpLRL,           //0   convert sensor lower range limit
    &sysRamData.dp[0].f32DpURL,           //1   convert sensor upper range limit
    &infoB.snsDpCfg[0].f32DpSpanCal,     //6   convert span calibration apply pressure
    &infoB.snsDpCfg[0].f32DpLRV,         //7   convert scale lower operating limit
    &infoB.snsDpCfg[0].f32DpURV,         //8   convert scale upper operating limit
//    &infoB.snsDpCfg[0].f32ScaleDpX1,     //9
//    &infoB.snsDpCfg[0].f32ScaleDpX2,     //10
  };

  CAL_PV_STRUC inRef;
  tU8  u8Idx = 0;         // the recalculate variable index, variable associate DP unit change
  tF32 f32DpConvtFac;     // DP unit canvert factor

  // clear flag
  u8DoDiagErrorProcess &= (tU8)~CALC_DP_UNIT_CHANGE;

  inRef.u8InUnit = sysRamData.dp[0].u8DpOldUnits;
  inRef.u8OutUnit = infoD.xmt.u8DpUnits;
  inRef.f32NewValue = *chgDpUnit[u8Idx];
  *chgDpUnit[u8Idx] = convertPresUnitsStd(&inRef);

  f32DpConvtFac = inRef.f32UnitsCnvtF;
  // write sysBuf.mfgSns[0].f32DpLRL (4 bytes) to EEPROM, because u8Idx = 0
  //reqeustRightAccessFlash(); //u8Layer1TaskScheduler |= INTERF_WRITE_FLASH;

  for (u8Idx = 1; u8Idx < LINK_DP_UNIT_CHANGE; u8Idx++)
  {
    // recalcuate new value for DP unit associate variables
    *chgDpUnit[u8Idx] = f32DpConvtFac * (*chgDpUnit[u8Idx]);
    // write following u8Idx = 0 veriables (4 bytes/each) to EEPROM
 //    reqeustRightAccessFlash(); //u8Layer1TaskScheduler |= INTERF_WRITE_FLASH;
  }
  sysRamData.dp[0].u8DpOldUnits = infoD.xmt.u8DpUnits;
}
void calcSpUnitchange (void)
{
  tF32 *chgSpUnit[LINK_SP_UNIT_CHANGE] =
  {
    &sysRamData.sp[0].f32SpLRL,           //0   convert sensor lower range limit
    &sysRamData.sp[0].f32SpURL,           //1   convert sensor upper range limit
    &infoC.snsSpCfg[0].f32SpSpanCal,       //6   convert span calibration apply pressure
    &infoC.snsSpCfg[0].f32SpLRV,           //7   convert scale lower operating limit
    &infoC.snsSpCfg[0].f32SpURV,           //8   convert scale upper operating limit
//    &infoC.snsSpCfg[0].f32ScaleSpX1,       //9
//    &infoC.snsSpCfg[0].f32ScaleSpX2,       //10
  };
  CAL_PV_STRUC inRef;
  tU8  u8Idx = 0;         // the recalculate variable index, variable associate DP unit change
  tF32 f32SpConvtFac;     // SP unit canvert factor
  // clear flag
  u8DoDiagErrorProcess &= (tU8)~CALC_SP_UNIT_CHANGE;

  inRef.u8InUnit = sysRamData.sp[0].u8SpOldUnits;
  inRef.u8OutUnit = infoD.xmt.u8SpUnits;
  inRef.f32NewValue = *chgSpUnit[u8Idx];
  *chgSpUnit[u8Idx] = convertPresUnitsStd(&inRef);

  f32SpConvtFac = inRef.f32UnitsCnvtF;
  // write sysBuf.mfgSet[0].f32DpLRL (4 bytes) to EEPROM, because u8Idx = 0
  //reqeustRightAccessFlash(); //u8Layer1TaskScheduler |= INTERF_WRITE_FLASH;

  for (u8Idx = 1; u8Idx < LINK_SP_UNIT_CHANGE; u8Idx++)
  {
    // recalcuate new value for DP unit associate variables
    *chgSpUnit[u8Idx] = f32SpConvtFac * (*chgSpUnit[u8Idx]);
    // write following u8Idx = 0 veriables (4 bytes/each) to EEPROM
//    reqeustRightAccessFlash(); //u8Layer1TaskScheduler |= INTERF_WRITE_FLASH;
  }
    sysRamData.sp[0].u8SpOldUnits = infoD.xmt.u8SpUnits;
}
void calcPtUnitchange (void)
{
  tF32 *chgDpUnit[LINK_PT_UNIT_CHANGE] =
  {
    &sysRamData.pt.f32PtLRL,           //0   convert sensor lower range limit
    &sysRamData.pt.f32PtURL,           //1   convert sensor upper range limit
    &infoA.pt.f32RtdLRV,          //7   convert scale lower operating limit
    &infoA.pt.f32RtdURV,          //8   convert scale upper operating limit
    &infoA.pt.f32ScalePtX1,       //9
    &infoA.pt.f32ScalePtX2,       //10
    &infoA.pt.f32RtdZeroOffset,   //11
  };
  CAL_PV_STRUC inRef;
  tU8  u8Idx;         // the recalculate variable index, variable associate DP unit change

  inRef.u8InUnit = sysRamData.pt.u8PtOldUnits;
  inRef.u8OutUnit = infoD.xmt.u8RtdUnits;
  for (u8Idx = 0; u8Idx < LINK_PT_UNIT_CHANGE; u8Idx++)
  {
    inRef.f32NewValue = *chgDpUnit[u8Idx];

    *chgDpUnit[u8Idx] = convertPtUnitsStd(&inRef);
    // write following u8Idx = 0 veriables (4 bytes/each) to EEPROM
//     reqeustRightAccessFlash(); //u8Layer1TaskScheduler |= INTERF_WRITE_FLASH;
  }
    sysRamData.pt.u8PtOldUnits = infoD.xmt.u8RtdUnits;
}

//
// FUNCTION: packAscii
//
// DESCRIPTION:
//  This function converts 4 bytes ASCII string into 3 bytes of packed string.
//  the asciiLength should be an ordinal multiple of 4.
//  ASCII string length = 4   Packed string length = 3
//                      = 8                        = 6
//                      = 12                       = 9
//                      ....
//                      = 32                       = 24
// INPUTS:
//             asciiSrc    tU8 *        Where to start reading.
//             packedDest  tU8 *        Where to start writing.
//             asciiLength tU8 *        Number of ascii characters to convert.
//
// RETURNS:
//      NONE
//
// CAVEATS:
//      none
//
//
void packAscii(tU8 *asciiSrc, tU8 *packedDest, tU8 asciiLength)
{
  tU8 i, j;

  // prepare ascii source
  for(i = 0; i < asciiLength; i++)
  {
    // Check for end of string
    if (asciiSrc[i] == 0)
    {
      break;
    }
    // Check for invalid characters
    if (asciiSrc[i] < ASCII_SPACE_HEX)
    {
      // force ASCII space
      asciiSrc[i] = ASCII_SPACE_HEX;
    }
    // Check for lower case
    if (asciiSrc[i] > ASCII_LOW_CASE_ST)
    {
      // force upper case
      asciiSrc[i] -= ASCII_SPACE_HEX;
    }
  }

  // Write ASCII spaces in the remaining bytes
  for(; i < asciiLength; i++)
  {
    // force space
    asciiSrc[i] = ASCII_SPACE_HEX;
  }

  // pack ascii source to packed destination
  j = 0;
  for (i = 0; i < asciiLength; i+=4)
  {
    packedDest[j]      = (tU8)(asciiSrc[i]     & USE_LOW6BITS) << 2;
    packedDest[j]     += (tU8)(asciiSrc[i + 1] & USE_LOW6BITS) >> 4;
    packedDest[j + 1]  = (tU8)(asciiSrc[i + 1] & USE_LOW6BITS) << 4;
    packedDest[j + 1] += (tU8)(asciiSrc[i + 2] & USE_LOW6BITS) >> 2;
    packedDest[j + 2]  = (tU8)(asciiSrc[i + 2] & USE_LOW6BITS) << 6;
    packedDest[j + 2] += (tU8)(asciiSrc[i + 3] & USE_LOW6BITS);
    j+=3;
  }
}


//
// FUNCTION:  unpackAscii
//
//
// DESCRIPTION:
//  This function unpacks 3 bytes packed string into 4 bytes ASCII string.
//  the packedLength should be an ordinal multiple of 3.
//  Packed string length = 3    ASCII string length = 4
//                       = 6                        = 8
//                       = 9                        = 12
//                      ....
//                       = 24                       = 32
//     Converts with the following rules:
//     (1) set unPacked (8 bits) = packed (6 bits)
//     (2) set unPacked bit 7 = 0
//     (3) set unPacked bit 6 = complement of unPacked bit 5.
//
// INPUTS:
//             packedSrc    tU8 *        Where to start reading.
//             asciiDest    tU8 *        Where to start writing.
//             packedLength tU8          Number of bytes to unpack.
//
// RETURNS:
//      Unpacked Length
//
// CAVEATS:
//      none
//
//
tU8 unpackAscii(tU8 *packedSrc, tU8 *asciiDest, tU8 packedLength)
{
  tU8 i, j, k;
  k = 0;
  tU8 asciiTemp[4];

  // unpack packed source to ascii destination
  for(i=0; i<packedLength; i+=3)
  {
    asciiTemp[0] = packedSrc[i]      >> 2;
    asciiTemp[1] = (tU8)(packedSrc[i]   & USE_LOW2BITS) << 4;
    asciiTemp[1]+= packedSrc[i+1] >> 4;
    asciiTemp[2] = (tU8)(packedSrc[i+1] & USE_LOW4BITS) << 2;
    asciiTemp[2]+= packedSrc[i+2] >> 6;
    asciiTemp[3] = packedSrc[i+2]   & USE_LOW6BITS;

    for(j=0; j<4; j++)
    {
      // test unpacked ASCII char bit5
      if ((asciiTemp[j] & ASCII_SPACE_HEX) == ASCII_SPACE_HEX)
      {
        // unpacked ASCII bit5 is 1, set bit6 = 0
        asciiDest[k] = asciiTemp[j];
      }
      else // must be in the 0x40s or 0x50s
      {
        // unpacked ASCII bit5 is 0, set bit6 = 1
        asciiDest[k] = asciiTemp[j] | SET_BIT6_ONE;
      }
      k++;
    }
  }
  return ((tU8)k);
}

void cmdFbMspReset(void)
{
  //this instruction forces system reset
  WDTCTL = WRONG_WDTPW + WDTHOLD;
}


tF32 takeAwayBigSmalF32Data(tU8 u8Type, tF32 f32Live)
{
  tF32 f32Temp, f32LargerOne, f32SmallOne;
  tU8 i;
  // save sample data in buffer
  f32Samples[u8Type][u8BufIdx[u8Type]] = f32Live;
  u8BufIdx[u8Type]++;                           // buffer point increase 1
  if (u8BufIdx[u8Type] >= 0x02)
  {
    u8BufIdx[u8Type] = 0x00;
  }

  f32Temp = f32LargerOne = f32SmallOne = f32Samples[u8Type][0];
  for (i = 1; i < 3; i++)
  {
    if (f32Samples[u8Type][i] > f32LargerOne)
    {
       f32LargerOne = f32Samples[u8Type][i];
    }
    else if (f32Samples[u8Type][i] < f32SmallOne)
    {
      f32SmallOne = f32Samples[u8Type][i];
    }
    else
    {
      f32Temp = f32Samples[u8Type][i];
    }
  }
  return f32Temp;
}

tF32 averageF32Data(tU8 u8Type, tF32 f32Live)
{
  tF32 f32Temp, f32LargerOne, f32SmallOne;
  tU8 i;
  f32Temp = f32Live;
  // save sample data in buffer
  f32Samples[u8Type][u8BufIdx[u8Type]] = f32Live;
  u8BufIdx[u8Type]++;                           // buffer point increase 1
  u8BufIdx[u8Type] &= TOTAL_SAMPLES - 1;        // save point not over size

  if (u8SmpIdx[u8Type] >= (TOTAL_SAMPLES - 1))
  {
    // alwasy 16
    u8SmpIdx[u8Type] = TOTAL_SAMPLES;
    f32LargerOne = f32SmallOne = f32Samples[u8Type][0];
    for (i = 1; i < u8SmpIdx[u8Type]; i++)
    {
      if (f32Samples[u8Type][i] > f32LargerOne)
      {
        f32LargerOne = f32Samples[u8Type][i];
      }
      else if (f32Samples[u8Type][i] < f32SmallOne)
      {
        f32SmallOne = f32Samples[u8Type][i];
      }
      else
      {
        // between larger and small do the avarage
      }
    }
    f32Temp = 0.0f;
    for (i = 0; i < u8SmpIdx[u8Type]; i++)
    {
      f32Temp += f32Samples[u8Type][i];
    }
    f32Temp = f32Temp - f32LargerOne - f32SmallOne;

    f32Temp /= f32AverageOver[u8SmpIdx[u8Type] - 2];
  }
  else
  {
    // 1, 2, 3,...16
    u8SmpIdx[u8Type]++;
  }

  return f32Temp;
}

void reInitCalVaribles(void)
{
  infoB.snsDpCfg[0].f32DpZeroTrim = DEFAULT_ZERO_OFFSET;
  infoB.snsDpCfg[0].f32DpSpanTrim = DEFAULT_SPAN_TRIM;
  infoB.snsDpCfg[0].f32DpSpanCal = 249.791f;
  infoC.snsSpCfg[0].f32SpZeroTrim = DEFAULT_ZERO_OFFSET;
  infoC.snsSpCfg[0].f32SpSpanTrim = DEFAULT_SPAN_TRIM;
  infoC.snsSpCfg[0].f32SpSpanCal = 800.0f;
  infoA.pt.f32RtdZeroTrim = DEFAULT_ZERO_OFFSET;
  infoA.pt.f32RtdSpanTrim = DEFAULT_SPAN_TRIM;
  //infoA.pt.f32PtCalZero = -40.0f;
  //infoA.pt.f32PtCalSpan = 752.0f;
}
//=============================================================================
// end file by xuedong
//=============================================================================
