//=============================================================================
// Header: mspDataExchange
// Date Time: 2/21/2013 12:06:26 PM
// Author: Xuedong.Liu
//
// Description: This file has two MSP430 SPI RAM data exchange funcitons.
//     each MSP use two RAM block to send request, status request line and
//     one toggle line.  if other MSP not present, directly use the FLash/RAM
//     if other MSP is present use this to exachange data.  each block is
//     128 bytes = MSP430F5438A information flash size. or ADS1218 flash size.
//     the MSP430F5638 may not need information on ADS1218.  it do need
//     MSP430F5438A information flash data. it is user configuration data.
//
//=============================================================================
// constant definitions, type definitions
#include "masterHeader.h"

// global volatile variables
#include "externGlobalVar.h"

// global function prototypes definitions:
#include "externFunc.h"

// local statice function prototypes definitions:
static void done5638WritRamData(void);
static void txRamData(void);
static void endUseRamSpi(void);
static void abbitritioRamSpiBus(void);
static tU8 modifyIdx(tU8 idx);
static void done5638ReadRamData(void);
static tU8 findInBuf(tU8 data);
static void startTxRamData(void);
static tU8 ramRdWtDummy(RAM_DATA_PROC *inInfor);
static void addoneStartAllOver(void);

static tU8 cpData4LCD(RAM_DATA_PROC *inInfor);
static tU8 wt5638Flash(RAM_DATA_PROC *inInfor);

// local static variables
static tU8 u8TempType,   // which type of data exchange in proress
           u8RamBusFree, // this MSP control the RAM SPI bus
           u8RxStateOn;  // this MSP is in Rx mode or Tx mode1
static tU8 u8Retry;
tU8 askExchgType[MAX_TYPE_SIZE];  //remember the all the type exchanges
tU8 u8AddInIdx,   // from where to add new type
    u8DoneIdx;    // from where to process the type
const RAM_SPI_ACCESS ramCtrl[] = {
    {RAM_EX_BLK1_START, (tU8 *)&infoA,              USED_INFOA_SIZE,   TYPE_INFOA,        NOTHING_TO_DO, ramRdWtDummy},
    {RAM_EX_BLK1_START, (tU8 *)&infoB,              USED_INFOB_SIZE,   TYPE_INFOB,        NOTHING_TO_DO, ramRdWtDummy},
    {RAM_EX_BLK1_START, (tU8 *)&infoC,              USED_INFOC_SIZE,   TYPE_INFOC,        NOTHING_TO_DO, ramRdWtDummy},
    {RAM_EX_BLK1_START, (tU8 *)&infoD,              USED_INFOD_SIZE,   TYPE_INFOD,        CPY_SNS1_4LCD, ramRdWtDummy},
    {RAM_EX_BLK1_START, (tU8 *)&sysRamData.pt,      RAM_PT_SIZE,       TYPR_RAM_PT,       NOTHING_TO_DO, ramRdWtDummy},
    {RAM_EX_BLK1_START, (tU8 *)&sysRamData.rtcV,    RAM_CALENDER_SIZE, TYPE_CALENDER,     NOTHING_TO_DO, ramRdWtDummy},
    {RAM_EX_BLK1_START, (tU8 *)&sysRamData.diag,    RAM_DIAG_SIZE,     TYPE_DIAG,         NOTHING_TO_DO, ramRdWtDummy},
    {RAM_EX_BLK1_START, (tU8 *)&sysRamData.xmt,     RAM_XMT_SIZE,      TYPE_RAM_XMT,      NOTHING_TO_DO, ramRdWtDummy},
    {RAM_EX_BLK1_START, (tU8 *)&sysRamData.dp[0],   RAM_DP_SIZE,       TYPE_RAM_DP_SNS1,  NOTHING_TO_DO, ramRdWtDummy},
    {RAM_EX_BLK1_START, (tU8 *)&sysRamData.sp[0],   RAM_SP_SIZE,       TYPE_RAM_SP_SNS1,  NOTHING_TO_DO, ramRdWtDummy},
    {RAM_EX_BLK1_START, (tU8 *)&sysRamData.st[0],   RAM_ST_SIZE,       TYPE_RAM_ST_SNS1,  NOTHING_TO_DO, ramRdWtDummy},
    {RAM_EX_BLK1_START, (tU8 *)&sysRamData.dp[1],   RAM_DP_SIZE,       TYPE_RAM_DP_SNS2,  NOTHING_TO_DO, ramRdWtDummy},
    {RAM_EX_BLK1_START, (tU8 *)&sysRamData.sp[1],   RAM_SP_SIZE,       TYPE_RAM_SP_SNS2,  NOTHING_TO_DO, ramRdWtDummy},
    {RAM_EX_BLK1_START, (tU8 *)&sysRamData.st[1],   RAM_ST_SIZE,       TYPE_RAM_ST_SNS2,  NOTHING_TO_DO, ramRdWtDummy},
    {RAM_EX_BLK1_START, (tU8 *)&sysBuf.mfgPg1[0],   MFG_PAGE1_SIZE,    TYPE_MFG_PG1_SNS1, NOTHING_TO_DO, ramRdWtDummy},
    {RAM_EX_BLK1_START, (tU8 *)&sysBuf.mfgPg1[1],   MFG_PAGE1_SIZE,    TYPE_MFG_PG1_SNS2, NOTHING_TO_DO, ramRdWtDummy},
    {RAM_EX_BLK1_START, (tU8 *)&sysRamData.apiPlng, PLNG_VAR_SIZE,     TYPE_AP_PLNG_LIFT, NOTHING_TO_DO, ramRdWtDummy},

    {RAM_EX_BLK1_START, (tU8 *)&sysRamData.exchg8A[0], RAM_EX8A_SIZE,  TYPE_EX8A_SNS1,    CPY_SNS1_4LCD, cpData4LCD},
    {RAM_EX_BLK1_START, (tU8 *)&sysRamData.exchg8A[1], RAM_EX8A_SIZE,  TYPE_EX8A_SNS2,    CPY_SNS2_4LCD, cpData4LCD},
    {RAM_EX_BLK1_START, (tU8 *)&sysRamData.exchg6Pv,   RAM_EX6V_SIZE,  TYPE_EX6_PVS,      NOTHING_TO_DO, ramRdWtDummy},        
    {RAM_EX_BLK1_START, (tU8 *)&info5638A,          INFO_5638A_SIZE,   TYPE_INFO5638A,    WT_INFOA,      wt5638Flash}, 
    {RAM_EX_BLK1_START, (tU8 *)&info5638B,          INFO_5638B_SIZE,   TYPE_INFO5638B,    WT_INFOB,      wt5638Flash},  
    {RAM_EX_BLK1_START, (tU8 *)&info5638C,          INFO_5638C_SIZE,   TYPE_INFO5638C,    WT_INFOC,      wt5638Flash},  
};
#define LKUP_ITEM  lengthof(ramCtrl)

// global & local function implementation
void initExchgIdx(void)
{
    u8RamBusFree = YES;
    if(u8RxStateOn != ALLOW_OTHER_USE_BUS)
    {
      u8RxStateOn = INITIAL_STATE;
    }
    endUseRamSpi();
    u8AddInIdx = u8DoneIdx = 0;
    memset(askExchgType, NO_XCHG, MAX_TYPE_SIZE);
}

static void addoneStartAllOver(void)
{
  addOneTypeExchg(TYPE_INFO5638A);
}

static void endUseRamSpi(void)
{
    // SPI RAM bus is free
    GPIO_setOutputLowOnPin(
        __MSP430_BASEADDRESS_PORT4_R__,
        GPIO_PORT_P4,
        P4_3_5638_USE_RAM_OUT);
    __no_operation();
}

static tU8 modifyIdx(tU8 idx)
{
    // increase index
    (idx)++;
    // modify index
    (idx) &= (tU8)(MAX_TYPE_SIZE - 1);
    return idx;
}

static tU8 findInBuf(tU8 data)
{
  tU8 i, temp;
  for (i = 0; i < MAX_TYPE_SIZE; i++)
  {
    if(askExchgType[i] == data)
    { 
      // no need add this type in buffer
      temp = 1;
      break; 
    }
    if(askExchgType[i] == NO_XCHG)
    {
      // not find in buffer, add it in
      temp = 0;
      break;
    }
    if(i > u8AddInIdx)
    {
      // something wrong fix it
      while(1)
      {
         temp = 0;     
      }
    }
  }
  return temp;
}
void addOneTypeExchg(tU8 data)
{    
    tU8 temp = findInBuf(data);

    if (temp == 0)
    {
      WTVALUE(1);
      WTVALUE(data);

      // record which block of data exchanges
      askExchgType[u8AddInIdx] = data;
      // point to next record space.
      u8AddInIdx = modifyIdx(u8AddInIdx);
    }

    if (u8RxStateOn == INITIAL_STATE)
    {
      if(u8AddInIdx > u8DoneIdx)
      {
        WTVALUE(2);
        // initiates data exchange
        startTxRamData();
      }
      else
      {
        // keep inintial state, nothing to do
      }
    }
    else
    {
      if(u8RxStateOn != ALLOW_OTHER_USE_BUS)
      {
        WTVALUE(3);
        // wait current state is done
        u8RxStateOn = HAS_EXCHAGE_TODO;
        // set timer for re-request, ACK
        ramDataExchangeTimerStart(BUS_ACK_WAIT);
      }
      // wait othe rdone write data to RAM
    }
}

tU8 getEchgType(void)
{
  return u8TempType;
}

static void abbitritioRamSpiBus(void)
{
    u8RamBusFree = NO;
    // check bus in use
    if ((P4IN & P4_1_5438A_USE_RAM_IN) != P4_1_5438A_USE_RAM_IN)
    {
        __delay_cycles(EXCNG_RECEK);
        //check again
        if ((P4IN & P4_1_5438A_USE_RAM_IN) != P4_1_5438A_USE_RAM_IN)
        {
            // this msp use SPI RAM bus
            GPIO_setOutputHighOnPin(__MSP430_BASEADDRESS_PORT4_R__,
                GPIO_PORT_P4,
                P4_3_5638_USE_RAM_OUT);
            u8RamBusFree = YES;
        }
    }
}

static void requestRamSpiBus(void)
{
  // toggle the request line
  GPIO_setOutputHighOnPin(__MSP430_BASEADDRESS_PORT5_R__,
       GPIO_PORT_P5,
       P5_3_5638_REQ_RAM_DATA_EX_OUT);
  __delay_cycles(5);

  GPIO_setOutputLowOnPin (__MSP430_BASEADDRESS_PORT5_R__,
       GPIO_PORT_P5,
       P5_3_5638_REQ_RAM_DATA_EX_OUT);
  __no_operation();  
}

//******************Tx start******************
// request use SPI RAM bus
// wait for ACK or wait for timeout re-try request 
static void startTxRamData(void)
{
        WTVALUE(4);
  // it starts data exchange
  u8RxStateOn = THIS_START_RAM_EXCH;
  // request use SPI RAM bus
  requestRamSpiBus();
  // set timer for re-request, ACK
  ramDataExchangeTimerStart(BUS_ACK_WAIT);            
}

static tU8 getXchgType(void)
{
  while ((askExchgType[u8DoneIdx] == NO_XCHG) && (u8DoneIdx != u8AddInIdx))
  {
    u8DoneIdx = modifyIdx(u8DoneIdx);
  }
  return u8DoneIdx;
}
// this msp get the ack from other MSP
// it can start to send data in RAM
void mspAckReqTxRx(void)
{
  // not request, wil not get ACK
  u8Layer1TaskScheduler &= (tU8)~MSP_ACK_REQ_RAM;
  ctrlRamBusTxData();
}

static void ctrlRamBusTxData(void)
{
  // take over SPI bus
  abbitritioRamSpiBus();

  if (u8RamBusFree == YES)
  {
      // state wait othe rto read
      u8RxStateOn = WAIT_OTHER_DONE_READ;

      // do the start
      u8DoneIdx = getXchgType();
      u8TempType = askExchgType[u8DoneIdx];
       WTVALUE(4);
       WTVALUE(u8TempType);
      
      // if write is not successful, the other MSP read all zero, or last in RAM data
      txRamData();
      // free SPI bus
      endUseRamSpi();

      // release RAM SPI bus control
      u8RamBusFree = YES;

      __no_operation();
      
      // toggle pin to casue MSP 5638 interrupt service
      done5638WritRamData();
  }
  else
  {
      // state unchange
      u8RxStateOn = THIS_START_RAM_EXCH;
  }
  // set timer for re-request or done read
  ramDataExchangeTimerStart(It_WAIT_DONE_RD);            
}

static void txRamData(void)
{
    tU8 i;
    RAM_SPI_ACCESS tempRamCtrl;
    tU8 tempdata[128];
    // enable the RAM spi CS
    ramCsEnable();
    
    memset(tempdata, 0, 128);
    // set RAM SPI bus
    enableRamSpi();

    for (i = 0; i < LKUP_ITEM ; i ++)
    {
        if (ramCtrl[i].u8Type == u8TempType)
        {
            tempRamCtrl.u32RamAddr = ramCtrl[i].u32RamAddr;
            memmove(tempdata, ramCtrl[i].u8DataAddr, ramCtrl[i].u8cnt);
            tempRamCtrl.u8DataAddr = tempdata;
            tempRamCtrl.u8cnt = ramCtrl[i].u8cnt;
            tempRamCtrl.u8Type = ramCtrl[i].u8Type;

            // send temp data out
            ramWtSpiData(&tempRamCtrl);

           break;
        }
    }

    // reset RAM SPI bus
    disableRamSpi();
    
    // disbale the RAM spi CS
    ramCsDisable();
}

void ramTimerWaitDone(void)
{
            WTVALUE(12);
 // clear flag
  u8Layer2TaskScheduler &= (tU8)~TIMEOUT_RAM_WAIT;
  if(u8Retry < 10)
  { 
    u8Retry++;
  }
  if((u8RxStateOn == THIS_START_RAM_EXCH) ||
     (u8RxStateOn == HAS_EXCHAGE_TODO))
  {
      // test bus free & try Tx again
      startTxRamData();
  }
  else if(u8RxStateOn == WAIT_OTHER_DONE_READ)
  {
      // toggle pin to casue MSP 5638 interrupt service
      done5638WritRamData(); 
      // set tiem for othe rto read
      ramDataExchangeTimerStart(BUS_ACK_WAIT);            
  }
  else if((u8RxStateOn == READ_RAM_DATA_START) ||
          (u8RxStateOn == RD_THEN_WT))
  {
      // last time did not read data, re-try 
      ctrlRamBusRxData();
  }
  else
  {

  }
}

void doneTxRxRamData(void)
{
  tU8 temp = getEchgType();
  // clear flag
  u8Layer2TaskScheduler &= (tU8)~RD_RAM_DATA_DONE;
  u8DoneIdx = modifyIdx(u8DoneIdx);
  u8RxStateOn = INITIAL_STATE;
  if((temp < TYPE_INFOD) && (temp >= TYPE_INFOA))
  {
    temp++;
    addOneTypeExchg(temp);
       WTVALUE(61);
  }
  else
  {
       WTVALUE(5);
    // more data to Tx and Rx
    if (u8AddInIdx > u8DoneIdx)
    {
      // has more data do it again
      startTxRamData();
       WTVALUE(6);
    }
    else
    {
         WTVALUE(7);
      // stop timer
      RamTimerStop();
      // done all data exchange from this MSP
      initExchgIdx();
    }
  }
}

static void done5638WritRamData(void)
{
            WTVALUE(11);
   // toggle the request line
    GPIO_setOutputHighOnPin(__MSP430_BASEADDRESS_PORT5_R__,
         GPIO_PORT_P5,
         P5_4_5638_WT_RAM_DONE_OUT);

     __delay_cycles(5);

    GPIO_setOutputLowOnPin (__MSP430_BASEADDRESS_PORT5_R__,
         GPIO_PORT_P5,
         P5_4_5638_WT_RAM_DONE_OUT);
}

//******************Rx start******************
// otehr MSP write data to RAM
// let this MSP to read data, this MSP receiver the 
// read rquest, and start to read data from RAM
// the max data exchange size is 128 bytes
void ctrlRamBusRxData(void)
{
  tU8 i, u8tempBuf[DATA_BUF_SIZE], errCode;
  RAM_SPI_ACCESS rdCtrl;
  RAM_DATA_PROC InfoIn;
  // clear flag
  u8Layer2TaskScheduler &= (tU8)~RX_RD_RAM_DATA;
  if(u8RxStateOn != INITIAL_STATE)
  {
    u8RxStateOn = RD_THEN_WT;
  }
  else
  {
    u8RxStateOn = READ_RAM_DATA_START;
  }

  // init temperal read buffer
  memset(u8tempBuf, 0, DATA_BUF_SIZE);
  rdCtrl.u8DataAddr = u8tempBuf;
  rdCtrl.u32RamAddr = RAM_EX_BLK1_START;

  // test is RAM SPI aviable
  abbitritioRamSpiBus();
  if (u8RamBusFree == YES)
  {
#if 1
    GPIO_setOutputHighOnPin(__MSP430_BASEADDRESS_PORT6_R__,
        GPIO_PORT_P6,
        P6_2_ACLK_DEBUG);
#endif
      // enable the RAM spi CS
      ramCsEnable();
      // set RAM SPI bus
      enableRamSpi();
      // read data from RAM
      ramRdSpiData(&rdCtrl);

      // reset RAM SPI bus
      disableRamSpi();

      // release RAM SPI bus control
      u8RamBusFree = YES;

      // find wich block save this RAM data
      for (i = 0; i < LKUP_ITEM ; i ++)
      {
        if (ramCtrl[i].u8Type == rdCtrl.u8Type)
        {
           WTVALUE(8);
           WTVALUE(rdCtrl.u8Type);
           // move the data from temperal to this MSP
           memmove(ramCtrl[i].u8DataAddr, u8tempBuf, rdCtrl.u8cnt);
           
           if(ramCtrl[i].u8Opt != NOTHING_TO_DO)
           {
             // which block to do this process
             InfoIn.u8optDo = ramCtrl[i].u8Opt;
             // size right now not in use
             errCode = ramCtrl[i].procFunc(&InfoIn);
           }
           __no_operation();
          break;
        }
      }
      // disbale the RAM spi CS
      ramCsDisable();
      // each time doen read, add this MSP data to exchange
      done5638ReadRamData();
      
      // is this one has data to Tx to other
      if ((u8RxStateOn == RD_THEN_WT) || (u8AddInIdx > u8DoneIdx))
      {
        u8RxStateOn = THIS_START_RAM_EXCH;
        u8DoneIdx = modifyIdx(u8DoneIdx);
        if(u8AddInIdx == u8DoneIdx)
        {
          endUseRamSpi();
          initExchgIdx();         
        }
        else
        {
          // start Tx data 
          mspAckReqTxData();
        }
      }
      else
      {
        // low Port4 Pin2, done use RAM
        endUseRamSpi();
        // this MSP no data need to use RAM SPI bus
        initExchgIdx();
      }
      // no data to other, other can done or keep going.
      __no_operation();
      if(u8Retry >= 10)
      {
        // timeout restart all over, on this side
        initExchgIdx();
        addoneStartAllOver();
      }
      // reset error
      u8Retry = 0;
#if 1
      GPIO_setOutputLowOnPin(__MSP430_BASEADDRESS_PORT6_R__,
        GPIO_PORT_P6,
        P6_2_ACLK_DEBUG);
#endif
  }
  else 
  {
               WTVALUE(14);
      u8Retry++;
      // set timer for re-request
      ramDataExchangeTimerStart(BUS_ACK_WAIT); 
  }
}

static void done5638ReadRamData(void)
{
  // toggle the request line
  GPIO_setOutputHighOnPin(__MSP430_BASEADDRESS_PORT4_R__,
        GPIO_PORT_P4,
        P4_5_5638_DONE_RD_RAM_D_OUT); 

  __delay_cycles(5);
           WTVALUE(10);

  GPIO_setOutputLowOnPin (__MSP430_BASEADDRESS_PORT4_R__,
         GPIO_PORT_P4,
         P4_5_5638_DONE_RD_RAM_D_OUT); 
}

//********************MSP430F5638 can has SPI RAM bus*************
void testGiveAck(void)
{
  if((u8RxStateOn == INITIAL_STATE) || 
     (u8RxStateOn == WAIT_OTHER_DONE_READ))
  {
           WTVALUE(9);
           // send ACK to other MSP,
      GPIO_setOutputHighOnPin(__MSP430_BASEADDRESS_PORT4_R__,
          GPIO_PORT_P4,
          P4_7_ACK_RAM_REQ_OUT); 

      __delay_cycles(5);

      GPIO_setOutputLowOnPin (__MSP430_BASEADDRESS_PORT4_R__,
           GPIO_PORT_P4,
           P4_7_ACK_RAM_REQ_OUT); 
  }
}
//=============================================================================
// end file by xuedong
// do process after data received
//=============================================================================
static tU8 ramRdWtDummy(RAM_DATA_PROC *inInfor)
{
  __no_operation();
  return 0;
}

static tU8 cpData4LCD(RAM_DATA_PROC *inInfor)
{
  sysRamData.dp[0].f32DpLive = sysRamData.exchg8A[0].f32DpLive;
  sysRamData.sp[0].f32SpLive = sysRamData.exchg8A[0].f32SpLive;
  sysRamData.st[0].f32StLive = sysRamData.exchg8A[0].f32StLive;
  sysRamData.pt.f32PtLive = sysRamData.exchg8A[0].f32PtLive;
  return 0;
}

static tU8 wt5638Flash(RAM_DATA_PROC *inInfor)
{
    // write data to EEPROM
  if((inInfor->u8optDo & WT_INFOA) == WT_INFOA)
  {
    u8OtherTaskScheduler |= DYNAMIC_WT_IMFOA;  // do EEP section A write process
  }
  if((inInfor->u8optDo & WT_INFOB) == WT_INFOB)
  {
    u8OtherTaskScheduler |= DYNAMIC_WT_IMFOB;  // do EEP section B write process
    chge5638CtrlItems();             // msp5438A changed control items
  }
  if((inInfor->u8optDo & WT_INFOC) == WT_INFOC)
  {
    u8OtherTaskScheduler |= DYNAMIC_WT_IMFOC;  // do EEP section C write process
  }
#if 0
  // it is free at 2014-10-09
  if((inInfor->u8optDo & WT_INFOA) == WT_INFOA)
  {
    u8OtherTaskScheduler |= DYNAMIC_WT_IMFOD;  // do EEP section D write process
  }
#endif
  return 0;
}
