//=============================================================================
// Header: processLcdMsg.c
// Date Time: 2/7/2013 1:58 AM
// Author: Xuedong.Liu
//
// Description: process message display on LCD.
//
//=============================================================================
//constant definitions, type definitions
#include "masterHeader.h"

//global volatile variables
#include "externGlobalVar.h"

//global function prototypes definitions
#include "externFunc.h"

static tU16 u16idxLcd;
static tU8  u8OnceDone, lcdChange, u8UserDefIdx;
const tF32 f32tenMultiple[8] =
{
  0.0f,          //0
  1.0f,          //1
  10.0f,         //2
  100.0f,        //3
  1000.0f,       //4
  10000.0f,      //5
  100000.0f,     //6
  1000000.0f,    //7
};

const tF32 f32LcdNumLimits[7][2] =
{
  {0.0f,       9.9f},
  {10.0f,      99.9f},
  {100.0f,     999.9f},
  {1000.0f,    9999.9f},
  {10000.0f,   99999.9f},
  {100000.0f,  999999.9f},
  {1000000.0f, 9999999.9f},
};

const tU8 u8DecPointLocate[7] =
{
  0,              // no decimal point
  DEC_ONE_BIT0,   // The first decimal point
  DEC_TWO_BIT1,   // starts from the right of
  DEC_THREE_BIT2, // the display number[1]
  DEC_FOUR_BIT3,
  DEC_FIVE_BIT4,
  DEC_SIX_BIT5,
};

void LcdLayer2Driver(void);
void u32ToLcdNum(tU32 num, tU8 *bufPtr, tU8 u8Size);
void displayFloatV(tDisplayData *proLcdData, tF32 *in_float);
void displayIntV(tDisplayData *proLcdData, tS32 *inIntV);

// Note : must be all upper case, Top line Max 5 chars, Bottom line Max 6 chars
//        Numerical 7 digits
const tLcdShowData lcdData[]=
{
  // topChar, BotChar,  LcdFunc,              LcdCtrl,       LcdVarAddr,                                      LcdType
  {"xxxxxxx", "xxxxxx", setAllLcdSupports,    DISPAY_PWR_ON, UNUSED_SNS_VAR_ADDR,                             DATA_TYPE_UNUSED}, //0
  {" Y-M-D ", " DATE ", updateDate,           DISPAY_OFF,    UNUSED_SNS_VAR_ADDR,                             DATA_TYPE_UNUSED},
  {" BD-ID ", "SERAIL", hwSerialLCD,          DISPAY_OFF,    UNUSED_SNS_VAR_ADDR,                             DATA_TYPE_UNUSED},
  {"CPU SW ", " REV  ", fw0ReleaseNum,        LCD_FR_REV,    UNUSED_SNS_VAR_ADDR,                             DATA_TYPE_UNUSED},  // WAS DISPAY_PWR_ON
  {" F REV ", " 5638 ", fw1ReleaseNum,        DISPAY_OFF,    UNUSED_SNS_VAR_ADDR,                             DATA_TYPE_UNUSED},  // WAS DISPAY_PWR_ON
  {"  ID   ", " SEN-1", sensor1Id,            DISPAY_OFF,    UNUSED_SNS_VAR_ADDR,                             DATA_TYPE_UNUSED},
  {"  ID   ", " SEN-2", sensor2Id,            DISPAY_OFF,    UNUSED_SNS_VAR_ADDR,                             DATA_TYPE_UNUSED},
  {" H-M-S ", " TIME ", updateDigitalClock,   DISPAY_OFF,    UNUSED_SNS_VAR_ADDR,                             DATA_TYPE_UNUSED},
  {" JT400 ", "ADDRES", mdbsAddrLcd,          LCD_ADDR,      UNUSED_SNS_VAR_ADDR,                             DATA_TYPE_UNUSED},
  {" RS485 ", " BAUD ", rs485BaudLcd,         LCD_485BD,     UNUSED_SNS_VAR_ADDR,                             DATA_TYPE_UNUSED},
  {" RS232 ", " BAUD ", rs232BaudLcd,         LCD_232BD,     UNUSED_SNS_VAR_ADDR,                             DATA_TYPE_UNUSED},
  {" MSP-T ", "  C   ", tempFeatureBd,        LCD_BD_TEMP,   UNUSED_SNS_VAR_ADDR,                             DATA_TYPE_UNUSED},
  {" BATTER", " VOLT ", vcc0FeatureBd,        LCD_VCC_V,     UNUSED_SNS_VAR_ADDR,                             DATA_TYPE_UNUSED},
  {" SOLAR ", " VOLT ", vcc1FeatureBd,        LCD_SOLAR_V,   UNUSED_SNS_VAR_ADDR,                             DATA_TYPE_UNUSED},
  // sensor data start
  {" S1-DP ", "R-CNTS", snsVarToLcd,          DISPAY_OFF,    &sysRamData.dp[TOP_SENSOR_ID].s32RawDPVal,       VER_S32DATA},
  {" S1-SP ", "R-CNTS", snsVarToLcd,          DISPAY_OFF,    &sysRamData.sp[TOP_SENSOR_ID].s32RawSpVal,       VER_S32DATA},
  {" S1-BR ", "R-CNTS", snsVarToLcd,          DISPAY_OFF,    &sysRamData.dp[TOP_SENSOR_ID].s32RawBrVolts,     VER_S32DATA},
  {" S1-V0 ", "R-CNTS", snsVarToLcd,          DISPAY_OFF,    &sysRamData.dp[TOP_SENSOR_ID].s32RawVzero,       VER_S32DATA},
  {" S1-VR ", "R-CNTS", snsVarToLcd,          DISPAY_OFF,    &sysRamData.dp[TOP_SENSOR_ID].s32RawVref,        VER_S32DATA},
  {"S1-D/GP", "INH2O ", snsVarToLcd,          LCD_DPV_S1,    &sysRamData.exchg8A[0].f32DpLive,                VER_F32DATA},
  {" S1-SP ", " PSI  ", snsVarToLcd,          LCD_SPV_S1,    &sysRamData.exchg8A[0].f32SpLive,                VER_F32DATA},
  {" S1-ST ", "  C   ", snsVarToLcd,          LCD_ESTV_S1,   &sysRamData.exchg8A[0].f32StLive,                VER_F32DATA},

  {" S2-DP ", "R-CNTS", snsVarToLcd,          DISPAY_OFF,    &sysRamData.dp[BTM_SENSOR_ID].s32RawDPVal,       VER_S32DATA},
  {" S2-SP ", "R-CNTS", snsVarToLcd,          DISPAY_OFF,    &sysRamData.sp[BTM_SENSOR_ID].s32RawSpVal,       VER_S32DATA},
  {" S2-BR ", "R-CNTS", snsVarToLcd,          DISPAY_OFF,    &sysRamData.dp[BTM_SENSOR_ID].s32RawBrVolts,     VER_S32DATA},
  {" S2-V0 ", "R-CNTS", snsVarToLcd,          DISPAY_OFF,    &sysRamData.dp[BTM_SENSOR_ID].s32RawVzero,       VER_S32DATA},
  {" S2-VR ", "R-CNTS", snsVarToLcd,          DISPAY_OFF,    &sysRamData.dp[BTM_SENSOR_ID].s32RawVref,        VER_S32DATA},
  {"S2-G/SP", " PSI  ", snsVarToLcd,          DISPAY_OFF,    &sysRamData.exchg8A[1].f32DpLive,                VER_F32DATA},
  {" S2-SP ", " PSI  ", snsVarToLcd,          DISPAY_OFF,    &sysRamData.exchg8A[1].f32SpLive,                VER_F32DATA},
  {" S2-ST ", "  C   ", snsVarToLcd,          DISPAY_OFF,    &sysRamData.exchg8A[1].f32StLive,                VER_F32DATA},
  // pt data start
  {"PROC-T ", " C RTD", pvPtValue,            LCD_RTD_V,     UNUSED_SNS_VAR_ADDR,                             DATA_TYPE_UNUSED},
  // LCD user define content
  {"       ", "      ", userDefLcd,           LCD_USER1,    &info5638B.f32UserLcdValue[0],        DATA_TYPE_UNUSED},
  {"       ", "      ", userDefLcd,           LCD_USER2,    &info5638B.f32UserLcdValue[1],        DATA_TYPE_UNUSED},
  {"       ", "      ", userDefLcd,           LCD_USER3,    &info5638B.f32UserLcdValue[2],        DATA_TYPE_UNUSED},
  {"       ", "      ", userDefLcd,           LCD_USER4,    &info5638B.f32UserLcdValue[3],        DATA_TYPE_UNUSED},
  {"       ", "      ", userDefLcd,           LCD_USER5,    &info5638B.f32UserLcdValue[4],        DATA_TYPE_UNUSED},
  {"       ", "      ", userDefLcd,           LCD_USER6,    &info5638B.f32UserLcdValue[5],        DATA_TYPE_UNUSED},
  {"       ", "      ", userDefLcd,           LCD_USER7,    &info5638B.f32UserLcdValue[6],        DATA_TYPE_UNUSED},
  {"       ", "      ", userDefLcd,           LCD_USER8,    &info5638B.f32UserLcdValue[7],        DATA_TYPE_UNUSED},
  {"       ", "      ", userDefLcd,           LCD_USER9,    &info5638B.f32UserLcdValue[8],        DATA_TYPE_UNUSED},
};
#define NUM_ELEMENT_LCD  lengthof(lcdData)

static const code2LcdText pressUnit2LcdTxt[] =  // map pressure unit to LCD text
{
  {U_inH2OAt68F,      0, "in H2O"},  // map inH2O@68F to LCD text
  {U_inHgAt0C,        0, "in Hg "},  // map inHg@0C to LCD text
  {U_ftH20At68F,      0, "FT H2O"},  // map ftH2O@68F to LCD text
  {U_mmH20At68F,      0, "mm H2O"},  // map mmH2O@68F to LCD text
  {U_mmHgAt0C,        0, "mm Hg "},  // map mmHg@0C to LCD text
  {U_psi,             0, "  PSI "},  // map psi to LCD text
  {U_bars,            0, " bar  "},  // map bar to LCD text
  {U_milliBars,       0, " mbar "},  // map mbar to LCD text
  {U_gramPerSqCm,     0, " g/cm2"},  // map g/Cm2 to LCD text
  {U_kiloGramPerSqCm, 0, "kg/cm2"},  // map Kg/Cm2 to LCD text
  {U_pascals,         0, "  Pa  "},  // map Pa to LCD text
  {U_kiloPascals,     0, "  kPa "},  // map KPa to LCD text
  {U_torr,            0, " Torr "},  // map torr to LCD text
  {U_atm,             0, "  atm "},  // map atmospheres to LCD text
  {U_megaPascals,     0, "  MPa "},  // map MPa to LCD text
  {U_inH20At4C,       0, "in H2O"},  // map inH2O@4C to LCD text
  {U_mmH20At4C,       0, "mm H2O"},  // map mmH2O@4C to LCD text
  {U_inH2OAt60F,      0, "in H2O"},  // map inH2O@60F to LCD text , this is 3095 default
  {255,               0, "UNKNOW"},  // no unit can display
};
#define LOOKUP_LCD_TXT_SIZE   lengthof(pressUnit2LcdTxt)

static const code2LcdText ptUnit2LcdTxt[]=  // convert to Celsius
{
  {U_celsius,     0, "  C   "},   // U_celsius to LCD text
  {U_fahrenheit,  0, "  F   "},   // U_fahrenheit to LCD text
  {U_rankine,     0, "  R   "},   // U_rankine to LCD text
  {U_kelvin,      0, "  K   "},   // U_kelvin to LCD text
  {255,           0, "UNKNOW"},   // no unit can display
};
#define PT_LCD_TXT_SIZE   lengthof(ptUnit2LcdTxt)

void initLcdRowNum(void)
{
  u16idxLcd = 0;
  u8OnceDone = NO;
  lcdChange = 0;
}

static void BcdToLcdChar(tU8 *u8BcdPtr, tU8 *u8LcdPtr, tU8 u8Num)
{
  tU8 u8CharIndex, i;
  for (i = 0; i < u8Num; i++)
  {
    // from BCD to LCD char, low nibble value as LCD numerical index
    u8CharIndex = *u8BcdPtr & 0x0F;
    // get LCD numerical char
    *u8LcdPtr = u8LcdNum[u8CharIndex];
    // point to next LCD buffer
    u8LcdPtr++;

    // high nibble value as LCD numerical index
    u8CharIndex = ((tU8)(*u8BcdPtr & (tU8)0xF0) >> (tU8)4);
    // get LCD numerical char
    *u8LcdPtr = u8LcdNum[u8CharIndex];
    // point to next LCD buffer
    u8LcdPtr++;

    // Point to next BCD char
    u8BcdPtr++;
   }
}

static void clearLcdSupports(tDisplayData *tLcdData)
{
  tLcdData->decimals = 0;
  tLcdData->bargraph = 0;
  tLcdData->annunc = 0;
}

static tReturnStatus setAllLcdSupports(tDisplayData *tLcdData)
{
  memset(tLcdData->topLine, 0xff, NUM_TOP_LINE_CHARS);
  memset(tLcdData->bottomLine, 0xff, NUM_BOTTOM_LINE_CHARS);
  memset(tLcdData->numLine, '8', NUM_NUMERIC_LINE_CHARS);
  tLcdData->decimals = 0xff;
  tLcdData->bargraph = 0xff;
  tLcdData->annunc = 0xff;
  return RS_good;
}

/**********************************************************************//**
 * @brief  Updates the hours, minutes and seconds on the digital clock
 *
 * @param  none
 *
 * @return none
 *************************************************************************/
static tReturnStatus updateDigitalClock(tDisplayData *tLcdData)
{
  tU8 u8BcdTime[3];
  u8BcdTime[2] = sysRamData.rtcV.Hours;
  u8BcdTime[1] = sysRamData.rtcV.Minutes;
  u8BcdTime[0] = sysRamData.rtcV.Seconds;

  BcdToLcdChar(u8BcdTime, tLcdData->numLine, 3);

  tLcdData->decimals = BIT1 + BIT3;
  tLcdData->bargraph = 0;
  tLcdData->annunc = 0;
  return RS_good;
}

/**********************************************************************//**
 * @brief  Updates the hours, minutes and seconds on the digital clock
 *
 * @param  none
 *
 * @return none
 *************************************************************************/
static tReturnStatus updateDate(tDisplayData *tLcdData)
{
  tU8 u8BcdTime[3];
  u8BcdTime[0]  = sysRamData.rtcV.Year;
  u8BcdTime[1]  = sysRamData.rtcV.Month;
  u8BcdTime[2]  = sysRamData.rtcV.DayOfMonth;

  tLcdData->numLine[6] = ((u8BcdTime[0] & 0xF0) >> 4) + 0x30;
  tLcdData->numLine[5] = (u8BcdTime[0] & 0x0F) + 0x30;
  tLcdData->numLine[4] = '-';
  tLcdData->numLine[3] = ((u8BcdTime[1] & 0xF0) >> 4) + 0x30;
  tLcdData->numLine[2] = (u8BcdTime[1] & 0x0F) + 0x30;
  tLcdData->numLine[1] = ((u8BcdTime[2] & 0xF0) >> 4) + 0x30;
  tLcdData->numLine[0] = (u8BcdTime[2] & 0x0F) + 0x30;

  tLcdData->decimals = BIT1 + BIT3;
  tLcdData->bargraph = 0;
  tLcdData->annunc = 0;
  return RS_good;
}

/**********************************************************************//**
 * @brief  Updates the hours, minutes and seconds on the digital clock
 *
 * @param  none
 *
 * @return none
 *************************************************************************/
static tReturnStatus hwSerialLCD(tDisplayData *tLcdData)
{
  u32ToLcdNum(infoD.xmt.u8HwRev, tLcdData->numLine, NUM_NUMERIC_LINE_CHARS);
  clearLcdSupports(tLcdData);
  return RS_good;
}

static void u32ToLcdNum(tU32 num, tU8 *bufPtr, tU8 u8Size)
{
  tS32 s32div = 1;
  tU8 u8len = 1, i;
  tU32 nbr = num;
  // clear will use buffer
  memset(bufPtr, 0x00, u8Size);
  // how many bytes will use
  while (nbr/s32div >= 10)
  {
    s32div *= 10;
    u8len++;
  }
  // should not over size, if do set to the size, number high size will display
  if (u8len > u8Size)
  {
    nbr %= s32div;
    s32div /= (10 * (u8len - u8Size));
    u8len = u8Size;
  }
  // put data in LCD know char
  while (u8len)
  {
    i = u8len-1;
    *(bufPtr + i) = '0' + (nbr/s32div);
    u8len--;
    nbr %= s32div;
    s32div /= 10;
  }
}

static void displayFloatV(tDisplayData *proLcdData, tF32 *in_float)
{
  tU8 u8Temp[10], procTemp;
  tF32 f32temp;
  tS16 i;
  proLcdData->decimals = 0;

  // put on minus sign
  if (*in_float < 0.0f)
  {
    // set "-" sign; absolute number
    proLcdData->annunc |= ANNUNC_MINUS;
    proLcdData->decimals = DEC_NEG_SIGN_BIT6;
    f32temp = -*in_float;
  }
  else
  {
    // clear "-" sign; absolute number
    proLcdData->annunc &= (tU8)~ANNUNC_MINUS;
    proLcdData->decimals &= (tU8)~DEC_NEG_SIGN_BIT6;
    f32temp = *in_float;
  }

  memset(u8Temp, ' ', 10);
  {
    sprintf((char *)u8Temp, "%.3f", f32temp);  //.8
    procTemp = 0;
  }

  // where to put decimal point
  for (i = 0; i < 10; i++)
  {
    if (u8Temp[i] == '.')
    {
      proLcdData->decimals |= u8DecPointLocate[7 - i];
      memmove(&u8Temp[i], &u8Temp[i+1], 10-i);
      i = 11; // to end the loop
    }
  }
  if (procTemp == 0)
  {
    for (i= 0; i < NUM_NUMERIC_LINE_CHARS; i++)
    {
      proLcdData->numLine[NUM_NUMERIC_LINE_CHARS-i-1] = u8Temp[i];
    }
  }
  else
  {
    for (i= 0; i < NUM_NUMERIC_LINE_CHARS; i++)
    {
      if (i < 3)
      {
        proLcdData->numLine[NUM_NUMERIC_LINE_CHARS-i-1] = u8Temp[i];
      }

      if(u8Temp[i] == 0x45)  // 0x45 is ASCII "E" hex value
      {
        proLcdData->numLine[2] = 0x45;
        if(u8Temp[i+1] == 0x22)  //0x22 is ASCII "+" hex value
        {
          proLcdData->numLine[1] = u8Temp[i+2];
          proLcdData->numLine[0] = ' ';
          i = 10;  // end loop
        }
        else
        {
          proLcdData->numLine[1] = u8Temp[i+1];
          proLcdData->numLine[0] = u8Temp[i+2];
        }
      }
    }
  }
  __no_operation();
}

static void displayIntV(tDisplayData *proLcdData, tS32 *inIntV)
{
  tU8 u8Temp[8], i;

  if ((*inIntV >= -999999) && (*inIntV <= 9999999))
  {
    sprintf((char *)u8Temp, "%7ld", *inIntV);
    for(i = 0; i < NUM_NUMERIC_LINE_CHARS; i++)
    {
       proLcdData->numLine[NUM_NUMERIC_LINE_CHARS - 1 - i] = u8Temp[i];
    }
    __no_operation();
  }
  else
  {
    memset(proLcdData->numLine, 0x45, NUM_NUMERIC_LINE_CHARS);
  }
}

static tReturnStatus fw0ReleaseNum(tDisplayData *tLcdData)
{
  u32ToLcdNum(BuildFwInfo.u8VerBeta, &tLcdData->numLine[0], 2);
  u32ToLcdNum(BuildFwInfo.u8VerMinor, &tLcdData->numLine[2], 2);
  u32ToLcdNum(infoD.xmt.u8FwProdCode[5], &tLcdData->numLine[4], 2);
  u32ToLcdNum(infoD.xmt.u8FwProdCode[4], &tLcdData->numLine[6], 2);
  if(tLcdData->numLine[5] == 0x00)
  {
    tLcdData->numLine[5] = 0x30;
  }
  if(tLcdData->numLine[1] == 0x00)
  {
    tLcdData->numLine[1] = 0x30;
  }
  tLcdData->decimals = BIT1 + BIT5;
  tLcdData->bargraph = 0;
  tLcdData->annunc = 0;
  return RS_good;
}

static tReturnStatus fw1ReleaseNum(tDisplayData *tLcdData)
{
  u32ToLcdNum(BuildFwInfo.u8VerBeta,  &tLcdData->numLine[0], 2);
  u32ToLcdNum(BuildFwInfo.u8VerMinor, &tLcdData->numLine[2], 2);
  u32ToLcdNum(BuildFwInfo.u8VerMajor, &tLcdData->numLine[4], 2);
  tLcdData->decimals = BIT1 + BIT3;
  tLcdData->bargraph = 0;
  tLcdData->annunc = 0;
  return RS_good;
}

static tReturnStatus mdbsAddrLcd(tDisplayData *tLcdData)
{
  u32ToLcdNum(infoA.comm.u8ModbusAddr, tLcdData->numLine, NUM_NUMERIC_LINE_CHARS);
  clearLcdSupports(tLcdData);
  return RS_good;
}

static tReturnStatus rs485BaudLcd(tDisplayData *tLcdData)
{
  tU32 u32Temp = (tU32)infoA.comm.f32BaudRate;
  u32ToLcdNum(u32Temp, tLcdData->numLine, NUM_NUMERIC_LINE_CHARS);
  clearLcdSupports(tLcdData);
  return RS_good;
}

static tReturnStatus rs232BaudLcd(tDisplayData *tLcdData)
{
  tU32 u32Temp = (tU32)infoA.comm.f32Baud232Rate;
  // for find error
  u32ToLcdNum(u32Temp, tLcdData->numLine, NUM_NUMERIC_LINE_CHARS);
  clearLcdSupports(tLcdData);
  return RS_good;
}

static tReturnStatus tempFeatureBd(tDisplayData *tLcdData)
{
    tF32 f32temp;
    f32temp = sysRam56D.exchg6Pv.f32FbdTemp;
    clearLcdSupports(tLcdData);
    if (infoD.xmt.u8RtdUnits == U_fahrenheit)
    {
      tLcdData->bottomLine[0] = 'F';
    }
    else
    {
      tLcdData->bottomLine[0] = 'C';
    }
    displayFloatV(tLcdData, &f32temp);
    return RS_good;
}

static tReturnStatus vcc0FeatureBd(tDisplayData *tLcdData)
{
  clearLcdSupports(tLcdData);
  displayFloatV(tLcdData, (void *)&sysRam56D.exchg6Pv.f32BatV);
  return RS_good;
}
static tReturnStatus vcc1FeatureBd(tDisplayData *tLcdData)
{
  clearLcdSupports(tLcdData);
  displayFloatV(tLcdData, (void *)&sysRam56D.exchg6Pv.f32SolarV);
  return RS_good;
}

static tReturnStatus sensor1Id(tDisplayData *tLcdData)
{
  u32ToLcdNum(infoD.xmt.u8SenSeries[0], tLcdData->numLine, NUM_NUMERIC_LINE_CHARS);
  clearLcdSupports(tLcdData);
  return RS_good;
}

static tReturnStatus sensor2Id(tDisplayData *tLcdData)
{
  u32ToLcdNum(infoD.xmt.u8SenSeries[1], tLcdData->numLine, NUM_NUMERIC_LINE_CHARS);
  clearLcdSupports(tLcdData);
  return RS_good;
}

static tReturnStatus ptUnit2Txt(tDisplayData *tLcdData, tU8 u8PressUnit)
{
  tU8 i;
  tReturnStatus errCode = RS_good;
  for (i = 0; i < PT_LCD_TXT_SIZE; i++)
  {
    if (ptUnit2LcdTxt[i].u8SysCode == u8PressUnit)
    {
      errCode = RS_good;
      break;
    }
  }
  if (i >= LOOKUP_LCD_TXT_SIZE)
  {
    // do not find LCD text, diaply default.
    i = LOOKUP_LCD_TXT_SIZE - 1;
    errCode = RER_ERROR;
  }
  memmove(tLcdData->bottomLine, ptUnit2LcdTxt[i].u8LcdTxt, NUM_BOTTOM_LINE_CHARS);
  return errCode;
}

static tReturnStatus userDefLcd(tDisplayData *tLcdData)
{
  tReturnStatus errCode = RS_good;
  tF32 f32temp;
  // clear support
  clearLcdSupports(tLcdData);
  
  f32temp = info5638B.f32UserLcdValue[u8UserDefIdx];
  // middle or numerica line display
  displayFloatV(tLcdData, &f32temp);
  // Bottom line text
  memmove(tLcdData->bottomLine, info5638A.u8UserDefUnit[u8UserDefIdx], NUM_BOTTOM_LINE_CHARS);
  // top line text
  memmove(tLcdData->topLine, info5638A.u8VarLabel[u8UserDefIdx], NUM_TOP_LINE_CHARS);
    
  return errCode;  
}

static tReturnStatus pvPtValue(tDisplayData *tLcdData)
{
    tReturnStatus errCode = RS_good;
    // this function run 1.21msec @12Mhz, displayFloatV use 99% of time
    tF32 f32temp;
    if ((sysRamData.diag.u16XmtStatus[1] & RTD_DISCONNECT) == RTD_DISCONNECT)
    {
        // error in PT, stop display this one, pickup next one
        errCode = RS_rtd_error;
        f32temp = sysRamData.exchg8A[0].f32PtLive; //infoA.pt.f32RtdZeroOffset;
    }
    else
    {
        // for find error
        f32temp = sysRamData.exchg8A[0].f32PtLive;
    }
    clearLcdSupports(tLcdData);
    displayFloatV(tLcdData, &f32temp);
    errCode = ptUnit2Txt(tLcdData, infoD.xmt.u8RtdUnits);

    return errCode;
}

static tReturnStatus presUnit2Txt(tDisplayData *tLcdData, tU8 u8PressUnit)
{
  tU8 i;
  tReturnStatus errCode = RS_good; //RER_ERROR;;
  for (i = 0; i < LOOKUP_LCD_TXT_SIZE; i++)
  {
    if (pressUnit2LcdTxt[i].u8SysCode == u8PressUnit)
    {
      errCode = RS_good;
      break;
    }
  }
  if (i >= LOOKUP_LCD_TXT_SIZE)
  {
    // do not find LCD text, diaply default.
    i = LOOKUP_LCD_TXT_SIZE - 1;
    errCode = RER_ERROR;
  }
  memmove(tLcdData->bottomLine, pressUnit2LcdTxt[i].u8LcdTxt, NUM_BOTTOM_LINE_CHARS);
  return errCode;
}

static tReturnStatus snsVarToLcd(tDisplayData *tLcdData)
{
    tS32 s32Temp;
    tF32 f32temp;

    tReturnStatus errCode = RS_good; // FIX ME later

    clearLcdSupports(tLcdData);
    // multi sensor, display DP
    if (lcdData[u16idxLcd].u8DataType == VER_S32DATA)
    {
        s32Temp = *(tS32 *)lcdData[u16idxLcd].u32SnsVerAddr;
        displayIntV(tLcdData, &s32Temp);
    }
    else if (lcdData[u16idxLcd].u8DataType == VER_F32DATA)
    {
        f32temp = *(tF32 *)lcdData[u16idxLcd].u32SnsVerAddr;
        //memmove(&f32temp, lcdData[u16idxLcd].u32SnsVerAddr, 4);
        displayFloatV(tLcdData, &f32temp);
    }
    else
    {

    }
    return errCode;
}

tReturnStatus notImplement(tDisplayData *tLcdData)
{
    // move to next, allow display line
    return RER_ERROR;
}

static void displayIt(tU16 u16Temp)
{
    const tU8 dp_top[7] = "S1-D/GP";
    const tU8 sp_top[7] = " S1-SP ";
//    const tU8 st_top[7] = " S1-ST ";
    const tU8 RTD_top[7] = "PROC-T ";
    
    tDisplayData pDisplayData;
    pDisplayData.u8LcdIdx = u16Temp;
    u16idxLcd = u16Temp;
    if(u16Temp >= (NUM_ELEMENT_LCD - NUM_USER_DEF_LCD))
    {
      // do user defined LCD
      u8UserDefIdx = u16Temp - (NUM_ELEMENT_LCD - NUM_USER_DEF_LCD);
      //tReturnStatus errCode = RS_good;  // FIX Me Later
      (void)lcdData[u16idxLcd].LcdNumerical(&pDisplayData);
    }
    else
    {
      //tReturnStatus errCode = RS_good;  // FIX Me Later
      (void)lcdData[u16idxLcd].LcdNumerical(&pDisplayData);
      // top line and bottom line text
      memmove(pDisplayData.topLine, lcdData[u16idxLcd].u8TopChars, NUM_TOP_LINE_CHARS);
      if((memcmp((void *)pDisplayData.topLine, dp_top, 7) == 0) ||
         (memcmp((void *)pDisplayData.topLine, "S2-G/SP", 7) == 0))
      {
          (void)presUnit2Txt(&pDisplayData, infoD.xmt.u8DpUnits);
      }
      else if((memcmp((void *)pDisplayData.topLine, sp_top, 7) == 0) ||
              (memcmp((void *)pDisplayData.topLine, " S2-SP ", 7) == 0))
      {
          (void)presUnit2Txt(&pDisplayData, infoD.xmt.u8SpUnits);
      }
      else if(/*(memcmp((void *)pDisplayData.topLine, st_top, 7) == 0) ||*/
              (memcmp((void *)pDisplayData.topLine, " S2-ST ", 7) == 0) ||
              (memcmp((void *)pDisplayData.topLine, RTD_top, 7) == 0))
      {
          (void)ptUnit2Txt(&pDisplayData, infoD.xmt.u8RtdUnits);
      }
      else
      {
        if(u16idxLcd != 0)
        {
          memmove(pDisplayData.bottomLine, lcdData[u16idxLcd].u8BottomChars, NUM_BOTTOM_LINE_CHARS);
        }
      }
    }
    
    LcdDataCheck(&pDisplayData);
    
    // setup LCD display
    updateGlass(&pDisplayData);
    __no_operation();   // For debugger only
}

void LcdLayer2Driver(void)
{
    tU32 u32LcdOnCtrl; //u8temp;
    tU16 i;

    if (u8OnceDone == YES)
    {
      // only display this items
      //u8temp = DISPAY_SNS1;
      u32LcdOnCtrl = (tU32)info5638A.u16LcdControlBits + LCD_ADDR;  
    }
    else
    {
      // at power on time display normal and power on items
      //u8temp = DISPAY_SNS1 + DISPAY_PWR_ON;
      u32LcdOnCtrl = (tU32)info5638A.u16LcdControlBits + DISPAY_PWR_ON + LCD_ADDR;
    }
    u32LcdOnCtrl += (((tU32)info5638A.u16LcdCtrlHiWord) << 16);
    // update Numerical
    // if next layer is not in right state, it may stop in wrong state
    // reset it to the right state, not wait any more
    I2cStateReg = CH1_STOP;

    // update for next time
    for (i = u16idxLcd; i < NUM_ELEMENT_LCD; i++) // find next display element from u16idxLcd to NUM_ELEMENT_LCD
    { 
        //if ((u8temp & lcdData[i].u32LcdCtrl) == lcdData[i].u32LcdCtrl)
        if ((u32LcdOnCtrl & lcdData[i].u32LcdCtrl) == lcdData[i].u32LcdCtrl)
        {
            if ((lcdData[i].u8DataType != VER_S32DATA) && (lcdData[i].u8DataType != VER_F32DATA))
            {
                __no_operation();   // For debugger only
                lcdChange = 1;
                break;
            }
            else if ((lcdData[i].u8DataType == VER_S32DATA) && (*((tU32 *)lcdData[i].u32SnsVerAddr) != 0))
            {
                __no_operation();   // For debugger only
                lcdChange = 1;
                break;
            }
            else if ((lcdData[i].u8DataType == VER_F32DATA) && (*((tF32 *)lcdData[i].u32SnsVerAddr) != 0.0f))
            {
                __no_operation();   // For debugger only
                lcdChange = 1;
                break;
            }
        }
    }        
    
    if (i >= NUM_ELEMENT_LCD)  // from u16idxLcd to NUM_ELEMENT_LCD, nothing to display
    {
        u8OnceDone = YES;
        i = u16idxLcd = 0;
    }
    
    if( lcdChange == 1)
    {
       // find the diplay item, update lcd and clear flag for next search
       lcdChange = 0;
       displayIt(i);
       i++;  // point to next look place
       u16idxLcd = i;
       u8Layer1TaskScheduler |= INTERF_SEND_LCD_CTRL1;         // start to send data to LCD
       u8Layer2TaskScheduler &= (tU8)~PRO_LCD_UPDATE_REQ;      // clear the request update LCD flag
    }
    else
    {
      // do not clear flag, find next diplay item
      __no_operation();                     // For debugger only
    }
}

tU16 getLcdIdx(void)
{
  return u16idxLcd;
}
//=============================================================================
// end file by xuedong
//=============================================================================
