//=============================================================================
// Header: MSP430F5438A interrupt service functions
// Date Time: 1/19/2013 9:10 AM
// Author: Xuedong.Liu
//
// Description: define all the interrupt service enter point.
//
//=============================================================================

// constant definitions, type definitions
#include "masterHeader.h"

// global volatile variables
#include "externGlobalVar.h"

// global function prototypes definitions:
#include "externFunc.h"

// local static variables
//static tU32 u32SecondCnt;

//                                             for MSP430F5438A
//__interrupt void switch power on          // Reset, power, password violation //0x0FFFE   priority highest
//__interrupt void JTage                    // JTAG mail box                    //0x0FFFC
__interrupt void nmiIsr(void);              // user NMI (above non Maskable)    //0x0FFFA
__interrupt void timer0B0Isr(void);         // TB0 CCR0 (below     maskable)    //0X0FFf8
__interrupt void timer0B1B6Isr(void);       // TB0 CCR1 to CCR6
__interrupt void wdtIsr(void);              // watchdog
__interrupt void radioInterruptIsr(void);   // UCSI A0 Rx/Tx, Radio UART            5638 -> Radio interface
__interrupt void sensor1SpiMasterIsr(void); // USCI B0 Rx/Tx, sensor 1 SPI          5638 -> LCD interrupt
__interrupt void adc12Isr(void);            // ADC12_A
__interrupt void timer0A0Isr(void);         // TA0 CCR0
__interrupt void timer0A1A4Isr(void);       // TA0 CCR1 to CCR4
__interrupt void fskInterruptIsr(void);     // UCSI A2 Rx/Tx, FSK UART
__interrupt void ramSpiMasterIsr(void);     // UCSI B2 Rx/Tx, RAM 23A1024 SPI
__interrupt void dmaIsr(void);              // DMA
__interrupt void timer1A0Isr(void);         // TA1 CCR0
__interrupt void timer1A1A2Isr(void);       // TA1 CCR1 to CCR2
__interrupt void port1Isr(void);            // Prot1
__interrupt void sensor2SpiMasterIsr(void); // UCSI A1 Rx/Tx, snesor 2 SPI
__interrupt void flashSpiMasterIsr(void);   // UCSI B1 Rx/Tx, Flash W25Q80 SPI
__interrupt void rtdSpiMasterIsr(void);     // UCSI B3 Rx/Tx, RTD SPI
__interrupt void port2Isr(void);            // Port2
__interrupt void rtcIsr(void);              // RTC_A   // Priority lowest 0xFFD2

// interrupt service function inplementation
//*********###################################*********************************
// special interrupts function implementation
//=============================================================================
// XT1 fail interrupt process
//=============================================================================
#pragma vector=UNMI_VECTOR
__interrupt void nmiIsr(void)
{
    tU16 status;
    do {
        // If it still can't clear the oscillator fault flags after the timeout,
        // trap and wait here.
        status = UCS_clearAllOscFlagsWithTimeout(
            __MSP430_BASEADDRESS_UCS__,
            1000);
    } while(status != 0);
}

//=============================================================================
// wachdog timer interrupt, should reset system. something wrong made
// program not good to reset watchdog timer in kernel
//=============================================================================
#pragma vector = WDT_VECTOR
__interrupt void wdtIsr(void)
{
    __no_operation();
  // this instruction forces system reset
  WDTCTL = WRONG_WDTPW + WDTHOLD;
}

//==============================================================================
// MSP430F5438A internal ADC conversion, it is use sequance, only one interrupt
// used by this mode. Hi speed AI on the MSP430F5638
//==============================================================================
#pragma vector=ADC12_VECTOR
__interrupt void adc12Isr(void)
{
  switch (__even_in_range(ADC12IV,34))
  {
     case  0: break;   //Vector  0:  No interrupt
     case  2: break;   //Vector  2:  ADC overflow
     case  4: break;   //Vector  4:  ADC timing overflow
     case  6:          //Vector  6:  ADC12IFG0
       break;        
     case  8: break;   //Vector  8:  ADC12IFG1
     case 10: break;   //Vector 10:  ADC12IFG2
     case 12:          //Vector 12:  ADC12IFG3, MSP430F54, or 56 done A/D conversion
       break;
     case 14:          //Vector 14:  ADC12IFG4

         ADC12_disableInterrupt (__MSP430_BASEADDRESS_ADC12_PLUS__,
                ADC12IE4);  

         REF_disableReferenceVoltage(__MSP430_BASEADDRESS_REF__);
           
          // not pluger lift, do other task
          // 5638 use it 
          u8Layer2TaskScheduler |= CHARGE_BAT_HS_AI;
          
          // MSP temrpeature
          u16MspAdcCnt[0] = ADC12_getResults(
                     __MSP430_BASEADDRESS_ADC12_PLUS__,
                     ADC12_MEMORY_0);   // MSP temperature
          u16MspAdcCnt[1] = ADC12_getResults(
                     __MSP430_BASEADDRESS_ADC12_PLUS__,
                     ADC12_MEMORY_1);   // VCM 
          u16MspAdcCnt[2] = ADC12_getResults(
                     __MSP430_BASEADDRESS_ADC12_PLUS__,
                     ADC12_MEMORY_2);   // SP_Volts
          u16MspAdcCnt[3] = ADC12_getResults(
                     __MSP430_BASEADDRESS_ADC12_PLUS__,
                     ADC12_MEMORY_3);   // BAT_Volts
          u16MspAdcCnt[4] = ADC12_getResults(
                     __MSP430_BASEADDRESS_ADC12_PLUS__,
                     ADC12_MEMORY_4);   // HS_AI
                     
          __bic_SR_register_on_exit(LPM3_bits);
          break;   
     case 16: break;   //Vector 16:  ADC12IFG5
     case 18: break;   //Vector 18:  ADC12IFG6
     case 20: break;   //Vector 20:  ADC12IFG7
     case 22: break;   //Vector 22:  ADC12IFG8
     case 24: break;   //Vector 24:  ADC12IFG9
     case 26: break;   //Vector 26:  ADC12IFG10
     case 28: break;   //Vector 28:  ADC12IFG11
     case 30: break;   //Vector 30:  ADC12IFG12
     case 32: break;   //Vector 32:  ADC12IFG13
     case 34: break;   //Vector 34:  ADC12IFG14
     default: break;
  }
}

//==============================================================================
//
// FUNCTION:        DMA_ISR
//
//                  MSP430F5638 has 6 internal DMA channels, allocation:
//                      DMA0, 1, 2, for USB, USB bridge
//                      DMA3 for LCD -> I2C Tx
//                      DMA4 fpr Radio UART Tx
//                      DAM5 is free
//==============================================================================
#pragma vector=DMA_VECTOR
__interrupt void dmaIsr(void)
{
  switch(__even_in_range(DMAIV,16))
  {
    case VECTOR_NOINT:              // no interrupt
    case VECTOR_DMA0:               // DMA0IFG = DMA Channel 0
      __no_operation();             // Channel use for USB
      // search -> starts DMA transfer to/from USB memory into/from RAM
      break;
    case VECTOR_DMA1:               // DMA1IFG = DMA Channel 1
      __no_operation();             // Channel use for USB
      // search -> starts DMA transfer to/from USB memory into/from RAM
      break;
    case VECTOR_DMA2:               // DMA2IFG = DMA Channel 2
      __no_operation();             // Channel use for USB
      // search -> starts DMA transfer to/from USB memory into/from RAM
     break;
    case VECTOR_DMA3:               // DMA3IFG = DMA Channel 3
        I2C_masterMultiByteSendStop(__MSP430_BASEADDRESS_USCI_B1__);
        u8Layer1TaskScheduler |= INTERF_SEND_LCD_CTRL1;// set flag, start send next set of LCD data
        __bic_SR_register_on_exit(EMV_LPM_ON);
          break;
    case VECTOR_DMA4:                       // DMA4IFG = DMA Channel 4, Radio Tx is done in MSP430F5638
        //start timer radio msg out
        setupRadioDmaRxCtrl();
        radioTimerStart(50, TX_RADIO_DONE);  // about 91 msecond 
        break;
    case VECTOR_DMA5:                       // DMA5IFG = DMA Channel 5      
      // use this channel as radio Rx message
      u8Layer1TaskScheduler |= RADIO_RECEIVED_CHARS;
      __no_operation();
      __bic_SR_register_on_exit(EMV_LPM_ON);    
      break;
    case VECTOR_DMA6:                       // DMA6IFG = DMA Channel 6, JT808 no this channel
    case VECTOR_DMA7:                       // DMA7IFG = DMA Channel 7, JT808 no this channel
    default:
      __no_operation();                     // For debugger only
      break;
  }
}
tU8 u8Run30Sec;
//==============================================================================
// RTC interrupt process, interrup tevery second, interrupt every minute
//==============================================================================
#pragma vector=RTC_VECTOR
__interrupt void rtcIsr (void)
{
    switch (__even_in_range(RTCIV,16)){
        case 0: break;  //No interrupts
        case 2:         //RTCRDYIFG once every second          
          if(u8Run30Sec >= 30)
          {
            //sendJ6RtcToJ4();
            u8Run30Sec = 0;
          }
          else
          {
            u8Run30Sec++;
          }
          __no_operation();
           break;
        case 4:         //RTCEVIFG
            //Interrupts every minute
            __no_operation();
            break;
        case 6:         //RTCAIFG
            //Interrupts 5:00pm on 5th day of week
            __no_operation();
            break;
        case 8: break;  //RT0PSIFG
        case 10: break; //RT1PSIFG
        case 12: break; //Reserved
        case 14: break; //Reserved
        case 16: break; //Reserved
        default: break;
    }
}

//*********###################################*********************************
// port 1 and 2 interrupts function implementation
//==============================================================================
// port 1 interrupt
//==============================================================================
#pragma vector = PORT1_VECTOR
__interrupt void port1Isr(void)
{
    switch(__even_in_range(P1IV, P1IV_P1IFG7))
    {
        case P1IV_NONE: break;  // No Interrupt pending
        case P1IV_P1IFG0:       // P1IV P1IFG.0, start LCD updates
            u8Layer2TaskScheduler |= RD_RAM_DATA_DONE;
            stopWaitOtherReadTimer();
            __bic_SR_register_on_exit(EMV_LPM_ON);    // weak up CPU
            __no_operation();          
            break;
        case P1IV_P1IFG1:
            u8Layer1TaskScheduler |= READ_OTHER_RAM_DATA;
            __bic_SR_register_on_exit(EMV_LPM_ON);    // weak up CPU
            __no_operation();
            break;
        case P1IV_P1IFG2:       // P1IV P1IFG.2,
            __no_operation();
            break;
        case P1IV_P1IFG3:       // P1IV P1IFG.3,
            __no_operation();
            break;
        case P1IV_P1IFG4:       // P1IV P1IFG.4,
            // update firmware, user request through other MSP
            otherMspRequestFwUpdate();
            __bic_SR_register_on_exit(EMV_LPM_ON);    // weak up CPU          
            __no_operation();
            break;
        case P1IV_P1IFG5:       // P1IV P1IFG.5,
            // done update is J4M, clear Status, for next run
            updateDoneClrNextRun();
            __no_operation();                                
            break;
        case P1IV_P1IFG6:       // P1IV P1IFG.6,
            __no_operation();
            break;
        case P1IV_P1IFG7:       // P1IV P1IFG.7,
            break;
        default:
            break;
    }
}

//==============================================================================
// port 2 interrupt
//==============================================================================
#pragma vector = PORT2_VECTOR
__interrupt void port2Isr(void)
{
    switch(__even_in_range(P2IV, P2IV_P2IFG7))
    {
        case P2IV_NONE:	/* No Interrupt pending */
            break;
        case P2IV_P2IFG0:	/* P2IV P2IFG.0 */
            __no_operation();
            break;
        case P2IV_P2IFG1:	/* P2IV P2IFG.1 */
            break;
        case P2IV_P2IFG2:	/* P2IV P2IFG.2 */
            break;
        case P2IV_P2IFG3:	/* P2IV P2IFG.3 */
            break;
        case P2IV_P2IFG4:	// P2IV P2IFG.4,
            break;
        case P2IV_P2IFG5:	// P2IV P2IFG.5,
            break;
        case P2IV_P2IFG6:	/* P2IV P2IFG.6 */
            break;
        case P2IV_P2IFG7:	/* P2IV P2IFG.7 */
            break;
        default:
            break;
    }
}

//******************************************************************************
// USCI interrupts function implementation
//==============================================================================
// Radio -> USCI_A0 only Tx, no Rx, becasue the power usage, Rx interrupt enable
//==============================================================================
// DESCRIPTION:     Radio Rx interrupts, Tx not use interrupt or Tx interrupt is
//                  disabled, interrupt on Rx char or error triggled
//                  by the communication line for receiving.  Tx message by DMA.
//
//==============================================================================
#pragma vector=USCI_A0_VECTOR
__interrupt void radioInterruptIsr(void)
{
#if 0
  tU8 u8WeakUpMsp, u8RxChar;          
  __no_operation();
  switch(__even_in_range(UCA0IV,4))
  {
    case VECTOR_NOINT:      // Vector 0 - no interrupt
         break;
    case VECTOR_RXIFG:      // Vector 2 - RXIFG
         u8RxChar =  UART_receiveData(__MSP430_BASEADDRESS_USCI_A0__);
         __no_operation();
         u8WeakUpMsp = radioInterrupRxChar(u8RxChar);
  
         //if(u8WeakUpMsp == 0)
         {
            __bic_SR_register_on_exit(EMV_LPM_ON);    // weak up CPU          
            __no_operation();    
         }
         break;
    case VECTOR_TXIFG:      // Vector 4 - TXIFG
         break;
    default:
         break;
  }
#else
  // radio only Tx message, should not get in here
    FOREVER
    {
        // FIX ME
        __no_operation();
    }
#endif
}

//==============================================================================
// sensor 2 -> USCI_A1 SPI master.  SPI master does not need intrrept service
//==============================================================================
#pragma vector=USCI_A1_VECTOR
__interrupt void sensor2SpiMasterIsr(void)
{
    // MSP430F5638 USCIA1 -> RAM SPI
    FOREVER
    {
        // FIX ME
        __no_operation();
    }
}


//==============================================================================
// FLash SPI in MSP430F5638 no interrupt need
//==============================================================================
#pragma vector = USCI_B0_VECTOR
__interrupt void sensor1SpiMasterIsr(void)
{
    FOREVER
    {
        //FIX ME
        __no_operation();
    }
}

//==============================================================================
// I2C -> LCD interface in MSP430F5638, I2C error interrupt, fix later
//==============================================================================
#pragma vector = USCI_B1_VECTOR
__interrupt void flashSpiMasterIsr(void)
{
    switch(__even_in_range(UCB1IV, 12))
    {
    case  0: break;               // Vector  0: No interrupts
    case  2: break;               // Vector  2: ALIFG
    case  4:                      // Vector  4: NACKIFG
      break;
    case  6: break;               // Vector  6: STTIFG
    case  8: break;               // Vector  8: STPIFG
    case 10:                      // Vector 10: RXIFG
      break;
    case 12:                      // Vector 12: TXIFG
      UCB1IFG |= UCTXIFG;
      break;
    default: break;
  }
}


//******************************************************************************
// timer interrupts function implementation
//==============================================================================
// TA0.0 is free, LCD connect to the MSP430F5638
//==============================================================================
#pragma vector = TIMER0_A0_VECTOR
__interrupt void timer0A0Isr(void)
{
    // last request did not response, start request agian.
    FOREVER
    {
        // FIX ME
        __no_operation();
    }
}

//==============================================================================
// in MSP430F5638  it has 5 timers
//==============================================================================
#pragma vector = TIMER0_A1_VECTOR
__interrupt void timer0A1A4Isr(void)
{
    switch(__even_in_range(TA0IV, 14))
    {
        case 0: break;           // CCR0
        case 2:                  // CCR1
            break;
        case 4:                  // CCR2
            break;
        case 6:                  // CCR3
            __no_operation();
            break;
        case 8:                  // CCR4
            __no_operation();
            break;
        case 10: break;          // Reserved not used,
        case 12: break;          // Reserved not used,
        case 14:                 // overflow
            __no_operation();
            break;
        default: break;
    }
}

//==============================================================================
// TA1.0 is FSK current loop output low
//==============================================================================
#pragma vector = TIMER1_A0_VECTOR
__interrupt void timer1A0Isr(void)
{
    FOREVER
    {
        //FIX ME
        __no_operation();
    }
}

//==============================================================================
// in MSP430F5438A
// interrupt seriver from TA1.1 to TA1.2
// match with ctrlTimer.c definitions
// T1A1-3 use SKCLK as clock source, overflow interrupt as 8.192mSec interval
//    to count pluger lift two samples interval timer. 32 bits long can count
//    8.947minutes
// it has A2.0; A2.1; A2.2; free
//
// in MSP430F5638 has A1.0; A1.1; A1.2; free
//             it has A2.0; A2.1; A2.2; free
//==============================================================================
#pragma vector =  TIMER1_A1_VECTOR
__interrupt void timer1A1A2Isr(void)
{
    switch(__even_in_range(TA1IV, TA1IV_TA1CCR2))
    {
        case 0: break;           // CCR0 none interrupt
        case TA1IV_TA1CCR1:      // CCR1
            __no_operation();
            break;
        case TA1IV_TA1CCR2:      // CCR2
            __no_operation();
            break;
        case TA1IV_TA1IFG:       // overflow interrupt
          __no_operation();
            break;
        default:
            __no_operation();
            break;
    }
}

//=============================================================================
// TB0.0 is schedule tasks timer, each task start 250mSec
//      schedule plan is 250mSec start one task,
//      250mSec start sensor 1
//      500mSec start sensor 2
//      750mSec start RTD
//      1000mSec start Vcc, MSP temp, and AIs, high speed counter etc.
//=============================================================================
#pragma vector =  TIMER0_B0_VECTOR           // TB0 CC0
__interrupt void timer0B0Isr(void)
{
    FOREVER
    {
        // FIX ME
        __no_operation();
    }
}

static tU8 u8TxIt, u8LpTxType[5] =
{                       //u8TxIt
    TYPE_INFO5638A,     // 0, after this when change do exchange, no change
    TYPE_INFO5638B,     // 1, each MSP has same content copy locally
    TYPE_INFO5638C,     // 2,
    TYPE_INFO5638D,     // 3,
    TYPE_EX6_PVS,       // 4,
};

#ifdef __JT808_MSP430F5638__ 
tU8 timerloop;
#endif
//=============================================================================
// MSP430F5638 has TB0.0; scheduler
//          it has TB0.1; TB0.2; TB0.3; TB0.4; TB0.5; TB0.6; free
//=============================================================================
static tU8 inLcdStates, lcdState, u8ExchOn; //, scheduleExchange;  // how long LCD hold in seconds 
static tU16 lcdHoldTime;
#pragma vector =  TIMER0_B1_VECTOR
__interrupt void timer0B1B6Isr(void)
{ 
  switch(__even_in_range(TB0IV, TB0IV_TB0IFG))
  {  
    case 0:               // No Interrupt pending
      break;
    case TB0IV_TBCCR1:   // TBCCR
      break;
    case TB0IV_TBCCR2:   // TBCCR2
      break;
    case TB0IV_TBCCR3:   // TBCCR3
      // radio quiet time, expired 
#if 1
      radioTimerStop();
      //u8Layer2TaskScheduler |= DONE_TX_RADIO_MSG;
      __bic_SR_register_on_exit(EMV_LPM_ON); // weak up CPU      
#endif
      break;
    case TB0IV_TBCCR4:   // TBCCR4
      // exchg Sleep Timer Expired
      stopExchgSleepTimer();
      // weakup, do "done read" steps
      u8Layer2TaskScheduler |= RD_RAM_DATA_DONE;
      __bic_SR_register_on_exit(EMV_LPM_ON); // weak up CPU
      break;
    case TB0IV_TBCCR5:   // TBCCR5
      otherMspNoReadReTx();
      __bic_SR_register_on_exit(EMV_LPM_ON); // weak up CPU
      break;
    case TB0IV_TBCCR6:   // TBCCR6
      //to here is 0.125seconds with/without USB 
      u8ExchOn++;
      if(u8ExchOn > LCD_UPDATE_INTERVAL)  // 2seconds
      {
        if(u8TxIt >= 5)                 // 4 exchange type
        {                               // update LCD, do one data exchange
          u8TxIt = 0;                   // done 4 all of them, reset and repeats
        }
        addOneTypeExchg(u8LpTxType[u8TxIt]);  // 5638 use it
        u8TxIt++;
        __bic_SR_register_on_exit(EMV_LPM_ON); // weak up CPU
        u8ExchOn = 0;
      }

      if (I2cStateReg >= LCD_HOLD_TIME_ON)
      {
        lcdHoldTime++;
        if(lcdHoldTime > (LCD_BASE_HOLD * info5638A.u16LcdHoldTime)) // 2seconds
        {               
          I2cStateReg = CH1_STOP;         // update LCD
          lcdHoldTime = 0;                // clear hold time
        }
      }
      
      if (I2cStateReg == CH1_STOP)
      {
         // do LCD, power on LCD
         GPIO_setOutputHighOnPin(
              __MSP430_BASEADDRESS_PORT8_R__,
              GPIO_PORT_P8,
              P8_7_LCD_PWR_OUT);
                
          // LCD in idle state, start update
          u8Layer2TaskScheduler |= PRO_LCD_UPDATE_REQ;
          __bic_SR_register_on_exit(EMV_LPM_ON); // weak up CPU
     }

      // LCD stuck in one state to long
      if((I2cStateReg > CH1_STOP) && (I2cStateReg < LCD_HOLD_TIME_ON))
      { 
        if(I2cStateReg != lcdState)      
        {       
          lcdState = I2cStateReg;        
          inLcdStates = 0;
        }
        else
        {
          inLcdStates++;
          if(inLcdStates > 10)
          {       
            u8Layer1TaskScheduler |= INTERF_SEND_LCD_CTRL1;
            __bic_SR_register_on_exit(EMV_LPM_ON); // weak up CPU
            inLcdStates = 0;
          }
        }
      }

      // keep this time interrupt come in
      startTaskScheduler();
      __no_operation();
      break; 
    case 14:              // over flow
      //correctRtcValue();
      __no_operation();
      break;
    default:
      break;
  }
}

//=============================================================================
// end file by xuedong
//=============================================================================

