//=============================================================================
// Header: Controls SPI port
// Date Time: 1/21/2013 12:06:26 PM
// Author: Xuedong.Liu
//
// Description: This file has SPI port read/write functions.  each device has
//     its own SPI bus, the device are RTD, Sensor1, sensor 2, Flash, and RAM.
//
//     write and read Flash and RAM need sychnize with other MSP.  This SPI port
//     initial and unuse state, they are input pin, when in-use are function pins.
//     at inital development (only one MSP functional) the interaction are not in
//     function, they are out/input pin, output pins are output low.
//
//     SPI port is call/activated from the upper layer
//=============================================================================
// constant definitions, type definitions
#include "masterHeader.h"

// global volatile variables
#include "externGlobalVar.h"

// global function prototypes definitions:
#include "externFunc.h"

// local statice function prototypes definitions:
tU8 spiRxtU8(tU8 device);

// local static variables
static void wtRamSpi(tU8 data);
static tU8 rdRamSpi(void);
static void commonEraseFlash(tU32 addr, tU8 u8ErseCmd);

// global & local function implementation
//=============================================================================
// Flash chip read and write, JT808 is W25Q80,
//    the chip select control by the upper layer
//=============================================================================
tU8 rdFlashSpi(void)
{
  tU8 data;
  __no_operation(); 
  //USCI_B0 TX buffer ready?
  data = SPI_receiveData(__MSP430_BASEADDRESS_USCI_B0__);

  //Transmit dummy byte, to generate CLK
  SPI_transmitData(__MSP430_BASEADDRESS_USCI_B0__, 0);

  //USCI_B0 TX buffer ready?
  while (!SPI_getInterruptStatus(__MSP430_BASEADDRESS_USCI_B0__,
               SPI_RECEIVE_INTERRUPT)) 
  {
    __no_operation(); 
  }
  // SPI when Tx & Rx complete at same time
  data = SPI_receiveData(__MSP430_BASEADDRESS_USCI_B0__);
  
  __no_operation(); 
  return (data);
}

void wtFlashSpi(tU8 data)
{
  __no_operation(); 
  SPI_transmitData(__MSP430_BASEADDRESS_USCI_B0__, data);
  while (!SPI_getInterruptStatus(__MSP430_BASEADDRESS_USCI_B0__,
             SPI_RECEIVE_INTERRUPT)) 
  {
    __no_operation();    
  }
  SPI_receiveData(__MSP430_BASEADDRESS_USCI_B0__);
  __no_operation(); 
}

void flashCsEnable(void)
{
#if 1
  //Set output low,
  P3DIR |= P3_6_5638_FLASH_CS_OUT;
  //Set output direction
  // set low, CS enable
  P3OUT &= ~P3_6_5638_FLASH_CS_OUT;  
#else
  GPIO_setAsOutputPin(
        __MSP430_BASEADDRESS_PORT3_R__,
        GPIO_PORT_P3,
        P3_6_5638_FLASH_CS_OUT);
    
    //Set output low
    GPIO_setOutputLowOnPin(
        __MSP430_BASEADDRESS_PORT3_R__,
        GPIO_PORT_P3,
        P3_6_5638_FLASH_CS_OUT);
#endif
}

void flashCsDisbale(void)
{
#if 1
  // set low, CS enable
  P3OUT |= P3_6_5638_FLASH_CS_OUT;  
  //Set output low,
  P3DIR &= ~P3_6_5638_FLASH_CS_OUT;
  //Set output direction
#else
    GPIO_setOutputHighOnPin(
        __MSP430_BASEADDRESS_PORT3_R__,
        GPIO_PORT_P3,
        P3_6_5638_FLASH_CS_OUT);

    // becasue has pull up resistor, set to input will make CS go high
    GPIO_setAsInputPin(
        __MSP430_BASEADDRESS_PORT3_R__,
        GPIO_PORT_P3,
        P3_6_5638_FLASH_CS_OUT);
#endif
}

void enableFlashSpi(void)
{
    GPIO_setAsPeripheralModuleFunctionOutputPin(
        __MSP430_BASEADDRESS_PORT2_R__,
        GPIO_PORT_P2,
        (P2_3_USB0_SPI_FLASH_CLK_FUNC +
         P2_1_USB0_SPI_FLASH_SIMO_FUNC));

    GPIO_setAsPeripheralModuleFunctionInputPin(
        __MSP430_BASEADDRESS_PORT2_R__,
        GPIO_PORT_P2,
        P2_2_USB0_SPI_FLASH_SOMI_FUNC);
    
    if (STATUS_FAIL == SPI_masterInit(__MSP430_BASEADDRESS_USCI_B0__,
            SPI_CLOCKSOURCE_SMCLK,
            UCS_getSMCLK(__MSP430_BASEADDRESS_UCS__),
            FLASH_SPI_CLK,
            SPI_MSB_FIRST,
            SPI_PHASE_DATA_CHANGED_ONFIRST_CAPTURED_ON_NEXT,
            SPI_CLOCKPOLARITY_INACTIVITY_HIGH)
        )
    {
      // set status bit, TDB, do not use Sensor 1 any more
      // loop here foreve to fix problem, FIXME
      __no_operation();  // debug, set break point here
    }

    //Enable SPI module
    SPI_enable(__MSP430_BASEADDRESS_USCI_B0__);
}

void disableFlashSpi(void)
{
    SPI_disable (__MSP430_BASEADDRESS_USCI_B0__);

    GPIO_setAsInputPin(
        __MSP430_BASEADDRESS_PORT2_R__,
        GPIO_PORT_P2,
        (P2_1_USB0_SPI_FLASH_SIMO_FUNC +
        P2_3_USB0_SPI_FLASH_CLK_FUNC +
        P2_2_USB0_SPI_FLASH_SOMI_FUNC));
}

//=============================================================================
// RAM chip read and write, JT808 used 23A1024,  SPI Port is USCI B2
//    the chip select control by the upper layer
//=============================================================================
//extern tU8 rdRamSpi(void);
static tU8 rdRamSpi(void)
{
  tU8 data;
  //USCI_A1 TX buffer ready?
  while (!SPI_getInterruptStatus(__MSP430_BASEADDRESS_USCI_A1__,
                  SPI_TRANSMIT_INTERRUPT))
  {
    ;
  }
  SPI_clearInterruptFlag (__MSP430_BASEADDRESS_USCI_A1__,
             SPI_TRANSMIT_INTERRUPT);

  //__disable_interrupt();
  //Transmit dummy byte, to generate CLK
  SPI_transmitData(__MSP430_BASEADDRESS_USCI_A1__, 0);

  while (!SPI_getInterruptStatus(__MSP430_BASEADDRESS_USCI_A1__,
                  SPI_TRANSMIT_INTERRUPT))
  {
    ;
  }
  //__enable_interrupt(); 

  // SPI when Tx & Rx complete at same time
  data = SPI_receiveData(__MSP430_BASEADDRESS_USCI_A1__);
  return (data);
}

static void wtRamSpi(tU8 data)
{
  //USCI_B2 TX buffer ready?
  while (!SPI_getInterruptStatus(__MSP430_BASEADDRESS_USCI_A1__,
              SPI_TRANSMIT_INTERRUPT)) ;
  //__disable_interrupt();
  //Transmit Data to slave
  SPI_transmitData(__MSP430_BASEADDRESS_USCI_A1__, data);
  while (!SPI_getInterruptStatus(__MSP430_BASEADDRESS_USCI_A1__,
              SPI_TRANSMIT_INTERRUPT)) ;  

  //__enable_interrupt(); 
}

void ramCsEnable(void)
{
    //Set output low
    GPIO_setOutputLowOnPin(
        __MSP430_BASEADDRESS_PORT3_R__,
        GPIO_PORT_P3,
        P3_7_5638_RAM_CS_OUT);

    //Set output direction
    GPIO_setAsOutputPin(
        __MSP430_BASEADDRESS_PORT3_R__,
        GPIO_PORT_P3,
        P3_7_5638_RAM_CS_OUT);    
}

void ramCsDisable(void)
{
  GPIO_setOutputHighOnPin(
        __MSP430_BASEADDRESS_PORT3_R__,
        GPIO_PORT_P3,
        P3_7_5638_RAM_CS_OUT);

  // becasue has pull up resistor, set to input will make CS go high
  GPIO_setAsInputPinWithPullUpresistor (
        __MSP430_BASEADDRESS_PORT3_R__,
        GPIO_PORT_P3,
        P3_7_5638_RAM_CS_OUT);
}

void enableRamSpi(void)
{
    GPIO_setAsPeripheralModuleFunctionOutputPin(
        __MSP430_BASEADDRESS_PORT8_R__,
        GPIO_PORT_P8,
        ( P8_1_UCA1_SPI_RAM_Clk_FUNC +
          P8_2_UCA1_SPI_RAM_SIMO_FUNC +
          P8_3_UCA1_SPI_RAM_SOMI_FUNC));

    if (STATUS_FAIL == SPI_masterInit(__MSP430_BASEADDRESS_USCI_A1__,
            SPI_CLOCKSOURCE_SMCLK,
            UCS_getSMCLK(__MSP430_BASEADDRESS_UCS__),
            RAM_SPICLK,
            SPI_MSB_FIRST,
            SPI_PHASE_DATA_CAPTURED_ONFIRST_CHANGED_ON_NEXT,
            SPI_CLOCKPOLARITY_INACTIVITY_LOW)
        )
    {
      // set status bit, TDB, do not use Sensor 1 any more
      // loop here foreve to fix problem, FIXME
      __no_operation();  // debug, set break point here
    }
    
    // set max speed
    UCA1BRW = 0x0001;

    //Enable SPI module
    SPI_enable(__MSP430_BASEADDRESS_USCI_A1__);

    ramCsDisable();
}

void disableRamSpi(void)
{
    SPI_disable (__MSP430_BASEADDRESS_USCI_A1__);

    GPIO_setAsInputPinWithPullUpresistor (
        __MSP430_BASEADDRESS_PORT8_R__,
        GPIO_PORT_P8,
        ( P8_1_UCA1_SPI_RAM_Clk_FUNC +
          P8_2_UCA1_SPI_RAM_SIMO_FUNC +
          P8_3_UCA1_SPI_RAM_SOMI_FUNC));
        
    waste_time(100);  // 100 usec at 8Mhz clk
}

void ramRdSpiData(RAM_SPI_ACCESS *rdCtrl)
{
    union
    {
        tU8 u8TempB[4];
        tU32 u32Temp;
    } u32ToU8;
    tU8 u8SizeTemp;

    u32ToU8.u32Temp = rdCtrl->u32RamAddr;
 
    __disable_interrupt();

    // write SPI CMD
    wtRamSpi(RAM_RD_DATA_CMD);
    // write SPI bit24 - bit0 ADDRESS
    wtRamSpi(u32ToU8.u8TempB[2]);
    wtRamSpi(u32ToU8.u8TempB[1]);
    wtRamSpi(u32ToU8.u8TempB[0]);

    // data exchange use type, cnts, chart recorder not use those
    if(rdCtrl->u8Type != TYPE_CHART_REC)
    {
      u8SizeTemp = rdCtrl->u8cnt = rdRamSpi();
      rdCtrl->u8Type = rdRamSpi();
    }
    else
    {
      u8SizeTemp = rdCtrl->u8cnt;
    }

        // here is data
    while (u8SizeTemp > 0)
    {
        *rdCtrl->u8DataAddr = rdRamSpi();
        rdCtrl->u8DataAddr++;
        u8SizeTemp--;
    }

    __enable_interrupt(); 

    __no_operation();
}

void ramWtSpiData(RAM_SPI_ACCESS *wtCtrl)
{
    union
    {
        tU8 u8TempB[4];
        tU32 u32Temp;
    } u32ToU8;

    tU8 u8temp;

    RAM_SPI_ACCESS wtCtrlRamV;
    memmove(&wtCtrlRamV, wtCtrl, RAM_ACCESS_SIZE);

    u32ToU8.u32Temp = wtCtrlRamV.u32RamAddr;
    
    __disable_interrupt();
    
    // write SPI CMD
    wtRamSpi(RAM_WT_DATA_CMD);
    // write SPI bit24 - bit0 ADDRESS
    wtRamSpi(u32ToU8.u8TempB[2]);
    wtRamSpi(u32ToU8.u8TempB[1]);
    wtRamSpi(u32ToU8.u8TempB[0]);

    // number bytes in updates
    wtRamSpi(wtCtrl->u8cnt);

    // type exchange
    wtRamSpi(wtCtrl->u8Type);

    // here is data
    while (wtCtrlRamV.u8cnt > 0)
    {
        u8temp = *wtCtrlRamV.u8DataAddr;
        wtRamSpi(u8temp);
        wtCtrlRamV.u8DataAddr++;
        wtCtrlRamV.u8cnt--;
    }

    __enable_interrupt();

    __no_operation();  // debug, set break point here
}

//*****************************************
// Flash function
//*****************************************
void flashQ80PowerDown(void)
{
  //enableFlashSpi();
  flashCsEnable();                // Select the device, referenc eto the device datasheet
  wtFlashSpi(EEPWDN);             // Enable power down mode
  flashCsDisbale();               // Deselect the device to set the write enable latch     
  waste_time(7);                  // is about 7uSec, it is greater than the datasheet request 2uSec
  // debug following 3lines
  flashCsEnable();                // Select the device, referenc eto the device datasheet
  wtFlashSpi(0x07);               // debuge SPI message
  flashCsDisbale();               // Deselect the device to set the write enable latch     
  //disableFlashSpi();  
}

void flashQ80NormalOp(void)
{
  // relase flash from power down
  //enableFlashSpi();
  flashCsEnable();                // Select the device, referenc eto the device datasheet
  wtFlashSpi(EEPNORMAL);          // release from power down mode
  flashCsDisbale();               // Deselect the device to set the write enable latch     
  waste_time(7);                  // is about 7uSec, it is greater than the datasheet request 2uSec
  // debug following 3lines
  flashCsEnable();                // Select the device, referenc eto the device datasheet
  wtFlashSpi(0x08);               // debuge SPI message
  flashCsDisbale();               // Deselect the device to set the write enable latch     
  //disableFlashSpi();  
}

tU8 readQ80Status(void)
{
  //note: interrupt, and process interrupt disabled 
  tU8 u8Status;
  
  __disable_interrupt();        //no two SPI device access same time
  
  flashCsEnable();                // Select the device, referenc eto the device datasheet
  wtFlashSpi(EEPRDSR);            // Enable Writes
  u8Status = rdFlashSpi();        // get status value
  flashCsDisbale();               // Deselect the device to set the write enable latch     
  
  __enable_interrupt();

  return u8Status;
}

void erase64kW25Q80Flash(tU32 addr)
{
  commonEraseFlash(addr, EEPBLOCK_E);
}

void eraseW25Q80Flash(tU32 addr)
{
  commonEraseFlash(addr, EEPSECTOR_E);
}

static void commonEraseFlash(tU32 addr, tU8 u8ErseCmd)
{
    union
    {
        tU8 u8AddressB[4];
        tU32 u32InAddress;
    } temp;
    temp.u32InAddress = addr;
    //enableFlashSpi();

    __disable_interrupt();        //no two SPI device access same time
    flashCsEnable();                // Select the device, referenc eto the device datasheet
    // check device busy may need,  FIXME
    wtFlashSpi(EEPWREN);            // Enable Writes
    flashCsDisbale();               // Deselect the device to set the write enable latch

    flashCsEnable();                // Select the device again
    wtFlashSpi(u8ErseCmd);        // Send the Write command to the device
    // address 16 bits
    wtFlashSpi(temp.u8AddressB[2]); // send 24 bits address, MSB 7 bits
    wtFlashSpi(temp.u8AddressB[1]); // send 24 bits address, middle 7 bits
    wtFlashSpi(temp.u8AddressB[0]); // send 24 bits address, LSB 7 bits
    __enable_interrupt();

    flashCsDisbale();               // Deselect the device to set the write enable latch

    u8FlashStatus = ERASE_START;
      
    //disableFlashSpi();   
}

//==============================================================================
// FUNCTION:        eepGetEEpromData()
//
// DESCRIPTION:     read the configuration from the flash, Hardware uses W25Q80BV
//
// PARAMETERS:      NAME              TYPE   DESCRIPTION
//                  ----------        ------ ------------------------------------------
//                  addr              tU16   flash address, place to start read data
//                  count             tU16   number bytes to read
//                  *data             tU8 *  point to the data will store
//
// RETURNS:         None
//==============================================================================
void eepGetEEpromData(tU32 addr, tU16 count, tU8 *data)
{
    union
    {
        tU8 u8AddressB[4];
        tU32 u32InAddress;
    } temp;

    temp.u32InAddress = addr;

    //enableFlashSpi();
    flashCsEnable();

    wtFlashSpi(EEPREAD);                     // send the Read command
    wtFlashSpi(temp.u8AddressB[2]);          // send 24 bits address, MSB 7 bits
    wtFlashSpi(temp.u8AddressB[1]);          // send 24 bits address, middle 7 bits
    wtFlashSpi(temp.u8AddressB[0]);          // send 24 bits address, LSB 7 bits
    while(count--)
    {
        *data++ = rdFlashSpi();              // Read the data
    }

    flashCsDisbale();
    //disableFlashSpi();
}

//==============================================================================
//  Function:  eepWriteEEprom()
//______________________________________________________________________________
//
//  Description: Writes data to the serial EEPROM, write size need less than
//               data sheet define block size
//
//  Parameters: none
//
//==============================================================================
#ifdef USE_DMA_TX_FLASH
tU8 dmaTxBuff[256];
#endif
void eepWriteEEprom(tU32 addr, tS16 count, tU8 *data)
{
    union
    {
        tU8 u8AddressB[4];
        tU32 u32InAddress;
    } temp;
    temp.u32InAddress = addr;
    
    //enableFlashSpi();
#ifndef USE_DMA_TX_FLASH
    __disable_interrupt();        //no two SPI device access same time
    flashCsEnable();                // Select the device, referenc eto the device datasheet
    // check device busy may need,  FIXME
    wtFlashSpi(EEPWREN);            // Enable Writes
    flashCsDisbale();               // Deselect the device to set the write enable latch

    flashCsEnable();                // Select the device again
    wtFlashSpi(EEPWRITE);           // Send the Write command to the device
    // address 16 bits
    wtFlashSpi(temp.u8AddressB[2]); // send 24 bits address, MSB 7 bits
    wtFlashSpi(temp.u8AddressB[1]); // send 24 bits address, middle 7 bits
    wtFlashSpi(temp.u8AddressB[0]); // send 24 bits address, LSB 7 bits
    do {
        wtFlashSpi(*data++);        // write all data bytes
    } while (--count > 0);
    // delay let data to set may need, FIXME
    flashCsDisbale();

    __enable_interrupt();         // allow interrupts again
#else
    sysRamData.diag.u16XmtStatus[3] |= WT_FLASH_ON;
    dmaTxBuff[0] = EEPWRITE;
    dmaTxBuff[1] = temp.u8AddressB[2];
    dmaTxBuff[2] = temp.u8AddressB[1];
    dmaTxBuff[3] = temp.u8AddressB[0];
    memmove(&dmaTxBuff[4], data, count);
    count += 4;
    // setup DMA Tx data to flash
    //Initialize and Setup DMA Channel 5 -> SPI B0 
    // Base Address of the DMA Module
    // Configure DMA channel 5
    // Configure channel for single transfer
    // DMA transfers will be disabled and interrupt flag will be set after every
    //   size transfer
    // Use DMA Trigger Source 19 (UCB0TXIFG)
    // Transfer Byte-to-byte
    // Trigger transfer on signal held high
    DMA_init(__MSP430_BASEADDRESS_DMAX_6__,
        DMA_CHANNEL_5,
        DMA_TRANSFER_SINGLE,
        count,                              // ist byte at start DMA send out
        DMA_TRIGGERSOURCE_19,               // UCSI B0 Tx_IFG
        DMA_SIZE_SRCBYTE_DSTBYTE,
        DMA_TRIGGER_HIGH);

    // Base Address of the DMA Module
    // Configure DMA channel 5
    // Use TxString as source
    // Increment source address after every transfe
    DMA_setSrcAddress(__MSP430_BASEADDRESS_DMAX_6__,
        DMA_CHANNEL_5,
        (unsigned long)(&dmaTxBuff),       // set send address, next is function code
        DMA_DIRECTION_INCREMENT);
    
    // Base Address of the DMA Module
    // Configure DMA channel 5
    // Use UCSI B0 Tx Buffer as destination
    // Don't move the destination address after every transfer
    DMA_setDstAddress(__MSP430_BASEADDRESS_DMAX_6__,
        DMA_CHANNEL_5,
        UART_getTransmitBufferAddressForDMA(__MSP430_BASEADDRESS_USCI_B0__),
        DMA_DIRECTION_UNCHANGED);

    flashCsEnable();                // Select the device, referenc eto the device datasheet
    // check device busy may need,  FIXME
    wtFlashSpi(EEPWREN);            // Enable Writes
    flashCsDisbale();               // Deselect the device to set the write enable latch

    flashCsEnable();                // Select the device, referenc eto the device datasheet

    DMA_enableInterrupt(__MSP430_BASEADDRESS_DMAX_6__,
        DMA_CHANNEL_5);

    DMA_enableTransfers(__MSP430_BASEADDRESS_DMAX_6__,
        DMA_CHANNEL_5);

    __no_operation();
    //SPI_transmitData(__MSP430_BASEADDRESS_USCI_B0__, EEPWRITE);

#endif
    //disableFlashSpi();
}

void dynamicWriteinfoA(void)
{
    u8OtherTaskScheduler &= (tU8)~DYNAMIC_WT_IMFOA;
    eepWtMspInfoAData((tU8 *)MSP_INFOA_ADDR, MSP_INFO_FALSH_SIZE, (tU8 *)&info5638A);
    //eepWtMspInfoData((tU8 *)MSP_INFOD_ADDR, MSP_INFO_FALSH_SIZE, (tU8 *)&info5638D);

    // after update MSP430F5438A flash, the update should send to MSP430F5638
}

void dynamicWriteinfoB(void)
{
    u8OtherTaskScheduler &= (tU8)~DYNAMIC_WT_IMFOB;
    if((info5638B.u8Io5638Control & CTRL_CHANGED) == CTRL_CHANGED)
    {
      turnOnOffDispLed(info5638B.u8Io5638Control);
    }
    {
      eepWtMspInfoData((tU8 *)MSP_INFOB_ADDR, MSP_INFO_FALSH_SIZE, (tU8 *)&info5638B);
      // after update MSP430F5438A flash, the update should send to MSP430F5638
    }
}

void dynamicWriteinfoC(void)
{
    u8OtherTaskScheduler &= (tU8)~DYNAMIC_WT_IMFOC;
    eepWtMspInfoData((tU8 *)MSP_INFOC_ADDR, MSP_INFO_FALSH_SIZE, (tU8 *)&info5638C);
    // after update MSP430F5438A flash, the update should send to MSP430F5638
}

void dynamicWriteinfoD(void)
{
    u8OtherTaskScheduler &= (tU8)~DYNAMIC_WT_IMFOD;
    eepWtMspInfoData((tU8 *)MSP_INFOD_ADDR, MSP_INFO_FALSH_SIZE, (tU8 *)&info5638D);
    // after update MSP430F5438A flash, the update should send to MSP430F5638
}

void eepWtMspInfoData(tU8 *addr, tU16 count, tU8 *data)
{
    tU16 status;
    tU8 *u8ByteAddr = (tU8 *)addr; 

    do
    {
        //Erase INFO
        Flash_SegmentErase(u8ByteAddr);
        status = Flash_EraseCheck(u8ByteAddr,
            128);
    } while (status == FLASH_STATUS_ERROR);
    // writ eto info A flash, need to clear LOCKA flage,
    //Write a 1 to this bit to change its state. Writing 0 has no effect.
    // read the reigter in user guide FCTL3 Register for detail.
#if 1
    // do 32 bits, the counter devided by 4
    FlashWrite_8(data, addr, count);
#endif
}

void eepWtMspInfoAData(tU8 *addr, tU16 count, tU8 *data)
{
    tU16 status;
    tU8 *u8ByteAddr = (tU8 *)addr; 

    // writ eto info A flash, need to clear LOCKA flage,
    //Write a 1 to this bit to change its state. Writing 0 has no effect.
    // read the reigter in user guide FCTL3 Register for detail.
    // when update TI Lib check following function, I made change in it
    FLASH_unlockInfoA(__MSP430_BASEADDRESS_FLASH__);  
    __no_operation();  // debug, 
    
    do
    {
        //Erase INFO
        Flash_SegmentErase(u8ByteAddr);
        status = Flash_EraseCheck(u8ByteAddr,
            128);
    } while (status == FLASH_STATUS_ERROR);
    
    // do 32 bits, the counter devided by 4
    FlashWrite_8(data, addr, count);
    
    //Lock Info Segment A
    FLASH_lockInfoA(__MSP430_BASEADDRESS_FLASH__);

}
// FLash read
void eepGetMspInfoData(tU8 *addr, tU16 count, tU8 *data)
{
    count = count;      // 8 bits read
    while (count > 0)
    {
        //Read from Flash
        *(data++) =  *(addr++);
        count--;
    }
}


//=============================================================================
// end file by xuedong
//=============================================================================

