/*
 * Copyright (c) 2020 Texas Instruments Incorporated - http://www.ti.com
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "board.h"

void Board_init()
{
	EALLOW;

	PinMux_init();
	ASYSCTL_init();
	ADC_init();
	CMPSS_init();
	CPUTIMER_init();
	DAC_init();
	EPWM_init();
	EPWMXBAR_init();
	GPIO_init();
	I2C_init();
	OUTPUTXBAR_init();
	SCI_init();
	SPI_init();
	INTERRUPT_init();

	EDIS;
}

void PinMux_init()
{
	//
	// EPWM5 -> FAN_PWM_GPIO Pinmux
	//
	GPIO_setPinConfig(GPIO_8_EPWM5_A);
	GPIO_setPinConfig(GPIO_9_EPWM5_B);
	//
	// EPWM1 -> DRI_PRI Pinmux
	//
	GPIO_setPinConfig(GPIO_0_EPWM1_A);
	GPIO_setPinConfig(GPIO_1_EPWM1_B);
	//
	// EPWM2 -> DRI_2ND Pinmux
	//
	GPIO_setPinConfig(GPIO_2_EPWM2_A);
	GPIO_setPinConfig(GPIO_3_EPWM2_B);
	// GPIO13 -> DEBUG_CLA Pinmux
	GPIO_setPinConfig(GPIO_13_GPIO13);
	// GPIO12 -> DEBUG_C28 Pinmux
	GPIO_setPinConfig(GPIO_12_GPIO12);
	// GPIO15 -> WORK_LED Pinmux
	GPIO_setPinConfig(GPIO_15_GPIO15);
	// GPIO14 -> FAULT_LED Pinmux
	GPIO_setPinConfig(GPIO_14_GPIO14);
	// GPIO39 -> PFC_OK_2ND_GPIO Pinmux
	GPIO_setPinConfig(GPIO_39_GPIO39);
	// GPIO59 -> HIGH_LINE_2ND_GPIO Pinmux
	GPIO_setPinConfig(GPIO_59_GPIO59);
	// GPIO40 -> PFC_VIN_CROSS_DET_2ND Pinmux
	GPIO_setPinConfig(GPIO_40_GPIO40);
	// GPIO10 -> FAN_RPM_GPIO Pinmux
	GPIO_setPinConfig(GPIO_10_GPIO10);
	// GPIO7 -> PFC_OV_DET_GPIO Pinmux
	GPIO_setPinConfig(GPIO_7_GPIO7);
	//
	// I2CA -> I2C_SLAVE Pinmux
	//
	GPIO_setPinConfig(GPIO_32_I2CA_SDA);
	GPIO_setPadConfig(32, GPIO_PIN_TYPE_PULLUP);
	GPIO_setQualificationMode(32, GPIO_QUAL_ASYNC);
	GPIO_setPinConfig(GPIO_33_I2CA_SCL);
	GPIO_setPadConfig(33, GPIO_PIN_TYPE_PULLUP);
	GPIO_setQualificationMode(33, GPIO_QUAL_ASYNC);
	//
	// OUTPUTXBAR1 -> myOUTPUTXBAR0 Pinmux
	//
	GPIO_setPinConfig(GPIO_34_OUTPUTXBAR1);
	//
	// SCIA -> DEBUG_SCI Pinmux
	//
	GPIO_setPinConfig(GPIO_28_SCIA_RX);
	GPIO_setPinConfig(GPIO_29_SCIA_TX);
	//
	// SCIB -> PRI_SCI Pinmux
	//
	GPIO_setPinConfig(GPIO_11_SCIB_RX);
	GPIO_setPinConfig(GPIO_18_SCIB_TX);
	//
	// SPIA -> SPIA_SLAVE Pinmux
	//
	GPIO_setPinConfig(GPIO_16_SPIA_SIMO);
	GPIO_setPinConfig(GPIO_17_SPIA_SOMI);
	GPIO_setPinConfig(GPIO_56_SPIA_CLK);
	GPIO_setPinConfig(GPIO_57_SPIA_STE);
	//
	// SPIB -> SPIB_RAM Pinmux
	//
	GPIO_setPinConfig(GPIO_24_SPIB_SIMO);
	GPIO_setPinConfig(GPIO_25_SPIB_SOMI);
	GPIO_setPinConfig(GPIO_26_SPIB_CLK);
	GPIO_setPinConfig(GPIO_27_SPIB_STE);

}

void ADC_init(){
	//VOUT_SEN_AD initialization

	// ADC Initialization: Write ADC configurations and power up the ADC
	// Configures the ADC module's offset trim
	ADC_setOffsetTrimAll(ADC_REFERENCE_INTERNAL,ADC_REFERENCE_3_3V);
	// Configures the analog-to-digital converter module prescaler.
	ADC_setPrescaler(VOUT_SEN_AD_BASE, ADC_CLK_DIV_1_0);
	// Sets the timing of the end-of-conversion pulse
	ADC_setInterruptPulseMode(VOUT_SEN_AD_BASE, ADC_PULSE_END_OF_CONV);
	// Powers up the analog-to-digital converter core.
	ADC_enableConverter(VOUT_SEN_AD_BASE);
	// Delay for 1ms to allow ADC time to power up
	DEVICE_DELAY_US(5000);

	// SOC Configuration: Setup ADC EPWM channel and trigger settings
	// Disables SOC burst mode.
	ADC_disableBurstMode(VOUT_SEN_AD_BASE);
	// Sets the priority mode of the SOCs.
	ADC_setSOCPriority(VOUT_SEN_AD_BASE, ADC_PRI_ALL_ROUND_ROBIN);
	// Start of Conversion 0 Configuration
	// Configures a start-of-conversion (SOC) in the ADC and its interrupt SOC trigger.
	// 	  	SOC number		: 0
	//	  	Trigger			: ADC_TRIGGER_EPWM2_SOCA
	//	  	Channel			: ADC_CH_ADCIN8
	//	 	Sample Window	: 12 SYSCLK cycles
	//		Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	ADC_setupSOC(VOUT_SEN_AD_BASE, ADC_SOC_NUMBER0, ADC_TRIGGER_EPWM2_SOCA, ADC_CH_ADCIN8, 12U);
	ADC_setInterruptSOCTrigger(VOUT_SEN_AD_BASE, ADC_SOC_NUMBER0, ADC_INT_SOC_TRIGGER_NONE);
	// ADC Interrupt 1 Configuration
	// 		SOC/EOC number	: 0
	// 		Interrupt Source: enabled
	// 		Continuous Mode	: disabled
	ADC_setInterruptSource(VOUT_SEN_AD_BASE, ADC_INT_NUMBER1, ADC_SOC_NUMBER0);
	ADC_enableInterrupt(VOUT_SEN_AD_BASE, ADC_INT_NUMBER1);
	ADC_clearInterruptStatus(VOUT_SEN_AD_BASE, ADC_INT_NUMBER1);
	ADC_disableContinuousMode(VOUT_SEN_AD_BASE, ADC_INT_NUMBER1);

	//TEMP_2ND_AD initialization

	// ADC Initialization: Write ADC configurations and power up the ADC
	// Configures the ADC module's offset trim
	ADC_setOffsetTrimAll(ADC_REFERENCE_INTERNAL,ADC_REFERENCE_3_3V);
	// Configures the analog-to-digital converter module prescaler.
	ADC_setPrescaler(TEMP_2ND_AD_BASE, ADC_CLK_DIV_1_0);
	// Sets the timing of the end-of-conversion pulse
	ADC_setInterruptPulseMode(TEMP_2ND_AD_BASE, ADC_PULSE_END_OF_ACQ_WIN);
	// Sets the timing of early interrupt generation.
	ADC_setInterruptCycleOffset(TEMP_2ND_AD_BASE, 0U);
	// Powers up the analog-to-digital converter core.
	ADC_enableConverter(TEMP_2ND_AD_BASE);
	// Delay for 1ms to allow ADC time to power up
	DEVICE_DELAY_US(5000);

	// SOC Configuration: Setup ADC EPWM channel and trigger settings
	// Disables SOC burst mode.
	ADC_disableBurstMode(TEMP_2ND_AD_BASE);
	// Sets the priority mode of the SOCs.
	ADC_setSOCPriority(TEMP_2ND_AD_BASE, ADC_PRI_ALL_ROUND_ROBIN);
	// Start of Conversion 0 Configuration
	// Configures a start-of-conversion (SOC) in the ADC and its interrupt SOC trigger.
	// 	  	SOC number		: 0
	//	  	Trigger			: ADC_TRIGGER_EPWM2_SOCA
	//	  	Channel			: ADC_CH_ADCIN6
	//	 	Sample Window	: 12 SYSCLK cycles
	//		Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	ADC_setupSOC(TEMP_2ND_AD_BASE, ADC_SOC_NUMBER0, ADC_TRIGGER_EPWM2_SOCA, ADC_CH_ADCIN6, 12U);
	ADC_setInterruptSOCTrigger(TEMP_2ND_AD_BASE, ADC_SOC_NUMBER0, ADC_INT_SOC_TRIGGER_NONE);
	// Start of Conversion 1 Configuration
	// Configures a start-of-conversion (SOC) in the ADC and its interrupt SOC trigger.
	// 	  	SOC number		: 1
	//	  	Trigger			: ADC_TRIGGER_EPWM2_SOCA
	//	  	Channel			: ADC_CH_ADCIN3
	//	 	Sample Window	: 12 SYSCLK cycles
	//		Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	ADC_setupSOC(TEMP_2ND_AD_BASE, ADC_SOC_NUMBER1, ADC_TRIGGER_EPWM2_SOCA, ADC_CH_ADCIN3, 12U);
	ADC_setInterruptSOCTrigger(TEMP_2ND_AD_BASE, ADC_SOC_NUMBER1, ADC_INT_SOC_TRIGGER_NONE);

}
void ASYSCTL_init(){
	// asysctl initialization
	// Disables the temperature sensor output to the ADC.
	ASysCtl_disableTemperatureSensor();
	// Set the analog voltage reference selection to internal.
	ASysCtl_setAnalogReferenceInternal( ASYSCTL_VREFHIA | ASYSCTL_VREFHIB | ASYSCTL_VREFHIC );
	// Set the internal analog voltage reference selection to 1.65V.
	ASysCtl_setAnalogReference1P65( ASYSCTL_VREFHIA | ASYSCTL_VREFHIB | ASYSCTL_VREFHIC );
}
 
void CMPSS_init(){
	// Select the value for CMPHNMXSEL.
	ASysCtl_selectCMPHNMux(ASYSCTL_CMPHNMUX_SELECT_7 | ASYSCTL_CMPHNMUX_SELECT_4 | ASYSCTL_CMPHNMUX_SELECT_2 | ASYSCTL_CMPHNMUX_SELECT_3 | ASYSCTL_CMPHNMUX_SELECT_1 | ASYSCTL_CMPHNMUX_SELECT_5);
	// Select the value for CMPLNMXSEL.
	ASysCtl_selectCMPLNMux(ASYSCTL_CMPLNMUX_SELECT_7 | ASYSCTL_CMPLNMUX_SELECT_4 | ASYSCTL_CMPLNMUX_SELECT_2 | ASYSCTL_CMPLNMUX_SELECT_3 | ASYSCTL_CMPLNMUX_SELECT_1 | ASYSCTL_CMPLNMUX_SELECT_5);
	// Select the value for CMPHPM[object Object]SEL.
	ASysCtl_selectCMPHPMux(ASYSCTL_CMPHPMUX_SELECT_7,0U);
	// Select the value for CMPLPM[object Object]SEL.
	ASysCtl_selectCMPLPMux(ASYSCTL_CMPLPMUX_SELECT_7,0U);
	// Select the value for CMPHPM[object Object]SEL.
	ASysCtl_selectCMPHPMux(ASYSCTL_CMPHPMUX_SELECT_4,0U);
	// Select the value for CMPLPM[object Object]SEL.
	ASysCtl_selectCMPLPMux(ASYSCTL_CMPLPMUX_SELECT_4,0U);
	// Select the value for CMPHPM[object Object]SEL.
	ASysCtl_selectCMPHPMux(ASYSCTL_CMPHPMUX_SELECT_2,0U);
	// Select the value for CMPLPM[object Object]SEL.
	ASysCtl_selectCMPLPMux(ASYSCTL_CMPLPMUX_SELECT_2,0U);
	// Select the value for CMPHPM[object Object]SEL.
	ASysCtl_selectCMPHPMux(ASYSCTL_CMPHPMUX_SELECT_3,0U);
	// Select the value for CMPLPM[object Object]SEL.
	ASysCtl_selectCMPLPMux(ASYSCTL_CMPLPMUX_SELECT_3,0U);
	// Select the value for CMPHPM[object Object]SEL.
	ASysCtl_selectCMPHPMux(ASYSCTL_CMPHPMUX_SELECT_1,0U);
	// Select the value for CMPLPM[object Object]SEL.
	ASysCtl_selectCMPLPMux(ASYSCTL_CMPLPMUX_SELECT_1,0U);
	// Select the value for CMPHPM[object Object]SEL.
	ASysCtl_selectCMPHPMux(ASYSCTL_CMPHPMUX_SELECT_5,0U);
	// Select the value for CMPLPM[object Object]SEL.
	ASysCtl_selectCMPLPMux(ASYSCTL_CMPLPMUX_SELECT_5,0U);

	//SEC_CT_DET initialization
	// Sets the configuration for the high comparator.
	CMPSS_configHighComparator(SEC_CT_DET_BASE,(CMPSS_INSRC_DAC));
	// Sets the configuration for the high comparator.
	CMPSS_configLowComparator(SEC_CT_DET_BASE,(CMPSS_INSRC_DAC));
	// Sets the configuration for the internal comparator DACs.
	CMPSS_configDAC(SEC_CT_DET_BASE,(CMPSS_DACVAL_SYSCLK | CMPSS_DACREF_VDDA | CMPSS_DACSRC_SHDW));
	// Sets the value of the internal DAC of the high comparator.
	CMPSS_setDACValueHigh(SEC_CT_DET_BASE,0U);
	// Sets the value of the internal DAC of the low comparator.
	CMPSS_setDACValueLow(SEC_CT_DET_BASE,0U);
	//  Configures the digital filter of the high comparator.
	CMPSS_configFilterHigh(SEC_CT_DET_BASE, 0U, 1U, 1U);
	// Configures the digital filter of the low comparator.
	CMPSS_configFilterLow(SEC_CT_DET_BASE, 0U, 1U, 1U);
	// Sets the output signal configuration for the high comparator.
	CMPSS_configOutputsHigh(SEC_CT_DET_BASE,(CMPSS_TRIPOUT_ASYNC_COMP | CMPSS_TRIP_ASYNC_COMP));
	// Sets the output signal configuration for the low comparator.
	CMPSS_configOutputsLow(SEC_CT_DET_BASE,(CMPSS_TRIPOUT_ASYNC_COMP | CMPSS_TRIP_ASYNC_COMP));
	// Sets the comparator hysteresis settings.
	CMPSS_setHysteresis(SEC_CT_DET_BASE,0U);
	// Configures the comparator subsystem's ramp generator.
	CMPSS_configRamp(SEC_CT_DET_BASE,0U,0U,0U,1U,true);
	// Disables reset of HIGH comparator digital filter output latch on PWMSYNC
	CMPSS_disableLatchResetOnPWMSYNCHigh(SEC_CT_DET_BASE);
	// Disables reset of LOW comparator digital filter output latch on PWMSYNC
	CMPSS_disableLatchResetOnPWMSYNCLow(SEC_CT_DET_BASE);
	// Sets the ePWM module blanking signal that holds trip in reset.
	CMPSS_configBlanking(SEC_CT_DET_BASE,1U);
	// Disables an ePWM blanking signal from holding trip in reset.
	CMPSS_disableBlanking(SEC_CT_DET_BASE);
	// Configures whether or not the digital filter latches are reset by PWMSYNC
	CMPSS_configLatchOnPWMSYNC(SEC_CT_DET_BASE,false,false);
	// Disables the CMPSS module.
	CMPSS_disableModule(SEC_CT_DET_BASE);

	//PRI_CT_DET initialization
	// Sets the configuration for the high comparator.
	CMPSS_configHighComparator(PRI_CT_DET_BASE,(CMPSS_INSRC_DAC));
	// Sets the configuration for the high comparator.
	CMPSS_configLowComparator(PRI_CT_DET_BASE,(CMPSS_INSRC_DAC));
	// Sets the configuration for the internal comparator DACs.
	CMPSS_configDAC(PRI_CT_DET_BASE,(CMPSS_DACVAL_SYSCLK | CMPSS_DACREF_VDDA | CMPSS_DACSRC_SHDW));
	// Sets the value of the internal DAC of the high comparator.
	CMPSS_setDACValueHigh(PRI_CT_DET_BASE,0U);
	// Sets the value of the internal DAC of the low comparator.
	CMPSS_setDACValueLow(PRI_CT_DET_BASE,0U);
	//  Configures the digital filter of the high comparator.
	CMPSS_configFilterHigh(PRI_CT_DET_BASE, 0U, 1U, 1U);
	// Configures the digital filter of the low comparator.
	CMPSS_configFilterLow(PRI_CT_DET_BASE, 0U, 1U, 1U);
	// Sets the output signal configuration for the high comparator.
	CMPSS_configOutputsHigh(PRI_CT_DET_BASE,(CMPSS_TRIPOUT_ASYNC_COMP | CMPSS_TRIP_ASYNC_COMP));
	// Sets the output signal configuration for the low comparator.
	CMPSS_configOutputsLow(PRI_CT_DET_BASE,(CMPSS_TRIPOUT_ASYNC_COMP | CMPSS_TRIP_ASYNC_COMP));
	// Sets the comparator hysteresis settings.
	CMPSS_setHysteresis(PRI_CT_DET_BASE,0U);
	// Configures the comparator subsystem's ramp generator.
	CMPSS_configRamp(PRI_CT_DET_BASE,0U,0U,0U,1U,true);
	// Disables reset of HIGH comparator digital filter output latch on PWMSYNC
	CMPSS_disableLatchResetOnPWMSYNCHigh(PRI_CT_DET_BASE);
	// Disables reset of LOW comparator digital filter output latch on PWMSYNC
	CMPSS_disableLatchResetOnPWMSYNCLow(PRI_CT_DET_BASE);
	// Sets the ePWM module blanking signal that holds trip in reset.
	CMPSS_configBlanking(PRI_CT_DET_BASE,1U);
	// Disables an ePWM blanking signal from holding trip in reset.
	CMPSS_disableBlanking(PRI_CT_DET_BASE);
	// Configures whether or not the digital filter latches are reset by PWMSYNC
	CMPSS_configLatchOnPWMSYNC(PRI_CT_DET_BASE,false,false);
	// Disables the CMPSS module.
	CMPSS_disableModule(PRI_CT_DET_BASE);

	//VOUT_OV_DET initialization
	// Sets the configuration for the high comparator.
	CMPSS_configHighComparator(VOUT_OV_DET_BASE,(CMPSS_INSRC_DAC));
	// Sets the configuration for the high comparator.
	CMPSS_configLowComparator(VOUT_OV_DET_BASE,(CMPSS_INSRC_DAC));
	// Sets the configuration for the internal comparator DACs.
	CMPSS_configDAC(VOUT_OV_DET_BASE,(CMPSS_DACVAL_SYSCLK | CMPSS_DACREF_VDDA | CMPSS_DACSRC_SHDW));
	// Sets the value of the internal DAC of the high comparator.
	CMPSS_setDACValueHigh(VOUT_OV_DET_BASE,2600U);
	// Sets the value of the internal DAC of the low comparator.
	CMPSS_setDACValueLow(VOUT_OV_DET_BASE,0U);
	//  Configures the digital filter of the high comparator.
	CMPSS_configFilterHigh(VOUT_OV_DET_BASE, 0U, 1U, 1U);
	// Configures the digital filter of the low comparator.
	CMPSS_configFilterLow(VOUT_OV_DET_BASE, 0U, 1U, 1U);
	// Sets the output signal configuration for the high comparator.
	CMPSS_configOutputsHigh(VOUT_OV_DET_BASE,(CMPSS_TRIPOUT_ASYNC_COMP | CMPSS_TRIP_ASYNC_COMP));
	// Sets the output signal configuration for the low comparator.
	CMPSS_configOutputsLow(VOUT_OV_DET_BASE,(CMPSS_TRIPOUT_ASYNC_COMP | CMPSS_TRIP_ASYNC_COMP));
	// Sets the comparator hysteresis settings.
	CMPSS_setHysteresis(VOUT_OV_DET_BASE,0U);
	// Configures the comparator subsystem's ramp generator.
	CMPSS_configRamp(VOUT_OV_DET_BASE,0U,0U,0U,1U,true);
	// Disables reset of HIGH comparator digital filter output latch on PWMSYNC
	CMPSS_disableLatchResetOnPWMSYNCHigh(VOUT_OV_DET_BASE);
	// Disables reset of LOW comparator digital filter output latch on PWMSYNC
	CMPSS_disableLatchResetOnPWMSYNCLow(VOUT_OV_DET_BASE);
	// Sets the ePWM module blanking signal that holds trip in reset.
	CMPSS_configBlanking(VOUT_OV_DET_BASE,1U);
	// Disables an ePWM blanking signal from holding trip in reset.
	CMPSS_disableBlanking(VOUT_OV_DET_BASE);
	// Configures whether or not the digital filter latches are reset by PWMSYNC
	CMPSS_configLatchOnPWMSYNC(VOUT_OV_DET_BASE,false,false);
	// Enables the CMPSS module.
	CMPSS_enableModule(VOUT_OV_DET_BASE);
	//SEC_CT_integral initialization
	// Sets the configuration for the high comparator.
	CMPSS_configHighComparator(SEC_CT_integral_BASE,(CMPSS_INSRC_DAC));
	// Sets the configuration for the high comparator.
	CMPSS_configLowComparator(SEC_CT_integral_BASE,(CMPSS_INSRC_DAC));
	// Sets the configuration for the internal comparator DACs.
	CMPSS_configDAC(SEC_CT_integral_BASE,(CMPSS_DACVAL_SYSCLK | CMPSS_DACREF_VDDA | CMPSS_DACSRC_SHDW));
	// Sets the value of the internal DAC of the high comparator.
	CMPSS_setDACValueHigh(SEC_CT_integral_BASE,0U);
	// Sets the value of the internal DAC of the low comparator.
	CMPSS_setDACValueLow(SEC_CT_integral_BASE,0U);
	//  Configures the digital filter of the high comparator.
	CMPSS_configFilterHigh(SEC_CT_integral_BASE, 0U, 1U, 1U);
	// Configures the digital filter of the low comparator.
	CMPSS_configFilterLow(SEC_CT_integral_BASE, 0U, 1U, 1U);
	// Sets the output signal configuration for the high comparator.
	CMPSS_configOutputsHigh(SEC_CT_integral_BASE,(CMPSS_TRIPOUT_ASYNC_COMP | CMPSS_TRIP_ASYNC_COMP));
	// Sets the output signal configuration for the low comparator.
	CMPSS_configOutputsLow(SEC_CT_integral_BASE,(CMPSS_TRIPOUT_ASYNC_COMP | CMPSS_TRIP_ASYNC_COMP));
	// Sets the comparator hysteresis settings.
	CMPSS_setHysteresis(SEC_CT_integral_BASE,0U);
	// Configures the comparator subsystem's ramp generator.
	CMPSS_configRamp(SEC_CT_integral_BASE,0U,0U,0U,1U,true);
	// Disables reset of HIGH comparator digital filter output latch on PWMSYNC
	CMPSS_disableLatchResetOnPWMSYNCHigh(SEC_CT_integral_BASE);
	// Disables reset of LOW comparator digital filter output latch on PWMSYNC
	CMPSS_disableLatchResetOnPWMSYNCLow(SEC_CT_integral_BASE);
	// Sets the ePWM module blanking signal that holds trip in reset.
	CMPSS_configBlanking(SEC_CT_integral_BASE,1U);
	// Disables an ePWM blanking signal from holding trip in reset.
	CMPSS_disableBlanking(SEC_CT_integral_BASE);
	// Configures whether or not the digital filter latches are reset by PWMSYNC
	CMPSS_configLatchOnPWMSYNC(SEC_CT_integral_BASE,false,false);
	// Disables the CMPSS module.
	CMPSS_disableModule(SEC_CT_integral_BASE);

	//PRI_CT_integral initialization
	// Sets the configuration for the high comparator.
	CMPSS_configHighComparator(PRI_CT_integral_BASE,(CMPSS_INSRC_DAC));
	// Sets the configuration for the high comparator.
	CMPSS_configLowComparator(PRI_CT_integral_BASE,(CMPSS_INSRC_DAC));
	// Sets the configuration for the internal comparator DACs.
	CMPSS_configDAC(PRI_CT_integral_BASE,(CMPSS_DACVAL_SYSCLK | CMPSS_DACREF_VDDA | CMPSS_DACSRC_SHDW));
	// Sets the value of the internal DAC of the high comparator.
	CMPSS_setDACValueHigh(PRI_CT_integral_BASE,0U);
	// Sets the value of the internal DAC of the low comparator.
	CMPSS_setDACValueLow(PRI_CT_integral_BASE,0U);
	//  Configures the digital filter of the high comparator.
	CMPSS_configFilterHigh(PRI_CT_integral_BASE, 0U, 1U, 1U);
	// Configures the digital filter of the low comparator.
	CMPSS_configFilterLow(PRI_CT_integral_BASE, 0U, 1U, 1U);
	// Sets the output signal configuration for the high comparator.
	CMPSS_configOutputsHigh(PRI_CT_integral_BASE,(CMPSS_TRIPOUT_ASYNC_COMP | CMPSS_TRIP_ASYNC_COMP));
	// Sets the output signal configuration for the low comparator.
	CMPSS_configOutputsLow(PRI_CT_integral_BASE,(CMPSS_TRIPOUT_ASYNC_COMP | CMPSS_TRIP_ASYNC_COMP));
	// Sets the comparator hysteresis settings.
	CMPSS_setHysteresis(PRI_CT_integral_BASE,0U);
	// Configures the comparator subsystem's ramp generator.
	CMPSS_configRamp(PRI_CT_integral_BASE,0U,0U,0U,1U,true);
	// Disables reset of HIGH comparator digital filter output latch on PWMSYNC
	CMPSS_disableLatchResetOnPWMSYNCHigh(PRI_CT_integral_BASE);
	// Disables reset of LOW comparator digital filter output latch on PWMSYNC
	CMPSS_disableLatchResetOnPWMSYNCLow(PRI_CT_integral_BASE);
	// Sets the ePWM module blanking signal that holds trip in reset.
	CMPSS_configBlanking(PRI_CT_integral_BASE,1U);
	// Disables an ePWM blanking signal from holding trip in reset.
	CMPSS_disableBlanking(PRI_CT_integral_BASE);
	// Configures whether or not the digital filter latches are reset by PWMSYNC
	CMPSS_configLatchOnPWMSYNC(PRI_CT_integral_BASE,false,false);
	// Disables the CMPSS module.
	CMPSS_disableModule(PRI_CT_integral_BASE);

	//IOUT_OC_DET initialization
	// Sets the configuration for the high comparator.
	CMPSS_configHighComparator(IOUT_OC_DET_BASE,(CMPSS_INSRC_DAC));
	// Sets the configuration for the high comparator.
	CMPSS_configLowComparator(IOUT_OC_DET_BASE,(CMPSS_INSRC_DAC | CMPSS_INV_INVERTED));
	// Sets the configuration for the internal comparator DACs.
	CMPSS_configDAC(IOUT_OC_DET_BASE,(CMPSS_DACVAL_SYSCLK | CMPSS_DACREF_VDDA | CMPSS_DACSRC_SHDW));
	// Sets the value of the internal DAC of the high comparator.
	CMPSS_setDACValueHigh(IOUT_OC_DET_BASE,3846U);
	// Sets the value of the internal DAC of the low comparator.
	CMPSS_setDACValueLow(IOUT_OC_DET_BASE,150U);
	//  Configures the digital filter of the high comparator.
	CMPSS_configFilterHigh(IOUT_OC_DET_BASE, 5U, 10U, 6U);
	// Configures the digital filter of the low comparator.
	CMPSS_configFilterLow(IOUT_OC_DET_BASE, 5U, 10U, 6U);
	// Initializes the digital filter of the high comparator.
	CMPSS_initFilterHigh(IOUT_OC_DET_BASE);
	// Initializes the digital filter of the low comparator.
	CMPSS_initFilterLow(IOUT_OC_DET_BASE);
	// Sets the output signal configuration for the high comparator.
	CMPSS_configOutputsHigh(IOUT_OC_DET_BASE,(CMPSS_TRIPOUT_FILTER | CMPSS_TRIP_FILTER));
	// Sets the output signal configuration for the low comparator.
	CMPSS_configOutputsLow(IOUT_OC_DET_BASE,(CMPSS_TRIPOUT_FILTER | CMPSS_TRIP_FILTER));
	// Sets the comparator hysteresis settings.
	CMPSS_setHysteresis(IOUT_OC_DET_BASE,0U);
	// Configures the comparator subsystem's ramp generator.
	CMPSS_configRamp(IOUT_OC_DET_BASE,0U,0U,0U,1U,true);
	// Disables reset of HIGH comparator digital filter output latch on PWMSYNC
	CMPSS_disableLatchResetOnPWMSYNCHigh(IOUT_OC_DET_BASE);
	// Disables reset of LOW comparator digital filter output latch on PWMSYNC
	CMPSS_disableLatchResetOnPWMSYNCLow(IOUT_OC_DET_BASE);
	// Sets the ePWM module blanking signal that holds trip in reset.
	CMPSS_configBlanking(IOUT_OC_DET_BASE,1U);
	// Disables an ePWM blanking signal from holding trip in reset.
	CMPSS_disableBlanking(IOUT_OC_DET_BASE);
	// Configures whether or not the digital filter latches are reset by PWMSYNC
	CMPSS_configLatchOnPWMSYNC(IOUT_OC_DET_BASE,false,false);
	// Enables the CMPSS module.
	CMPSS_enableModule(IOUT_OC_DET_BASE);
	// Delay for CMPSS DAC to power up.
	DEVICE_DELAY_US(500);
	// Causes a software reset of the high comparator digital filter output latch.
	CMPSS_clearFilterLatchHigh(IOUT_OC_DET_BASE);
	// Causes a software reset of the low comparator digital filter output latch.
	CMPSS_clearFilterLatchLow(IOUT_OC_DET_BASE);
}
void CPUTIMER_init(){
	//SWTIRMER initialization 
	CPUTimer_setEmulationMode(SWTIRMER_BASE, CPUTIMER_EMULATIONMODE_RUNFREE);
	CPUTimer_selectClockSource(SWTIRMER_BASE, CPUTIMER_CLOCK_SOURCE_SYS, CPUTIMER_CLOCK_PRESCALER_2);
	CPUTimer_setPreScaler(SWTIRMER_BASE, 1U);
	CPUTimer_setPeriod(SWTIRMER_BASE, 50000000U);
	CPUTimer_disableInterrupt(SWTIRMER_BASE);
	CPUTimer_stopTimer(SWTIRMER_BASE);

	CPUTimer_reloadTimerCounter(SWTIRMER_BASE);
	CPUTimer_startTimer(SWTIRMER_BASE);
}
void DAC_init(){
	// DAC1 initialization
	// Set DAC reference voltage.
	DAC_setReferenceVoltage(DAC1_BASE, DAC_REF_ADC_VREFHI);
	// Set DAC gain mode.
	DAC_setGainMode(DAC1_BASE, DAC_GAIN_TWO);
	// Set DAC load mode.
	DAC_setLoadMode(DAC1_BASE, DAC_LOAD_SYSCLK);
	// Enable the DAC output
	DAC_enableOutput(DAC1_BASE);
	// Set the DAC shadow output
	DAC_setShadowValue(DAC1_BASE, 0U);

	// DAC2 initialization
	// Set DAC reference voltage.
	DAC_setReferenceVoltage(DAC2_BASE, DAC_REF_VDAC);
	// Set DAC load mode.
	DAC_setLoadMode(DAC2_BASE, DAC_LOAD_SYSCLK);
	// Disable the DAC output
	DAC_disableOutput(DAC2_BASE);
	// Set the DAC shadow output
	DAC_setShadowValue(DAC2_BASE, 0U);

	// Delay for buffered DAC to power up.
	DEVICE_DELAY_US(5000);

}

void EPWM_init(){
	//FAN_PWM_GPIO initialization
	//DRI_PRI initialization
	//DRI_2ND initialization
}
void EPWMXBAR_init(){
	//VOUT_OVP_EPWMXBAR0 initialization
		
	XBAR_setEPWMMuxConfig(XBAR_TRIP4, XBAR_EPWM_MUX02_CMPSS2_CTRIPH);
	XBAR_enableEPWMMux(XBAR_TRIP4, XBAR_MUX02);

	//IOUT_OCP_EPWMXBAR1 initialization
		
	XBAR_setEPWMMuxConfig(XBAR_TRIP5, XBAR_EPWM_MUX08_CMPSS5_CTRIPH);
	XBAR_enableEPWMMux(XBAR_TRIP5, XBAR_MUX08);

	//IOUT_OCP_EPWMXBAR2 initialization
		
	XBAR_setEPWMMuxConfig(XBAR_TRIP7, XBAR_EPWM_MUX09_CMPSS5_CTRIPL);
	XBAR_enableEPWMMux(XBAR_TRIP7, XBAR_MUX09);

}
void GPIO_init(){
		
	//DEBUG_CLA initialization
	GPIO_setDirectionMode(DEBUG_CLA, GPIO_DIR_MODE_OUT);
	GPIO_setPadConfig(DEBUG_CLA, GPIO_PIN_TYPE_STD);
	GPIO_setMasterCore(DEBUG_CLA, GPIO_CORE_CPU1_CLA1);
	GPIO_setQualificationMode(DEBUG_CLA, GPIO_QUAL_SYNC);
		
	//DEBUG_C28 initialization
	GPIO_setDirectionMode(DEBUG_C28, GPIO_DIR_MODE_OUT);
	GPIO_setPadConfig(DEBUG_C28, GPIO_PIN_TYPE_STD);
	GPIO_setMasterCore(DEBUG_C28, GPIO_CORE_CPU1);
	GPIO_setQualificationMode(DEBUG_C28, GPIO_QUAL_SYNC);
		
	//WORK_LED initialization
	GPIO_setDirectionMode(WORK_LED, GPIO_DIR_MODE_OUT);
	GPIO_setPadConfig(WORK_LED, GPIO_PIN_TYPE_STD);
	GPIO_setMasterCore(WORK_LED, GPIO_CORE_CPU1);
	GPIO_setQualificationMode(WORK_LED, GPIO_QUAL_SYNC);
		
	//FAULT_LED initialization
	GPIO_setDirectionMode(FAULT_LED, GPIO_DIR_MODE_OUT);
	GPIO_setPadConfig(FAULT_LED, GPIO_PIN_TYPE_STD);
	GPIO_setMasterCore(FAULT_LED, GPIO_CORE_CPU1);
	GPIO_setQualificationMode(FAULT_LED, GPIO_QUAL_SYNC);
		
	//PFC_OK_2ND_GPIO initialization
	GPIO_setDirectionMode(PFC_OK_2ND_GPIO, GPIO_DIR_MODE_IN);
	GPIO_setPadConfig(PFC_OK_2ND_GPIO, GPIO_PIN_TYPE_PULLUP);
	GPIO_setMasterCore(PFC_OK_2ND_GPIO, GPIO_CORE_CPU1);
	GPIO_setQualificationMode(PFC_OK_2ND_GPIO, GPIO_QUAL_SYNC);
		
	//HIGH_LINE_2ND_GPIO initialization
	GPIO_setDirectionMode(HIGH_LINE_2ND_GPIO, GPIO_DIR_MODE_IN);
	GPIO_setPadConfig(HIGH_LINE_2ND_GPIO, GPIO_PIN_TYPE_PULLUP);
	GPIO_setMasterCore(HIGH_LINE_2ND_GPIO, GPIO_CORE_CPU1);
	GPIO_setQualificationMode(HIGH_LINE_2ND_GPIO, GPIO_QUAL_SYNC);
		
	//PFC_VIN_CROSS_DET_2ND initialization
	GPIO_setDirectionMode(PFC_VIN_CROSS_DET_2ND, GPIO_DIR_MODE_IN);
	GPIO_setPadConfig(PFC_VIN_CROSS_DET_2ND, GPIO_PIN_TYPE_PULLUP);
	GPIO_setMasterCore(PFC_VIN_CROSS_DET_2ND, GPIO_CORE_CPU1);
	GPIO_setQualificationMode(PFC_VIN_CROSS_DET_2ND, GPIO_QUAL_SYNC);
		
	//FAN_RPM_GPIO initialization
	GPIO_setDirectionMode(FAN_RPM_GPIO, GPIO_DIR_MODE_IN);
	GPIO_setPadConfig(FAN_RPM_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setMasterCore(FAN_RPM_GPIO, GPIO_CORE_CPU1);
	GPIO_setQualificationMode(FAN_RPM_GPIO, GPIO_QUAL_SYNC);
		
	//PFC_OV_DET_GPIO initialization
	GPIO_setDirectionMode(PFC_OV_DET_GPIO, GPIO_DIR_MODE_IN);
	GPIO_setPadConfig(PFC_OV_DET_GPIO, GPIO_PIN_TYPE_PULLUP);
	GPIO_setMasterCore(PFC_OV_DET_GPIO, GPIO_CORE_CPU1);
	GPIO_setQualificationMode(PFC_OV_DET_GPIO, GPIO_QUAL_SYNC);
}
void I2C_init(){
	//I2C_SLAVE initialization
	
	I2C_disableModule(I2C_SLAVE_BASE);
	I2C_configureModuleFrequency(I2C_SLAVE_BASE, DEVICE_SYSCLK_FREQ);
	I2C_setConfig(I2C_SLAVE_BASE, I2C_SLAVE_RECEIVE_MODE);
	I2C_setOwnSlaveAddress(I2C_SLAVE_BASE, 0);
	I2C_setBitCount(I2C_SLAVE_BASE, I2C_BITCOUNT_8);
	I2C_setDataCount(I2C_SLAVE_BASE, 1);
	I2C_setAddressMode(I2C_SLAVE_BASE, I2C_ADDR_MODE_7BITS);
	I2C_enableFIFO(I2C_SLAVE_BASE);
	I2C_setEmulationMode(I2C_SLAVE_BASE, I2C_EMULATION_FREE_RUN);
	I2C_enableModule(I2C_SLAVE_BASE);

}
void INTERRUPT_init(){
	
	// Interrupt Setings for INT_VOUT_SEN_AD_1
	Interrupt_register(INT_VOUT_SEN_AD_1, &INT_VOUT_SEN_AD_1_ISR);
	Interrupt_enable(INT_VOUT_SEN_AD_1);
}

void OUTPUTXBAR_init(){
	
	//myOUTPUTXBAR0 initialization
	XBAR_setOutputLatchMode(XBAR_OUTPUT1, false);
	XBAR_invertOutputSignal(XBAR_OUTPUT1, false);
		
	//Mux configuration
	XBAR_setOutputMuxConfig(XBAR_OUTPUT1, XBAR_OUT_MUX02_CMPSS2_CTRIPOUTH);
	XBAR_enableOutputMux(XBAR_OUTPUT1, XBAR_MUX02);
}
void SCI_init(){
	
	//DEBUG_SCI initialization
	SCI_clearInterruptStatus(DEBUG_SCI_BASE, SCI_INT_RXFF | SCI_INT_TXFF | SCI_INT_FE | SCI_INT_OE | SCI_INT_PE | SCI_INT_RXERR | SCI_INT_RXRDY_BRKDT | SCI_INT_TXRDY);
	SCI_clearOverflowStatus(DEBUG_SCI_BASE);

	SCI_resetTxFIFO(DEBUG_SCI_BASE);
	SCI_resetRxFIFO(DEBUG_SCI_BASE);
	SCI_resetChannels(DEBUG_SCI_BASE);

	SCI_setConfig(DEBUG_SCI_BASE, DEVICE_LSPCLK_FREQ, DEBUG_SCI_BAUDRATE, (SCI_CONFIG_WLEN_8|SCI_CONFIG_STOP_ONE|SCI_CONFIG_PAR_EVEN));
	SCI_disableLoopback(DEBUG_SCI_BASE);
	SCI_performSoftwareReset(DEBUG_SCI_BASE);
	SCI_enableFIFO(DEBUG_SCI_BASE);
	SCI_enableModule(DEBUG_SCI_BASE);
	
	//PRI_SCI initialization
	SCI_clearInterruptStatus(PRI_SCI_BASE, SCI_INT_RXFF | SCI_INT_TXFF | SCI_INT_FE | SCI_INT_OE | SCI_INT_PE | SCI_INT_RXERR | SCI_INT_RXRDY_BRKDT | SCI_INT_TXRDY);
	SCI_clearOverflowStatus(PRI_SCI_BASE);

	SCI_resetTxFIFO(PRI_SCI_BASE);
	SCI_resetRxFIFO(PRI_SCI_BASE);
	SCI_resetChannels(PRI_SCI_BASE);

	SCI_setConfig(PRI_SCI_BASE, DEVICE_LSPCLK_FREQ, PRI_SCI_BAUDRATE, (SCI_CONFIG_WLEN_8|SCI_CONFIG_STOP_ONE|SCI_CONFIG_PAR_EVEN));
	SCI_disableLoopback(PRI_SCI_BASE);
	SCI_performSoftwareReset(PRI_SCI_BASE);
	SCI_enableFIFO(PRI_SCI_BASE);
	SCI_enableModule(PRI_SCI_BASE);
}
void SPI_init()
{
	
	//SPIA_SLAVE initialization
	SPI_disableModule(SPIA_SLAVE_BASE);
	SPI_setConfig(SPIA_SLAVE_BASE, DEVICE_LSPCLK_FREQ, SPI_PROT_POL0PHA0,
				  SPI_MODE_SLAVE, 1250000, 	16);
	SPI_enableFIFO(SPIA_SLAVE_BASE);
	SPI_disableLoopback(SPIA_SLAVE_BASE);
	SPI_setEmulationMode(SPIA_SLAVE_BASE, SPI_EMULATION_STOP_MIDWAY);
	SPI_enableModule(SPIA_SLAVE_BASE);
	
	//SPIB_RAM initialization
	SPI_disableModule(SPIB_RAM_BASE);
	SPI_setConfig(SPIB_RAM_BASE, DEVICE_LSPCLK_FREQ, SPI_PROT_POL0PHA0,
				  SPI_MODE_SLAVE, 1250000, 	16);
	SPI_enableFIFO(SPIB_RAM_BASE);
	SPI_disableLoopback(SPIB_RAM_BASE);
	SPI_setEmulationMode(SPIB_RAM_BASE, SPI_EMULATION_STOP_MIDWAY);
	SPI_enableModule(SPIB_RAM_BASE);
}
