/*****************************************************************************
 * CM4Uart.c
 *****************************************************************************/
#include <stdio.h>
#include <string.h>
#include "cm.h"
#include "driverlib_cm.h"
#include "CM4Uart.h"

/* FIFO RX/TX MAX 16Byte. Rx 1data 8+4bit. Tx 1data 8+4bit.  Max 8char... */

/* define-------------------------- */
#define INPUT_CHAR_MAX (MSG_MAX_BUFF - 1)     // char255 + '\0'
#define NEWLINE CM4Uart_sendMsgSet( titleBuff ); // NewLine

/* Global Variable----------------- */
/* Variable------------------------ */
static char     rBuff[MSG_MAX_BUFF];  // readBuff
static char     eBuff[MSG_MAX_BUFF];  // enterBuff
static uint16_t rLen = 0;    // readLen
static uint16_t eLen = 0;    // enterLen
static char     titleBuff[36];

/* Prototype----------------------- */
static uint16_t RxMain(void);
static uint16_t EnterMain(void);

/* RAM Section Functions ---------- */
//#pragma CODE_SECTION(CM4Uart_init, ".TI.ramfunc");
//#pragma CODE_SECTION(CM4Uart_Exec, ".TI.ramfunc");
//#pragma CODE_SECTION(CM4Uart_sendMsgSet, ".TI.ramfunc");
//#pragma CODE_SECTION(RxMain, ".TI.ramfunc");
//#pragma CODE_SECTION(EnterMain, ".TI.ramfunc");


/* *** A RX interrupt is issued when there are 2 more bytes in the FIFO buffer. *** */
/* *** 1-byte interrupt is not allowed.                                         *** */
//#define UART_RXISR_ENABLE
#ifdef UART_RXISR_ENABLE
static uint16_t m_UartIsrFlg = 0;
__interrupt void UART_RX_IntHandler(void);
#endif // UART_RXISR_ENABLE


/* CM4Uart_init() */
void CM4Uart_init( void )
{
    // Configure UART0 for echoback.Set up to transfer data at 19200 baud.
    UART_setConfig(UART0_BASE, UART_CLK_FREQ, 19200, (UART_CONFIG_WLEN_8 | UART_CONFIG_STOP_ONE | UART_CONFIG_PAR_NONE));
    // Put a character to show start of example.  This will display on the terminal.
    CM4Uart_title( NULL ); // ">"
    NEWLINE;
#ifdef UART_RXISR_ENABLE
    // Enable the UART0 interrupt on the processor (NVIC).
    UART_registerInterrupt(INT_UART0, UART_RX_IntHandler);
#endif // UART_RXISR_ENABLE
    // FIFO enable
    UART_enableFIFO(UART0_BASE);
    // FIFO interrupt levels are set to generate an interrupt
    // when the TX FIFO is less than or equal to 7/8 empty and the
    // RX FIFO is greater than or equal to 1/8 full.
    UART_setFIFOLevel(UART0_BASE, UART_FIFO_TX1_8, UART_FIFO_RX1_8);
#ifdef UART_RXISR_ENABLE
    // FIFO receive interrupt configuration
    UART_clearInterruptStatus(UART0_BASE, UART_INT_RX | UART_INT_RT);
    UART_enableInterrupt(UART0_BASE, UART_INT_RX);
#endif // UART_RXISR_ENABLE
}

/* CM4Uart_Exec() */
void CM4Uart_Exec( void )
{
    /* 1cycle 1exe. */
    if ( 0 == EnterMain() ) { // Enter CMD
        RxMain();             // RX
    }
}

/* CM4Uart_title() */
void CM4Uart_title( const char* c )
{
    if ( c != NULL ) {
        snprintf( titleBuff, 35, "\r\n%s> ", c );
    }
    else {
        snprintf( titleBuff, 35, "\r\n> " );
    }
}

/*************************************************
 * debug shell implement functions
*************************************************/
void wxdbg_printf(char *format, ...)
{
    va_list argp;
    char msg[MSG_MAX_BUFF];

    va_start(argp, format);
    vsnprintf(msg, MSG_MAX_BUFF, format, argp);
    va_end(argp);

    CM4Uart_sendMsgSet( msg );
    return;
}

void wxdbg_shell_printf(char *format, ...)
{
    va_list argp;
    char msg[MSG_MAX_BUFF];

    va_start(argp, format);
    vsnprintf(msg, MSG_MAX_BUFF, format, argp);
    va_end(argp);

    CM4Uart_sendMsgSet( msg );
    return;
}

/* CM4Uart_sendMsgSet() */
void CM4Uart_sendMsgSet( char* msg )
{
    uint16_t temp = strlen( (const char*)msg );
    uint16_t i = 0;
    for ( ; i < temp; i++ ) {
        while (1){
            if( true == UART_writeCharNonBlocking(UART0_BASE, msg[i]) ) break;
        }
    }
}

/* RxMain() */
static uint16_t RxMain(void)
{
    uint16_t ret = 0;
    static char prev_char = 0x00;

#ifdef UART_RXISR_ENABLE
    if ( m_UartIsrFlg == 0 ) return 0;
    m_UartIsrFlg = 0;

    // Loop while there are characters in the receive FIFO.
    while ( UART_isDataAvailable(UART0_BASE) ) {
#else // UART_RXISR_ENABLE
    /* 1Cycle, Get 1Char. */
    if ( UART_isDataAvailable(UART0_BASE) != 0 ) {
#endif // UART_RXISR_ENABLE
        ret = 1;
        char temp = UART_readCharNonBlocking(UART0_BASE);
        if ( temp == 0x08 ) { // BS
            if ( rLen == 0 ) {/* none */;}
            else {
                rLen--;
                rBuff[rLen] = '\0';
                UART_writeCharNonBlocking(UART0_BASE, 0x08); // BS,  PC term output
                UART_writeCharNonBlocking(UART0_BASE, 0x20); // SPC, PC term output
                UART_writeCharNonBlocking(UART0_BASE, 0x08); // BS,  PC term output
            }
        }
        else if ( temp == '\n' || temp == '\r' ) { // Enter
            if ( prev_char == '\r' && temp == '\n' ) {/* none */;} // check "\r\n"
            else if ( rLen == 0 ) {NEWLINE;} // char blank
            else { // with input.
                rBuff[rLen] = '\0';
                rLen++;
                memcpy( eBuff, rBuff, rLen );
                eLen = rLen;
//                memset( rBuff, '\0', rLen );
                rLen = 0;
            }
        }
        else { // other key
            /* ASCII General Characters. */
            if ( temp >= 0x20 && temp <= 0x7E ) { /* ' ' - '~' */
                if ( (rLen+1) > INPUT_CHAR_MAX ) { /* none */;} /* buff over */
                else {
                    rBuff[rLen] = temp;
                    UART_writeCharNonBlocking(UART0_BASE, rBuff[rLen]); // PC term output
                    rLen++;
                }
            }
        }
        prev_char = temp;
    }
    return ret;
}

/* EnterMain() */
static uint16_t EnterMain(void)
{
    uint16_t i, ret, len1, len2;
    char cmd[256];

    int test1 = 0;

    if ( eLen != 0 ) {
        i = 0;
        test1 = 0;

        /* loop Start. */
        while ( debug_shell_addin[i].cmd_func != NULL ) {
            sscanf( eBuff, "%s", cmd );
            len1 = strlen( cmd );
            len2 = strlen( debug_shell_addin[i].cmd );
            ret = memcmp( debug_shell_addin[i].cmd, cmd, (len1 > len2 ? len1 : len2) );
            if ( ret == 0 ) {
                CM4Uart_sendMsgSet("\r\n"); // NewLine ">" ignore
                /* Uart Cmd Exec. */
                debug_shell_addin[i].cmd_func( eBuff, test1 );
                /* Compleate. */
//                memset( eBuff, '\0', eLen );
                eLen = 0;
                NEWLINE; // last
                return 1;
            }
            i++;
        }
        /* Loop End. */

        /* Cmd None. */
        memset( eBuff, '\0', eLen );
        eLen = 0;
        NEWLINE; // last
        return 1;
    }
    return 0;
}

#ifdef UART_RXISR_ENABLE
__interrupt void UART_RX_IntHandler(void)
{
    uint32_t ui32Status;
    /* Get the interrupt status. */
    ui32Status = UART_getInterruptStatus(UART0_BASE, UART_RAW_INT);
    /* Clear the asserted interrupts. */
    UART_clearInterruptStatus(UART0_BASE, ui32Status);

    m_UartIsrFlg = 1;
}
#endif // UART_RXISR_ENABLE
