/* wxemac_sharc.c */
#include <stdio.h>
#include <string.h>
#include "cm.h"
#include "driverlib_cm.h"
#include <wxemac.h>

/* BUF_RX(20) + BUFF_TX (2) = memory 0x0A490. memory all = 0x18000.  WxEmac = 42% memory used. */
#define WXEMAC_DMALINE_TXRX  (ETHERNET_MAX_NUM_DMA_CHANNELS)           // DMA RX/TX のライン数 1-2 // default 2
#define WXEMAC_BUFFER_RX     (ETHERNET_DESCRIPTORS_NUM_RX_PER_CHANNEL) // Receive Buffer
#define WXEMAC_BUFFER_TX     (ETHERNET_DESCRIPTORS_NUM_TX_PER_CHANNEL) // Transmit Buffer
#define WXEMAC_BUFFER_RX_MAX (ETHERNET_DESCRIPTORS_NUM_RX)             // (DMALINE_TXRX * WXEMAC_BUFFER_RX) + SURPLUS_DESCRIPTORS
#define WXEMAC_BUFFER_TX_MAX (ETHERNET_DESCRIPTORS_NUM_TX)             // (DMALINE_TXRX * WXEMAC_BUFFER_TX)
#define WXEMAC_BUFFER_RX_SURPLUS (WXEMAC_BUFFER_RX_MAX - (WXEMAC_DMALINE_TXRX * WXEMAC_BUFFER_RX)) // SURPLUS_DESCRIPTORS

#define WXEMAC_PACKET_FULL  (WXEMAC_FRAME_LENGTH + 4)

/* Variable------------------------ */
static WXEMAC_LINK m_preEtherLinkStatus = WXEMAC_LINK_DOWN;                // pre Link Status
static uint16_t m_RxNextDescCnt = 0;                                       // Rx Init　assigned Descripter counter
static uint16_t m_RxSurplusDescCnt = 0;                                    // Rx Surplus Descripter counter
static Ethernet_Pkt_Desc* m_RxSurplusDescArray[WXEMAC_BUFFER_RX_SURPLUS];  // Rx Surplus Descripter Address Array
static bool bInit = false; // Initialize flag
static uint16_t m_TxFrameCnt = 0;

/* Prototype----------------------- */
Ethernet_Pkt_Desc* CM4Emac_CB_pfcbRxPacket( Ethernet_Handle handleApplication, Ethernet_Pkt_Desc *pPacket );
Ethernet_Pkt_Desc* CM4Emac_CB_pfcbGetPacket( void );
void CM4Emac_CB_pfcbFreePacket( Ethernet_Handle handleApplication, Ethernet_Pkt_Desc *pPacket );
uint16_t PHY_MdioRegRead( uint16_t regAdr );
uint16_t PHY_MdioRegWrite( uint16_t regAdr, uint16_t regDat );

typedef struct _wxemac_device {
    WXEMAC_INFO                 info;
    wxemac_cbfunc               cbfunc;
    void                        *cbparam;

    // TI
    Ethernet_Handle             etHandel;
    Ethernet_MACAddrConfig      emacMACaddrCfg;

    // Buffer
    WXEMAC_FRAME                *rxframe;
    Ethernet_Pkt_Desc           *rxbuffer;

    volatile Ethernet_Pkt_Desc  *rxhead;
    volatile Ethernet_Pkt_Desc  *rxtail;
} WXEMAC_DEVICE;

enum {
    WXEMAC_FRAME_USE_RX0 = 0,
    WXEMAC_FRAME_USE_RX1 = 1,
    WXEMAC_FRAME_USE_TX = 2
};

typedef struct _wxemac_frame_driver {
    void                *reserved[13];
    uint32_t            use;            // 0:RX0, 1:RX1, 2:TX
    Ethernet_Pkt_Desc   *PktDesc;
} WXEMAC_FRAME_DRIVER;
typedef union _wxframe {
    uint8_t     buffer[WXEMAC_FRAME_LENGTH];
    uint16_t    data[2];
} WXFRAME;
#define WXEMAC_FRAME_DLEN(frame)    ((WXFRAME *)frame)->data[1]

// ethernet frame
//#pragma LOCATION( WxEmac_FBaseRx, 0x20008000 ) // mmap, Ethernet Rx buffer
static WXEMAC_FRAME      WxEmac_FBaseRx[WXEMAC_BUFFER_RX_MAX] __attribute__ ((aligned (64)));       // frame for rx
static Ethernet_Pkt_Desc WxEmac_BBaseRx[WXEMAC_BUFFER_RX_MAX] __attribute__ ((aligned (64)));       // buffer for rx
static WXEMAC_FRAME      WxEmac_FBaseTx[WXEMAC_BUFFER_TX_MAX] __attribute__ ((aligned (64)));       // frame for tx
static Ethernet_Pkt_Desc WxEmac_BBaseTx[WXEMAC_BUFFER_TX_MAX] __attribute__ ((aligned (64)));       // buffer for tx
static volatile Ethernet_Pkt_Desc *WxEmac_BFreeHead, *WxEmac_BFreeTail;

// WxEmac Devices
static WXEMAC_DEVICE WxEmac_Device = {
    {WXEMAC_DEV_0, 0, ""}, NULL, NULL, NULL, {0},
    WxEmac_FBaseRx, WxEmac_BBaseRx,
    NULL, NULL
};

static WXEMAC_STATISTIC statistics0[] = {
     { 0," txOctetCountGoodBad            ", 0 },
     { 0," txPacketCountGoodBad           ", 0 },
     { 0," txBroadcastPacketsGood         ", 0 },
     { 0," txMulticastPacketsGood         ", 0 },
     { 0," tx64OctetsPacketsGood          ", 0 },
     { 0," tx65to127OctetsPacketsGood     ", 0 },
     { 0," tx128to255OctetsPacketsGood    ", 0 },
     { 0," tx256to511OctetsPacketsGood    ", 0 },
     { 0," tx512to1023OctetsPacketsGood   ", 0 },
     { 0," tx1024toMaxOctetsPacketsGood   ", 0 },
     { 0," txUnicastPacketsGoodBad        ", 0 },
     { 0," txMulticastPacketsGoodBad      ", 0 },
     { 0," txBroadcastPacketsGoodBad      ", 0 },
     { 0," txUnderFlowErrorPackets        ", 0 },
     { 0," txSingleCollisionGoodPackets   ", 0 },
     { 0," txMultipleCollisionGoodPackets ", 0 },
     { 0," txDeferredPackets              ", 0 },
     { 0," txLateCollisionPackets         ", 0 },
     { 0," txExcessiveCollisionPackets    ", 0 },
     { 0," txCarrierErrorPackets          ", 0 },
     { 0," txOctetCountGood               ", 0 },
     { 0," txPacketCountGood              ", 0 },
     { 0," txExcessiveDeferralError       ", 0 },
     { 0," txPausePackets                 ", 0 },
     { 0," txVLANPacketsGood              ", 0 },
     { 0," txOSizePacketsGood             ", 0 },
     { 0," reserved_1", 0 },
     { 0," rxPacketsCountGoodBad          ", 0 },
     { 0," rxOctetCountGoodBad            ", 0 },
     { 0," rxOctetCountGood               ", 0 },
     { 0," rxBroadcastPacketsGood         ", 0 },
     { 0," rxMulticastPacketsGood         ", 0 },
     { 0," rxCRCErrorPackets              ", 0 },
     { 0," rxAlignmentErrorPackets        ", 0 },
     { 0," rxRuntErrorPackets             ", 0 },
     { 0," rxJabberErrorPackets           ", 0 },
     { 0," rxUnderSizePacketsGood         ", 0 },
     { 0," rxOverSizePacketsGood          ", 0 },
     { 0," rx64OctetsPacketsGoodBad       ", 0 },
     { 0," rx65to127OctetsPacketsGoodBad  ", 0 },
     { 0," rx128to255OctetsPacketsGoodBad ", 0 },
     { 0," rx256to511OctetsPacketsGoodBad ", 0 },
     { 0," rx512to1023OctetsPacketsGoodBad", 0 },
     { 0," rx1024toMaxOctetsPacketsGoodBad", 0 },
     { 0," rxUnicastPacketsGood           ", 0 },
     { 0," rxLengthErrorPackets           ", 0 },
     { 0," rxOutOfRangeTypePackets        ", 0 },
     { 0," rxPausePackets                 ", 0 },
     { 0," rxFIFOOverFlowPackets          ", 0 },
     { 0," rxVLANPacketsGoodBad           ", 0 },
     { 0," rxWatchDogErrorPackets         ", 0 },
     { 0," rxReceiveErrorPackets          ", 0 },
     { 0," rxControlPacketsGood           ", 0 },
     { 0," txLPIUsecCounter", 0 },
     { 0," txLPITranCounter", 0 },
     { 0," rxLPIUsecCounter", 0 },
     { 0," rxLPITranCounter", 0 }
};

/***************************/
/* CallBack--------------- */
/***************************/

/** Receive packet call back   called when a Packet is Received by the LLD　*/
Ethernet_Pkt_Desc* CM4Emac_CB_pfcbRxPacket( Ethernet_Handle handleApplication, Ethernet_Pkt_Desc *pPacket )
{
    WXEMAC_DEVICE *device = &WxEmac_Device;
    Ethernet_Pkt_Desc* nextBuffer;
    Ethernet_Pkt_Desc* pBuffer = (Ethernet_Pkt_Desc*)pPacket;
    Ethernet_Pkt_Desc* pBufferLast = (Ethernet_Pkt_Desc*)pPacket;
    int i;

    /* Count the number of received packets */
    device->info.rxcnt ++;
    while (pBufferLast->nextPacketDesc != NULL) {
        pBufferLast = pBufferLast->nextPacketDesc;
        device->info.rxcnt ++;
    }

    /* Put the received packets to the queue to be processed */
    Interrupt_disableInProcessor(); // Disable global interrupt.
    if (device->rxhead == NULL) {
        device->rxhead = pBuffer;
    }
    device->rxtail = pBufferLast;

    /* set next Rx Descripter. */
    nextBuffer = m_RxSurplusDescArray[0];
    m_RxSurplusDescCnt--; // update surplus counter.
    for ( i = 0; i < m_RxSurplusDescCnt; i++ ) {
        m_RxSurplusDescArray[i] = m_RxSurplusDescArray[i + 1]; // update surplus descripter.
    }
    Interrupt_enableInProcessor(); // Enable global interrupt.

    if (NULL != device->cbfunc) {
        device->cbfunc(WXEMAC_EVENT_RECEIVE_FRAME, device->cbparam);
    }

    return nextBuffer;
}

 /** Get packet call back   called by this driver when it needs a packet　from Application */
Ethernet_Pkt_Desc* CM4Emac_CB_pfcbGetPacket( void )
{
    // At initialization, it is called for the number of "descriptors to be used for RX".　"" Surplus desc not included ""

    uint16_t temp = m_RxNextDescCnt;
    if ( m_RxNextDescCnt < WXEMAC_BUFFER_RX_MAX ) m_RxNextDescCnt++;

    return &WxEmac_BBaseRx[ temp ]; // Ethernet_Pkt_Desc
}

 /** Free packet call back   called after the completion of Tx and　when the buffer needs to handled back to application */
void CM4Emac_CB_pfcbFreePacket( Ethernet_Handle handleApplication, Ethernet_Pkt_Desc *pPacket )
{
    Ethernet_Pkt_Desc *pBuffer = pPacket;
    Ethernet_Pkt_Desc *pBufferLast = pPacket;
    WXEMAC_DEVICE *device = &WxEmac_Device;

    if (device != NULL) {
        /* Count the number of transmitted packets */
        device->info.txcnt ++;
        while (pBufferLast->nextPacketDesc != NULL) {
            pBufferLast = pBufferLast->nextPacketDesc;
            device->info.txcnt ++;
        }

        /* Put the transmitted packets to the Tx Free Buffer List */
        Interrupt_disableInProcessor(); // Disable global interrupt.
        if (WxEmac_BFreeHead == NULL) {
            WxEmac_BFreeHead = pBuffer;
        } else {
            WxEmac_BFreeTail->nextPacketDesc = pBuffer;
        }
        WxEmac_BFreeTail = pBufferLast;
        m_TxFrameCnt++;
        Interrupt_enableInProcessor(); // Enable global interrupt.
    }
}

/***************************/
/* API ------------------- */
/***************************/

int32_t WxEmac_GetStatistic(WXEMAC_DEV dev, WXEMAC_STATISTIC stats[WXEMAC_STATISTIC_MAX])
{
    Ethernet_Statistics statsX;
    int i = 0;

    memset(&statsX, 0x0, sizeof(Ethernet_Statistics));
    Ethernet_getStatistics((Ethernet_Handle)WxEmac_Device.etHandel, &statsX);

    uint32_t* test_stats = (uint32_t*)&statsX;
    for ( i = 0; i < WXEMAC_STATISTIC_MAX; i++ ) {
        statistics0[i].count = *((uint32_t*)test_stats++);
    }
    memcpy(stats, statistics0, sizeof(WXEMAC_STATISTIC)*WXEMAC_STATISTIC_MAX);
    return 0;
}

void WxEmac_UpdateStats(WXEMAC_DEV dev)
{
    // none
}

void show_emacps(void)
{
    // none
}

static void init_rxbuffer(WXEMAC_FRAME *frame, Ethernet_Pkt_Desc *PktDesc, int32_t use)
{
    WXEMAC_FRAME_DRIVER *wfdriver;
    int32_t cnt = WXEMAC_BUFFER_RX_MAX;
    int32_t i;

    /* Clear the buffers */
    memset((char*)frame, 0, sizeof(WXEMAC_FRAME)*cnt);
    memset((char*)PktDesc, 0, sizeof(Ethernet_Pkt_Desc)*cnt);

    for (i = 0; i < cnt; i++) {
        wfdriver = (WXEMAC_FRAME_DRIVER *)frame[i].drivers;
        wfdriver->PktDesc = &PktDesc[i];
        wfdriver->use = use;

        PktDesc[i].bufferLength = WXEMAC_FRAME_LENGTH;
        PktDesc[i].dataOffset = 0;
        PktDesc[i].dataBuffer = (uint8_t *)&frame[i].buffer[0];
        PktDesc[i].nextPacketDesc = 0;
        PktDesc[i].flags = ETHERNET_PKT_FLAG_SOP | ETHERNET_PKT_FLAG_EOP | ETHERNET_PKT_FLAG_SA_INS;
        PktDesc[i].pktChannel = ETHERNET_DMA_CHANNEL_NUM_0;
        PktDesc[i].pktLength = WXEMAC_FRAME_LENGTH;
        PktDesc[i].validLength = WXEMAC_FRAME_LENGTH;
        PktDesc[i].numPktFrags = 1;
    }
}

static void init_txbuffer(void)
{
    WXEMAC_FRAME_DRIVER *wfdriver;
    int32_t cnt = WXEMAC_BUFFER_TX_MAX;
    int32_t i;

    WxEmac_BFreeHead = WxEmac_BFreeTail = NULL;
    /* Clear the buffers */
    memset((char*)WxEmac_BBaseTx, 0, sizeof(WxEmac_BBaseTx));

    for (i = 0; i < cnt; i++) {
        wfdriver = (WXEMAC_FRAME_DRIVER *)WxEmac_FBaseTx[i].drivers;
        wfdriver->PktDesc = &WxEmac_BBaseTx[i];
        wfdriver->use = WXEMAC_FRAME_USE_TX;

        WxEmac_BBaseTx[i].bufferLength = WXEMAC_FRAME_LENGTH;
        WxEmac_BBaseTx[i].dataOffset = 0;
        WxEmac_BBaseTx[i].dataBuffer = (uint8_t *)&WxEmac_FBaseTx[i].buffer[0];
        WxEmac_BBaseTx[i].nextPacketDesc = 0;
        WxEmac_BBaseTx[i].flags = ETHERNET_PKT_FLAG_SOP | ETHERNET_PKT_FLAG_EOP | ETHERNET_PKT_FLAG_SA_INS;
        WxEmac_BBaseTx[i].pktChannel = ETHERNET_DMA_CHANNEL_NUM_0;
        WxEmac_BBaseTx[i].pktLength = WXEMAC_FRAME_LENGTH;
        WxEmac_BBaseTx[i].validLength = WXEMAC_FRAME_LENGTH;
        WxEmac_BBaseTx[i].numPktFrags = 1;

        WxEmac_FreeFrame(&WxEmac_FBaseTx[i]);
        m_TxFrameCnt++;
    }
}

static void wxemac_system_init(void)
{
    // none
}

static WXEMAC_ERROR wxemac_check_handle(WXEMAC_HANDLE handle)
{
    WXEMAC_DEVICE *device = (WXEMAC_DEVICE *)handle;

    if (device != &WxEmac_Device) {
        return (WXEMAC_ERROR_HANDLE);
    }

    if (WXEMAC_STATUS_ACTIVE != (device->info.status & WXEMAC_STATUS_ACTIVE)) {
        return (WXEMAC_ERROR_HANDLE);
    }

    return (WXEMAC_ERROR_SUCCESS);
}


/***************************************************
 * comment    : A thing to check the Ethernet status every cycle.
 *              The TMS320F283888D cannot use "GPIO interrupts not assigned to peripherals" for CM cores.
 *              It is necessary to check the link status every cycle or to determine the GPIO interr_
 *              upt in the C28 core and interr_upt the CM core using IPC to make a decision.
 ***************************************************/
void WxEmac_Poll(void)
{
    WXEMAC_DEVICE *device = &WxEmac_Device;
    WXEMAC_LINK nowLinkStatus;
    uint16_t mdioRead_0, mdioRead_1, mdioRead_10;

    if ( device->etHandel == NULL ) {
        m_preEtherLinkStatus = WXEMAC_LINK_DOWN;
        return;
    }

    /* PHY Link Check. */
    mdioRead_0  = PHY_MdioRegRead( 0x0000 ); // BMCR
    mdioRead_1  = PHY_MdioRegRead( 0x0001 ); // BMSR
    if ( (mdioRead_1 & 0x4) == 0 ) nowLinkStatus = WXEMAC_LINK_DOWN;
    else { // LINK UP
        if ( (mdioRead_0 & 0x1000) != 0 ) {
            /* Auto-nego Enable. check Auto-nego Compleate. */
            if ( (mdioRead_1 & 0x20) == 0 ) nowLinkStatus = WXEMAC_LINK_DOWN;
            else                            nowLinkStatus = WXEMAC_LINK_UP;
        }
        else { // Auto-nego Disable.
            nowLinkStatus = WXEMAC_LINK_UP;
        }
    }

    /* Status change. */
    if ( nowLinkStatus != m_preEtherLinkStatus ) {
        if ( nowLinkStatus == WXEMAC_LINK_UP ) {
            if ( (mdioRead_0 & 0x1000) != 0 ) { // auto-nego Enable.
                mdioRead_10 = PHY_MdioRegRead( 0x0010 ); // PHYSTS
                if ( ((mdioRead_10 & 0x2) != 0) && ((mdioRead_10 & 0x4) == 0) )      device->info.status |= WXEMAC_STATUS_SPEED_10;
                else if ( ((mdioRead_10 & 0x2) != 0) && ((mdioRead_10 & 0x4) != 0) ) device->info.status |= WXEMAC_STATUS_SPEED_10 | WXEMAC_STATUS_FULL;
                else if ( ((mdioRead_10 & 0x2) == 0) && ((mdioRead_10 & 0x4) == 0) ) device->info.status |= WXEMAC_STATUS_SPEED_100;
                else if ( ((mdioRead_10 & 0x2) == 0) && ((mdioRead_10 & 0x4) != 0) ) device->info.status |= WXEMAC_STATUS_SPEED_100 | WXEMAC_STATUS_FULL;
                /* MAC Configuration Settings. ----------------------*/
                /* linkSpeed ( MAC_Configuration Register "FES" ) */
                if ( (mdioRead_10 & 0x2) != 0 ) Ethernet_clearMACConfiguration( EMAC_BASE, ((uint32_t)1 << 14) ); // EMAC 10Mbps
                else                            Ethernet_setMACConfiguration( EMAC_BASE, ((uint32_t)1 << 14) );   // EMAC 100Mbps
                /* linkMode ( MAC_Configuration Register "DM" ) */
                if ( (mdioRead_10 & 0x4) != 0 ) Ethernet_setMACConfiguration( EMAC_BASE, ((uint32_t)1 << 13) );   // EMAC Full Duplex
                else                            Ethernet_clearMACConfiguration( EMAC_BASE, ((uint32_t)1 << 13) ); // EMAC Half Duplex
            }
            device->info.status |= WXEMAC_STATUS_LINKUP;
            if ( NULL != device->cbfunc ) {
                device->cbfunc( WXEMAC_EVENT_LINK_UP, device->cbparam );
            }
        }
        else { // nowLinkStatus == WXEMAC_LINK_DOWN
            device->info.status &= WXEMAC_STATUS_ACTIVE;
            if ( NULL != device->cbfunc ) {
                device->cbfunc( WXEMAC_EVENT_LINK_DOWN, device->cbparam );
            }
        }
    }

    m_preEtherLinkStatus = nowLinkStatus; // status update
}

WXEMAC_ERROR WxEmac_GetInfo(WXEMAC_DEV dev, WXEMAC_INFO *pinfo)
{
    WXEMAC_DEVICE *device = &WxEmac_Device;
    uint16_t mdioRead_0, mdioRead_1;

    if (NULL == pinfo) {
        return (WXEMAC_ERROR_ARGS);
    }
    if ( device->etHandel == NULL ) {
        return (WXEMAC_ERROR_HANDLE);
    }


    /* PHY Link Check. */
    mdioRead_0 = PHY_MdioRegRead( 0x0000 );
    mdioRead_1 = PHY_MdioRegRead( 0x0001 );
    if ( (mdioRead_1 & 0x4) == 0 ) device->info.status &= ~WXEMAC_STATUS_LINKUP;
    else { // LINK UP
        if ( (mdioRead_0 & 0x1000) != 0 ) {
            /* Auto-nego Enable. check Auto-nego Compleate. */
            if ( (mdioRead_1 & 0x20) != 0 ) device->info.status |= WXEMAC_STATUS_LINKUP;
            else                            device->info.status &= ~WXEMAC_STATUS_LINKUP;
        }
        else {
            /* Auto-nego Disable. */
            device->info.status |= WXEMAC_STATUS_LINKUP;
        }
    }
    memcpy(pinfo, &device->info, sizeof(WXEMAC_INFO));

    return (WXEMAC_ERROR_SUCCESS);
}

bool WxEmac_TxFrameChk(void)
{
    if ( m_TxFrameCnt ) return true;
    else                return false;
}

WXEMAC_FRAME *WxEmac_AllocFrame(void)
{
    volatile Ethernet_Pkt_Desc *buffer = NULL;
    WXEMAC_FRAME *frame = NULL;

    /* Get one Tx buffer from the Tx free buffer list */
    Interrupt_disableInProcessor(); // Disable global interrupt.
    if (NULL != WxEmac_BFreeHead) {
        buffer = WxEmac_BFreeHead;
        WxEmac_BFreeHead = buffer->nextPacketDesc;
        if (NULL == WxEmac_BFreeHead) {
            WxEmac_BFreeTail = NULL;
        }
        buffer->nextPacketDesc = NULL;
        frame = (WXEMAC_FRAME *)(buffer->dataBuffer);
        m_TxFrameCnt--;
    }
    Interrupt_enableInProcessor(); // Enable global interrupt.

    return (frame);
}

WXEMAC_ERROR WxEmac_FreeFrame(WXEMAC_FRAME *pframe)
{
    WXEMAC_FRAME_DRIVER *wfdriver;

    int use;

    if (NULL == pframe) {
        return WXEMAC_ERROR_ARGS;
    }

    wfdriver = (WXEMAC_FRAME_DRIVER *)pframe->drivers;
    wfdriver->PktDesc->nextPacketDesc = 0;
    use = wfdriver->use;

    memset((char *)pframe, 0, WXEMAC_PACKET_FULL);

    switch (use) {
    case WXEMAC_FRAME_USE_RX0:
        /* Add Surplus. */
        m_RxSurplusDescArray[ m_RxSurplusDescCnt ] = wfdriver->PktDesc; // After completion of received data processing, the descriptor is put into surplus.
        m_RxSurplusDescCnt++;
        break;
    case WXEMAC_FRAME_USE_TX:
        Interrupt_disableInProcessor(); // Disable global interrupt.
        if (NULL == WxEmac_BFreeHead) {    // WxEmac_BFreeHead,WxEmac_BFreeTail
            WxEmac_BFreeHead = wfdriver->PktDesc;
            WxEmac_BFreeTail = wfdriver->PktDesc;
        } else {
            WxEmac_BFreeTail->nextPacketDesc = wfdriver->PktDesc;
            WxEmac_BFreeTail = wfdriver->PktDesc;
        }
        Interrupt_enableInProcessor(); // Enable global interrupt.
        break;
    default:
        break;
    }

    return WXEMAC_ERROR_SUCCESS;
}

WXEMAC_ERROR WxEmac_Read(WXEMAC_HANDLE handle, WXEMAC_FRAME **pframe)
{
    WXEMAC_DEVICE *device = (WXEMAC_DEVICE *)handle;
    volatile Ethernet_Pkt_Desc *buffer = NULL;
    WXEMAC_ERROR ret;

    ret = wxemac_check_handle(handle);
    if (WXEMAC_ERROR_SUCCESS != ret) {
        return (ret);
    }
    if (NULL == pframe) {
        return (WXEMAC_ERROR_ARGS);
    }

    *pframe = NULL;

    /* Get all the current Rx buffers that need to be processed */
    Interrupt_disableInProcessor(); // Disable global interrupt.
    if (NULL == device->rxhead) {
        Interrupt_enableInProcessor(); // Enable global interrupt.
        return (WXEMAC_ERROR_SUCCESS);
    }
    buffer = device->rxhead;
    device->rxhead = buffer->nextPacketDesc;
    if (device->rxtail == buffer) {
        device->rxtail = NULL;
    }
    Interrupt_enableInProcessor(); // Enable global interrupt.

    buffer->nextPacketDesc = NULL;
    (*pframe) = (WXEMAC_FRAME *)(buffer->dataBuffer);
    (*pframe)->len = WXEMAC_FRAME_DLEN((*pframe)) - 2 - 4;  // del "length field" size and FCS.

    return (WXEMAC_ERROR_SUCCESS);
}

WXEMAC_ERROR WxEmac_Write(WXEMAC_HANDLE handle, WXEMAC_FRAME *pframe)
{
    WXEMAC_DEVICE *device = (WXEMAC_DEVICE *)handle;
    WXEMAC_FRAME_DRIVER *wfdriver;
    WXEMAC_ERROR ret;
    uint16_t elen;

    ret = wxemac_check_handle(handle);
    if (WXEMAC_ERROR_SUCCESS != ret) {
        return (ret);
    }
    if (NULL == pframe) {
        return (WXEMAC_ERROR_ARGS);
    }

    /* Submit the buffer to the ethernet driver to be transmitted */
    if (pframe->len < 60) {
        uint8_t *padhead = WXEMAC_FRAME_HEAD(pframe) + pframe->len;
        memset(padhead, 0, 60-pframe->len);
        elen = 60;
    } else if (pframe->len > 1514) {
        elen = 1514;
    } else {
        elen = pframe->len;
    }

    wfdriver = (WXEMAC_FRAME_DRIVER *)pframe->drivers;
    wfdriver->PktDesc->validLength = elen;  //pframe->len; // add "length field" size
    wfdriver->PktDesc->pktLength = elen;    //pframe->len; // add "length field" size
    wfdriver->PktDesc->bufferLength = elen; //pframe->len; // add "length field" size
//    WXEMAC_FRAME_DLEN(pframe) = elen;       //pframe->len;
    wfdriver->PktDesc->nextPacketDesc = NULL;

    //Send the packet prepared
    uint32_t retEth = Ethernet_sendPacket( device->etHandel, wfdriver->PktDesc );
    if ( retEth != 0 ) {
        return WXEMAC_ERROR_SYSTEM;
    }

    return WXEMAC_ERROR_SUCCESS;
}

WXEMAC_ERROR WxEmac_Open(WXEMAC_DEV dev, WXEMAC_OPEN_PARAM *poparam, WXEMAC_HANDLE *phandle)
{
    WXEMAC_DEVICE *device = &WxEmac_Device;
    uint32_t mac[2] = {0};
    uint16_t mdioRead = 0;

    if (NULL == poparam) {
        return (WXEMAC_ERROR_ARGS);
    }

    *phandle = NULL;

    if ( bInit == false ) {
        return (WXEMAC_ERROR_ALREADY_OPEN);
    }
    if (WXEMAC_STATUS_ACTIVE == (device->info.status & WXEMAC_STATUS_ACTIVE)) {
        return (WXEMAC_ERROR_ALREADY_OPEN);
    }

    /* MAC Address Set */
    memcpy( &device->info.mac_address, poparam->mac_address, 6 );
    memcpy( &device->emacMACaddrCfg.addr, poparam->mac_address, 6 );
    device->emacMACaddrCfg.channelNum = 0; // instanseNUM  0 = ETHERNET_O_MAC_ADDRESS0, 1 = ETHERNET_O_MAC_ADDRESS1, 2... 0-7
    mac[0] = ((uint32_t)device->emacMACaddrCfg.addr[4] <<  8 | device->emacMACaddrCfg.addr[5]);
    mac[1] = ((uint32_t)device->emacMACaddrCfg.addr[0] << 24 | device->emacMACaddrCfg.addr[1] << 16 |
                        device->emacMACaddrCfg.addr[2] <<  8 | device->emacMACaddrCfg.addr[3]);
    Ethernet_setMACAddr( EMAC_BASE,           // Base Address
                         0,                   // InstansNum 0-7
                         mac[0],              // MAC Address HIGH
                         mac[1],              // MAC Address LOW
                         ETHERNET_CHANNEL_0); // CHANNEL0 or CHANNEL1

    /* MAC Configuration Settings. ----------------------*/
    mdioRead = PHY_MdioRegRead( 0x0010 );
    /* linkSpeed ( MAC_Configuration Register "FES" ) */
    if ( (mdioRead & 0x2) != 0 ) Ethernet_clearMACConfiguration( EMAC_BASE, ((uint32_t)1 << 14) ); // EMAC 10Mbps
    else                         Ethernet_setMACConfiguration( EMAC_BASE, ((uint32_t)1 << 14) );   // EMAC 100Mbps
    /* linkMode ( MAC_Configuration Register "DM" ) */
    if ( (mdioRead & 0x4) != 0 ) Ethernet_setMACConfiguration( EMAC_BASE, ((uint32_t)1 << 13) );   // EMAC Full Duplex
    else                         Ethernet_clearMACConfiguration( EMAC_BASE, ((uint32_t)1 << 13) ); // EMAC Half Duplex

    /* Initialize the variables */
    device->cbfunc = poparam->cbfunc;
    device->cbparam = poparam->cbparam;

    /* EMAC Enable TX/RX */
    Ethernet_setMACConfiguration( EMAC_BASE, 0x2 ); // TX Enable
    Ethernet_setMACConfiguration( EMAC_BASE, 0x1 ); // RX Enable

    device->info.status |= WXEMAC_STATUS_ACTIVE;
    *phandle = (WXEMAC_HANDLE)device;
    return (WXEMAC_ERROR_SUCCESS);
}

WXEMAC_ERROR WxEmac_Close(WXEMAC_HANDLE handle)
{
    WXEMAC_DEVICE *device = (WXEMAC_DEVICE *)handle;
    WXEMAC_ERROR ret;

    ret = wxemac_check_handle(handle);
    if (WXEMAC_ERROR_SUCCESS != ret) {
        return (ret);
    }
    /* (caution) Only Disable TX and RX, not device processing */
    Ethernet_clearHandle( device->etHandel );   // Disable MAC Transmission and Reception
    device->info.status = 0;                    // Status

    return (WXEMAC_ERROR_SUCCESS);

}

WXEMAC_ERROR WxEmac_Init(void)
{
    Ethernet_InitInterfaceConfig initInterfaceConfig;
    Ethernet_InitConfig *pInitCfg;
    WXEMAC_DEVICE *device;
    uint16_t mdioRead = 0;
    int i;

    if (true == bInit) {
        return (WXEMAC_ERROR_SUCCESS);
    }

    /* Initialize the system */
    wxemac_system_init();

    /* Prepare the buffers */
    device = &WxEmac_Device;
    init_rxbuffer(device->rxframe, device->rxbuffer, 0);
    init_txbuffer();

    /* Reduced Media-Independent Interface. 50Mhz Clk. 100Mbps. ( 100Mbps = 50Mhz * 2Databus )  */

    /* Select the MII interface of the module */
    initInterfaceConfig.ssbase = EMAC_SS_BASE;
    initInterfaceConfig.enet_base = EMAC_BASE;
    initInterfaceConfig.phyMode = ETHERNET_SS_PHY_INTF_SEL_RMII;
#ifdef ETHER_EMAC_CLK_INTERNAL
    initInterfaceConfig.clockSel = ETHERNET_SS_CLK_SRC_INTERNAL; // PHY : RMII Master mode. PHY output 50Mhz clock. EMAC input.
#else  // ETHER_EMAC_CLK
    initInterfaceConfig.clockSel = ETHERNET_SS_CLK_SRC_EXTERNAL; // PHY : RMII Master mode. PHY output 50Mhz clock. EMAC input.
#endif // ETHER_EMAC_CLK
//    initInterfaceConfig.clockSel = ETHERNET_SS_CLK_SRC_INTERNAL;
    /* Assign SoC specific functions for Enabling,Disabling interrupts
     * and for enabling the Peripheral at system level */
    initInterfaceConfig.ptrPlatformInterruptDisable = &Platform_disableInterrupt;
    initInterfaceConfig.ptrPlatformInterruptEnable = &Platform_enableInterrupt;
    initInterfaceConfig.ptrPlatformPeripheralEnable = &Platform_enablePeripheral;
    initInterfaceConfig.ptrPlatformPeripheralReset = &Platform_resetPeripheral;
    /* Assign the peripheral number at the SoC */
    initInterfaceConfig.peripheralNum = SYSCTL_PERIPH_CLK_ENET;
    /* Assign the default SoC specific interrupt numbers of Ethernet interrupts */
    initInterfaceConfig.interruptNum[0] = INT_EMAC;
    initInterfaceConfig.interruptNum[1] = INT_EMAC_TX0;
    initInterfaceConfig.interruptNum[2] = INT_EMAC_TX1;
    initInterfaceConfig.interruptNum[3] = INT_EMAC_RX0;
    initInterfaceConfig.interruptNum[4] = INT_EMAC_RX1;
    pInitCfg = Ethernet_initInterface(initInterfaceConfig);

    /* Get an initial configuration of known good parameters */
    Ethernet_getInitConfig(pInitCfg);

    /* Max Packet Size */
    pInitCfg->pktMTU = WXEMAC_FRAME_LENGTH;
    pInitCfg->numChannels = WXEMAC_DMALINE_TXRX;
    /*　Init of TX/RX DMA Channels　*/
    for ( i = 0U; i < pInitCfg->numChannels; i++ ) {
        /* WXEMAC_BUFFER_TX number of TX descriptors are required for each WXEMAC_DMALINE_TXRX */
        pInitCfg->chInfo[ETHERNET_CH_DIR_TX][i].numBD = WXEMAC_BUFFER_TX; // ETHERNET_DESCRIPTORS_NUM_TX_PER_CHANNEL;
        pInitCfg->chInfo[ETHERNET_CH_DIR_TX][i].dmaQueueSize = ETHERNET_MTL_Q_OP_MODE_QSIZE_4096;
        pInitCfg->chInfo[ETHERNET_CH_DIR_TX][i].chNum = i;
        /* Each WXEMAC_DMALINE_TXRX requires as many RX descriptors as WXEMAC_BUFFER_RX + α */
        pInitCfg->chInfo[ETHERNET_CH_DIR_RX][i].numBD = WXEMAC_BUFFER_RX; // ETHERNET_DESCRIPTORS_NUM_RX_PER_CHANNEL;
        pInitCfg->chInfo[ETHERNET_CH_DIR_RX][i].dmaBufferSize = 384; // dmaBuffSize <<= 2 ; // RX Only
        pInitCfg->chInfo[ETHERNET_CH_DIR_RX][i].dmaQueueSize = ETHERNET_MTL_Q_OP_MODE_QSIZE_4096;
        pInitCfg->chInfo[ETHERNET_CH_DIR_RX][i].chNum = i;
    }

    /* EMAC Subsystem Wrapper Settings */
    pInitCfg->emacSSConfig.phyIntfSel = ETHERNET_SS_CTRLSTS_PHY_INTF_SEL_RMII;
#ifdef ETHER_EMAC_CLK_INTERNAL
    pInitCfg->emacSSConfig.clkSrcSel  = ETHERNET_SS_CTRLSTS_CLK_SRC_SEL_INTERNAL;
#else  // ETHER_EMAC_CLK
    pInitCfg->emacSSConfig.clkSrcSel  = ETHERNET_SS_CTRLSTS_CLK_SRC_SEL_EXTERNAL;
#endif // ETHER_EMAC_CLK
    pInitCfg->emacSSConfig.LoopBackModeClkSel = ETHERNET_SS_CTRLSTS_LMCLKSEL_NORMAL;
    pInitCfg->emacSSConfig.flowControlEn = ETHERNET_SS_CTRLSTS_FLOW_CTRL_EN_DISABLED;

    /* MAC Loopback Settings */
#ifdef LOOPBACK_MODE_EMAC
    pInitCfg->loopbackMode = ETHERNET_MAC_CONFIGURATION_LM_LOOPBACK_ENABLED;
#else // LOOPBACK_MODE_EMAC
    pInitCfg->loopbackMode = ETHERNET_MAC_CONFIGURATION_LM_LOOPBACK_DISABLED;
#endif // LOOPBACK_MODE_EMAC

    /* Assign number of Packets for Buffer Allocation */
    pInitCfg->maxPacketLengthRxBuffer = WXEMAC_FRAME_LENGTH;
    /* Max Packet Size */
    pInitCfg->pktMTU = WXEMAC_FRAME_LENGTH; // Packet Maximum Transfer Unit

    /* DMA Mode */
    pInitCfg->dmaMode.TxRxArbitration  = ETHERNET_DMA_OPERATION_MODE_DA_ROUND_ROBIN; // DA    //
    pInitCfg->dmaMode.TransmitPriority = 0;                                          // TXPR  // Priority Rx
    pInitCfg->dmaMode.PriorityRatio    = ETHERNET_DMA_OPERATION_MODE_PR_TX4_RX1;     // PR    // Rx4:Tx1

    /**/
    pInitCfg->linkMode  = ETHERNET_MAC_CONFIGURATION_DM_FULL_DUPLEX;
    pInitCfg->linkSpeed = 1; // 100Mbps


    /* set Callback. ---------------------- */

    /** Get packet call back  called by this driver when it needs a packet from Application */
    pInitCfg->pfcbGetPacket = &CM4Emac_CB_pfcbGetPacket;   // Assign the callbacks for Getting packet buffer when needed

    /** Receive packet call back  called when a Packet is Received by the LLD */
    pInitCfg->pfcbRxPacket = &CM4Emac_CB_pfcbRxPacket;     // Receive packet callback on Receive packet completion interrupt

    /** Free packet call back called after the completion of Tx and when the buffer needs to handled back to application */
    pInitCfg->pfcbFreePacket = &CM4Emac_CB_pfcbFreePacket; // Releasing the TxPacketBuffer on Transmit interrupt callbacks

    /** Clear enqueued packets call back called during RBU error to drop all enqueued RX packets*/
    pInitCfg->pfcbDeletePackets = NULL; // &CM4Emac_CB_pfcbDeletePackets;
    /** Callback for Asynchronous EEE Events Magic Packet Received or Remote Wakeup Packet Received */
    pInitCfg->pfcbLPIPacketReceived = NULL; // &CM4Emac_CB_pfcbLPIPacketReceived;

    /* get Handle. ---------------------- */

    /* The Application handle is not used by this application Hence using a dummy value of 1 */
    i = Ethernet_getHandle((Ethernet_Handle)1, pInitCfg, &device->etHandel);
    if ( ETHERNET_RET_SUCCESS != i ) {
        m_RxNextDescCnt = 0;
        return WXEMAC_ERROR_SYSTEM;
    }

    /* EMAC Disable TX/RX */
    Ethernet_clearMACConfiguration( EMAC_BASE, 0x2 ); // TX Disable
    Ethernet_clearMACConfiguration( EMAC_BASE, 0x1 ); // RX Disable

    /* Set Surplus Receive Buffer */
    if ( (WXEMAC_BUFFER_RX_MAX - m_RxNextDescCnt) != WXEMAC_BUFFER_RX_SURPLUS ) {
        WxEmac_Fin();
        device->etHandel = NULL;
        m_RxNextDescCnt = 0;
        return WXEMAC_ERROR_SYSTEM;
    }
    for ( i = m_RxNextDescCnt; i < WXEMAC_BUFFER_RX_MAX; i++, m_RxSurplusDescCnt++ ) {
        m_RxSurplusDescArray[m_RxSurplusDescCnt] = &device->rxbuffer[i]; // set Surplus Descripter.
    }

    /* Do global Interrupt Enable */
    (void)Interrupt_enableInProcessor();
    /* Assign default ISRs */
    Interrupt_registerHandler(INT_EMAC_TX0, Ethernet_transmitISR);
    Interrupt_registerHandler(INT_EMAC_TX1, Ethernet_transmitISR);
    Interrupt_registerHandler(INT_EMAC_RX0, Ethernet_receiveISR);
    Interrupt_registerHandler(INT_EMAC_RX1, Ethernet_receiveISR);
    /* Enable the default interrupt handlers */
    Interrupt_enable(INT_EMAC_TX0);
    Interrupt_enable(INT_EMAC_TX1);
    Interrupt_enable(INT_EMAC_RX0);
    Interrupt_enable(INT_EMAC_RX1);

    bInit = true;

    /* Low Frequency
     * value of 5 for selecting the slowest possible MDIO Clock
     * Clause 22 mode */
    Ethernet_configureMDIO(EMAC_BASE,0,5,0);
    /* The DP83822 External PHY in Control Card
     * takes a PHY address of 1 by default
     * Configure the MDIO module to use PHY address of 0x1 */
    Ethernet_configurePHYAddress(EMAC_BASE,1);

    /* PHY Setting. Auto-nego OFF. 100Mbps. Full-Duplex. */
#if defined( PHY_100M_FULL )
    mdioRead = PHY_MdioRegRead( 0x0000 );
    mdioRead &= ~0x1000; // PHY AutoNego OFF.
    mdioRead |=  0x2000; // PHY 100Mbps
    mdioRead |=  0x100;  // PHY Full Duplex.
    PHY_MdioRegWrite( 0x0000, mdioRead );
    mdioRead = PHY_MdioRegRead( 0x0000 );
#elif defined( PHY_10M_FULL )
    mdioRead = PHY_MdioRegRead( 0x0000 );
    mdioRead &= ~0x1000; // PHY AutoNego OFF.
    mdioRead &= ~0x2000; // PHY 10Mbps
    mdioRead |=  0x100;  // PHY Full Duplex.
    PHY_MdioRegWrite( 0x0000, mdioRead );
    mdioRead = PHY_MdioRegRead( 0x0000 );
#elif defined( PHY_100M_HALF )
    mdioRead = PHY_MdioRegRead( 0x0000 );
    mdioRead &= ~0x1000; // PHY AutoNego OFF.
    mdioRead |=  0x2000; // PHY 100Mbps
    mdioRead &= ~0x100;  // PHY Half Duplex.
    PHY_MdioRegWrite( 0x0000, mdioRead );
    mdioRead = PHY_MdioRegRead( 0x0000 );
#elif defined( PHY_10M_HALF )
    mdioRead = PHY_MdioRegRead( 0x0000 );
    mdioRead &= ~0x1000; // PHY AutoNego OFF.
    mdioRead &= ~0x2000; // PHY 10Mbps
    mdioRead &= ~0x100;  // PHY Half Duplex.
    PHY_MdioRegWrite( 0x0000, mdioRead );
    mdioRead = PHY_MdioRegRead( 0x0000 );
#endif

    /* PHY RMII Setting */
    mdioRead = PHY_MdioRegRead( 0x0017 );
    mdioRead &= ~0x200;// RGMII Mode
    mdioRead &= ~0x80; // RMII Clock Select
    mdioRead |= 0x40;  // RMII Recovered Clock Async FIFO Bypass
    mdioRead |= 0x20;  // RMII Mode
    mdioRead &= ~0x10; // RMII Revision Select
    mdioRead &= ~0x2;  // Receive Elasticity Buffer Size #1 //
    mdioRead |= 0x1;   // Receive Elasticity Buffer Size #0 // 01 = 2 bit tolerance (up to 2400 byte packets)
    PHY_MdioRegWrite( 0x0017, mdioRead );
    mdioRead = PHY_MdioRegRead( 0x0017 );

    mdioRead = PHY_MdioRegRead( 0x0009 );
    mdioRead |= 0x20;  // Robust Auto-MDIX
    mdioRead |= 0x40;  // Fast Auto-MDIX
    PHY_MdioRegWrite( 0x0009, mdioRead );
    mdioRead = PHY_MdioRegRead( 0x0009 );

#ifdef LOOPBACK_MODE_PHY_MII
    mdioRead = PHY_MdioRegRead( 0x0000 );
    mdioRead |= 0x4000; // PHY Enable MII-Loopback
    PHY_MdioRegWrite( 0x0000, mdioRead );
    mdioRead = PHY_MdioRegRead( 0x0000 );
#endif // LOOPBACK_MODE_PHY_MII
#ifdef LOOPBACK_MODE_PHY_DEGITAL
    mdioRead = PHY_MdioRegRead( 0x0016 );
    mdioRead |= 0x4; // PHY Digital Loopback
    PHY_MdioRegWrite( 0x0016, mdioRead );
    mdioRead = PHY_MdioRegRead( 0x0016 );
#endif // LOOPBACK_MODE_PHY_DEGITAL

    mdioRead = 0x4000; // Digital reset
    PHY_MdioRegWrite( 0x001F, mdioRead );
    mdioRead = PHY_MdioRegRead( 0x001F );

#ifdef PHY_AUTONEGO_TEST
    DEVICE_DELAY_US(3000);  // 3ms PHY Init Delay. min 2.5ms
    mdioRead = PHY_MdioRegRead( 0x0000 ); // PHY Status Register
    mdioRead |= 0x1000; // Auto-nego Enable
    mdioRead |= 0x0200; // Auto-nego Restart
    PHY_MdioRegWrite( 0x0000, mdioRead );
    DEVICE_DELAY_US(500);
//    while ( 1 ) {
//        mdioRead = PHY_MdioRegRead( 0x0010 ); // PHY Status Register
//        if ( (mdioRead & 0x1)  == 0 ) continue;  // Link not Valid
//        if ( (mdioRead & 0x10) == 0 ) continue;  // Auto nego not completed
//        break;
//    }
#endif

    return (WXEMAC_ERROR_SUCCESS);
}

WXEMAC_ERROR WxEmac_Fin(void)
{
    int i = 0;

    /* Ethernet finalize. */
    Ethernet_shutdownInterface();                       // Ether end
    WxEmac_Device.rxhead = WxEmac_Device.rxtail = NULL; // RX buff array
    WxEmac_BFreeHead = WxEmac_BFreeTail = NULL;         // TX buff array
    m_RxNextDescCnt = 0;                                // RX Desc Counter Clear
    WxEmac_Device.etHandel = NULL;


    for ( i = 0; i < WXEMAC_BUFFER_RX_SURPLUS; i++ ) {
        m_RxSurplusDescArray[i] = NULL;                 // RX Surplus Desc Array Clear
    }
    m_RxSurplusDescCnt = 0;                             // RX Surplus Desc Counter Clear
    m_TxFrameCnt = 0;

    bInit = false;
    return (WXEMAC_ERROR_SUCCESS);
}


// MMD Register Read/write
#define CRADR 0x0000 // CR Target Address.
#define CRDAT 0x4000 // CR Target Data.
#define CRDEV_MMD  0x001F // CR Target Data.
#define CRDEV_MMD3 0x0003 // CR Target Data.
#define CRDEV_MMD7 0x0007 // CR Target Data.
#define REGCR 0x000D // REGCR
#define ADDAR 0x000E // ADDAR
/***************************************************
 * name       : PHY_MdioRegRead
 * input      : uint16_t   regAdr
 * return     : uint16_t   reg data.
 * comment    : Read PHY Register Data. PHY:dp83822i
 *            : !!! Available after Ethernet_getHandle()
 ***************************************************/
uint16_t PHY_MdioRegRead( uint16_t regAdr )
{
    WXEMAC_DEVICE *device = &WxEmac_Device;
    uint16_t cradr, crdat;

    if ( device->etHandel == NULL ) return 0x0;

    if ( regAdr < 0x0020 ) {        // Direct
        return Ethernet_readPHYRegister( EMAC_BASE, regAdr );   // read ADDAR.(target Addr Data)
    }
    else if ( regAdr < 0x04D7 ) {   // reg MMD
        cradr = CRADR + CRDEV_MMD;  // set CR. next target Addr.
        crdat = CRDAT + CRDEV_MMD;  // set CR. next get Data.
    }
    else if ( regAdr < 0x3017 ) {   // reg MMD3
        cradr = CRADR + CRDEV_MMD3; // set CR. next target Addr.
        crdat = CRDAT + CRDEV_MMD3; // set CR. next get Data.
    }
    else {                          // reg MMD7
        cradr = CRADR + CRDEV_MMD7; // set CR. next target Addr.
        crdat = CRDAT + CRDEV_MMD7; // set CR. next get Data.
    }
    Ethernet_writePHYRegister( EMAC_BASE, REGCR, cradr );  // set CR. target Addr.
    Ethernet_writePHYRegister( EMAC_BASE, ADDAR, regAdr ); // set Addr. Addr(Data -> ADDAR)
    Ethernet_writePHYRegister( EMAC_BASE, REGCR, crdat );  // set CR. get Data.
    return Ethernet_readPHYRegister( EMAC_BASE, ADDAR );   // read ADDAR.(target Addr Data)
}

/***************************************************
 * name       : PHY_MdioRegWrite
 * input      : uint16_t   regAdr
 *            : uint16_t   regDat
 * return     : uint16_t   0 = success. 1 = NG.
 * comment    : Write PHY Register Data. PHY:dp83822i
 *            : !!! Available after Ethernet_getHandle()
 ***************************************************/
uint16_t PHY_MdioRegWrite( uint16_t regAdr, uint16_t regDat )
{
    WXEMAC_DEVICE *device = &WxEmac_Device;
    uint16_t cradr, crdat;

    if ( device->etHandel == NULL ) return 0x0;

    if ( regAdr < 0x0020 ) {        // Direct
        Ethernet_writePHYRegister( EMAC_BASE, regAdr, regDat ); // Write ADDAR.(target Addr Data)
        return 0;
    }
    else if ( regAdr < 0x04D7 ) {   // reg MMD
        cradr = CRADR + CRDEV_MMD;  // set CR. next target Addr.
        crdat = CRDAT + CRDEV_MMD;  // set CR. next get Data.
    }
    else if ( regAdr < 0x3017 ) {   // reg MMD3
        cradr = CRADR + CRDEV_MMD3; // set CR. next target Addr.
        crdat = CRDAT + CRDEV_MMD3; // set CR. next get Data.
    }
    else {                          // reg MMD7
        cradr = CRADR + CRDEV_MMD7; // set CR. next target Addr.
        crdat = CRDAT + CRDEV_MMD7; // set CR. next get Data.
    }
    Ethernet_writePHYRegister( EMAC_BASE, REGCR, cradr );  // set CR. target Addr.
    Ethernet_writePHYRegister( EMAC_BASE, ADDAR, regAdr ); // set Addr. Addr(Data -> ADDAR)
    Ethernet_writePHYRegister( EMAC_BASE, REGCR, crdat );  // set CR. get Data.
    Ethernet_writePHYRegister( EMAC_BASE, ADDAR, regDat ); // read ADDAR.(target Addr Data)
    return 0;
}
