/*****************************************************************************
 * Dsp1Run.c
 *****************************************************************************/
#include <Dsp1Run.h>

/* Define-------------------------- */
/* Variable------------------------ */
/* Prototype----------------------- */
static void Dsp1Run_MainLoop( void );
static void Init_GSRAM( void );
static void gpioInitSet( uint32_t pin, GPIO_Direction pinIO, GPIO_CoreSelect core, GPIO_QualificationMode qual, uint32_t pinType, uint32_t pinConfig );
static void PerGpio_Init( void );
/* RAM Section Functions ---------- */
#pragma CODE_SECTION( Dsp1Run_MainLoop, ".TI.ramfunc" );


/* main                      */
/* User Program Entry Point. */
int main(void)
{
    DINT;
    IER = 0x0000;
    IFR = 0x0000;

    /* !!! Set before initialization for use in cache. !!! */
    /* Global Shared RAM & MSG RAM Initialize. Set GSRAM Master. */
    Init_GSRAM();

    /* Initialize device clock and peripherals */
    Device_init();

    /* Initialize GPIO and configure the GPIO pin as a push-pull output */
    Device_initGPIO();

/* GPIO & shared peripheral.------------------------------------------------------------ */
    /* User GPIO Setting. */
    PerGpio_Init(); // GPIO Init.

    /* Change Allocated shared peripheral. */
    EALLOW;
    CmConfRegs.PALLOCATE0.bit.CAN_A = 1;
    CmConfRegs.PALLOCATE0.bit.CAN_B = 1;
    CmConfRegs.PALLOCATE0.bit.ETHERCAT = 1;
    CmConfRegs.PALLOCATE0.bit.MCAN_A = 1;
    CmConfRegs.PALLOCATE0.bit.USB_A = 1;
    EDIS;

    /* Clear any IPC flags if set already */
    IPC_clearFlagLtoR(IPC_CPU1_L_CPU2_R, IPC_FLAG_ALL);
    IPC_clearFlagLtoR(IPC_CPU1_L_CM_R, IPC_FLAG_ALL);

    /* Boot CM&CPU2 core */
    Device_bootCM(BOOTMODE_BOOT_TO_FLASH_SECTOR0);
    Device_bootCPU2(BOOTMODE_BOOT_TO_FLASH_SECTOR0);


/* Ethernrt setting.---------------------------------------------------------------- */

    /* FG4 Usigng RMII. */
    SysCtl_setEnetClk(SYSCTL_ENETCLKOUT_DIV_2, SYSCTL_SOURCE_SYSPLL); // EnetClk : clock divider to 2.
    // SYSPLLCLK -> ETHDIV( SYSCTL_ENETCLKOUT_DIV_2 ) -> PTP_CLK (100Mhz)
    //                                                -> /2 -> ENET_RMII_CLK (Internal clock 50Mhz)

/* Setup Complete. CPU1,2 Sync. ---------------------------------------------------- */
    /* CPU12 Sync. */
    IPC_setFlagLtoR( IPC_CPU1_L_CPU2_R, IPC_FLAG4 ); // IPC Flg set.  [C1] => C2
    IPC_waitForAck( IPC_CPU1_L_CPU2_R, IPC_FLAG4 );  // IPC Ack wait. [C1] <= C2

    /* Enable Global Interrupt (INTM) and realtime interrupt (DBGM) */
    EINT; // Enable Global interrupt INTM
    ERTM; // Enable Global realtime interrupt DBGM

/* mainloop ------------------------------------------------------------------------- */
    Dsp1Run_MainLoop(); // not return.

    return 0;
}

/* MainLoop. */
static void Dsp1Run_MainLoop( void )
{
    static U16 LogCount = 0;

    while(1)
    {

    } // loop End
}

/***************************************************
 * name       : Init_GSRAM
 * comment    : Initialized Global Shared Ram
 ***************************************************/
static void Init_GSRAM( void )
{
    EALLOW;
    MemCfgRegs.GSxINIT.all = 0x0000FFFF;  // GS0-15 Initialized.
    /* CPU1toCPU2 MSGRAM1,2 & CPU1toCM MSGRAM1,2 Initialized. */
    MemCfgRegs.MSGxINIT.bit.INIT_CPUTOCPU_MSGRAM0 = 1;
    MemCfgRegs.MSGxINIT.bit.INIT_CPUTOCPU_MSGRAM1 = 1;
    MemCfgRegs.MSGxINIT.bit.INIT_CPUTOCM_MSGRAM0 = 1;
    MemCfgRegs.MSGxINIT.bit.INIT_CPUTOCM_MSGRAM1 = 1;
    EDIS;
    while ( 0x0000FFFF != MemCfgRegs.GSxINITDONE.all );                   // Initialize Done.
    while ( 0 == MemCfgRegs.MSGxINITDONE.bit.INITDONE_CPUTOCPU_MSGRAM0 ); // Initialize Done.
    while ( 0 == MemCfgRegs.MSGxINITDONE.bit.INITDONE_CPUTOCPU_MSGRAM1 ); // Initialize Done.
    while ( 0 == MemCfgRegs.MSGxINITDONE.bit.INITDONE_CPUTOCM_MSGRAM0 );  // Initialize Done.
    while ( 0 == MemCfgRegs.MSGxINITDONE.bit.INITDONE_CPUTOCM_MSGRAM1 );  // Initialize Done.
    EALLOW;
    MemCfgRegs.GSxMSEL.all = 0x00000080;  // GS0-15 CPU1 Master. (CPU1 W/R , CPU2 R)
    EDIS;
}

/************************************************************************
 * name   : gpioInitSet
 * input  : uint32_t               pin           GPIO No
 * input  : GPIO_Direction         pinIO         IN/OUT
 * input  : GPIO_CoreSelect        core          User Core
 * input  : GPIO_QualificationMode qual          Qualification
 * input  : uint32_t               pinType       Pad Config
 * input  : uint32_t               pinConfig     Mux/GMux   (check pin_map.h)
 ************************************************************************/
static void gpioInitSet( uint32_t pin, GPIO_Direction pinIO, GPIO_CoreSelect core, GPIO_QualificationMode qual, uint32_t pinType, uint32_t pinConfig )
{
    GPIO_setDirectionMode( pin, pinIO );
    GPIO_setMasterCore( pin, core );
    GPIO_setQualificationPeriod( pin, qual );
    GPIO_setPadConfig( pin, pinType );
    GPIO_setPinConfig( pinConfig );
}

/************************************************************************
 *  id      : PerGpio_Init
 *  comment : GPIO Init.
 ************************************************************************/
static void PerGpio_Init( void )
{
    /* GPIO Setting. ------- */
    EALLOW;

    GpioDataRegs.GPCSET.bit.GPIO67 = 1; // set HIGH // ETH_RST*.   Low to Reset. CM Setting.

    gpioInitSet(  84, GPIO_DIR_MODE_OUT, GPIO_CORE_CM,   GPIO_QUAL_ASYNC, GPIO_PIN_TYPE_STD,    GPIO_84_UARTA_TX ); // UARTA_TX : UART_TX
    gpioInitSet(  85, GPIO_DIR_MODE_IN,  GPIO_CORE_CM,   GPIO_QUAL_ASYNC, GPIO_PIN_TYPE_STD,    GPIO_85_UARTA_RX ); // UARTA_RX : UART_RX

    gpioInitSet(  38, GPIO_DIR_MODE_IN,  GPIO_CORE_CM,   GPIO_QUAL_SYNC,  GPIO_PIN_TYPE_STD,    GPIO_38_ENET_MII_RX_DV );  // ENET_MII_RX DV : ETH_CRS
    gpioInitSet(  39, GPIO_DIR_MODE_IN,  GPIO_CORE_CM,   GPIO_QUAL_SYNC,  GPIO_PIN_TYPE_STD,    GPIO_39_ENET_MII_RX_ERR ); // ENET_MII_RX_ERR : ETH_RX_ERR
    gpioInitSet(  42, GPIO_DIR_MODE_OUT, GPIO_CORE_CM,   GPIO_QUAL_SYNC,  GPIO_PIN_TYPE_STD,    GPIO_42_ENET_MDIO_CLK );  // ENET_MDIO_CLK : ETH_MDIO_CLK
    gpioInitSet(  43, GPIO_DIR_MODE_OUT, GPIO_CORE_CM,   GPIO_QUAL_SYNC,  GPIO_PIN_TYPE_STD,    GPIO_43_ENET_MDIO_DATA ); // ENET_MDIO_DATA : ETH_MDIO_DATA
    gpioInitSet(  53, GPIO_DIR_MODE_IN,  GPIO_CORE_CM,   GPIO_QUAL_SYNC,  GPIO_PIN_TYPE_STD,    GPIO_53_ENET_MII_RX_DATA1 ); // ENET_MII_RX_DATA1 : ETH_RX_DATA1
    gpioInitSet(  56, GPIO_DIR_MODE_OUT, GPIO_CORE_CM,   GPIO_QUAL_SYNC,  GPIO_PIN_TYPE_STD,    GPIO_56_ENET_MII_TX_EN );    // ENET_MII_TX_EN : ETH_TX_EN
    gpioInitSet(  59, GPIO_DIR_MODE_OUT, GPIO_CORE_CM,   GPIO_QUAL_SYNC,  GPIO_PIN_TYPE_STD,    GPIO_59_ENET_MII_TX_DATA0 ); // ENET_MII_TX_DATA0 : ETH_TX_DATA0
    gpioInitSet(  60, GPIO_DIR_MODE_OUT, GPIO_CORE_CM,   GPIO_QUAL_SYNC,  GPIO_PIN_TYPE_STD,    GPIO_60_ENET_MII_TX_DATA1 ); // ENET_MII_TX_DATA1 : ETH_TX_DATA1
    gpioInitSet(  66, GPIO_DIR_MODE_IN,  GPIO_CORE_CM,   GPIO_QUAL_SYNC,  GPIO_PIN_TYPE_STD,    GPIO_66_ENET_MII_RX_DATA0 ); // ENET_MII_RX_DATA0 : ETH_RX_DATA0
    gpioInitSet(  73, GPIO_DIR_MODE_IN,  GPIO_CORE_CM,   GPIO_QUAL_SYNC,  GPIO_PIN_TYPE_STD,    GPIO_73_ENET_RMII_CLK ); // ENET_RMII_CLK : ETH_REF_CLK

    EDIS;
}

