//*****************************************************************************
//
// Example to illustrate issues with CMPSS2 and EPWM9
//
//*****************************************************************************

#include "F28x_Project.h"

void InitDevice(void);
void InitApplication(void);
void InitApplicationGPIO(void);
void StartPwmGlobalClock(void);
void StopPwmGlobalClock(void);
void SetCmpssOutputs(void);
void InitPWMs(void);
void SetCmpssOutputs(void);
void InitCMPSS2(void);

//*****************************************************************************
// Name: main
// Input: None
// Output/Return Value: None
// Description: Initialize the device and the application and enter the CPU
//              into an infinite background loop.
//*****************************************************************************
void main(void)
{

	InitDevice();
    InitApplication();

	// Globally Enable Interrupts
	EINT;
	ERTM;

	// infinite loop for background tasks
	for(;;)
	{
	}
}

//*****************************************************************************
// Name: InitDevice
// Input: None
// Output/Return Value: None
// Description: Set up the system clock, enable all peripheral clocks, GPIO,
//              and assign default ISR vectors for each CPU interrupt.
//*****************************************************************************
void InitDevice(void)
{
	InitSysCtrl();
	DINT;					// Globally disable interrupts before changes
	InitPieCtrl();			// Reset all PIE registers
	IER = 0x0000;			// Globally clear PIE enables
	IFR = 0x0000;			// Globally clear PIE flags
	InitPieVectTable();		// Default ISR function assignments
	InitGpio();
}

//*****************************************************************************
// Name: InitApplication
// Input: None
// Output/Return Value: None
// Description: Initialize all the peripherals required to support the application's
//              main functionality.
//*****************************************************************************
void InitApplication(void)
{


	// Configure all pin functions required for the application
	InitApplicationGPIO();

    InitPWMs();
    SetCmpssOutputs();
    InitCMPSS2();
}

//*****************************************************************************
// Name: main
// Input: None
// Output/Return Value: None
// Description: Setup GPIO pins required to output signals
//*****************************************************************************
void InitApplicationGPIO()
{
    EALLOW;
    GpioCtrlRegs.GPAPUD.bit.GPIO4 = 1;     // Disable pull-up on GPIO4 (EPWM3A)
    GpioCtrlRegs.GPAMUX1.bit.GPIO4 = 1;    // Configure GPIO4 as EPWM3A

    GpioCtrlRegs.GPAPUD.bit.GPIO6 = 1;     // Disable pull-up on GPIO6 (EPWM4A)
    GpioCtrlRegs.GPAMUX1.bit.GPIO6 = 1;    // Configure GPIO6 as EPWM4A

    GpioCtrlRegs.GPAPUD.bit.GPIO16 = 1;    // Disable pull-up on GPIO16 (EPWM9A)
    GpioCtrlRegs.GPAGMUX2.bit.GPIO16 = 1;  // Configure GPIO16 as EPWM9A
    GpioCtrlRegs.GPAMUX2.bit.GPIO16 = 1;   // Configure GPIO16 as EPWM9A

    EDIS;
}


//*****************************************************************************
// Name: main
// Input: None
// Output/Return Value: None
// Description: Initialize the ePWM modules
//*****************************************************************************
void InitPWMs(void)
{
    EALLOW;
    const static float32 DUTY = .9;
    CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 0; // stop PWM clock

//    // set up PWM modules frequency
    Uint32 PWMFrequencyHz = 700000;
    Uint16 PWMClockCycles = (Uint16)(100000000 / PWMFrequencyHz);
    EPwm3Regs.TBCTL.bit.PRDLD = 1;          // immediate load
    EPwm3Regs.TBCTL.bit.CTRMODE = 0;        // count up mode
    EPwm3Regs.TBCTL.bit.HSPCLKDIV = 0;      // prescaler of 1 (100MHz clock)
    EPwm3Regs.TBCTL.bit.CLKDIV = 0;         // prescaler of 1
    EPwm3Regs.TBPRD = PWMClockCycles - 1;
    EPwm9Regs.TBCTL.bit.PRDLD = 1;          // immediate load
    EPwm9Regs.TBCTL.bit.CTRMODE = 0;        // count up mode
    EPwm9Regs.TBCTL.bit.HSPCLKDIV = 0;      // prescaler of 1 (100MHz clock)
    EPwm9Regs.TBCTL.bit.CLKDIV = 0;         // prescaler of 1
    EPwm9Regs.TBPRD = PWMClockCycles - 1;
    EPwm4Regs.TBCTL.bit.PRDLD = 1;          // immediate load
    EPwm4Regs.TBCTL.bit.CTRMODE = 0;        // count up mode
    EPwm4Regs.TBCTL.bit.HSPCLKDIV = 0;      // prescaler of 1 (100MHz clock)
    EPwm4Regs.TBCTL.bit.CLKDIV = 0;         // prescaler of 1
    EPwm4Regs.TBPRD = PWMClockCycles/2 - 1; // double the frequency

    Uint16 maxDutyClkCycles;
    Uint16 halfPrdClkCycles;
    Uint16 reDbClkCycles;
    Uint16 REdbClkCycles;
    Uint16 FEdbClkCycles;

    maxDutyClkCycles = (Uint16) ((DUTY * 0.5) * (float32) EPwm3Regs.TBPRD + 0.5);
    halfPrdClkCycles = (Uint16)((100000000 / PWMFrequencyHz - 1) * 0.5);
    reDbClkCycles = (Uint16) (50 / 10);
    REdbClkCycles = (Uint16) (50 / 10);
    FEdbClkCycles = (Uint16) (50 / 10);
    // PWM3 output A is used to drive Sync FET 1 which is just FB's switch A with deadband delays applied on each edge
    EPwm3Regs.CMPA.bit.CMPA = EPwm3Regs.TBPRD - (FEdbClkCycles + 1) ; // Use CMPA to create additional deadtime between disabling Sync FET and disabling GaN switching FET
    EPwm3Regs.CMPB.bit.CMPB = maxDutyClkCycles;       // Set to max duty to re-enable in case comparator does not trip
    EPwm3Regs.AQCTLA.bit.CAU = AQ_CLEAR;              // Set PWM5A low at compare A value
    EPwm3Regs.AQCTLA.bit.CBU = AQ_SET;                // Set PWM5A high at compare B value

    // Set the trip
    EPwm3Regs.DCTRIPSEL.bit.DCAHCOMPSEL = 3;          // Select TRIPIN4 for DCAH
    EPwm3Regs.TZDCSEL.bit.DCAEVT2 = TZ_DCAH_HI;       // Generate DCAEVT2 when DCAH is high (comparator output is high)
    // Let action qualifier handle the event, don't do anything through trip zone!
    EPwm3Regs.TZCTL.bit.DCAEVT2 = TZ_NO_CHANGE;
    EPwm3Regs.TZSEL.bit.DCAEVT2 = TZ_DISABLE;
    // Connect DCAEVT2 triggered by the comparator to actions on A and B
    EPwm3Regs.AQTSRCSEL.bit.T1SEL = DC_SRC_DCAEVT2;   // 1 = DCAEVT2 produced by DCAH HI
    EPwm3Regs.AQCTLA2.bit.T1U = AQ_SET;               // Set output A high on DCBEVT2
    EPwm3Regs.DBCTL.bit.IN_MODE = DBA_ALL;            // Set output A as source to all deadband logic
    EPwm3Regs.DBCTL.bit.OUT_MODE =  DBA_ENABLE;       // Enable rising edge delayed output
    // This is rising edge db on the trip
    // NOTE: Falling edge delays is created via CMPA
    EPwm3Regs.DBRED.bit.DBRED = reDbClkCycles + REdbClkCycles;

    // PWM9 output A is used to drive Sync FET 2 which is just FB's switch D with deadband delays applied on each edge
    EPwm9Regs.CMPA.bit.CMPA = halfPrdClkCycles - FEdbClkCycles;  // Use CMPA to create additional deadtime, note this one is shifted 180 degrees into the period
    EPwm9Regs.CMPB.bit.CMPB = halfPrdClkCycles + maxDutyClkCycles;     // Set to max duty in case comparator does not trip
    EPwm9Regs.AQCTLA.bit.CAU = AQ_CLEAR;              // Set PWM6A low at compare A value
    EPwm9Regs.AQCTLA.bit.CBU = AQ_SET;                // Set PWM6B high at compare B value

    // Set the trip for peak current mode control, Sync FET will trip on PCM just like switch C but with added delay
    EPwm9Regs.DCTRIPSEL.bit.DCAHCOMPSEL = 3;          // Select TRIPIN4 for DCAH
    EPwm9Regs.TZDCSEL.bit.DCAEVT2 = TZ_DCAH_HI;       // Generate DCAEVT2 when DCAH goes high (comparator output goes high)
    // Let action qualifier handle the event, don't do anything through trip zone!
    EPwm9Regs.TZCTL.bit.DCAEVT2 = TZ_NO_CHANGE;
    EPwm9Regs.TZSEL.bit.DCAEVT2 = TZ_DISABLE;
    // Connect DCAEVT2 triggered by the comparator to actions on A and B
    EPwm9Regs.AQTSRCSEL.bit.T1SEL = DC_SRC_DCAEVT2;   // DCAEVT2 produced by DCAH HI
    EPwm9Regs.AQCTLA2.bit.T1U = AQ_SET;               // Set output A high on filtered DCAEVT2
    EPwm9Regs.DBCTL.bit.IN_MODE = DBA_ALL;            // Set output A as source to all deadband logic
    EPwm9Regs.DBCTL.bit.OUT_MODE =  DBA_ENABLE;       // Enable rising edge delayed output

    // NOTE: Falling edge delays is created via CMPA
    EPwm9Regs.DBRED.bit.DBRED = reDbClkCycles + REdbClkCycles;

    // PWM4 is used only to reset the DAC at twice the sw freq
    // 50ns delay into the period is necessary to create min duty by allowing the PWM to go high first and only then reset
    // the comparator DAC in case it goes high right away
    EPwm4Regs.CMPA.bit.CMPA = 5;         // Max duty in pwm clk cycles
    EPwm4Regs.AQCTLA.bit.ZRO = AQ_SET;                  // Set output A high at CTR=0
    EPwm4Regs.AQCTLA.bit.CAU = AQ_CLEAR;                // Set output A low on max duty if comparator trip doesn't occur
    EPwm4Regs.CMPC = 5;
    EPwm4Regs.HRPCTL.bit.PWMSYNCSELX = 4;              // Sync pulse to reset the DAC is on CTR=CMPC counting up

    // enable PWMs

    EPwm1Regs.AQCSFRC.bit.CSFA = 0;
    EPwm3Regs.AQCSFRC.bit.CSFA = 0;
    EPwm4Regs.AQCSFRC.bit.CSFA = 0;
    CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 1; // start PWM clock

    EDIS;
}

//*****************************************************************************
// Name: SetCmpssOutputs
// Input: None
// Output/Return Value: None
// Description: Internal connection of the comparator outputs to the PWM trip
//              zone signals.
//*****************************************************************************
void SetCmpssOutputs(void)
{
    EALLOW;
    // Connect comparator trips as trip sources through EPWM X-bar
    EPwmXbarRegs.TRIP4MUX0TO15CFG.bit.MUX2 = 0; // Selection 0 (CMPSS2.CTRIPH) drives mux2
    EPwmXbarRegs.TRIP4MUXENABLE.bit.MUX2 = 1;   // Mux2 is enabled to drive EPWM's TRIP4

    // Connect comparator output to output X-BAR for debug
    GpioCtrlRegs.GPBPUD.bit.GPIO58 = 1;
    GpioCtrlRegs.GPBGMUX2.bit.GPIO58 = 1;          // Configure GPIO58 as OUTPUTXBAR1
    GpioCtrlRegs.GPBMUX2.bit.GPIO58 = 1;           // Configure GPIO58 as OUTPUTXBAR1
    // Connect comparator output to output X-BAR for debug
    OutputXbarRegs.OUTPUT1MUX0TO15CFG.bit.MUX2 = 0;  // Select input 0 (CMPSS2.CTRIPOUTH) for Mux2
    OutputXbarRegs.OUTPUT1MUXENABLE.bit.MUX2 = 1;    // Mux2 drives OUTPUTXBAR1

    EDIS;

}

//*****************************************************************************
// Name: main
// Input: None
// Output/Return Value: None
// Description: Initialize comparator 2
//*****************************************************************************
void InitCMPSS2(void)
{
    EALLOW;
    Cmpss2Regs.RAMPDECVALS = 1000;                     // Value to subtract from ramp for each CLK cycle
    Cmpss2Regs.RAMPMAXREFS = 9927; // 0.5V           // Value preloaded into ramp once PWMSYNC is received
    Cmpss2Regs.RAMPDLYS.bit.DELAY = 0;               // Time offset after PWMSYNC before ramp is decremented
    Cmpss2Regs.COMPCTL.bit.COMPDACE = 1;             // Enable comparator / DAC
    Cmpss2Regs.COMPDACCTL.bit.RAMPSOURCE = 3;        // sync of ePWM4
    Cmpss2Regs.COMPDACCTL.bit.RAMPLOADSEL = 0;       // Synchronize changes in RAMPMAXREF to CLK
    Cmpss2Regs.COMPDACCTL.bit.DACSOURCE = 1;         // Feed ramp out into DAC
    Cmpss2Regs.COMPDACCTL.bit.SWLOADSEL = 0;         // DACHVALA is updated from DACHVALS on SYSCLK
    Cmpss2Regs.COMPCTL.bit.COMPHINV = 0;             // Do not invert output of comparator; V+ > V- results in a logic high, low otherwise
    Cmpss2Regs.COMPCTL.bit.COMPHSOURCE = 0;          // Feed DAC output through negative terminal of comparator
    Cmpss2Regs.CTRIPHFILCTL.bit.SAMPWIN = 9;         // 10 samples will be used for the filter window  (SAMPWIN + 1)
    Cmpss2Regs.CTRIPHFILCTL.bit.THRESH = 8;          // 8 of the 10 samples must be high before the output will change
    Cmpss2Regs.CTRIPHFILCLKCTL.bit.CLKPRESCALE = 1;  // use half SYSCLK for sampling
    Cmpss2Regs.CTRIPHFILCTL.bit.FILINIT = 1;         // Initialize all samples to the filter input value - which should be low at this point in time
    Cmpss2Regs.COMPCTL.bit.CTRIPHSEL = 3;            // Filtered comparator output but latched for CTRIPH (pwm trip)
    Cmpss2Regs.COMPCTL.bit.CTRIPOUTHSEL = 3;         // Filtered comparator output but latched for CTRIPOUTH (debug on gpio)
    Cmpss2Regs.COMPSTSCLR.bit.HSYNCCLREN = 1;        // PWMSync will clear the comparator latch.
    EDIS;
}
