#include "F28x_Project.h" // this includes all headers needed to interact with peripherals (ADC, EPWM, etc.)
#include "GlobalVariables.h"

__interrupt void control_isr(void);
void stop_Sequence(void);

void configure_EPWM(void); // configure EPWM
void configure_ADC(void);  // configure ADC
void configure_GPIO(void); // configure GPIO
void configure_DAC(void);  // configure DAC

volatile Uint16 k = 0;
volatile Uint16 k_max = 100;
volatile Uint16 start_enabled = 0;
volatile Uint16 stop_all = 0;

volatile float32 duty_openloop = DUTY_INIT;

volatile Uint16 en_step = 0;

volatile float d_init = 0.35;
volatile float d_final = 0.63;

enum State
{
    STOPPED,
    ACTIVE
};

enum State sm_state = STOPPED;

void stop_Sequence(void)
{
    EPwm1Regs.ETSEL.bit.INTEN = 0;

    GpioDataRegs.GPACLEAR.bit.GPIO26 = 1; // trip signal is active low
    start_enabled = 0;
    sm_state = STOPPED;
    duty_openloop = 0.5;
}

int main(void)
{

    InitSysCtrl(); //Initialize SYSCTL (PLL, Watchdog, etc.)

    InitGpio();    //Initialize GPIO register states

    configure_GPIO();               // configure GPIO settings

    DINT;                           // Disable ST1.INTM
    IER = 0x0000;                   // Disable CPU interrupts
    IFR = 0x0000;                   // Clear all CPU interrupt flags

    InitPieCtrl();                  // Initialize PIE control registers
    InitPieVectTable();             // Initialize PIE vector table to default ISR locations...
                                    // This will typically be overwritten later

    configure_ADC();                // configure ADC settings
    configure_EPWM();               // configure EPWM settings
    configure_DAC();                // configure DAC settings

    // Write the ISR vector for each interrupt to the appropriate location in the PIE vector table
    EALLOW;
    PieVectTable.EPWM1_INT = &control_isr;        // main interrupt running at rate fpwm
    EDIS;

    // Enable interrupts on CPU level
    IER |= M_INT3;                  // PIE group1 --> CPU INT1 (contains ADCC1 INT)

    EINT;                           // Enable ST1.INTM
    ERTM;                           // Enable Global real time interrupt DBG

    // Enable interrupt on PIE level
    PieCtrlRegs.PIEIER3.bit.INTx1 = 1; // PIE TABLE INT3.1 is EPWM1 (see TRM Table 3.4.5)

    while(1)
    {
        if (sm_state == ACTIVE)
        {
            if (stop_all) // stop_all flag set
            {
                stop_Sequence();
            }
        }
        else // sm_state == STOPPED
        {
            if (start_enabled && !stop_all) // stop_all flag cleared and start_enabled flag set
            {
                GpioDataRegs.GPASET.bit.GPIO26 = 1; // reset PWM trip signal

                EALLOW;

                EPwm9Regs.TZCLR.bit.OST = 1; // clear Trip Zone OST for each PWM
                EPwm10Regs.TZCLR.bit.OST = 1;
                EPwm11Regs.TZCLR.bit.OST = 1;

                EDIS;

                sm_state = ACTIVE;

                EPwm1Regs.ETSEL.bit.INTEN = 1;
            }
        }
    }
}


__interrupt void control_isr(void)
{
    k++;

    static Uint16 PWM1_CMP_shdw = EPWM_TBPRD / 2; // the value loaded into the shadow register
    static Uint16 PWM2_CMP_shdw = EPWM_TBPRD / 2;
    static Uint16 PWM3_CMP_shdw = EPWM_TBPRD / 2;

    float duty = d_init;

    GpioDataRegs.GPASET.bit.GPIO27 = 1;

    while(!AdccRegs.ADCINTFLG.bit.ADCINT1) // wait for ADC2 ADCINT1 interrupt signal to go high
    {
        // we will wait till both conversions on ADC1 have finished
    }
    AdccRegs.ADCINTFLGCLR.bit.ADCINT1 = 1; // clear ADC2 ADCINT1 signal

    if (en_step)
    {
        GpioDataRegs.GPBSET.bit.GPIO53 = 1;
        duty = d_final;
    }
    else
    {
        GpioDataRegs.GPBCLEAR.bit.GPIO53 = 1;
    }

    Uint16 cmpval = (Uint16)(duty * EPWM_TBPRD);

    PWM1_CMP_shdw = cmpval;
    PWM2_CMP_shdw = cmpval;
    PWM3_CMP_shdw = cmpval;

    EPwm9Regs.CMPA.bit.CMPA = PWM1_CMP_shdw;
    EPwm10Regs.CMPA.bit.CMPA = PWM2_CMP_shdw;
    EPwm11Regs.CMPA.bit.CMPA = PWM3_CMP_shdw;

    GpioDataRegs.GPACLEAR.bit.GPIO27 = 1;

    EPwm1Regs.ETCLR.bit.INT = 1;
    PieCtrlRegs.PIEACK.all = PIEACK_GROUP3; // Clear PIE acknowledge register so that we are ready for next interrupt

}

void configure_EPWM(void)
{
    EALLOW;
    CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 0;           // stop TBCTR incrementing
    EDIS;


    // EPWM 1
    EPwm1Regs.TBCTL.bit.CLKDIV = TB_DIV1;           // CLKDIV=1     TBCLK=EPWMCLK/(HSPCLKDIV*CLKDIV)
    EPwm1Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;        // HSPCLKDIV=1
    EPwm1Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;  // Up-down mode

    EPwm1Regs.TBPRD = EPWM_TBPRD;                   // Counter period

    EPwm1Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;     // Shadow mode active for CMPA
    EPwm1Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;     // Shadow mode active for CMPB
    EPwm1Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD
    EPwm1Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD

    EPwm1Regs.CMPA.bit.CMPA = EPWM_TBPRD;        // Initial value of CMPA

    EPwm1Regs.AQCTLA.bit.CAU = AQ_CLEAR;            // Set EPWMA low on TBCTR=CMPA during up count
    EPwm1Regs.AQCTLA.bit.CAD = AQ_SET;              // Set EPWMA high on TBCTR=CMPA during down count

    EPwm1Regs.TBCTL.bit.SYNCOSEL = TB_SYNC_IN;      // Pass through SYNCIN signal (doesn't matter for what we're doing)

    EPwm1Regs.TBCTL.bit.PHSEN = TB_ENABLE;          // Enable loading of PHS register on sync event
                                                    // We will initially enable this so that we can force a sync to set up the phase shifts.
                                                    // After forcing the sync, we will disable this.
                                                    // Note: if this is enabled on EPWM1, the input X-BAR should be reconfigured to something
                                                    //       other than GPIO0 (otherwise TBCTR will be loaded with TBPHS when EPWM toggles)

    EPwm1Regs.TBPHS.bit.TBPHS = TBCTR_INIT_1;       // load TBCTR with this value on phase load event to set up phase shifts between carriers
    EPwm1Regs.TBCTR = 0;                            // start counter with value of 0 (TBCTR will be written with TBPHS when we force a sync)
    EPwm1Regs.TBCTL.bit.PHSDIR = DIR_1;             // some carriers will start counting up and some will start counting down
                                                    //    depending on their phase shifts

    EPwm1Regs.DBCTL.bit.POLSEL = DB_ACTV_HI;        // this switch is shorted, so we don't invert the B channel
    EPwm1Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;  // Enable both falling and rising edge delays
    EPwm1Regs.DBFED.bit.DBFED = EPWM_DEADTIME;      // Set delay for falling edge (DBFED)
    EPwm1Regs.DBRED.bit.DBRED = EPWM_DEADTIME;      // Set delay for rising edge (DBRED)

    // EPWM 2
    EPwm2Regs.TBCTL.bit.CLKDIV = TB_DIV1;           // CLKDIV=1     TBCLK=EPWMCLK/(HSPCLKDIV*CLKDIV)
    EPwm2Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;        // HSPCLKDIV=1
    EPwm2Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;  // Up-down mode

    EPwm2Regs.TBPRD = EPWM_TBPRD;                   // Counter period

    EPwm2Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;     // Shadow mode active for CMPA
    EPwm2Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;     // Shadow mode active for CMPB
    EPwm2Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD
    EPwm2Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD

    EPwm2Regs.CMPA.bit.CMPA = EPWM_TBPRD;        // Initial value of CMPA

    EPwm2Regs.AQCTLA.bit.CAU = AQ_CLEAR;            // Set EPWMA low on TBCTR=CMPA during up count
    EPwm2Regs.AQCTLA.bit.CAD = AQ_SET;              // Set EPWMA high on TBCTR=CMPA during down count

    EPwm2Regs.TBCTL.bit.SYNCOSEL = TB_SYNC_IN;      // Pass through SYNCIN signal (doesn't matter for what we're doing)
    EPwm2Regs.TBCTL.bit.PHSEN = TB_ENABLE;          // Enable loading of PHS register on sync event
                                                    // We will initially enable this so that we can force a sync to set up the phase shifts.
                                                    // After forcing the sync, we will disable this.
                                                    // Note: if this is enabled on EPWM1, the input X-BAR should be reconfigured to something
                                                    //       other than GPIO0 (otherwise TBCTR will be loaded with TBPHS when EPWM toggles)

    EPwm2Regs.TBPHS.bit.TBPHS = TBCTR_INIT_2;       // load TBCTR with this value on phase load event to set up phase shifts between carriers
    EPwm2Regs.TBCTR = 0;                            // start counter with value of 0 (TBCTR will be written with TBPHS when we force a sync)
    EPwm2Regs.TBCTL.bit.PHSDIR = DIR_2;             // some carriers will start counting up and some will start counting down
                                                    //    depending on their phase shifts

    EPwm2Regs.DBCTL.bit.POLSEL = DB_ACTV_HI;       // Active high complementary - EPWMxB is inverted.
    EPwm2Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;  // Enable both falling and rising edge delays
    EPwm2Regs.DBFED.bit.DBFED = EPWM_DEADTIME;      // Set delay for falling edge (DBFED)
    EPwm2Regs.DBRED.bit.DBRED = EPWM_DEADTIME;      // Set delay for rising edge (DBRED)

    // EPWM 3
    EPwm3Regs.TBCTL.bit.CLKDIV = TB_DIV1;           // CLKDIV=1     TBCLK=EPWMCLK/(HSPCLKDIV*CLKDIV)
    EPwm3Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;        // HSPCLKDIV=1
    EPwm3Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;  // Up-down mode

    EPwm3Regs.TBPRD = EPWM_TBPRD;                   // Counter period

    EPwm3Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;     // Shadow mode active for CMPA
    EPwm3Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;     // Shadow mode active for CMPB
    EPwm3Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD
    EPwm3Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD

    EPwm3Regs.CMPA.bit.CMPA = EPWM_TBPRD;        // Initial value of CMPA

    EPwm3Regs.AQCTLA.bit.CAU = AQ_CLEAR;            // Set EPWMA low on TBCTR=CMPA during up count
    EPwm3Regs.AQCTLA.bit.CAD = AQ_SET;              // Set EPWMA high on TBCTR=CMPA during down count

    EPwm3Regs.TBCTL.bit.SYNCOSEL = TB_SYNC_IN;      // Pass through SYNCIN signal (doesn't matter for what we're doing)
    EPwm3Regs.TBCTL.bit.PHSEN = TB_ENABLE;          // Enable loading of PHS register on sync event
                                                    // We will initially enable this so that we can force a sync to set up the phase shifts.
                                                    // After forcing the sync, we will disable this.
                                                    // Note: if this is enabled on EPWM1, the input X-BAR should be reconfigured to something
                                                    //       other than GPIO0 (otherwise TBCTR will be loaded with TBPHS when EPWM toggles)

    EPwm3Regs.TBPHS.bit.TBPHS = TBCTR_INIT_3;       // load TBCTR with this value on phase load event to set up phase shifts between carriers
    EPwm3Regs.TBCTR = 0;                            // start counter with value of 0 (TBCTR will be written with TBPHS when we force a sync)
    EPwm3Regs.TBCTL.bit.PHSDIR = DIR_3;             // some carriers will start counting up and some will start counting down
                                                    //    depending on their phase shifts

    EPwm3Regs.DBCTL.bit.POLSEL = DB_ACTV_HI;       // Active high complementary - EPWMxB is inverted.
    EPwm3Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;  // Enable both falling and rising edge delays
    EPwm3Regs.DBFED.bit.DBFED = EPWM_DEADTIME;      // Set delay for falling edge (DBFED)
    EPwm3Regs.DBRED.bit.DBRED = EPWM_DEADTIME;      // Set delay for rising edge (DBRED)

    // EPWM 4
    EPwm4Regs.TBCTL.bit.CLKDIV = TB_DIV1;           // CLKDIV=1     TBCLK=EPWMCLK/(HSPCLKDIV*CLKDIV)
    EPwm4Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;        // HSPCLKDIV=1
    EPwm4Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;  // Up-down mode

    EPwm4Regs.TBPRD = EPWM_TBPRD;                   // Counter period

    EPwm4Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;     // Shadow mode active for CMPA
    EPwm4Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;     // Shadow mode active for CMPB
    EPwm4Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD
    EPwm4Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD

    EPwm4Regs.CMPA.bit.CMPA = EPWM_TBPRD;        // Initial value of CMPA

    EPwm4Regs.AQCTLA.bit.CAU = AQ_CLEAR;            // Set EPWMA low on TBCTR=CMPA during up count
    EPwm4Regs.AQCTLA.bit.CAD = AQ_SET;              // Set EPWMA high on TBCTR=CMPA during down count

    EPwm4Regs.TBCTL.bit.SYNCOSEL = TB_SYNC_IN;      // Pass through SYNCIN signal (doesn't matter for what we're doing)
    EPwm4Regs.TBCTL.bit.PHSEN = TB_ENABLE;          // Enable loading of PHS register on sync event
                                                    // We will initially enable this so that we can force a sync to set up the phase shifts.
                                                    // After forcing the sync, we will disable this.
                                                    // Note: if this is enabled on EPWM1, the input X-BAR should be reconfigured to something
                                                    //       other than GPIO0 (otherwise TBCTR will be loaded with TBPHS when EPWM toggles)

    EPwm4Regs.TBPHS.bit.TBPHS = TBCTR_INIT_4;       // load TBCTR with this value on phase load event to set up phase shifts between carriers
    EPwm4Regs.TBCTR = 0;                            // start counter with value of 0 (TBCTR will be written with TBPHS when we force a sync)
    EPwm4Regs.TBCTL.bit.PHSDIR = DIR_4;             // some carriers will start counting up and some will start counting down
                                                    //    depending on their phase shifts

    EPwm4Regs.DBCTL.bit.POLSEL = DB_ACTV_HI;       // Active high complementary - EPWMxB is inverted.
    EPwm4Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;  // Enable both falling and rising edge delays
    EPwm4Regs.DBFED.bit.DBFED = EPWM_DEADTIME;      // Set delay for falling edge (DBFED)
    EPwm4Regs.DBRED.bit.DBRED = EPWM_DEADTIME;      // Set delay for rising edge (DBRED)

    // EPWM 5
    EPwm5Regs.TBCTL.bit.CLKDIV = TB_DIV1;           // CLKDIV=1     TBCLK=EPWMCLK/(HSPCLKDIV*CLKDIV)
    EPwm5Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;        // HSPCLKDIV=1
    EPwm5Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;  // Up-down mode

    EPwm5Regs.TBPRD = EPWM_TBPRD;                   // Counter period

    EPwm5Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;     // Shadow mode active for CMPA
    EPwm5Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;     // Shadow mode active for CMPB
    EPwm5Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD
    EPwm5Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD

    EPwm5Regs.CMPA.bit.CMPA = EPWM_TBPRD;        // Initial value of CMPA

    EPwm5Regs.AQCTLA.bit.CAU = AQ_CLEAR;            // Set EPWMA low on TBCTR=CMPA during up count
    EPwm5Regs.AQCTLA.bit.CAD = AQ_SET;              // Set EPWMA high on TBCTR=CMPA during down count

    EPwm5Regs.TBCTL.bit.SYNCOSEL = TB_SYNC_IN;      // Pass through SYNCIN signal (doesn't matter for what we're doing)
    EPwm5Regs.TBCTL.bit.PHSEN = TB_ENABLE;          // Enable loading of PHS register on sync event
                                                    // We will initially enable this so that we can force a sync to set up the phase shifts.
                                                    // After forcing the sync, we will disable this.
                                                    // Note: if this is enabled on EPWM1, the input X-BAR should be reconfigured to something
                                                    //       other than GPIO0 (otherwise TBCTR will be loaded with TBPHS when EPWM toggles)

    EPwm5Regs.TBPHS.bit.TBPHS = TBCTR_INIT_5;       // load TBCTR with this value on phase load event to set up phase shifts between carriers
    EPwm5Regs.TBCTR = 0;                            // start counter with value of 0 (TBCTR will be written with TBPHS when we force a sync)
    EPwm5Regs.TBCTL.bit.PHSDIR = DIR_5;             // some carriers will start counting up and some will start counting down
                                                    //    depending on their phase shifts

    EPwm5Regs.DBCTL.bit.POLSEL = DB_ACTV_HI;       // Active high complementary - EPWMxB is inverted.
    EPwm5Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;  // Enable both falling and rising edge delays
    EPwm5Regs.DBFED.bit.DBFED = EPWM_DEADTIME;      // Set delay for falling edge (DBFED)
    EPwm5Regs.DBRED.bit.DBRED = EPWM_DEADTIME;      // Set delay for rising edge (DBRED)

    // EPWM 6
    EPwm6Regs.TBCTL.bit.CLKDIV = TB_DIV1;           // CLKDIV=1     TBCLK=EPWMCLK/(HSPCLKDIV*CLKDIV)
    EPwm6Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;        // HSPCLKDIV=1
    EPwm6Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;  // Up-down mode

    EPwm6Regs.TBPRD = EPWM_TBPRD;                   // Counter period

    EPwm6Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;     // Shadow mode active for CMPA
    EPwm6Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;     // Shadow mode active for CMPB
    EPwm6Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD
    EPwm6Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD

    EPwm6Regs.CMPA.bit.CMPA = EPWM_TBPRD;        // Initial value of CMPA

    EPwm6Regs.AQCTLA.bit.CAU = AQ_CLEAR;            // Set EPWMA low on TBCTR=CMPA during up count
    EPwm6Regs.AQCTLA.bit.CAD = AQ_SET;              // Set EPWMA high on TBCTR=CMPA during down count

    EPwm6Regs.TBCTL.bit.SYNCOSEL = TB_SYNC_IN;      // Pass through SYNCIN signal (doesn't matter for what we're doing)
    EPwm6Regs.TBCTL.bit.PHSEN = TB_ENABLE;          // Enable loading of PHS register on sync event
                                                    // We will initially enable this so that we can force a sync to set up the phase shifts.
                                                    // After forcing the sync, we will disable this.
                                                    // Note: if this is enabled on EPWM1, the input X-BAR should be reconfigured to something
                                                    //       other than GPIO0 (otherwise TBCTR will be loaded with TBPHS when EPWM toggles)

    EPwm6Regs.TBPHS.bit.TBPHS = TBCTR_INIT_6;       // load TBCTR with this value on phase load event to set up phase shifts between carriers
    EPwm6Regs.TBCTR = 0;                            // start counter with value of 0 (TBCTR will be written with TBPHS when we force a sync)
    EPwm6Regs.TBCTL.bit.PHSDIR = DIR_6;             // some carriers will start counting up and some will start counting down
                                                    //    depending on their phase shifts

    EPwm6Regs.DBCTL.bit.POLSEL = DB_ACTV_HI;       // Active high complementary - EPWMxB is inverted.
    EPwm6Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;  // Enable both falling and rising edge delays
    EPwm6Regs.DBFED.bit.DBFED = EPWM_DEADTIME;      // Set delay for falling edge (DBFED)
    EPwm6Regs.DBRED.bit.DBRED = EPWM_DEADTIME;      // Set delay for rising edge (DBRED)

    // EPWM 7
    EPwm7Regs.TBCTL.bit.CLKDIV = TB_DIV1;           // CLKDIV=1     TBCLK=EPWMCLK/(HSPCLKDIV*CLKDIV)
    EPwm7Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;        // HSPCLKDIV=1
    EPwm7Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;  // Up-down mode

    EPwm7Regs.TBPRD = EPWM_TBPRD;                   // Counter period

    EPwm7Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;     // Shadow mode active for CMPA
    EPwm7Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;     // Shadow mode active for CMPB
    EPwm7Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD
    EPwm7Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD

    EPwm7Regs.CMPA.bit.CMPA = EPWM_TBPRD;        // Initial value of CMPA

    EPwm7Regs.AQCTLA.bit.CAU = AQ_CLEAR;            // Set EPWMA low on TBCTR=CMPA during up count
    EPwm7Regs.AQCTLA.bit.CAD = AQ_SET;              // Set EPWMA high on TBCTR=CMPA during down count

    EPwm7Regs.TBCTL.bit.SYNCOSEL = TB_SYNC_IN;      // Pass through SYNCIN signal (doesn't matter for what we're doing)
    EPwm7Regs.TBCTL.bit.PHSEN = TB_ENABLE;          // Enable loading of PHS register on sync event
                                                    // We will initially enable this so that we can force a sync to set up the phase shifts.
                                                    // After forcing the sync, we will disable this.
                                                    // Note: if this is enabled on EPWM1, the input X-BAR should be reconfigured to something
                                                    //       other than GPIO0 (otherwise TBCTR will be loaded with TBPHS when EPWM toggles)

    EPwm7Regs.TBPHS.bit.TBPHS = TBCTR_INIT_7;       // load TBCTR with this value on phase load event to set up phase shifts between carriers
    EPwm7Regs.TBCTR = 0;                            // start counter with value of 0 (TBCTR will be written with TBPHS when we force a sync)
    EPwm7Regs.TBCTL.bit.PHSDIR = DIR_7;             // some carriers will start counting up and some will start counting down
                                                    //    depending on their phase shifts

    EPwm7Regs.DBCTL.bit.POLSEL = DB_ACTV_HI;       // Active high complementary - EPWMxB is inverted.
    EPwm7Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;  // Enable both falling and rising edge delays
    EPwm7Regs.DBFED.bit.DBFED = EPWM_DEADTIME;      // Set delay for falling edge (DBFED)
    EPwm7Regs.DBRED.bit.DBRED = EPWM_DEADTIME;      // Set delay for rising edge (DBRED)

    // EPWM 8
    EPwm8Regs.TBCTL.bit.CLKDIV = TB_DIV1;           // CLKDIV=1     TBCLK=EPWMCLK/(HSPCLKDIV*CLKDIV)
    EPwm8Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;        // HSPCLKDIV=1
    EPwm8Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;  // Up-down mode

    EPwm8Regs.TBPRD = EPWM_TBPRD;                   // Counter period

    EPwm8Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;     // Shadow mode active for CMPA
    EPwm8Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;     // Shadow mode active for CMPB
    EPwm8Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD
    EPwm8Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD

    EPwm8Regs.CMPA.bit.CMPA = EPWM_TBPRD;        // Initial value of CMPA

    EPwm8Regs.AQCTLA.bit.CAU = AQ_CLEAR;            // Set EPWMA low on TBCTR=CMPA during up count
    EPwm8Regs.AQCTLA.bit.CAD = AQ_SET;              // Set EPWMA high on TBCTR=CMPA during down count

    EPwm8Regs.TBCTL.bit.SYNCOSEL = TB_SYNC_IN;      // Pass through SYNCIN signal (doesn't matter for what we're doing)
    EPwm8Regs.TBCTL.bit.PHSEN = TB_ENABLE;          // Enable loading of PHS register on sync event
                                                    // We will initially enable this so that we can force a sync to set up the phase shifts.
                                                    // After forcing the sync, we will disable this.
                                                    // Note: if this is enabled on EPWM1, the input X-BAR should be reconfigured to something
                                                    //       other than GPIO0 (otherwise TBCTR will be loaded with TBPHS when EPWM toggles)

    EPwm8Regs.TBPHS.bit.TBPHS = TBCTR_INIT_8;       // load TBCTR with this value on phase load event to set up phase shifts between carriers
    EPwm8Regs.TBCTR = 0;                            // start counter with value of 0 (TBCTR will be written with TBPHS when we force a sync)
    EPwm8Regs.TBCTL.bit.PHSDIR = DIR_8;             // some carriers will start counting up and some will start counting down
                                                    //    depending on their phase shifts

    EPwm8Regs.DBCTL.bit.POLSEL = DB_ACTV_HI;       // Active high complementary - EPWMxB is inverted.
    EPwm8Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;  // Enable both falling and rising edge delays
    EPwm8Regs.DBFED.bit.DBFED = EPWM_DEADTIME;      // Set delay for falling edge (DBFED)
    EPwm8Regs.DBRED.bit.DBRED = EPWM_DEADTIME;      // Set delay for rising edge (DBRED)

    // EPWM 9
    EPwm9Regs.TBCTL.bit.CLKDIV = TB_DIV1;           // CLKDIV=1     TBCLK=EPWMCLK/(HSPCLKDIV*CLKDIV)
    EPwm9Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;        // HSPCLKDIV=1
    EPwm9Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;  // Up-down mode

    EPwm9Regs.TBPRD = EPWM_TBPRD;                   // Counter period

    EPwm9Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;     // Shadow mode active for CMPA
    EPwm9Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;     // Shadow mode active for CMPB
    EPwm9Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD
    EPwm9Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD

    EPwm9Regs.CMPA.bit.CMPA = EPWM_CMP_INIT;        // Initial value of CMPA

    EPwm9Regs.AQCTLA.bit.CAU = AQ_CLEAR;            // Set EPWMA low on TBCTR=CMPA during up count
    EPwm9Regs.AQCTLA.bit.CAD = AQ_SET;              // Set EPWMA high on TBCTR=CMPA during down count

    EPwm9Regs.TBCTL.bit.SYNCOSEL = TB_SYNC_IN;      // Pass through SYNCIN signal (doesn't matter for what we're doing)
    EPwm9Regs.TBCTL.bit.PHSEN = TB_ENABLE;          // Enable loading of PHS register on sync event
                                                    // We will initially enable this so that we can force a sync to set up the phase shifts.
                                                    // After forcing the sync, we will disable this.
                                                    // Note: if this is enabled on EPWM1, the input X-BAR should be reconfigured to something
                                                    //       other than GPIO0 (otherwise TBCTR will be loaded with TBPHS when EPWM toggles)

    EPwm9Regs.TBPHS.bit.TBPHS = TBCTR_INIT_9;       // load TBCTR with this value on phase load event to set up phase shifts between carriers
    EPwm9Regs.TBCTR = 0;                            // start counter with value of 0 (TBCTR will be written with TBPHS when we force a sync)
    EPwm9Regs.TBCTL.bit.PHSDIR = DIR_9;             // some carriers will start counting up and some will start counting down
                                                    //    depending on their phase shifts

    EPwm9Regs.DBCTL.bit.POLSEL = DB_ACTV_HIC;       // Active high complementary - EPWMxB is inverted.
    EPwm9Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;  // Enable both falling and rising edge delays
    EPwm9Regs.DBFED.bit.DBFED = EPWM_DEADTIME;      // Set delay for falling edge (DBFED)
    EPwm9Regs.DBRED.bit.DBRED = EPWM_DEADTIME;      // Set delay for rising edge (DBRED)

    // EPWM 10
    EPwm10Regs.TBCTL.bit.CLKDIV = TB_DIV1;           // CLKDIV=1     TBCLK=EPWMCLK/(HSPCLKDIV*CLKDIV)
    EPwm10Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;        // HSPCLKDIV=1
    EPwm10Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;  // Up-down mode

    EPwm10Regs.TBPRD = EPWM_TBPRD;                   // Counter period

    EPwm10Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;     // Shadow mode active for CMPA
    EPwm10Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;     // Shadow mode active for CMPB
    EPwm10Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD
    EPwm10Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD

    EPwm10Regs.CMPA.bit.CMPA = EPWM_CMP_INIT;        // Initial value of CMPA

    EPwm10Regs.AQCTLA.bit.CAU = AQ_CLEAR;            // Set EPWMA low on TBCTR=CMPA during up count
    EPwm10Regs.AQCTLA.bit.CAD = AQ_SET;              // Set EPWMA high on TBCTR=CMPA during down count

    EPwm10Regs.TBCTL.bit.SYNCOSEL = TB_SYNC_IN;      // Pass through SYNCIN signal (doesn't matter for what we're doing)
    EPwm10Regs.TBCTL.bit.PHSEN = TB_ENABLE;          // Enable loading of PHS register on sync event
                                                     // We will initially enable this so that we can force a sync to set up the phase shifts.
                                                     // After forcing the sync, we will disable this.
                                                     // Note: if this is enabled on EPWM1, the input X-BAR should be reconfigured to something
                                                     //       other than GPIO0 (otherwise TBCTR will be loaded with TBPHS when EPWM toggles)

    EPwm10Regs.TBPHS.bit.TBPHS = TBCTR_INIT_10;      // load TBCTR with this value on phase load event to set up phase shifts between carriers
    EPwm10Regs.TBCTR = 0;                            // start counter with value of 0 (TBCTR will be written with TBPHS when we force a sync)
    EPwm10Regs.TBCTL.bit.PHSDIR = DIR_10;            // some carriers will start counting up and some will start counting down
                                                     //    depending on their phase shifts

    EPwm10Regs.DBCTL.bit.POLSEL = DB_ACTV_HIC;       // Active high complementary - EPWMxB is inverted.
    EPwm10Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;  // Enable both falling and rising edge delays
    EPwm10Regs.DBFED.bit.DBFED = EPWM_DEADTIME;      // Set delay for falling edge (DBFED)
    EPwm10Regs.DBRED.bit.DBRED = EPWM_DEADTIME;      // Set delay for rising edge (DBRED)

    // EPWM 11
    EPwm11Regs.TBCTL.bit.CLKDIV = TB_DIV1;           // CLKDIV=1     TBCLK=EPWMCLK/(HSPCLKDIV*CLKDIV)
    EPwm11Regs.TBCTL.bit.HSPCLKDIV = TB_DIV1;        // HSPCLKDIV=1
    EPwm11Regs.TBCTL.bit.CTRMODE = TB_COUNT_UPDOWN;  // Up-down mode

    EPwm11Regs.TBPRD = EPWM_TBPRD;                   // Counter period

    EPwm11Regs.CMPCTL.bit.SHDWAMODE = CC_SHADOW;     // Shadow mode active for CMPA
    EPwm11Regs.CMPCTL.bit.SHDWBMODE = CC_SHADOW;     // Shadow mode active for CMPB
    EPwm11Regs.CMPCTL.bit.LOADAMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD
    EPwm11Regs.CMPCTL.bit.LOADBMODE = CC_CTR_ZERO;   // Load on TBCTR=TBPRD

    EPwm11Regs.CMPA.bit.CMPA = EPWM_CMP_INIT;        // Initial value of CMPA

    EPwm11Regs.AQCTLA.bit.CAU = AQ_CLEAR;            // Set EPWMA low on TBCTR=CMPA during up count
    EPwm11Regs.AQCTLA.bit.CAD = AQ_SET;              // Set EPWMA high on TBCTR=CMPA during down count

    EPwm11Regs.TBCTL.bit.SYNCOSEL = TB_SYNC_IN;      // Pass through SYNCIN signal (doesn't matter for what we're doing)
    EPwm11Regs.TBCTL.bit.PHSEN = TB_ENABLE;          // Enable loading of PHS register on sync event
                                                     // We will initially enable this so that we can force a sync to set up the phase shifts.
                                                     // After forcing the sync, we will disable this.
                                                     // Note: if this is enabled on EPWM1, the input X-BAR should be reconfigured to something
                                                     //       other than GPIO0 (otherwise TBCTR will be loaded with TBPHS when EPWM toggles)

    EPwm11Regs.TBPHS.bit.TBPHS = TBCTR_INIT_11;      // load TBCTR with this value on phase load event to set up phase shifts between carriers
    EPwm11Regs.TBCTR = 0;                            // start counter with value of 0 (TBCTR will be written with TBPHS when we force a sync)
    EPwm11Regs.TBCTL.bit.PHSDIR = DIR_11;            // some carriers will start counting up and some will start counting down
                                                     //    depending on their phase shifts

    EPwm11Regs.DBCTL.bit.POLSEL = DB_ACTV_HIC;       // Active high complementary - EPWMxB is inverted.
    EPwm11Regs.DBCTL.bit.OUT_MODE = DB_FULL_ENABLE;  // Enable both falling and rising edge delays
    EPwm11Regs.DBFED.bit.DBFED = EPWM_DEADTIME;      // Set delay for falling edge (DBFED)
    EPwm11Regs.DBRED.bit.DBRED = EPWM_DEADTIME;      // Set delay for rising edge (DBRED)

    // setting the trip zone registers
    // we will use the trip zone to short out the upper 8 pairs of switches to use the 12-level hardware as a 4-Level
    EALLOW;

    EPwm9Regs.TZCTL.bit.TZA = TZ_FORCE_LO;          // Set EPWMA high when the trip-zone is triggered
    EPwm9Regs.TZCTL.bit.TZB = TZ_FORCE_LO;          // Set EPWMB high when the trip-zone is triggered
    EPwm9Regs.TZSEL.bit.OSHT1 = 1;                  // Enable TZ1 as a one-shot TZ event source
    EPwm9Regs.TZEINT.bit.OST = 0;                   // Enable TZ interrupt
    EPwm9Regs.TZCLR.bit.OST = 1;                    // Clear OST flag

    EPwm10Regs.TZCTL.bit.TZA = TZ_FORCE_LO;          // Set EPWMA high when the trip-zone is triggered
    EPwm10Regs.TZCTL.bit.TZB = TZ_FORCE_LO;          // Set EPWMB high when the trip-zone is triggered
    EPwm10Regs.TZSEL.bit.OSHT1 = 1;                  // Enable TZ1 as a one-shot TZ event source
    EPwm10Regs.TZEINT.bit.OST = 0;                   // Enable TZ interrupt
    EPwm10Regs.TZCLR.bit.OST = 1;                    // Clear OST flag

    EPwm11Regs.TZCTL.bit.TZA = TZ_FORCE_LO;          // Set EPWMA high when the trip-zone is triggered
    EPwm11Regs.TZCTL.bit.TZB = TZ_FORCE_LO;          // Set EPWMB high when the trip-zone is triggered
    EPwm11Regs.TZSEL.bit.OSHT1 = 1;                  // Enable TZ1 as a one-shot TZ event source
    EPwm11Regs.TZEINT.bit.OST = 0;                   // Enable TZ interrupt
    EPwm11Regs.TZCLR.bit.OST = 1;                    // Clear OST flag

    EDIS;

    // configure EPWMs to load CMPA and CMPB only when a sync is received

    EPwm9Regs.CMPCTL.bit.LOADAMODE = 0;
    EPwm9Regs.CMPCTL.bit.LOADASYNC = 2;

    EPwm9Regs.CMPCTL.bit.LOADBMODE = 0;
    EPwm9Regs.CMPCTL.bit.LOADBSYNC = 2;

    EPwm10Regs.CMPCTL.bit.LOADAMODE = 0;
    EPwm10Regs.CMPCTL.bit.LOADASYNC = 2;

    EPwm10Regs.CMPCTL.bit.LOADBMODE = 0;
    EPwm10Regs.CMPCTL.bit.LOADBSYNC = 2;

    EPwm11Regs.CMPCTL.bit.LOADAMODE = 0;
    EPwm11Regs.CMPCTL.bit.LOADASYNC = 2;

    EPwm11Regs.CMPCTL.bit.LOADBMODE = 0;
    EPwm11Regs.CMPCTL.bit.LOADBSYNC = 2;

    // SOC trigger for sampling
    EPwm1Regs.ETSEL.bit.SOCASEL = ET_CTR_ZERO;      // ADCSOCA on TBCTR=TBPRD
    EPwm1Regs.ETPS.bit.SOCAPRD = 1;                 // Generate SOCA on 1st event
    EPwm1Regs.ETSEL.bit.SOCAEN = 1;                 // Enable SOCA generation

    // set up PWM1 interrupt for main control loop
    EPwm1Regs.ETSEL.bit.INTSEL = ET_CTR_ZERO;      // Select INT on TBPRD event
    EPwm1Regs.ETSEL.bit.INTEN = 0;                // Disable INT
    EPwm1Regs.ETPS.bit.INTPRD = 1;           // Generate INT on 1st event
    EPwm1Regs.ETCLR.bit.INT = 1;             // Clear interrupt flag

    /*
     * EPWM signals propagate via a daisy chain depicted in TRM section 15.4.3.3
     * According to the TRM, the longest daisy chain propagation path must not exceed four EPWM modules.
     * If necessary, the sync chain can be modified with appropriate settings of bits in SyncSocRegs.SYNCSELECT
     *
     * However, in practice, it is unclear how many modules can be chained before the syncin->syncout pulse propagation delay is]
     * greater than 1 EPWMCLK.
     *
     * To set sync phase shifts properly, we need to modify the default EXTSYNC1 setup such that it does not interfere with the SWFSYNC
     * that we use to sync all the modules.
     *
     * By default, EXTSYNC1 is connected to INPUT5 on the Input XBAR, and INPUT5 is connected to GPIO0 which we are using for EPWM1A.
     *
     * SWFSYNC and the synchronization input signal EPWMxSYNCI both trigger EPWM sync (TRM section 15.4.3.3 and Fig. 15-5).
     * Leaving the default value for EXTSYNC1 can cause improper phase alignment if a rising edge on EPWM1 occurs before PHSEN is disabled
     * (for example if this routine is interrupted between the SWFSYNC = 1 and PHSEN = 0 operations). To ensure reliable phase alignment,
     * we will connect INPUT5 on the Input XBAR to an unused GPIO pin such that the only synchronizing signal is the SWFSYNC.
     */

    /*
     * EPWM sync triggers are latched until TBCTR is loaded with the value of TBPHS on the next TBCLK edge (TRM section 15.4.3.3)
     * Using SWFSYNC to synchronize a chain while TBCLKs are incrementing will result in a delay of TBCLK between the first EPWM module
     * in the chain and the subsequent ones, as the SYNCOUT signal only results in a synchronization at the next TBCLK edge.
     *
     * Therefore, when possible, the SWFSYNC should be set (and consequently latched) before the TBCLKSYNC bit is set to 1.
     */

    EALLOW;
    InputXbarRegs.INPUT1SELECT = 26;              // route INPUT1 to GPIO 26 that we will use exclusively for tripping PWMs
    InputXbarRegs.INPUT5SELECT = 25;              // route INPUT5 to an unused GPIO pin to not interfere with EXTSYNC1
    EDIS;

    EALLOW;

    EPwm1Regs.TBCTL.bit.SYNCOSEL = 3; // SYNCOSEL for EPWM1 on CTR = 0
    EPwm1Regs.TBCTL2.bit.SYNCOSELX = 1;
    EPwm1Regs.CMPC = 0;

    SyncSocRegs.SYNCSELECT.bit.EPWM4SYNCIN = 0; // these are the default values
    SyncSocRegs.SYNCSELECT.bit.EPWM7SYNCIN = 0;
    SyncSocRegs.SYNCSELECT.bit.EPWM10SYNCIN = 0;
    SyncSocRegs.SYNCSELECT.bit.SYNCOUT = 0;

    // see TRM 9.4 for where these settings come from
    // we will use Output 1 on the Output XBAR
    OutputXbarRegs.OUTPUT1MUX0TO15CFG.bit.MUX14 = 3; // select EXTSYNCOUT in its corresponding mux (see TRM Table 9.4)
    OutputXbarRegs.OUTPUT1MUXENABLE.bit.MUX14 = 1;   // enable OUTPUT1

    EDIS;

    EALLOW;
    EPwm1Regs.TBCTL.bit.SWFSYNC = 1;                // force synchronization of EPWMs in chain starting with EPWM1
    CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 1;           // start all TBCTR incrementing

//    EPwm1Regs.TBCTL.bit.PHSEN = TB_DISABLE;         // turn off PHSEN for all EPWMs to prevent subsequent phase loading
//    EPwm2Regs.TBCTL.bit.PHSEN = TB_DISABLE;
//    EPwm3Regs.TBCTL.bit.PHSEN = TB_DISABLE;
//    EPwm4Regs.TBCTL.bit.PHSEN = TB_DISABLE;
//    EPwm5Regs.TBCTL.bit.PHSEN = TB_DISABLE;
//    EPwm6Regs.TBCTL.bit.PHSEN = TB_DISABLE;
//    EPwm7Regs.TBCTL.bit.PHSEN = TB_DISABLE;
//    EPwm8Regs.TBCTL.bit.PHSEN = TB_DISABLE;
//    EPwm9Regs.TBCTL.bit.PHSEN = TB_DISABLE;
//    EPwm10Regs.TBCTL.bit.PHSEN = TB_DISABLE;
//    EPwm11Regs.TBCTL.bit.PHSEN = TB_DISABLE;


    EDIS;







}

void configure_ADC(void)
{
    EALLOW;

    // Note: Maximum value of ADCCLK is 50 MHz according to F28379D datasheet, so we have to set prescaler accordingly

#ifdef LAUNCHPAD
    AdcbRegs.ADCCTL2.bit.PRESCALE = 2; // use prescaler to divide by 2 for LaunchPad
    AdccRegs.ADCCTL2.bit.PRESCALE = 2;
    AdcdRegs.ADCCTL2.bit.PRESCALE = 2;
#else
    AdcbRegs.ADCCTL2.bit.PRESCALE = 6; // use prescaler to divide by 4 for ControlCard
    AdccRegs.ADCCTL2.bit.PRESCALE = 6;
    AdcdRegs.ADCCTL2.bit.PRESCALE = 6;
#endif

    // ADCB setup:

    //   No interrupts are triggered for ADCB

    // Note: TI recommends using the AdcSetMode driver function for configuring the ADC resolution and signal polarity
    //       The function also calls relevant calibration functions

    AdcSetMode(ADC_ADCB, ADC_RESOLUTION_12BIT, ADC_SIGNALMODE_SINGLE);  // Set the resolution and signal mode for a given ADC
    AdcbRegs.ADCCTL1.bit.INTPULSEPOS = 1;                               // Set pulse positions to late
    AdcbRegs.ADCCTL1.bit.ADCPWDNZ = 1;                                  // Power up the ADC
    DELAY_US(1000);                                                     // Delay for 1ms to allow ADC time to power up

    AdcbRegs.ADCSOC0CTL.bit.CHSEL = 2;                                  // SOC0 will convert ADCINB2, result is ADCRESULT0
    AdcbRegs.ADCSOC0CTL.bit.ACQPS = ADCACQ_12Bit;                       // Acquisition window is ACQPS+1 SYSCLK cycles
    AdcbRegs.ADCSOC0CTL.bit.TRIGSEL = 5;                                // Trigger on EPWM1 ADCSOCA

    AdcbRegs.ADCSOC1CTL.bit.CHSEL = 3;                                  // SOC1 will convert ADCINB3, result is ADCRESULT1
    AdcbRegs.ADCSOC1CTL.bit.ACQPS = ADCACQ_12Bit;                       // Acquisition window is ACQPS+1 SYSCLK cycles
    AdcbRegs.ADCSOC1CTL.bit.TRIGSEL = 5;                                // Trigger on EPWM1 ADCSOCA

    AdcbRegs.ADCSOCPRICTL.bit.SOCPRIORITY = 0;                          // don't need any of the SOCs to be high priority
                                                                        // we will just use the default round robin functionality to sample all the channels

    AdcbRegs.ADCINTSEL1N2.bit.INT1SEL = 1;                              // EOC1 will set ADCINT1 flag
    AdcbRegs.ADCINTSEL1N2.bit.INT1E = 0;                                // Disable interrupt
    AdcbRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;                              // Make sure INT1 flag is cleared


    // ADCC converts:

    // We will trigger the current control interrupt after the conversion of the inductor current

    AdcSetMode(ADC_ADCC, ADC_RESOLUTION_12BIT, ADC_SIGNALMODE_SINGLE);  // Set the resolution and signal mode for a given ADC
    AdccRegs.ADCCTL1.bit.INTPULSEPOS = 1;                               // Set pulse positions to late
    AdccRegs.ADCCTL1.bit.ADCPWDNZ = 1;                                  // Power up the ADC
    DELAY_US(1000);                                                     // Delay for 1ms to allow ADC time to power up

    AdccRegs.ADCSOC0CTL.bit.CHSEL = 2;                                  // SOC0 will convert ADCINC2, result is ADCRESULT0
    AdccRegs.ADCSOC0CTL.bit.ACQPS = ADCACQ_12Bit;                       // Acquisition window is ACQPS+1 SYSCLK cycles
    AdccRegs.ADCSOC0CTL.bit.TRIGSEL = 5;                                // Trigger on ePWM1 SOCA

    AdccRegs.ADCSOC1CTL.bit.CHSEL = 3;                                  // SOC1 will convert ADCINC3, result is ADCRESULT0
    AdccRegs.ADCSOC1CTL.bit.ACQPS = ADCACQ_12Bit;                       // Acquisition window is ACQPS+1 SYSCLK cycles
    AdccRegs.ADCSOC1CTL.bit.TRIGSEL = 5;                                // Trigger on ePWM1 SOCA

    AdccRegs.ADCSOCPRICTL.bit.SOCPRIORITY = 1;                          // SOC0 is high priority

    AdccRegs.ADCINTSEL1N2.bit.INT1SEL = 1;                              // EOC1 will set ADCINT1 flag
    AdccRegs.ADCINTSEL1N2.bit.INT1E = 1;                                // Enable interrupt
    AdccRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;                              // Make sure INT1 flag is cleared


// ADCD converts:

    AdcSetMode(ADC_ADCD, ADC_RESOLUTION_12BIT, ADC_SIGNALMODE_SINGLE);  // Set the resolution and signal mode for a given ADC
    AdcdRegs.ADCCTL1.bit.INTPULSEPOS = 1;                               // Set pulse positions to late
    AdcdRegs.ADCCTL1.bit.ADCPWDNZ = 1;                                  // Power up the ADC
    DELAY_US(1000);                                                     // Delay for 1ms to allow ADC time to power up

    AdcdRegs.ADCSOC0CTL.bit.CHSEL = 0;                                  // SOC0 will convert ADCIND0, result is ADCRESULT0
    AdcdRegs.ADCSOC0CTL.bit.ACQPS = ADCACQ_12Bit;                       // Acquisition window is ACQPS+1 SYSCLK cycles
    AdcdRegs.ADCSOC0CTL.bit.TRIGSEL = 5;                                // Trigger on ePWM1 SOCA

    AdcdRegs.ADCSOCPRICTL.bit.SOCPRIORITY = 1;                          // SOC0 is high priority

    AdcdRegs.ADCINTSEL1N2.bit.INT1SEL = 0;                              // EOC0 will set ADCINT1 flag
    AdcdRegs.ADCINTSEL1N2.bit.INT1E = 0;                                // Disable interrupt
    AdcdRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;                              // Make sure INT1 flag is cleared

    EDIS;
}

void configure_GPIO(void)
{
    EALLOW;

    // GPIOs for EPWM
    // see TRM Table 8.8 for configuration

    // Configure as EPWM1A output (leading cell carrier - high side)
    GpioCtrlRegs.GPADIR.bit.GPIO0 = 1;          // Configure as output
    GpioCtrlRegs.GPAMUX1.bit.GPIO0 = 1;         // Mux to ePWM1A
    GpioDataRegs.GPACLEAR.bit.GPIO0 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO0 = 1;

    // Configure as EPWM1B output (leading cell carrier - low side)
    GpioCtrlRegs.GPADIR.bit.GPIO1 = 1;          // Configure as output
    GpioCtrlRegs.GPAMUX1.bit.GPIO1 = 1;         // Mux to ePWM1B
    GpioDataRegs.GPACLEAR.bit.GPIO1 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO1 = 1;

    // Configure as EPWM2A output (#2 cell - high side)
    GpioCtrlRegs.GPADIR.bit.GPIO2 = 1;          // Configure as output
    GpioCtrlRegs.GPAMUX1.bit.GPIO2 = 1;         // Mux to ePWM2A
    GpioDataRegs.GPACLEAR.bit.GPIO2 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO2 = 1;

    // Configure as EPWM2A output (#2 cell - low side)
    GpioCtrlRegs.GPADIR.bit.GPIO3 = 1;          // Configure as output
    GpioCtrlRegs.GPAMUX1.bit.GPIO3 = 1;         // Mux to ePWM2B
    GpioDataRegs.GPACLEAR.bit.GPIO3 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO3 = 1;

    // Configure as EPWM3A output (#3 cell - high side)
    GpioCtrlRegs.GPADIR.bit.GPIO4 = 1;          // Configure as output
    GpioCtrlRegs.GPAMUX1.bit.GPIO4 = 1;         // Mux to ePWM3A
    GpioDataRegs.GPACLEAR.bit.GPIO4 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO4 = 1;

    // Configure as EPWM3B output (#3 cell - low side)
    GpioCtrlRegs.GPADIR.bit.GPIO5 = 1;          // Configure as output
    GpioCtrlRegs.GPAMUX1.bit.GPIO5 = 1;         // Mux to ePWM3B
    GpioDataRegs.GPACLEAR.bit.GPIO5 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO5 = 1;

    // Configure as EPWM4A output (#4 cell - high side)
    GpioCtrlRegs.GPADIR.bit.GPIO6 = 1;          // Configure as output
    GpioCtrlRegs.GPAMUX1.bit.GPIO6 = 1;         // Mux to ePWM4A
    GpioDataRegs.GPACLEAR.bit.GPIO6 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO6 = 1;

    // Configure as EPWM4B output (#4 cell - low side)
    GpioCtrlRegs.GPADIR.bit.GPIO7 = 1;          // Configure as output
    GpioCtrlRegs.GPAMUX1.bit.GPIO7 = 1;         // Mux to ePWM4B
    GpioDataRegs.GPACLEAR.bit.GPIO7 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO7 = 1;

    // Configure as EPWM5A output (#5 cell - high side)
    GpioCtrlRegs.GPADIR.bit.GPIO8 = 1;          // Configure as output
    GpioCtrlRegs.GPAMUX1.bit.GPIO8 = 1;         // Mux to ePWM5A
    GpioDataRegs.GPACLEAR.bit.GPIO8 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO8 = 1;

    // Configure as EPWM5B output (#5 cell - low side)
    GpioCtrlRegs.GPADIR.bit.GPIO9 = 1;          // Configure as output
    GpioCtrlRegs.GPAMUX1.bit.GPIO9 = 1;         // Mux to ePWM5B
    GpioDataRegs.GPACLEAR.bit.GPIO9 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO9 = 1;

    // Configure as EPWM6A output (#6 cell - high side)
    GpioCtrlRegs.GPADIR.bit.GPIO10 = 1;          // Configure as output
    GpioCtrlRegs.GPAMUX1.bit.GPIO10 = 1;         // Mux to ePWM6A
    GpioDataRegs.GPACLEAR.bit.GPIO10 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO10 = 1;

    // Configure as EPWM6B output (#6 cell - low side)
    GpioCtrlRegs.GPADIR.bit.GPIO11 = 1;          // Configure as output
    GpioCtrlRegs.GPAMUX1.bit.GPIO11 = 1;         // Mux to ePWM6B
    GpioDataRegs.GPACLEAR.bit.GPIO11 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO11 = 1;

    // Configure as EPWM7A output (#7 cell - high side)
    GpioCtrlRegs.GPADIR.bit.GPIO12 = 1;          // Configure as output
    GpioCtrlRegs.GPAMUX1.bit.GPIO12 = 1;         // Mux to ePWM7A
    GpioDataRegs.GPACLEAR.bit.GPIO12 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO12 = 1;

    // Configure as EPWM7B output (#7 cell - low side)
    GpioCtrlRegs.GPADIR.bit.GPIO13 = 1;          // Configure as output
    GpioCtrlRegs.GPAMUX1.bit.GPIO13 = 1;         // Mux to ePWM7B
    GpioDataRegs.GPACLEAR.bit.GPIO13 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO13 = 1;

    // Configure as EPWM8A output (#8 cell - high side)
    GpioCtrlRegs.GPADIR.bit.GPIO14 = 1;          // Configure as output
    GpioCtrlRegs.GPAMUX1.bit.GPIO14 = 1;         // Mux to ePWM8A
    GpioDataRegs.GPACLEAR.bit.GPIO14 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO14 = 1;

    // Configure as EPWM8B output (#8 cell - low side)
    GpioCtrlRegs.GPADIR.bit.GPIO15 = 1;          // Configure as output
    GpioCtrlRegs.GPAMUX1.bit.GPIO15 = 1;         // Mux to ePWM8B
    GpioDataRegs.GPACLEAR.bit.GPIO15 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO15 = 1;

    // Configure as EPWM9A output (#9 cell - high side)
    GpioCtrlRegs.GPADIR.bit.GPIO16 = 1;          // Configure as output
    GpioCtrlRegs.GPAGMUX2.bit.GPIO16 = 1;         //Mux to ePWM9A
    GpioCtrlRegs.GPAMUX2.bit.GPIO16 = 1;         // Mux to ePWM9A
    GpioDataRegs.GPACLEAR.bit.GPIO16 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO16 = 1;

    // Configure as EPWM9B output (#9 cell - low side)
    GpioCtrlRegs.GPADIR.bit.GPIO17 = 1;          // Configure as output
    GpioCtrlRegs.GPAGMUX2.bit.GPIO17 = 1;         //Mux to ePWM9B
    GpioCtrlRegs.GPAMUX2.bit.GPIO17 = 1;         // Mux to ePWM9B
    GpioDataRegs.GPACLEAR.bit.GPIO17 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO17 = 1;

    // Configure as EPWM10A output (#10 cell - high side)
    GpioCtrlRegs.GPADIR.bit.GPIO18 = 1;          // Configure as output
    GpioCtrlRegs.GPAGMUX2.bit.GPIO18 = 1;         //Mux to ePWM10A
    GpioCtrlRegs.GPAMUX2.bit.GPIO18 = 1;         // Mux to ePWM10A
    GpioDataRegs.GPACLEAR.bit.GPIO18 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO18 = 1;

    // Configure as EPWM10B output (#9 cell - low side)
    GpioCtrlRegs.GPADIR.bit.GPIO19 = 1;          // Configure as output
    GpioCtrlRegs.GPAGMUX2.bit.GPIO19 = 1;         //Mux to ePWM10B
    GpioCtrlRegs.GPAMUX2.bit.GPIO19 = 1;         // Mux to ePWM10B
    GpioDataRegs.GPACLEAR.bit.GPIO19 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO19 = 1;

    // Configure as EPWM11A output (#11 cell - high side)
    GpioCtrlRegs.GPADIR.bit.GPIO20 = 1;          // Configure as output
    GpioCtrlRegs.GPAGMUX2.bit.GPIO20 = 1;         //Mux to ePWM11A
    GpioCtrlRegs.GPAMUX2.bit.GPIO20 = 1;         // Mux to ePWM11A
    GpioDataRegs.GPACLEAR.bit.GPIO20 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO20 = 1;

    // Configure as EPWM11B output (#11 cell - low side)
    GpioCtrlRegs.GPADIR.bit.GPIO21 = 1;          // Configure as output
    GpioCtrlRegs.GPAGMUX2.bit.GPIO21 = 1;         //Mux to ePWM11B
    GpioCtrlRegs.GPAMUX2.bit.GPIO21 = 1;         // Mux to ePWM11B
    GpioDataRegs.GPACLEAR.bit.GPIO21 = 1; // Set low, until otherwise specified by control algorithm
    GpioCtrlRegs.GPAPUD.bit.GPIO21 = 1;

    // Configure as ordinary GPIO for tripping PWMs
    GpioCtrlRegs.GPADIR.bit.GPIO25 = 1;   // Configure as output
    GpioCtrlRegs.GPAMUX2.bit.GPIO25 = 0;  // leave configured as GPIO
    GpioDataRegs.GPACLEAR.bit.GPIO25 = 1; // clear

    // Configure as ordinary GPIO for tripping PWMs
    GpioCtrlRegs.GPADIR.bit.GPIO26 = 1;   // Configure as output
    GpioCtrlRegs.GPAMUX2.bit.GPIO26 = 0;  // leave configured as GPIO
    GpioDataRegs.GPACLEAR.bit.GPIO26 = 1; // clear

    // Configure as ordinary GPIO for debugging
    // CSB pin on 12-level board
    GpioCtrlRegs.GPADIR.bit.GPIO27 = 1;   // Configure as output
    GpioCtrlRegs.GPAMUX2.bit.GPIO27 = 0;  // leave configured as GPIO
    GpioDataRegs.GPACLEAR.bit.GPIO27 = 1; // clear

    // CS pin on 12-level board
    GpioCtrlRegs.GPBDIR.bit.GPIO53 = 1;   // Configure as output
    GpioCtrlRegs.GPBMUX2.bit.GPIO53 = 0;  // leave configured as GPIO
    GpioDataRegs.GPBCLEAR.bit.GPIO53 = 1; // clear

    // Configure GPIO24 to output EPWM EXTSYNC via OUTPUTXBAR1
    GpioCtrlRegs.GPADIR.bit.GPIO24 = 1;   // Configure as output
    GpioCtrlRegs.GPAMUX2.bit.GPIO24 = 1;  // Connect to OUTPUTXBAR1

    EDIS;
}

void configure_DAC(void)
{

    EALLOW;

    // DACOUTA

    //Use VREFHI (ADC) as the reference for DAC
    DacaRegs.DACCTL.bit.DACREFSEL = 1;
    // Load on next sysclk
    DacaRegs.DACCTL.bit.LOADMODE = 0;
    //Set DAC to some initial value
    DacaRegs.DACVALS.bit.DACVALS = 0;

    // DACOUTB
    //Use VREFHI (ADC) as the reference for DAC
    DacbRegs.DACCTL.bit.DACREFSEL = 1;
    // Load on next sysclk
    DacbRegs.DACCTL.bit.LOADMODE = 0;
    //Set DAC to some initial value
    DacbRegs.DACVALS.bit.DACVALS = 0;

    // DACOUTC
    //Use VREFHI (ADC) as the reference for DAC
    DaccRegs.DACCTL.bit.DACREFSEL = 1;
    // Load on next sysclk
    DaccRegs.DACCTL.bit.LOADMODE = 0;
    //Set DAC to some initial value
    DaccRegs.DACVALS.bit.DACVALS = 0;

    //Enable DAC outputs
    DacaRegs.DACOUTEN.bit.DACOUTEN = 1;
    DacbRegs.DACOUTEN.bit.DACOUTEN = 1;
    DaccRegs.DACOUTEN.bit.DACOUTEN = 1;

    EDIS;
}


