//#############################################################################
//
// FILE:   f29x_cpu1_echoback.c
//
// TITLE:  EtherCAT Echoback Reference Solution for F29X CPU1
//
// This reference solution demonstrates usage of the EtherCAT stack using
// CAN-over-EtherCAT (CoE) mailbox protocol to perform a basic loopback of
// data received from the EtherCAT master and then transmitted back to the
// master.
//
// This file initializes the device and EtherCAT hardware before starting
// the EtherCAT state machine handled by the slave stack code files. The
// slave stack code will call these "APPL" functions during state transitions
// as well as while running its main loop to copy data from the EtherCAT
// RAM to device RAM and from device RAM to EtherCAT RAM.
//
// Important: Before running this example, refer to the EtherCAT Slave
//            Controller User Guide in C2000Ware for the proper
//            setup/execution procedure of this example
//
// Note
//  Slave Stack Code (SSC) tool must be used to generate the stack files
//  required by this solution.
//
// External Connections
//  - The controlCARD RJ45 port 0 is connected to PC running TwinCAT master
//  - If distributed clocks enabled, connect and observe SYNC0/1 signals on
//    GPIO127 and GPIO128
//
// Watch Variables
//  - Switches0x6000 - 8-Bit switches data to send from slave to master
//  - LEDS0x7000 - 8-Bit LEDs data received from master
//  - DataToMaster0x6010 - 32-Bit general data to send from slave to master
//  - DatafromMaster0x7010 - 32-Bit general data received from master
//  - TargetModeResponse0x6012 - 16-Bit mode data to send from slave to master
//  - TargetMode0x7012 - 16-Bit mode data received from master
//  - TargetSpeedPosFeedback0x6014 - 32-Bit speed/position data to send from
//                                   slave to master
//  - TargetSpeedPosReq0x7014 - 32-Bit speed/position data received from master
//
//#############################################################################
// $TI Release: F29X EtherCAT Software v2.01.00.00 $
// $Release Date: August 31 2020 $
// $Copyright: Copyright (C) 2020 Texas Instruments Incorporated -
//             http://www.ti.com/ ALL RIGHTS RESERVED $
//#############################################################################

//
// Included Files
//
#include <stdint.h>
#include "ecat_def.h"
#include "applInterface.h"

#define _F29X_ECHOBACK_ 1
#include "f29x_cpu1_echoback.h"
#undef _F29X_ECHOBACK_

//
// PDO_ResetOutputs - Resets the Output data to zero
//
void PDO_ResetOutputs(void)
{
    LEDS0x7000.LED1 = 0x0U;
    LEDS0x7000.LED2 = 0x0U;
    LEDS0x7000.LED3 = 0x0U;
    LEDS0x7000.LED4 = 0x0U;
    LEDS0x7000.LED5 = 0x0U;
    LEDS0x7000.LED6 = 0x0U;
    LEDS0x7000.LED7 = 0x0U;
    LEDS0x7000.LED8 = 0x0U;

    DatafromMaster0x7010.DatafromMaster = 0x0UL;
    TargetMode0x7012.Mode = 0x0U;
    TargetSpeedPosReq0x7014.SpeedPosReq = 0x0UL;
}

//
// APPL_AckErrorInd - Called when error state was acknowledged by the master
//
void APPL_AckErrorInd(UINT16 stateTrans)
{
    //
    // No implementation for this application
    //
}

//
// APPL_StartMailboxHandler - Informs application of state transition from INIT
//                            to PREOP
//
UINT16 APPL_StartMailboxHandler(void)
{
    return(ALSTATUSCODE_NOERROR);
}

//
// APPL_StopMailboxHandler - Informs application of state transition from PREOP
//                           to INIT
//
UINT16 APPL_StopMailboxHandler(void)
{
    return(ALSTATUSCODE_NOERROR);
}

//
// APPL_StartInputHandler - Informs application of state transition from PREOP
//                          to SAFEOP
//
UINT16 APPL_StartInputHandler(UINT16 *pIntMask)
{
    return(ALSTATUSCODE_NOERROR);
}

//
// APPL_StopInputHandler - Informs application of state transition from SAFEOP
//                         PREOP
//
UINT16 APPL_StopInputHandler(void)
{
    return(ALSTATUSCODE_NOERROR);
}

//
// APPL_StartOutputHandler - Informs application of state transition from SAFEOP
//                           to OP
//
UINT16 APPL_StartOutputHandler(void)
{
    return(ALSTATUSCODE_NOERROR);
}

//
// APPL_StopOutputHandler - Informs application of state transition from OP to
//                          SAFEOP
//
UINT16 APPL_StopOutputHandler(void)
{
    //
    // Reset output data to zero
    //
    PDO_ResetOutputs();

    return(ALSTATUSCODE_NOERROR);
}

//
// APPL_GenerateMapping - Calculates the input and output process data sizes
//
UINT16 APPL_GenerateMapping(UINT16 *pInputSize, UINT16 *pOutputSize)
{
    UINT16 result = ALSTATUSCODE_NOERROR;
    UINT16 inputSize = 0U;
    UINT16 outputSize = 0U;
    UINT16 PDOAssignEntryCnt = 0U;
    OBJCONST TOBJECT OBJMEM * pPDO = NULL;
    UINT16 PDOSubindex0 = 0U;
    UINT16 *pPDOEntry = NULL;
    UINT16 PDOEntryCnt = 0U;

    //
    // Scan Object 0x1C12 (SyncManager 2)
    // (Sums up sizes of all SyncManager assigned RxPDOs subindexes)
    //
    for(PDOAssignEntryCnt = 0U; PDOAssignEntryCnt < sRxPDOassign.u16SubIndex0;
        PDOAssignEntryCnt++)
    {
        //
        // Get object handle for specified RxPDO assigned to SyncManager
        //
        pPDO = OBJ_GetObjectHandle(sRxPDOassign.aEntries[PDOAssignEntryCnt]);

        //
        // Confirm that Object isn't NULL before continuing
        //
        if(pPDO != NULL)
        {
            //
            // Get number of Object Entry subindexes
            //
            PDOSubindex0 = *((UINT16 *)pPDO->pVarPtr);

            //
            // Add up sizes of all the Object Entry subindexes
            //
            for(PDOEntryCnt = 0U; PDOEntryCnt < PDOSubindex0; PDOEntryCnt++)
            {
                pPDOEntry = ((UINT16 *)pPDO->pVarPtr +
                             (OBJ_GetEntryOffset((PDOEntryCnt + 1U),
                                                 pPDO) >> 4U));
                //
                // Increment the expected output size
                // depending on the mapped entry
                //
                outputSize += (UINT16)((*pPDOEntry) & BYTE_MASK);
            }
        }
        else
        {
            //
            // Assigned PDO was not found in object dictionary.
            // Return invalid mapping status.
            //
            outputSize = 0U;
            result = ALSTATUSCODE_INVALIDOUTPUTMAPPING;
            break;
        }
    }

    outputSize = (outputSize + 7U) >> 3U;

    //
    // Continue scanning of TXPDO if no error during RXPDO scanning
    //
    if(result == ALSTATUSCODE_NOERROR)
    {
        //
        // Scan Object 0x1C13 (SyncManager 3)
        // (Sums up sizes of all SyncManager assigned TXPDO subindexes)
        //
        for(PDOAssignEntryCnt = 0U;
            PDOAssignEntryCnt < sTxPDOassign.u16SubIndex0;
            PDOAssignEntryCnt++)
        {
            //
            // Get object handle for specified TxPDO assigned to SyncManager
            //
            pPDO =
             OBJ_GetObjectHandle(sTxPDOassign.aEntries[PDOAssignEntryCnt]);

            //
            // Confirm that Object isn't NULL before continuing
            //
            if(pPDO != NULL)
            {
                //
                // Get number of Object Entry subindexes
                //
                PDOSubindex0 = *((UINT16 *)pPDO->pVarPtr);

                //
                // Add up sizes of all the Object Entry subindexes
                //
                for(PDOEntryCnt = 0U; PDOEntryCnt < PDOSubindex0;
                    PDOEntryCnt++)
                {
                    pPDOEntry = ((UINT16 *)pPDO->pVarPtr +
                                 (OBJ_GetEntryOffset((PDOEntryCnt + 1U),
                                                     pPDO) >> 4U));
                    //
                    // Increment the expected output size
                    // depending on the mapped entry
                    //
                    inputSize += (UINT16)((*pPDOEntry) & BYTE_MASK);
                }
            }
            else
            {
                //
                // Assigned PDO was not found in object dictionary.
                // Return invalid mapping status.
                //
                inputSize = 0U;
                result = ALSTATUSCODE_INVALIDINPUTMAPPING;
                break;
            }
        }
    }

    inputSize = (inputSize + 7U) >> 3U;

    //
    // Assign calculated sizes
    //
    *pInputSize = inputSize;
    *pOutputSize = outputSize;

    return(result);
}

//
// APPL_InputMapping - Copies the input data from local device memory to ESC
//                     memory
//
void APPL_InputMapping(UINT16 *pData)
{
    UINT16 j = 0U;
    UINT16 *pTmpData = (UINT16 *)pData;
    UINT16 data;

    //
    // Loop through all TxPDO entries and write out data
    //
    for(j = 0U; j < sTxPDOassign.u16SubIndex0; j++)
    {
        switch(sTxPDOassign.aEntries[j])
        {
            //
            // TxPDO 0 (Data from ESC to master)
            //
            case 0x1A00U:
                //
                // Switches (8-bit (Byte) Data)
                //
                data = ((Switches0x6000.Switch8 << 7U) |
                        (Switches0x6000.Switch7 << 6U) |
                        (Switches0x6000.Switch6 << 5U) |
                        (Switches0x6000.Switch5 << 4U) |
                        (Switches0x6000.Switch4 << 3U) |
                        (Switches0x6000.Switch3 << 2U) |
                        (Switches0x6000.Switch2 << 1U) |
                        Switches0x6000.Switch1);

                *(volatile UINT16 *)pTmpData = data;
                break;
            //
            // TxPDO 1 (Data from ESC to master)
            //
            case 0x1A01U:
                //
                // DataToMaster (32 bits Data)
                //
                *(volatile UINT16 *)pTmpData |=
                 ((DataToMaster0x6010.DataToMaster) & BYTE_MASK) << 8U;

                pTmpData++;

                *(volatile UINT16 *)pTmpData =
                 ((DataToMaster0x6010.DataToMaster) & DWORD_WORD_MASK) >> 8U;

                pTmpData++;

                *(volatile UINT16 *)pTmpData =
                 ((DataToMaster0x6010.DataToMaster) & DWORD_MSB_MASK) >> 24U;

                //
                // ModeResponse (16 bits Data)
                //
                *(volatile UINT16 *)pTmpData |=
                 ((TargetModeResponse0x6012.ModeResponse) & BYTE_MASK) << 8U;

                pTmpData++;

                *(volatile UINT16 *)pTmpData =
                 ((TargetModeResponse0x6012.ModeResponse) &
                  WORD_MSB_MASK) >> 8U;

                //
                // SpeedPosFbk (32 bits Data)
                //
                *(volatile UINT16 *)pTmpData |=
                 ((TargetSpeedPosFeedback0x6014.SpeedPosFbk) &
                  BYTE_MASK) << 8U;

                pTmpData++;

                *(volatile UINT16 *)pTmpData =
                 ((TargetSpeedPosFeedback0x6014.SpeedPosFbk) &
                  DWORD_WORD_MASK) >> 8U;

                pTmpData++;

                *(volatile UINT16 *)pTmpData =
                 ((TargetSpeedPosFeedback0x6014.SpeedPosFbk) &
                  DWORD_MSB_MASK) >> 24U;
                break;
        }
    }
}

//
// APPL_OutputMapping - Copies the output data from ESC memory to local device
//                      memory
//
void APPL_OutputMapping(UINT16 *pData)
{
    UINT16 j = 0U;
    UINT16 *pTmpData = (UINT16 *)pData;
    UINT16 data = 0U;

    //
    // Loop through all RxPDO entries and read in data
    //
    for(j = 0U; j < sRxPDOassign.u16SubIndex0; j++)
    {
        switch(sRxPDOassign.aEntries[j])
        {
            //
            // RxPDO 0 (Data from Master to ESC)
            //
            case 0x1600U:
                //
                // LEDS (8-bit (Byte) Data)
                //
                data = (*(volatile UINT16 *)pTmpData);

                (LEDS0x7000.LED1) = data & BIT_MASK;
                data = data >> 1U;
                (LEDS0x7000.LED2) = data & BIT_MASK;
                data = data >> 1U;
                (LEDS0x7000.LED3) = data & BIT_MASK;
                data = data >> 1U;
                (LEDS0x7000.LED4) = data & BIT_MASK;
                data = data >> 1U;
                (LEDS0x7000.LED5) = data & BIT_MASK;
                data = data >> 1U;
                (LEDS0x7000.LED6) = data & BIT_MASK;
                data = data >> 1U;
                (LEDS0x7000.LED7) = data & BIT_MASK;
                data = data >> 1U;
                (LEDS0x7000.LED8) = data & BIT_MASK;
                break;
            //
            // RxPDO 1 (Data from Master to ESC)
            //
            case 0x1601U:
                //
                // DatafromMaster (32 bits Data)
                //
                DatafromMaster0x7010.DatafromMaster =
                 ((*(volatile UINT16 *)pTmpData) & WORD_MSB_MASK) >> 8U;

                pTmpData++;

                DatafromMaster0x7010.DatafromMaster |=
                 (UINT32)((*(volatile UINT16 *)pTmpData) & WORD_MASK) << 8U;

                pTmpData++;

                DatafromMaster0x7010.DatafromMaster |=
                 (UINT32)((*(volatile UINT16 *)pTmpData) & BYTE_MASK) << 24U;

                //
                // Mode (16 bits Data)
                //
                TargetMode0x7012.Mode =
                 (UINT16)((*(volatile UINT16 *)pTmpData) &
                          WORD_MSB_MASK) >> 8U;

                pTmpData++;

                TargetMode0x7012.Mode |=
                 (UINT16)((*(volatile UINT16 *)pTmpData) & BYTE_MASK) << 8U;

                //
                // SpeedPosReq (32 bits Data)
                //
                TargetSpeedPosReq0x7014.SpeedPosReq =
                 (UINT32)((*(volatile UINT16 *)pTmpData) &
                          WORD_MSB_MASK) >> 8U;

                pTmpData++;

                TargetSpeedPosReq0x7014.SpeedPosReq |=
                 (UINT32)((*(volatile UINT16 *)pTmpData) & WORD_MASK) << 8U;

                pTmpData++;

                TargetSpeedPosReq0x7014.SpeedPosReq |=
                 (UINT32)((*(volatile UINT16 *)pTmpData) & BYTE_MASK) << 24U;
                break;
        }
    }
}

//
// APPL_Application - Called as part of main stack loop. Assigns the output
//                    values to the input values for EtherCAT master
//
void APPL_Application(void)
{
    Switches0x6000.Switch1 = LEDS0x7000.LED1;
    Switches0x6000.Switch2 = LEDS0x7000.LED2;
    Switches0x6000.Switch3 = LEDS0x7000.LED3;
    Switches0x6000.Switch4 = LEDS0x7000.LED4;

    Switches0x6000.Switch5 = LEDS0x7000.LED5;
    Switches0x6000.Switch6 = LEDS0x7000.LED6;
    Switches0x6000.Switch7 = LEDS0x7000.LED7;
    Switches0x6000.Switch8 = LEDS0x7000.LED8;

    DataToMaster0x6010.DataToMaster = DatafromMaster0x7010.DatafromMaster;

    TargetModeResponse0x6012.ModeResponse = TargetMode0x7012.Mode;

    TargetSpeedPosFeedback0x6014.SpeedPosFbk =
     TargetSpeedPosReq0x7014.SpeedPosReq;
}

#if EXPLICIT_DEVICE_ID
//
// APPL_GetDeviceID - Return explicit device ID for f29x.CPU1
//
UINT16 APPL_GetDeviceID(void)
{
    return(0x5U);
}
#endif // EXPLICIT_DEVICE_ID

#if USE_DEFAULT_MAIN
//
// Main Echoback Function
//
int main(void)
{
    //
    // Initialize device hardware and EtherCAT slave controller
    //
    HW_Init();

    //
    // Initialize Slave Stack
    //
    MainInit();

    bRunApplication = TRUE;
    do
    {
        //
        // Run Slave Stack
        //
        MainLoop();
    } while(bRunApplication == TRUE);

    //
    // De-Initialize device and resources
    //
    HW_Release();

    return(0U);
}
#endif // USE_DEFAULT_MAIN

//
// End of File
//
