; H**************************************************************************
; $Archive:: /TI/product/c2xxhll/prg_algo/at29c257.asm                      $
; $Revision:: 2                                                             $
; $Date:: 12/31/97 5:19p                                                    $
; $Author:: Tonyc                                                           $
;
; DESCRIPTION:
;   Programmer functions called by the loader.
;
; GLOBALS:
;
;   unsigned * PRG_bufaddr      Address of buffer for program data
;   unsigned   PRG_bufsize      Size of program data buffer
;   unsigned   PRG_devsize	Size of programmable device
;   unsigned * PRG_paddr        First programming address
;   unsigned   PRG_page         Programming page
;   unsigned   PRG_length       Length of programming data
;   unsigned   PRG_status       Status of programming functions
;
; PUBLIC FUNCTIONS:
;   PRG_init                    Initialize system for programming
;   PRG_program                 Program a block
;   PRG_erase                   Erase a programmable device
;   PRG_verify                  Verify a block
;
; PRIVIATE FUNCTIONS:
;   None
;
; USAGE/LIMITATIONS
;   The global and public function symbol values must be available in the
;   COFF file for the loader to work properly.
;
;   The functions are not intended to be called by another program.
;   
;   This program supports the Atmel AT29C257 in 8 bit boot loader mode
;   for the C203 located in global data space.  The host program will
;   put on the header block and place the data into MSB first format.
;   The host will write data to the C203 buffer as 16 bit words.  The
;   FLASH algo will then store the data as *addr = data, *addr+1 = data>>8.
;   The Atmel AT29C257 device has a 64 byte sector.  The host will always
;   start programming a boot rom from 0x8000.  The program BUFFER_SIZE must
;   be set to a even multiple of the FLASH sector size/2.  So for At29C257
;   BUFFER_SIZE should be N*32, where N is the number of sectors you want
;   at one time.
;
; NOTES:
;
;   The program needs three memory sections to operate:
;
;       PRG_text    Executable section for this program
;       PRG_parms   Variable section for this program
;       PRG_data    Data buffer used to hold programming data
;
;   The size and location of each section  is user defined.  The
;   loader program will read the value of PRG_bufaddr and PRG_bufsize to
;   determine the location and size of the PRG_data section.
;
; H**************************************************************************

;
; ***************************************************************************
; INCLUDE FILES
; ***************************************************************************
	.mmregs

; ***************************************************************************
; PUBLIC DECLARATIONS
; ***************************************************************************
        .global PRG_init, PRG_program, PRG_erase, PRG_verify, PRG_stop
	  .global PRG_bufaddr, PRG_bufsize, PRG_devsize, PRG_paddr, PRG_page
        .global PRG_length, PRG_status, TimerDelay

; ***************************************************************************
; PRIVATE DECLARATIONS
; ***************************************************************************
; ***************************************************************************
; Address definitions
; ***************************************************************************
C203_FLASH_BASE      .set  08000h
C203_RAM_B2          .set  00060h
C203_RAM_B1          .set  00300h

; Define the C203 I/O mapped registers
C203_REG_IFR         .set  00006h
C203_REG_CLK         .set  0ffe8h
C203_REG_IC   	   .set  0ffech
C203_REG_SDTR 	   .set  0fff0h
C203_REG_SSPCR	   .set  0fff1h
C203_REG_ADTR        .set  0fff4h
C203_REG_ASPCR	   .set  0fff5h
C203_REG_IOSR        .set  0fff6h
C203_REG_BRD         .set  0fff7h
C203_REG_TCR         .set  0fff8h
C203_REG_PRD  	   .set  0fff9h
C203_REG_TIM 	   .set  0fffah
C203_REG_WSGR        .set  0fffch

; Define flash control addresses
FLASH_CTL_55         .set  0d555h 
FLASH_CTL_2A         .set  0aaaah

; ***************************************************************************
; Define the PRG_parms section
; ***************************************************************************
						
BUFFER_SIZE	.set	0100h		;Size of program buffer size
					;The loader uses max buffer of 256
					;so it does not make sense to use
					;value larger than 256

DEVICE_SIZE	.set	08000h      ;Size of device to be programmed.
                              ;Maximum size for the AT29C257 is 32K
				
; ***************************************************************************
; Define the PRG_parms section
; ***************************************************************************
	.sect	"PRG_parms"
PARMS:
PRG_bufaddr  	.word	PrgBuffer	;Address of buffer for program
PRG_bufsize	      .word	BUFFER_SIZE	;Tells host of buffer size
PRG_devsize	      .word	DEVICE_SIZE	;Tells host of device size
PRG_paddr	      .word	0		;First address to program
PRG_page	      .word	0		;Page to program
PRG_length    	.word	0		;Length of block to program
PRG_status	      .word	0		;Status of programming functions
Temp		      .word	0		;Temp location for verify
SectMask          .word 0003fh      ;Section size is 64 
CmpMask           .word 0           ;Compare mask to tell if prog complete
CmpData           .word 0           ;Data word to compare
TimeOut           .word 10000       ;Delay loop is 1us, Flash needs 10 ms
                                    ;loop based on 50ns clock cycle and
                                    ;20 instruction clocks in the loop

; ***************************************************************************
; Define the PRG_data section
; ***************************************************************************
	.sect	"PRG_data"
PrgBuffer	.space	BUFFER_SIZE*16	;Buffer for program data

;
; ***************************************************************************
; Define the PRG_text section
; ***************************************************************************
	.sect	"PRG_text"

; F**************************************************************************
; Name:	PRG_init
; 
; Description
;	Initialize the C2xx for programming external devices
; 
; Inputs
;	None
; Outputs
;	int PRG_status		Pass =0; Fail = 1;
;
; Notes:
;
; F**************************************************************************
 
PRG_init:	ldp	  #PARMS
            mar     *,ar5         ; Set AR5 as default ar *
            setc    intm          ; Disable interrupts 
            clrc    sxm           ; Disable sign extension

            ; Set the waitstates as:
            ; pslws = 7, psuws = 7, dsws = 7, isws = 0
            splk    01ffh,Temp
            out     Temp,C203_REG_WSGR
            splk    #80h,greg

            splk    #0,PRG_status
            b       PRG_stop
;
; F**************************************************************************
; Name:	PRG_program
; 
; Description
;	Transfers a block of data from RAM to programmable device
; 
; Inputs
;	unsigned *PRG_paddr	Pointer to first address to program
;	unsigned  PRG_page	Page to program, Program == 0, Data == 1
;	unsigned  PRG_length	Number of words to program
;	unsigned *PRG_bufaddr	Pointer to program data buffer
;
; Outputs
;	int PRG_status		Pass =0; Fail = 1;
;
; Notes:
;	The PRG_page value is setup by the host but not used in this case.
;	The assumption is that the programmable device is in program space.
; F**************************************************************************
 
PRG_program:
            ; Setup parameters
            ; ar7 points to the source data buffer in ram
            ; ar6 points to the destination in flash
            ; ar5 points to flash command addresses
            ldp     #PARMS
	      lar     ar7,PRG_bufaddr      ;Load source address
            lar     ar6,PRG_paddr        ;Load destination address  
            splk    #1,PRG_status        ;Set status to fail so we do not
                                         ;get a false pass   
            clrc    sxm                 ; Disable sign extension



InitSect:   mar     *,ar5
            lar     ar5, #FLASH_CTL_55   ; Send the programming start sequence.
            splk    #0aaaah, *           ; *5555 = aaaa 
            lar     ar5, #FLASH_CTL_2A   ; 
            splk    #05555h, *           ; *2aaa = 5555
            lar     ar5, #FLASH_CTL_55   ;
            splk    #0a0a0h, *           ; *5555 = a0a0        
            lacc    PRG_page             ; page == 8 is byte wide flash 
            sub     #8h
            bcnd    WriteSect8, eq

            ; Do 16 bit data writes
WriteSect16: 
            mar     *,ar7                ;arp = 7
            lacc    *+,ar6               ;Load source data, *ar7, arp6
            sacl    CmpData              ;Save the data value to temp
            sacl    *+,ar7               ;Store data, *ar6+, apr7
           
            lacc    PRG_length           ;Load the count
            sub     #1                   ;Decrement the program count
            sacl    PRG_length           ;Store count
            bcnd    SetCmp16,eq          ;Delay for last section before return
            sar     ar6,Temp             ;Store dest. address to Temp
            lacc    Temp                 ;Load dest address to acc
            and     SectMask             ;And in the section size mask
            bcnd    WriteSect16,neq      ;If address crossed sect boundary
SetCmp16:   splk    #08080h,CmpMask      ;Set data compare mask
            lacc    CmpData              ;Load the compare data
            and     CmpMask              ;Mask off all but bit 7/15
            sacl    CmpData              ;Store the data  
            b       Wdelay               ;delay to program section
  
            ; Do 8 bit data writes
WriteSect8: 
            mar     *,ar7                ;arp = 7
            lacc    *,ar6                ;Load source data, *ar7, arp6
            sacl    CmpData              ;Save the data value to temp
            sacl    *+,ar7               ;Store low byte, *ar6+, apr7
            lacc    *+,ar6               ;Reload source data, *ar7+, arp6
            rpt     #7                   ;Shift data right 8 bits
            sfr      
            sacl    *+,ar7               ;Store high byte, *ar6+, arp7
           
            lacc    PRG_length           ;Load the count
            sub     #1                   ;Decrement the program count
            sacl    PRG_length           ;Store count
            bcnd    SetCmp8,eq           ;Delay for last section before return
            sar     ar6,Temp             ;Store dest. address to Temp
            lacc    Temp                 ;Load dest address to acc
            and     SectMask             ;And in the section size mask
            bcnd    WriteSect8,neq       ;If address crossed sect boundary
                                         ;delay to program section
SetCmp8:    splk    #0080h,CmpMask       ;Set data compare mask
            lacc    CmpData              ;Load the compare data
            rpt     #7                   ;Shift down to get the MSB
            sfr
            and     CmpMask              ;Mask off all but bit 7
            sacl    CmpData              ;Store the data

Wdelay:     mar     *,ar6                ;Point to flash array
            mar     *-                   ;Set pointer to last value written
            lacc    TimeOut              ;Load the timeout value
            sacl    Temp                 ;Save in Temp for useage
CmpLoop:    lacc    *                    ;Load last flash value
            and     CmpMask              ;Mask off all but bit 7/15
            sub     CmpData              ;Sub with CmpData to get equal
            bcnd    CmpOk,eq             ;If equal then section programmed
            lacc    Temp                 ;Else decrrement Timeout
            sub     #1
            sacl    Temp
            bcnd    CmpLoop, neq         ;Timeout !0 then continue

            splk    #1,PRG_status        ;Else an error
            b       PRG_stop             ;Generate a breakpoint

CmpOk:      mar     *+                   ;Increment dest. address to previous
            lacc    PRG_length           ;Check for end of programming block
            bcnd    NoError, eq
            b       InitSect   
;
; F**************************************************************************
; Name:	PRG_erase
; 
; Description
;	
; 
; Inputs
;	None:
;
; Outputs
;	int PRG_status		Pass =0; Fail = 1;
;
; Notes:
;
; F**************************************************************************
PRG_erase:  
            ldp     #PARMS
            mar     *,ar5
                                         ; Address is 0x8000+FlashAddress
            lar     ar5, #FLASH_CTL_55   ; Send the programming start sequence.
            splk    #0aaaah, *           ; *5555 = aah
                                           
            lar     ar5, #FLASH_CTL_2A   ; 
            splk    #5555h, *            ; *2aaa = 55h

            lar     ar5, #FLASH_CTL_55   ;
            splk    #8080h, *            ; *5555 = 80h  

            lar     ar5, #FLASH_CTL_55   ;
            splk    #0aaaah, *           ; *5555 = aah 

            lar     ar5, #FLASH_CTL_2A   ;
            splk    #5555h, *            ; *2aaa = 55h  

            lar     ar5, #FLASH_CTL_55   ;
            splk    #1010h, *            ; *5555 = 10h 
   
            lacc    TimeOut              ;Load timeout value
ErDelay:    rpt     #19                  ;Create ~20 clock delay loop 
            nop
            sub     #1                   ;Loop until time expires
            bcnd    ErDelay, neq
    
            b       NoError

; F**************************************************************************
; Name:	PRG_verify
; 
; Description
;	Verify a block of programmed data
; 
; Inputs
;	unsigned *PRG_paddr	Pointer to first address to verify
;	unsigned  PRG_page	Page to verify, Program == 0, Data == 1
;	unsigned  PRG_length	Number of words to verify
;	unsigned *PRG_bufaddr	Pointer to verify data buffer
;
; Outputs
;	int PRG_status		Pass =0; Fail = 1;
;
; Notes:
;	The PRG_page value is setup by the host but not used in this case.
;	The assumption is that the programmable device is in program space.
; F**************************************************************************
 
PRG_verify:     
            ; Setup parameters
            ; ar7 points to the source data buffer in ram
            ; ar6 points to the destination in flash
            ; ar5 points to flash command addresses
            ldp     #PARMS
	      lar     ar7,PRG_bufaddr     ;Load source address
            lar     ar6,PRG_paddr       ;Load destination address     
            mar     *,ar6               ;Arp = 6 

            lacc    PRG_page            ;Verify based on TRG_page value
            sub     #8
            bcnd    Verify8,eq
            
Verify16:   lacc    *+,ar7              ;load flash data
            sub     *+,ar6              ;Subtract to find difference
            bcnd    Verror,neq

            lacc    PRG_length          ;Load verify count
            sub     #1                  ;Decrement the count
            sacl    PRG_length          ;Save the count
            bcnd    Verify16,neq        ;Loop if count not == 0
            b       NoError

Verify8:    lacc    *+                  ;load low byte
            and     #00ffh              ;mask upper bits
            sacl    Temp                ;save off
            lacc    *+,8,ar7            ;load upper byte
            or      Temp                ;Or in the lower bits
            sub     *+,ar6              ;Subtract to find difference
            sacl    Temp                ;Store and reload to get rid
            lacc    Temp                ;of AH.  This is faster then shl
            bcnd    Verror,neq

            lacc    PRG_length          ;Load verify count
            sub     #1                  ;Decrement the count
            sacl    PRG_length          ;Save the count
            bcnd    Verify8,neq         ;Loop if count not == 0
            b       NoError

Verror:     splk    #1,PRG_status       ;
            b       PRG_stop            ;Generate a breakpoint
;
; F**************************************************************************
; Name: PRG_stop
; 
; Description
;       Exit routine for all programming functions.
; 
; Inputs
;       None
;
;
; Outputs
;       None
;
; Notes:
;       Exit point for all programming functions
;       The ESTOP opcode gets executed as a NOP when not in emulation
;       mode.  The "b $" will keep the program from running off if
;       not in emulation mode.
;
; F**************************************************************************
NoError:    splk    #0,PRG_status        ;No error

PRG_stop:
		nop
		nop
            .word   0BE90h                  ;SWI instruction
		nop
		nop
            nop
 		b $

           .end
;
; F**************************************************************************
; Name: TimerDelay
; 
; Description
;       Delay loop using he C2xx timer.  Parameters are set to delay
;       4 seconds with 100ns C2xx clock.  
; 
; Inputs
;       None
;
;
; Outputs
;       None
;
; Notes:
;
; F**************************************************************************
TimerDelay:
           lacl  #66                     ; 4 secs of delay
Tloop1:    splk  #0010h,Temp             ; Stop timer
           out   Temp,C203_REG_TCR 
           splk  #0ea5fh,Temp            ; 60,000-1, in period
           out   Temp,C203_REG_PRD  
           splk  #00029h,Temp            ; Prescal of 10-1
           out   Temp,C203_REG_TCR
Tloop2:    bit   C203_REG_IFR,13
           bcnd  Tloop2,ntc
           splk  #0ffffh,C203_REG_IFR  
           sub   #1
           bcnd  Tloop1,neq	     
           ret
