;--------------------------------------------------------------------------------------------------
; Filename         : ERA_ALG.ASM	  
; Last Modified    : 28 Dec 2001.
; Version          : 1.3
; Originator       : Texas Instruments, DSP Digital Control Systems Group.
;--------------------------------------------------------------------------------------------------
; Description:
;
; This file contains the implementation of the core algorithm for ERASING the TMS320LF240x Flash.
; This algorithm is generic to any sector, and performs the pre-erase conditioning for the flash
; module on these parts.
;--------------------------------------------------------------------------------------------------
;--------------------------------------------------------------------------------------------------
; Revision History
; 
; Ver 1.0 (25 May 2001)
; 
;           Comments: Derived from previous source code. Several modifications in place for 
;                     enhanced embeddability.
; Ver 1.1 (10 Aug 2001)
;           Comments: Added code to save/restore the hardware stack. This excludes the possibility
;                     of hardware stack overflow.
; Ver 1.2 (10 Aug 2001)
;           Comments: Modified the timing delay routines to use AR2 instead of AR7. AR7 is used
;                     by the C-compiler in some cases, and cannot be used without first saving
;                     it. AR2 can be used directly.
; Ver 1.3 (28 Dec 2001)
;           Comments: 
;                   * Added support for the LF2401A. See the file ..\include\var.h on how to 
;                     set one of the available configurations.
;
;                   * Modified the macros ACCESS_REGS and ACCESS_ARRAY to not use a hard coded data
;                     mem address. This is to bring these macros in line with the rest of the algo
;                     which uses allocated addresses. These macros still use a hard coded address
;                     for the i/o space address 0xff0f (0ff0fh) since that is a special address,
;                     which will always address the i/o latch which controls the flash memory
;                     mode.
;--------------------------------------------------------------------------------------------------
;--------------------------------------------------------------------------------------------------

;--------------------------------------------------------------------------------------------------
; Symbols exported from this file
;--------------------------------------------------------------------------------------------------
            .globl  ERASE_FLASH

;--------------------------------------------------------------------------------------------------
; Import symbols (and other information) from VAR.H 
;--------------------------------------------------------------------------------------------------
            .include "..\include\var.h"

            .include "..\include\rundefs.h"

;--------------------------------------------------------------------------------------------------
; Program sections generated by this file:
; All code in this algorithm implementation is assembled into the named 
; section 'ALG_Text". Any relocation sensitive code is placed in the named
; section 'ESPL_text'.
;--------------------------------------------------------------------------------------------------
            .sect     "ERA_text"
;--------------------------------------------------------------------------------------------------
; Define macros to set the Flash Mode.
; ACCESS_REGS gives access to the Flash Registers in the Pgm Mem Space
; and ACCESS_ARRAY  gives access to the Flash ARRAY in Pgm Mem Space.
;
;--------------------------------------------------------------------------------------------------
ACCESS_REGS        .macro    
                OUT    flashAlgoVars.PAD,0ff0fh        
                .endm

ACCESS_ARRAY    .macro    
                IN     flashAlgoVars.PAD,0ff0fh
                .endm

;--------------------------------------------------------------------------------------------------
; Define Short DELAY loop macro. 
;  This will be used to generate a short delay upto 256 cycles.
;--------------------------------------------------------------------------------------------------
SDELAY          .macro  COUNT
                RPT     COUNT
                NOP
                .endm
;--------------------------------------------------------------------------------------------------
; ERASE_FLASH: Implementation of the user call for the flash erase routine. Call this with a valid
; sector mask in flashAlgoVars.SECTOR_MASK to erase any sector combination in the flash.
;--------------------------------------------------------------------------------------------------

        .label  EraseAlgoStartMain

ERASE_FLASH:                        ; Call label for assembly-language client applications.
_eraseFlash:                        ; Call label for C-language client applications.
        LDP     #flashAlgoVars.ADDR             ;Set the DP to the flash algo vars.

STACK_SAVE:
        LAR     AR2,#flashAlgoVars.STACK0       ;Use AR2 to save the hardware stack.
        MAR     *,AR2                           ;Make AR2 the current AR

        RPT     #7                              ;Save all the 8 hardware stack locations
        POPD    *+                              ;into RAM block.

SECTOR_0:
        LACC    flashAlgoVars.SECTOR_CMD        ;Get the sector processing command.
;--------------------------------------------------------------------------------------------------
        .if(DEV_TYPE = LF2407)
                                                ;Since the LF2401A has four sectors,
                                                ;allow only four bits.
        AND     #000Fh                          ;Mask out unwanted bits.

        .elseif (DEV_TYPE = LF2401A)

                                                ; Since the LF2401A has only two sectors,
                                                ; allow only two bits.
        AND     #0003h                          ;Mask out unwanted bits.

        .endif
;--------------------------------------------------------------------------------------------------
        BCND    NO_SECTORS,EQ                   ;If no sectors are set, then goto
                                                ;error reporting routine.

        SFR                                     ;Get Bit 0.
        SACL    flashAlgoVars.SECTOR_CMD        ;Store the rest of the command back.
;--------------------------------------------------------------------------------------------------
        BCND    SECTOR_1,NC                     ;If bit 0 is set, clear sector 0.
;--------------------------------------------------------------------------------------------------
        .if(DEV_TYPE = LF2407)

    ; Set up the limits for Sector 0 of LF2407 and call the core erase routine.

        SPLK    #0000h,flashAlgoVars.FL_SECST   ;Set up first addr of sector 0.
        SPLK    #0FFFh,flashAlgoVars.FL_SECEND  ;Set up last  addr of sector 0.
        SPLK    #0001h,flashAlgoVars.SECTOR_KEY ;Pass the sector enable key.
        CALL    ERASE_SECTOR                    ;Call clear_sector to clear sector 0.

        .elseif(DEV_TYPE = LF2401A)

    ; Set up the limits for Sector 0 of LF2401A and call the core erase routine.

        SPLK    #0000h,flashAlgoVars.FL_SECST   ;Set up first addr of sector 0.
        SPLK    #0FFFh,flashAlgoVars.FL_SECEND  ;Set up last  addr of sector 0.
        SPLK    #0001h,flashAlgoVars.SECTOR_KEY ;Pass the sector enable key.
        CALL    ERASE_SECTOR                    ;Call clear_sector to clear sector 0.
    
        .endif


;--------------------------------------------------------------------------------------------------
SECTOR_1:
        LACC    flashAlgoVars.SECTOR_CMD        ;Get the sector processing command.
        SFR                                     ;Mask out unwanted bits.
        SACL    flashAlgoVars.SECTOR_CMD        ;Store the rest of the command back.
        BCND    SECTOR_2,NC                     ;If bit 1 is set, clear sector 1.

        .if(DEV_TYPE = LF2407)

    ; Set up the limits for Sector 1 of LF2407 and call the core erase routine.

        SPLK    #1000h,flashAlgoVars.FL_SECST   ;Set up first addr of sector 1.
        SPLK    #3FFFh,flashAlgoVars.FL_SECEND  ;Set up last  addr of sector 1.
        SPLK    #0002h,flashAlgoVars.SECTOR_KEY ;Pass the sector enable key.
        CALL    ERASE_SECTOR                    ;Call clear_sector to clear
                                                ; Sector 1 if requested
        .elseif (DEV_TYPE = LF2401A)

    ; Set up the limits for Sector 1 of LF2401A and call the core erase routine.

        SPLK    #1000h,flashAlgoVars.FL_SECST   ;Set up first addr of sector 1.
        SPLK    #1FFFh,flashAlgoVars.FL_SECEND  ;Set up last  addr of sector 1.
        SPLK    #0002h,flashAlgoVars.SECTOR_KEY ;Pass the sector enable key.
        CALL    ERASE_SECTOR                    ;Call clear_sector to clear
                                                ; Sector 1 if requested

        .endif

;--------------------------------------------------------------------------------------------------


SECTOR_2:
        .if(DEV_TYPE = LF2407)

        ; For the LF2407, check if sector 2 is to be erased. If it is to be
        ; erased then call the core erase routine with the correct limits.


        LACC    flashAlgoVars.SECTOR_CMD        ;Get the sector processing command.
        SFR                                     ;Mask out unwanted bits.
        SACL    flashAlgoVars.SECTOR_CMD        ;Store the rest of the command back.
        BCND    SECTOR_3,NC                     ;If bit 2 is set, clear sector 2.

        SPLK    #4000h,flashAlgoVars.FL_SECST   ;Set up first address of sector 2.
        SPLK    #6FFFh,flashAlgoVars.FL_SECEND  ;Set up last address of sector 2.
        SPLK    #0004h,flashAlgoVars.SECTOR_KEY ;Pass the sector enable key.
        CALL    ERASE_SECTOR                    ;Call clear_sector to clear sector 2.

        ; For the LF2407, check if sector 2 is to be erased. If it is to be
        ; erased then call the core erase routine with the correct limits.

SECTOR_3:
        LACC    flashAlgoVars.SECTOR_CMD        ;Get the sector processing command.
        SFR                                     ;Mask out unwanted bits.
        SACL    flashAlgoVars.SECTOR_CMD        ;Store the rest of the command back.
        BCND    SECTORS_DONE,NC                 ;If bit 3 is set, clear sector 3.

        SPLK    #7000h,flashAlgoVars.FL_SECST   ;Set up first address of sector 3.
        SPLK    #7FFFh,flashAlgoVars.FL_SECEND  ;Set up last address of sector 3.
        SPLK    #0008h,flashAlgoVars.SECTOR_KEY ;Pass the sector enable key.
        CALL    ERASE_SECTOR                    ;Call clear_sector to clear sector 3.

        .endif

SECTORS_DONE:
        SPLK    #0,flashAlgoVars.ALGO_STATUS    ;No errors encountered.
        B       STACK_RESTORE

NO_SECTORS:
        SPLK    #11,flashAlgoVars.ALGO_STATUS   ;No sectors specified for erase.
        B       STACK_RESTORE

ERROR1: POPD    flashAlgoVars.PAD               ;Adjust stack depth.
        SPLK    #2,flashAlgoVars.ALGO_STATUS    ;Error encountered in ERASE.
        B       STACK_RESTORE

;--------------------------------------------------------------------------------------------------
; Restore the hardware stack and exit to caller.
;--------------------------------------------------------------------------------------------------
STACK_RESTORE:
        LAR     AR2,#flashAlgoVars.STACK7       ;Point AR2 to the last of the stored stack contents
        MAR     *,AR2                           ;Make AR2 the current AR
        
        RPT     #7                              ;Restore ALL stack locations.
        PSHD    *-
        MAR     *,AR1                           ;Make AR1 the current AR
        RET                                     ;Return to caller


;--------------------------------------------------------------------------------------------------    
; ERASE_SECTOR:
;--------------------------------------------------------------------------------------------------    
; This function is the implementation of the erase algorithm for the pre-program
; conditioning of the flash array. This function is desginated as a LEVEL_1 
; subroutine.
;--------------------------------------------------------------------------------------------------
ERASE_SECTOR:
        SPLK    #MX_ECNT,flashAlgoVars.PLS_CNT              ;Initialize the program pulse count. 
        SPLK    #0ffffh,flashAlgoVars.DATA                  ;Load 0ffffh for compare.
    
        BLDD    #flashAlgoVars.FL_SECST,flashAlgoVars.ADDR  ;Point address to the start of sector.
        SPLK    #0000,flashAlgoVars.LASTVER                 ;Initialize LASTVERIFY Flag 
;--------------------------------------------------------------------------------------------------
;   ENABLE
;--------------------------------------------------------------------------------------------------
; Enable the flash for erase:
; To accept commands the following must happen:
;   1. Place the flash in the register mode.
;   2. Copy the sector key to the SECT register.
;   3. Enable the core by setting Bit 0 of the ENAB Register
;--------------------------------------------------------------------------------------------------
ENABLE: ACCESS_REGS                                         ;Put the flash in register mode.

        LACC    #SECT                                       ;Enable sector.
        BLDD    #flashAlgoVars.SECTOR_KEY,flashAlgoVars.PAD ;Copy the sector key
        TBLW    flashAlgoVars.PAD                           ;
                                                            ;
        LACC    #ENAB                                       ;Enable core.
        SPLK    #0001h,flashAlgoVars.PAD                    ;
        TBLW    flashAlgoVars.PAD                           ;

        B    ERVER                                          ;Enter erase loop from the erase verify.

;--------------------------------------------------------------------------------------------------
;   PERASE:
;--------------------------------------------------------------------------------------------------
;   This loop erases the sector, based on the address passed by the caller in
;   flashAlgoVars.FL_SECST.
;--------------------------------------------------------------------------------------------------
PERASE:
        LACC    flashAlgoVars.PLS_CNT                       ;Get the pulse counter value.
        BCND    ERROR,EQ                                    ;If remaining pulses == 0 then
                                                            ;commence the error handling.
;--------------------------------------------------------------------------------------------------
        LACC    #WADDR                                      ;The sector is selected by writing
                                                            ;WADDR with the first address of 
                                                            ;the sector
        CALL    SETWADDR                                    ;Execute the write to WADDR.

        LACC    #WDATA                                      ;Load the WDATA register with 0xffff.
        TBLW    flashAlgoVars.DATA
;--------------------------------------------------------------------------------------------------
        LACC    #CTRL                                       ;Setup the erase mode.
        SPLK    #0001,flashAlgoVars.PAD
        TBLW    flashAlgoVars.PAD
;--------------------------------------------------------------------------------------------------
        SDELAY    #T_esu_er                                 ;T_esu(ER) Delay.
;--------------------------------------------------------------------------------------------------
        LACC    #PMPC                                       ;Activate the erase mode.
        SPLK    #0005h,flashAlgoVars.PAD    
        TBLW    flashAlgoVars.PAD    
;--------------------------------------------------------------------------------------------------
EPW:    LAR     AR2,#ERASE_PULSE                            ;Call the delay routine to define the
        CALL    DELAY,AR2                                   ;erase pulse width to T_erase(E)
;--------------------------------------------------------------------------------------------------
        SPLK    #0000h,flashAlgoVars.PAD1                   ;De-activate the erase mode.
        CALL    CLRCMD
;--------------------------------------------------------------------------------------------------
        SDELAY  #T_eh_er    ;T_eh(ER)                       ;Algo delay.
;--------------------------------------------------------------------------------------------------
        LACC    #CTRL                                       ;Exit the Erase mode.
        SPLK    #0000,flashAlgoVars.PAD
        TBLW    flashAlgoVars.PAD
;--------------------------------------------------------------------------------------------------
        SDELAY  #Teh_C                                     ;Wait in normal mode.
;--------------------------------------------------------------------------------------------------
        LACC    flashAlgoVars.PLS_CNT                       ;Advance the erase pulse counter.
        SUB     #1h
        SACL    flashAlgoVars.PLS_CNT
;--------------------------------------------------------------------------------------------------
; ERVER: ERASE Verify
;--------------------------------------------------------------------------------------------------
; This subroutine checks the flash to see if the flash is erased.
; To do this it does the followign steps:
;   1.  Reads the flash array in erase verify mode.
;   2.  Branches to the Erase routine if a word is non-FFFFh
;   3.  If all the words in the sector are ffffh then it transfers control
;       to PASSERVER.
;--------------------------------------------------------------------------------------------------
ERVER:  BLDD    #flashAlgoVars.FL_SECST,flashAlgoVars.ADDR  ;Init ADDR to first location in sector.
    
B0      LACC    #CTRL                                       ;Init the Erase verify mode.
        SPLK    #0002,flashAlgoVars.PAD
        TBLW    flashAlgoVars.PAD
        SDELAY  #T_eva_e                                    ;T_eva(E) ; Algo Delay.

B1

        CALL    READWORD                                    ;Read out the word in erase ver mode.

COMPARE:
        LACL    flashAlgoVars.READ                          ;Check to see if the word read out is
        XOR     #0FFFFh                                     ;0ffffh.
        BCND    PROCEED_TO_ERASE,NEQ                                  ;If not, then proceed to apply an erase 
                                                            ;pulse to the sector.
    
        LACC    flashAlgoVars.ADDR                          ;See if the current address is 
        SUB     flashAlgoVars.FL_SECEND                     ;end-of-sector. If so, transfer control
        BCND    PASSERVER,EQ                                ;to PASSERVER.

        ADD     flashAlgoVars.FL_SECEND                     ;Increment the ADDR counter variable.
        ADD     #1                                          ;to next address
        SACL    flashAlgoVars.ADDR                          ;Store back incremented variable.

        B       B1                                          ;Proceed to check the next word.

PROCEED_TO_ERASE:

        LACC    #CTRL                                       ;Exit the Erase verify mode.
        SPLK    #0000,flashAlgoVars.PAD
        TBLW    flashAlgoVars.PAD

        SDELAY  #Tevh_C                                     ;Wait in normal mode.

        B       PERASE

;--------------------------------------------------------------------------------------------------
;PASSERVER
;--------------------------------------------------------------------------------------------------
; This section of code is executed upon successful erase verify.
; This then
;       1. Checks to see if compaction verify has passed.
;       2. If yes, the erase operation is complete for this sector, and the
;          control is transferred to 'DONE' which finishes up.
;       3. If not, control is passed to the compaction verify routine.
;--------------------------------------------------------------------------------------------------
PASSERVER:

        LACC    #CTRL                                       ;Exit the Erase verify mode.
        SPLK    #0000,flashAlgoVars.PAD
        TBLW    flashAlgoVars.PAD

        SDELAY  #Tevh_C                                     ;Wait in normal mode.


        LACC    flashAlgoVars.LASTVER                       ;Check to see if compaction ver passed.
        BCND    DONE,NEQ                                    ;If yes, the erase is complete.
        B       CMPCTVER                                    ;else start compaction verify.

;--------------------------------------------------------------------------------------------------
;ERROR Error handling
;--------------------------------------------------------------------------------------------------
; This section of code is executed upon encountering errors.
;       1. Clears control registers.
;       2. Transfers control to an error exit point.
;--------------------------------------------------------------------------------------------------

ERROR:  SPLK    #6,flashAlgoVars.PAD1                       ; Set up for clearing 6 control regs
        CALL    CLRCMD                                      ; Call routine to do so.
        B       ERROR1

;--------------------------------------------------------------------------------------------------
; READWORD
; This routine performs the following operations
;    1.    Reads the word at the location pointed to by ADDR.
;    2.  Stores the word in the variable READ.
;    3.  Returns to the caller in Register Mode.
;--------------------------------------------------------------------------------------------------
READWORD
        ACCESS_ARRAY
        LACC    flashAlgoVars.ADDR                          ;Read word flash 
        TBLR    flashAlgoVars.READ                          ;store word in READ (data space)
        ACCESS_REGS

        RET

;--------------------------------------------------------------------------------------------------
; CMPCTVER: 
;--------------------------------------------------------------------------------------------------
; This section performs the following functions:
;   1.  Checks to see if compaction operation is needed.
;   2.  Calls the compaction routine if needed.
;--------------------------------------------------------------------------------------------------
CMPCTVER:
        BLDD    #flashAlgoVars.FL_SECST,flashAlgoVars.ADDR  ;Init ADDR to first location in sector.
        SPLK    #0deadh,flashAlgoVars.LASTVER               ;Sets the LASTVERIFY flag to indicate
                                                            ;that compaction verify is the last 
                                                            ;verification operation.    
;G1
       SPLK    #MX_CMPCT,flashAlgoVars.PLS_CNT             ;Initialize compaction pulse count.
;--------------------------------------------------------------------------------------------------
CMPCTVER1:
        LACC    #WADDR                                      ;Setup WADDR to the address to be 
        CALL    WADDR_SET                                   ;checked for compaction
;--------------------------------------------------------------------------------------------------
        LACC    #CTRL                                       ;Setup the compaction verify mode.
        SPLK    #0006,flashAlgoVars.PAD                     ; 
        TBLW    flashAlgoVars.PAD
;--------------------------------------------------------------------------------------------------
        SDELAY    #T_cva_cm                                 ;Algorithm Delay ~ 50ns(25ns MIN)     
                                                            ;T_cva(CM)
;--------------------------------------------------------------------------------------------------
        CALL    READWORD                                    ;Read out the word in compaction 
                                                            ;verify mode.
;--------------------------------------------------------------------------------------------------
        LACL    flashAlgoVars.READ                          ;Get the value read out.
        BCND    PROCEED_TO_CMPCT,NEQ                        ;If this is non-zero there are depleted
                                                            ;bits, and proceed to apply compaction 
                                                            ;pulses
;--------------------------------------------------------------------------------------------------
        LACC    flashAlgoVars.ADDR                          ;Increment the current address 
        ADD     #1                                          ;and store it back.
        SACL    flashAlgoVars.ADDR                          ;
;--------------------------------------------------------------------------------------------------
        SPLK    #MX_CMPCT,flashAlgoVars.PLS_CNT             ;Initialize compaction pulse count.
;--------------------------------------------------------------------------------------------------
        SUB     #STICKS                                     ;Subtract the stick count.
        SUB     flashAlgoVars.FL_SECST
        BCND    CMPCTVER1,NEQ                               ; CURRENT ADDR - FL_SECST is the
                                                            ; number of sticks checked.
                                                            ; SO CURRENT (ADDR-FL_SECST)-STICKS = 0
                                                            ; on the last stick to be checked.

                                                            ; If here done compaction check has 
                                                            ; passed. So exit compaction mode

;--------------------------------------------------------------------------------------------------
        LACC    #CTRL                                       ;Exit the compaction verify mode.
        SPLK    #0000,flashAlgoVars.PAD
        TBLW    flashAlgoVars.PAD
;--------------------------------------------------------------------------------------------------
        SDELAY  #Tcvh_CM                                    ; Wait in normal mode
;--------------------------------------------------------------------------------------------------
        B       ERVER
;--------------------------------------------------------------------------------------------------
PROCEED_TO_CMPCT:
;--------------------------------------------------------------------------------------------------

        LACC    #CTRL                                       ;Exit the compaction verify mode.
        SPLK    #0000,flashAlgoVars.PAD
        TBLW    flashAlgoVars.PAD


        SDELAY  #Tcvh_CM                                    ; Wait in normal mode

;--------------------------------------------------------------------------------------------------
        B       CMPCT
;--------------------------------------------------------------------------------------------------
; CMPCT: Implementation of the compaction routine, 
;--------------------------------------------------------------------------------------------------
; This code applies compaction pulses to the column addressed by flashAlgoVars.ADDR.
; It applies compaction pulses to the columns which read out as having depleted
; bits. 
;--------------------------------------------------------------------------------------------------
CMPCT:
;--------------------------------------------------------------------------------------------------
; DO NOT *EVER* REMOVE COMPACTION - THIS CAN DESTROY THE DEVICE. THIS IS VERY VERY STRICTLY
; FOR TESTING ROUTINES 
;;;;    B    DONE                                           ;To REMOVE COMPACTION uncomment this.
;--------------------------------------------------------------------------------------------------
        LACL    flashAlgoVars.PLS_CNT                       ;Get the number of pulses left.
        BCND    ERROR,EQ                                    ;If pulse count == 0 then commence 
                                                            ;error handling.

;       LACC    #WADDR                                      ;COLUMN ADDRESS => WADDR REG  
;       TBLW    flashAlgoVars.ADDR

        LACL    flashAlgoVars.READ                          ;Get the Data Read during compaction 
                                                            ;verify.
        XOR     #0FFFFh                                     ;Invert the bitmask.
        SACL    flashAlgoVars.READ                          ;Store the inverted bitmask.

        LACC    #WDATA                                      ;Write bitmask to the WDATA register.
        TBLW    flashAlgoVars.READ                    

        LACC    #CTRL                                       ;Initialize the compaction mode.
        SPLK    #0005,flashAlgoVars.PAD
        TBLW    flashAlgoVars.PAD

        SDELAY    #T_csu_cm                                 ;Wait T_csu(CM)

        LACC    #PMPC                                       ;Activate the compaction mode.
        SPLK    #0005h,flashAlgoVars.PAD
        TBLW    flashAlgoVars.PAD

CPW     LAR     AR2,#CMPCT_PULSE                            ;CMPCT PULSE WIDTH T_cmpct(E)
        CALL    DELAY,AR2                                   ;Generate a delay to define 
                                                            ;the compaction pulse width.

        SPLK    #0000h,flashAlgoVars.PAD1                   ;De-activate the compaction mode.
        CALL    CLRCMD

        SDELAY  #T_ch_cm                                    ;Wait T_ch(CM) 

        SPLK    #0001h,flashAlgoVars.PAD1                   ;Clear the control registers.
        CALL    CLRCMD

        SDELAY  #T_ch_C                                     ;Wait in normal read mode.


        LACC    flashAlgoVars.PLS_CNT                       ;Decrement the puse counter.
        SUB     #1h
        SACL    flashAlgoVars.PLS_CNT

        B       CMPCTVER1                                   ;Resume the compaction verify routine.
;--------------------------------------------------------------------------------------------------
DONE:    SPLK    #6,flashAlgoVars.PAD1                       ;Clear all control registers and
        CALL    CLRCMD                                      ;return to the caller, in ARRAY mode.
        ACCESS_ARRAY
        RET                                                 ;PASSED CMPCTVER AND ERVER    
;--------------------------------------------------------------------------------------------------


;--------------------------------------------------------------------------------------------------
; CLRCMD
;--------------------------------------------------------------------------------------------------
;     This routine performs the following operations
;   1.     Places the flash in normal read mode by writing 0000 to two control 
;        registers PMPC,CNTL=>0000)
;    2.  Returns to the caller in Register Mode.
;--------------------------------------------------------------------------------------------------
CLRCMD  ACCESS_REGS
        SPLK     #0,flashAlgoVars.PAD
        LACC     flashAlgoVars.FL_SECST
        RPT     flashAlgoVars.PAD1
        TBLW     flashAlgoVars.PAD
        RET
    

            
;--------------------------------------------------------------------------------------------------
;  SUBROUTINE: Delay
;
; DESCRIPTION:
;  This routine executes a delay approximately
;    (DLOOP + 8) + AR2 * (DLOOP + 6) * 1/freq = nsec
;    if AR2  = 0  Delay = 108 cycles
;    if AR2 != 0  Delay = 108 + AR6 * 106 cycles
;
;  A typical call to this subroutine is as follow:
;
;     lar     AR2,delay_param         1 cycles reg/#k, 2 cycles if #lk
;     call    Delay,*,AR2             4 cycles
;
;       minimum delay is when delay_parm=0
;       minimum delay = 113 cycles ( 108 + 5 )
;       for a cycle time of 50 nsec,
;         minimum delay = 5.65 usec   ( 113 cycles * 50 nsec)
;         delay time    = 5.65 usec + AR2 * 106 * 50 nsec
;                       = 5.65 usec + AR2 * 5.3 usec
;
; INPUTS:
;   AR2 = Passed on to this routine to set delay time as required
;   ARP => AR2    
; OUTPUTS:
;       none
;--------------------------------------------------------------------------------------------------
DELAY
        RPT     #DLOOP                  ;1 cycles
        NOP                             ;1+DLOOP = 101 CYCLES (FOR DLOOP = 100)
        BANZ    DELAY,*-,AR2            ;   4/2 (True/False)
        RET                             ;   4 cycles
                                        ; for delay_parm = 0
                                        ;   (1+101+2+4 = 108) cycles
                                        ; for delay_parm != 0
                                        ;   108 + AR2 * (1+101+4 = 106) cycles

        .label  EraseAlgoEndMain

;--------------------------------------------------------------------------------------------------
    .sect   "ESPL_text"
    
SETWADDR:

    .label  EraseAlgoStartSpl

    TBLW    flashAlgoVars.FL_SECST
    RET

    .loop 6
    nop
    .endloop
;--------------------------------------------------------------------------------------------------
WADDR_SET:
    TBLW    flashAlgoVars.ADDR
    RET

    .label  EraseAlgoEndSpl

