//###########################################################################
//
// FILE:    main.c
//
// TITLE:   Based upon DSP280x Device Getting Started Program.
//
// ASSUMPTIONS:
//
//    This program requires the DSP280x header files.  
//
//    This project is configured for "boot from Flash" operation.
//    The 280x Boot Mode table is shown below, based on the Cooling
//    System Board's GPIO18, GPIO29 and GPIO34 weak pull-ups.
//
//       Boot      GPIO18     GPIO29    GPIO34
//       Mode      SPICLKA    SCITXDA
//                 SCITXB
//       -------------------------------------
//       Flash       1          1        1  <- "Boot from Flash"
//       SCI-A       1          1        0
//       SPI-A       1          0        1
//       I2C-A       1          0        0
//       ECAN-A      0          1        1        
//       SARAM       0          1        0
//       OTP         0          0        1
//       I/0         0          0        0 
//
//
// DESCRIPTION:
//
//    This example configures CPU Timer1 and increments
//    a counter each time the timer asserts an interrupt.
//      
//       Watch Variables:
//          CpuTimer1.InterruptCount
//
//###########################################################################


#include "DSP280x_Device.h"     // DSP280x Headerfile Include File
#include "SysCtrl.h"
#include "Gpio.h"

// Definitions

// CPU-Timer1 Interval for control over cascaded
// ADC Collection and Processing, plus Background
// Executive (see "BackExec.c") Scheduling.
#define SCHED_microS  312.5

// Support Polling Version of CPU-Timer1 for use with
// the Simulator also, with zero value specified.
#define	HARD_TIMER_1	1


typedef enum sequence
{
   soc1initial,		// Initialization (first ADC)
   soc1sustain		// read ADC Data, restart cascade
} seqCase;


// External Functions
void InitSysCtrl (void);
void backExecutive (void);
void InitGPIO (void);
void InitPieCtrl (void);
void InitPieVectTable (void);

void initial_RS422 (void);
void initAdcModule (void);
void read_ADC_Casc (void);


// Local Variables

#if (HARD_TIMER_1 == 0)
static Uint32	currTimer1;
static Uint32	prevTimer1 = 0xFFFFFFFFL;
#endif

static seqCase  schedulADC = soc1initial;


// Local Functions
#if HARD_TIMER_1
interrupt void  cpuTimer1_ISR (void);			// ADC SEQ ISR  Surrogate
#else
void			cpuTimer1Poll (void);			// ADC SEQ Poll Surrogate
#endif



void main (void)
{
   // Initialize System Control:
   // PLL, WatchDog, enable Peripheral Modules' Clocks.
   // This function is found in "SysCtrl.c".
   InitSysCtrl();

   // Initialize GPIO: 
   // Enable GPIO34 for LED Output (via LED and weak pull-up, part
   // of "Boot from Flash" Configuration).
   InitGPIO();

   // Initialize the PIE control registers to their default state.
   // The default state is all PIE interrupts disabled and flags
   // are cleared.  
   // This function is found in "PieCtrl.c".
   InitPieCtrl();
   
   // Disable CPU interrupts and clear all CPU interrupt flags:
   IER = 0x0000;
   IFR = 0x0000;

   // Initialize the PIE vector table with pointers to the shell
   // Interrupt Service Routines (ISR).  
   // This will populate the entire table, even if the interrupt
   // is not used in this example.  This is useful for debug purposes.
   // The shell ISR routines are found in "DefaultIsr.c".
   // This function is found in "PieVect.c".
   InitPieVectTable();

   #if HARD_TIMER_1
   EALLOW;  // Needed to write to EALLOW protected registers
   PieVectTable.XINT13 = &cpuTimer1_ISR;
   XIntruptRegs.XNMICR.bit.ENABLE = 0;	 // Timer1 connected to XINT13
   XIntruptRegs.XNMICR.bit.SELECT = 0;	 // Disable NMI
   EDIS;    // Needed to disable write to EALLOW protected registers
   #endif

   // Initialize Timers used.
   InitCpuTimers();

   // Configure CPU-Timer1 count based upon System Clock & Period;
   // 20 CPU Frequency (in MHz), 156.25 Timer Period (in uSeconds).
   ConfigCpuTimer (&CpuTimer1, (float) SYSCLKOUT_MHZ, SCHED_microS);

   // Initialize the RS-422 Communications.
   initial_RS422();

   // Initialize the ADC Module.
   initAdcModule();

   // Enable CPU XINT13, for CPU-Timer1:
   IER |= M_INT13;

   // Enable global Interrupts and higher priority real-time debug events:
   EINT;   // Enable Global interrupt INTM
   ERTM;   // Enable Global real-time interrupt DBGM

   // Finally, start CPU-Timer1 going.
   CpuTimer1Regs.TCR.bit.TSS = 0;


   // Background loop. Just sit and loop forever.
   // Call the Background Scheduler every 156.25S.
   for (;;)
   {
      #if HARD_TIMER_1
      if (CpuTimer1.InterruptCount > 0)
      {
	      backExecutive();

	      CpuTimer1.InterruptCount = 0;
      }
	  #else
	  currTimer1 = CpuTimer1Regs.TIM.all;
	  if (currTimer1 > prevTimer1)
      {
         cpuTimer1Poll();

         backExecutive();
      }

      prevTimer1 = currTimer1;
	  #endif
   }
} 


#if HARD_TIMER_1
// CPU-Timer1 Interrupts are used as a surrogate
// for ADC Interrupts, as well as providing for
// Background Scheduling.
//
// Cascaded ADC Buffer Conversion is Software
// started, every Timer1 Interrupt.
//
// The original design was for half-Buffer
// triggering, upon every Timer1 Interrupt, with
// shorter duration.
//
// The same timing mechanism is applied, just to
// make a single cascaded ADC call.
// Sample processing (mainly ADC Peak and Valley
// Detection) is presently performed within the
// Timer1 interrupt path ('read_ADC_Casc').
interrupt void  cpuTimer1_ISR (void)
{
   CpuTimer1.InterruptCount++;

#else
void cpuTimer1Poll (void)
{
#endif
   switch (schedulADC)
   {
      case soc1initial:

         // Start ADC SEQ1 collecting.
         AdcRegs.ADCTRL2.bit.SOC_SEQ1 = 1;

		 // Prepare for next Case.
         schedulADC = soc1sustain;

         break;


      case soc1sustain:

         // Process cascaded SEQ ADC Data
         read_ADC_Casc();

         break;


      default:

         break;
   }
}

//===========================================================================
// No more.
//===========================================================================
