//-------------------------------------------------------------------------------------------------
//
// Main.c - C28x Benchmark BIOS vs. NON-BIOS Mg'd Interrupts
//
// Author: Eric Wilbur 01/06/13
//
// Interrupt #38 used for BIOS-Managed interrupt (INT1 Group)
// Interrupt #67 used for NON-BIOS-Managed Interrupt (INT5 Group, Zero Latency)
//
// Task is used because Tasks are ready when they are created (during BIOS_init, before main())
// and therefore they require no "trigger" to run.
//
// Interrupt #38 is managed by BIOS and therefore can be found in app.cfg GUI (INT1 Group)
// Interrupt #67 is NOT managed by BIOS. In app.cfg, if you click on the Hwi module (not the
// instance), you will see the "Zero Latency IER Mask" set to 0x10 (16 decimal) which is a
// "1" in the 5th bit - stating that INT5 group (all interrupts in this PIE block) are
// considered ZERO latency and will NOT be managed by BIOS.
//
// Interrupts that are used OUTSIDE of BIOS (i.e. zero-latency INTs) must be PLUGGED into
// the vector table using the BIOS API Hwi_plug() as shown in the hardware_init() fxn. This
// function call has a "funny" casting, not easy to figure out, but now you see it.
//
// When task runs:
// - The timestamp benchmark is taken (i.e. how long Timestamp_get32 take - Timestamp overhead)
// - BIOS managed interrupt (INT #38) is triggered and benchmark is taken
// - Non-bios managed interrupt (INT #67) is triggered and benchmark is taken
// - Results are printed to the console screen via System_printf()
//
// FYI - Hwi_post() is supported on the C28x processors and therefore is used to trigger
// each interrupt.
//
// Note that the boot module in the app.cfg file is set to 50MHz. Flash wait states do have an
// effect on the cycle count for this benchmark. If you run at, e.g. 80MHz, your results may
// be different. However, the cycle counts shown can be used as an estimate for any project
// using any frequency.
//
// TO RUN THIS CODE:
//
// Build, load and press Play (Resume). Watch the console window where the results will be
// displayed. You can then tweak the Hwi instance or anything else you'd like to see how
// those changes affect the benchmark.
//
// CONCLUSION:
//
// BIOS will set certain features on by default for all interrupts. This is the "DEFAULT" benchmark
// shown below. However, with ALL of the features turned off (which is not desirable), you can see
// the "MIN BIOS" benchmark below. The "ZERO LATENCY" benchmark is the latency of the interrupt
// without BIOS involvement at all.
//
// To obtain the BIOS (MIN) benchmark, simply go into the app.cfg GUI and click on the instance
// and the module for Hwi and UNCHECK every box you see. Again, this is probably not desirable
// for most applications - but hey, it's a benchmark and you know how we obtained it.
//
// The latency numbers shown below are from the approximate time of the trigger to the first line
// of code in the ISR. All numbers are approximate and your mileage may vary.
//
// C28x Interrupt Latency Results running at 50MHz using F28069 Control Stick (USB stick):
//
// BIOS (DEFAULT): 226 cycles
// BIOS (MIN):     193 cycles
// ZERO-LATENCY:    40 cycles
//---------------------------------------------------------------------------------------------------



//---------------------------------------
// BIOS Header Files first
//---------------------------------------
#include <xdc/std.h>  						//mandatory - have to include first, for BIOS types
#include <ti/sysbios/BIOS.h> 				//mandatory - if you call APIs like BIOS_start()
#include <xdc/cfg/global.h> 				//header file for statically defined objects/handles
#include <xdc/runtime/Log.h>				//used for Log_info() calls
#include <xdc/runtime/System.h>
#include <xdc/runtime/Timestamp.h>
#include <xdc/runtime/Types.h>
#include <string.h>
#include <xdc/runtime/Error.h>
#include <ti/sysbios/family/c28/Hwi.h>
#include <ti/sysbios/knl/Task.h>


//-----------------------------------------
// ControlSuite Header Files
//-----------------------------------------
#include "DSP28x_Project.h"


//---------------------------------------
// Prototypes
//---------------------------------------
void isr_BIOS(void);
void interrupt isr_nonBIOS();
void taskFxn (UArg a0, UArg a1);
void hardware_init(void);



//---------------------------------------
// Globals
//---------------------------------------
UInt16 t0, t1, timestamp_overhead;
UInt16 bios_isr_start, bios_isr_end;
UInt16 nonbios_isr_start, nonbios_isr_end;
UInt16 bios_int_latency, nonbios_int_latency;



//--------------------------------------------------------------
// taskFxn()
//
// Triggered by BIOS_Start()
// Triggers ISRs and performs benchmarks, then prints them
// to the console screen
//--------------------------------------------------------------

Void taskFxn(UArg a0, UArg a1)
{

	t0 =  Timestamp_get32();							// determine Timestamp overhead
	t1 =  Timestamp_get32();
	timestamp_overhead = t1 - t0;

	bios_isr_start = Timestamp_get32();					// start BIOS-managed interrupt snapshot
	Hwi_post(38);										// post BIOS-managed INT (38)

	nonbios_isr_start = Timestamp_get32();				// start non-BIOS-managed interrupt snapshot
	Hwi_post(67);  									 	// post non-BIOS-managed INT (67)

	// Calculate interrupt latencies
	bios_int_latency =  bios_isr_end -  bios_isr_start -  timestamp_overhead;
	nonbios_int_latency = nonbios_isr_end - nonbios_isr_start - timestamp_overhead;

	// print results to Console screen
	System_printf("TIMESTAMP FXN OVERHEAD = [%u] CYCLES\n", timestamp_overhead);
	System_printf("C28x BIOS Interrupt Latency = [%u] CYCLES\n", bios_int_latency);
	System_printf("C28x Zero Latency Interrupt Latency = [%u] CYCLES\n", nonbios_int_latency);
	System_flush();

	// falls into Idle loop here - forever.
}



//---------------------------------------------------------------
// isr_BIOS()
//
// Vector plugged by BIOS app.cfg - INT #38 (INT1)
// BIOS interrupt does NOT use interrupt keyword
//
// Take snapshot of Timestamp timer and then return
//---------------------------------------------------------------
void isr_BIOS(void)
{
	bios_isr_end = Timestamp_get32();		// get snapshot of timer
}



//------------------------------------------------------------------
// isr_nonBIOS()
//
// Vector plugged by Hwi_plug in hardware_init() - INT #67 (INT5)
// Non-bios interrupt uses interrupt keyword
//
// Take snapshot of Timestamp timer and then return
//------------------------------------------------------------------
void interrupt isr_nonBIOS()
{
	nonbios_isr_end = Timestamp_get32();  	// get snapshot of timer
	//asm(" NOP");
//	PieCtrlRegs.PIEACK.bit.ACK3 = 1; // Acknowledge interrupt to PIE or...
//	PieCtrlRegs.PIEACK.all = PIEACK_GROUP1;
}



//----------------------------------------------------------------------
// main()
//
// Call hardware_init() to perform hardware setup for this architecture
// Then call BIOS_start()
// Note:  as soon as BIOS_start() begins, it will run the taskFxn above
//----------------------------------------------------------------------
void main(void) {

	hardware_init();	// target-specific hardware setup

	BIOS_start();    	// does not return
}



//---------------------------------------------------------------------------------
// hardware_init()
//
// for C28x, must PLUG non-BIOS INT into vector table using Hwi_plug()
// and then enable the IER for INT5 and the PIEIER for INT #67
//
// Refer to F2806x_SysCtrl.c for the InitFlash() routine. In this file,
// you will notice a #pragma that specifies this routine to be allocated
// in the "ramfuncs" section per this line of code:
//
// #pragma CODE_SECTION(InitFlash, "ramfuncs");
//
// However, the USER must copy this code from the LOAD address (FLASH)
// to the RUN address (RAM) BEFORE it runs - hence the memcpy() shown
// below.
//
// This processor for this benchmark (F28069 Control Stick) is set to
// run at 50MHz. The DEFAULT wait states are set to 15 wait states
// so that ANYTHING will talk to it from boot. However, this affects the interrupt
// benchmarks. SO, the InitFlash() routine in the folder EWare_F28069_BIOS
// has been modified to set the proper wait states for 50MHz - 1,1,2.
//
// The # wait states you choose for the flash is dependent upon the
// frequency of the device. Reference this wiki topic for more details:
// http://processors.wiki.ti.com/index.php/C2000_Flash_Common_Issues/FAQs
// (almost near the end of that page)
//---------------------------------------------------------------------------------
void hardware_init(void) {

	//plug non-BIOS managed interrupt (67, Group 5) into the vector table and enable it
	Hwi_plug(67, (Hwi_PlugFuncPtr)isr_nonBIOS);			//plug 67 into vector table (group 5)
	Hwi_enableIER(0x10);								//enable INT5 (group 5)
	Hwi_enablePIEIER(5,0x0008);

	memcpy(&RamfuncsRunStart, &RamfuncsLoadStart, (Uint32)&RamfuncsLoadSize);

	InitFlash();

}


