#include "DSP280x_Device.h"     // DSP280x Header file Include File
#include "DSP280x_Examples.h"   // DSP280x Examples Include File

int     GPIO_val = 6;       // Hard-coded value. Choose bit ratio and Extended .

//###########################################################################
// ADC
// Determine when the shift to right justify the data takes place
// Only one of these should be defined as 1.
// The other two should be defined as 0.
#define POST_SHIFT   0  // Shift results after the entire sample table is full
#define INLINE_SHIFT 1  // Shift results as the data is taken from the results regsiter
#define NO_SHIFT     0  // Do not shift the results
// ADC start parameters
#define ADC_MODCLK 0x4   // HSPCLK = SYSCLKOUT/2*ADC_MODCLK2 = 100/(2*4)      = 12.5 MHz
                         //    for 60 MHz devices:    HSPCLK =  60/(2*4)      =  7.5 MHz
#define ADC_CKPS   0x0   // ADC module clock = HSPCLK/1      = 12.5MHz/(1)    = 12.5 MHz
                         //    for 60 MHz devices: ADC module clk = 7.5MHz/(1)=  7.5 MHz
#define ADC_SHCLK  0x1   // S/H width in ADC module periods                   = 2 ADC cycle
#define AVG        1000  // Average sample limit
#define ZOFFSET    0x00  // Average Zero offset
#define BUF_SIZE   1024  // Sample buffer size
// Prototype statements for functions found within this file.
//void Gpio_select(void);
interrupt void cpu_timer0_isr(void);
interrupt void adc_isr(void);        // ADC  End of Sequence ISR
//interrupt void xint1(void); // for rising edge detection
// Global Variables
Uint16 Voltage_VR1 = 0; //changed
Uint16 Current_CR2 = 0; //changed
Uint16 Tempera_TR3 = 0; //changed

Uint16 ADC_voltL = 0; // L- LSB bit
Uint16 ADC_voltM = 0; // M- MSB bit
Uint16 ADC_currL = 0;
Uint16 ADC_currM = 0;
Uint16 ADC_tempL = 0;
Uint16 ADC_tempM = 0;

// Charger Control values
// int ChargerDLC = 1;         // DLC for EV power charger (Data Length Code)
Uint16 Charger_Cu = 0;

Uint16 Charger_EN = 1;      // Factor 1
Uint16 Power_Refe = 10;     // 100 Factor 10 -> 1000 -> 0x3E8
Uint16 IDCVoltMax = 2500;    // 420 Factor 10 -> 4200 -> 0x1068
Uint16 IDCCurrMax = 4;      // 10 Factor 10 -> 100 -> 0x64
Uint16 Charger_Re = 0x0;    // Null 0x0

// Status 1
Uint16 ChargerSID = 0;
Uint16 ChargerFrq = 0;
float  ChargeOutV = 0;
float  ChargeOutA = 0;
float  ChargerInA = 0;
Uint16 ChargerSta = 0;

//Status 2
Uint16 CharAvailP = 0;
Uint16 ChargeMaxP = 0;
Uint16 ChargMainV = 0;
Uint16 ChargeSeTe = 0;
Uint16 ChargePrTe = 0;

//Errors
int E_DCUVS_b0 = 0;
int E_CNTCOMMF = 0;
int Er_MODFAIL = 0;
int E_DCOVS_b0 = 0;
int E_2DspCoFa = 0;
int E_HIGHMAIS = 0;
int E_LOWMAINS = 0;
int E_HIGHTEMP = 0;
int Er_LOWTEMP = 0;
int Er_CURRLIM = 0;

long    TX_A_loopcount = 0; // Counts the # of times eCAN-A transmitted successfully
long    RX_A_loopcount = 0; // Counts the # of times eCAN-A Received successfully

/************************************************************
 This function configure the GPIO port
*************************************************************/
void Gpio_select(void);

/************************************************************
 This function initialises all 32 MBOXes of CAN-A
*************************************************************/
void MBXwrA(void);

/************************************************************
 These functions are invoked based on the GPIO value read
*************************************************************/

void Use_Ext_ID(void);      // Use Extended ID for the Mailbox
void Use_Std_ID(void);      // Use Standard ID for the Mailbox

void Transmit_A(void);      // Transmit Mailbox Data function for eCAN-A
void Transmit_B(void);      // Turn Off Power charger
//void Transmit_C(void);      // Transmit ADC Data to the CAN bus
//void intruder(void); //for the combination of new interrupt rising edge detection

void Transmit_BCM_request(void);      // Transmit BCM_Request
void Transmit_chg_status1(void); // Transmit Charger_status1
void Transmit_chg_status2(void); // Transmit Charger_status2
void Transmit_NLG5_status(void);
void Transmit_NLG5_int_values(void);
void error_warning (void);
void charger_identification (void);
void Transmit_BCM_status (void);

void Transmit_MOS(void);

void Receive_A(void);       // Receive Mailbox Data function for eCAN-A
void chargerID(void);       // Read charger ID from power charger
void status1(void);         // Status message with measurements
void status2(void);         // Status message with measurements
void errors(void);          // Errors
void _500_kbps(void);       // Configure the modules for 500 kbps

void Xmit_Remote_frame(void);   // Transmit a Remote frame
void Resp_Remote_frame(void);   // Respond to a Remote frame

void CAN_traffic(void);         // Generates CAN traffic

void CCR_Enable(void);
void CCR_Disable(void);

void Enable_Mailboxes(void);
void Disable_Mailboxes(void);

/************************************************************
 This function is invoked for read the current value from BMS
 and control the LLC converter
*************************************************************/

void currentBMS(void);
void controLLC(void);

/************************************************************
 This function is invoked for time interrupt
*************************************************************/
void Time_Interrup_Count(void);


/* Create a shadow register structure for the CAN control registers.
This is needed, since, only 32-bit access is allowed to these registers. 16-bit access
 to these registers could potentially corrupt the register contents. This is
 especially true while writing to a bit (or group of bits) among bits 16 - 31 */

struct ECAN_REGS ECanaShadow;

void main(void)
{

// Step 1. Initialise System Control:
// Kill PLL, WatchDog. Enable Peripheral Clocks
// This example function is found in the DSP280x_SysCtrl.c file.
   InitSysCtrl();

//###########################################################################
// ADC
   // Specific clock setting for this example:
   EALLOW;
   SysCtrlRegs.HISPCP.all = ADC_MODCLK; // HSPCLK = SYSCLKOUT/ADC_MODCLK
   EDIS;
   EALLOW;
   SysCtrlRegs.WDCR= 0x00AF;   // Re-enable the watchdog
   EDIS;           // 0x00AF  to NOT disable the Watchdog, Prescaler = 64


// Step 2. Initialise GPIO:
// This example function is found in the DSP280x_Gpio.c file and
// illustrates how to set the GPIO to it's default state.
  InitGpio();

   // Just initialise eCAN pins for this example
   // This function is in DSP280x_ECan.c
  InitECanGpio();

// Step 3. Clear all interrupts and initialise PIE vector table:
// Disable CPU interrupts
   DINT;

   // Initialise GPIO0 to GPIO3 as output.
  Gpio_select();

// Initialise the PIE control registers to their default state.
// The default state is all PIE interrupts disabled and flags
// are cleared.
// This function is found in the DSP280x_PieCtrl.c file.
   InitPieCtrl();   // basic setup of PIE table;

// Disable CPU interrupts and clear all CPU interrupt flags:
   IER = 0x0000;
   IFR = 0x0000;

// Initialise the PIE vector table with pointers to the shell Interrupt
// Service Routines (ISR).
// This will populate the entire table, even if the interrupt
// is not used in this example.  This is useful for debug purposes.
// The shell ISR routines are found in DSP280x_DefaultIsr.c.
// This function is found in DSP280x_PieVect.c.
   InitPieVectTable();

// Interrupts that are used in this example are re-mapped to
// ISR functions found within this file.

// No interrupts used in this example.

// Step 4. Initialise all the Device Peripherals:
// This function is found in DSP280x_InitPeripherals.c
// InitPeripherals(); // Not required for this example

//###########################################################################
// ADC
   InitAdc();          // Basic ADC setup, incl. calibration
   // Specific ADC setup for this example:
   AdcRegs.ADCTRL1.all = 0;
   AdcRegs.ADCTRL1.bit.ACQ_PS = ADC_SHCLK;  // Sequential mode: Sample rate   = 1/[(2+ACQ_PS)*ADC clock in ns]
   //                     = 1/(3*80ns) =4.17MHz (for 100 MHz devices or
   //                     = 1/(3*133ns) = 2.51 MHz (for 60 MHz devices)
   // If Simultaneous mode enabled: Sample rate = 1/[(3+ACQ_PS)*ADC clock in ns]
   AdcRegs.ADCTRL1.bit.SEQ_CASC =1;    // 1=cascaded sequencer
//  AdcRegs.ADCTRL1.bit.CPS = 0;        // divide by 1
   AdcRegs.ADCTRL1.bit.CONT_RUN = 1;   // 1 Setup continuous run; 0 single run mode;
   AdcRegs.ADCTRL1.bit.SEQ_OVRD = 1;       // Enable Sequencer override feature
   AdcRegs.ADCTRL2.all = 0;
   AdcRegs.ADCTRL2.bit.INT_ENA_SEQ1 = 1;   // 1=enable SEQ1 interrupt
   AdcRegs.ADCTRL2.bit.INT_MOD_SEQ1 = 0;   // 0= interrupt after every end of sequence
   AdcRegs.ADCTRL2.bit.EPWM_SOCA_SEQ1 =1;  // 1=SEQ1 start from ePWM_SOCA trigger
   AdcRegs.ADCTRL3.bit.ADCCLKPS = ADC_CKPS;    // 3  ADC clock: FCLK = HSPCLK / 2 * ADCCLKPS
                                       // HSPCLK = 75MHz (see DSP2833x_SysCtrl.c)
                                       // FCLK = 12.5 MHz
   AdcRegs.ADCMAXCONV.all = 2;    // 3 conversions from Sequencer 1
   AdcRegs.ADCCHSELSEQ1.bit.CONV00 = 4; // Setup ADCINA4 (Voltage_VR1) as 1st SEQ1 conv.
   AdcRegs.ADCCHSELSEQ1.bit.CONV01 = 3; // Setup ADCINA3 (Tempera_TR3) as 2nd SEQ1 conv.
   AdcRegs.ADCCHSELSEQ1.bit.CONV02 = 2; // Setup ADCINA2 (Current_CR2) as 3rd SEQ1 conv.
   EPwm2Regs.TBCTL.all = 0xC030;   // Configure timer control register
   /*
    bit 15-14     11:     FREE/SOFT, 11 = ignore emulation suspend
    bit 13        0:      PHSDIR, 0 = count down after sync event
    bit 12-10     000:    CLKDIV, 000 => TBCLK = HSPCLK/1
    bit 9-7       000:    HSPCLKDIV, 000 => HSPCLK = SYSCLKOUT/1
    bit 6         0:      SWFSYNC, 0 = no software sync produced
    bit 5-4       11:     SYNCOSEL, 11 = sync-out disabled
    bit 3         0:      PRDLD, 0 = reload PRD on counter=0
    bit 2         0:      PHSEN, 0 = phase control disabled
    bit 1-0       00:     CTRMODE, 00 = count up mode
   */
   EPwm2Regs.TBPRD = 2999; // TPPRD +1  =  TPWM / (HSPCLKDIV * CLKDIV * TSYSCLK)
                           //           =  20 �s / 6.667 ns
   EPwm2Regs.ETPS.all = 0x0100;            // Configure ADC start by ePWM2
   /*
    bit 15-14     00:     EPWMxSOCB, read-only
    bit 13-12     00:     SOCBPRD, don't care
    bit 11-10     00:     EPWMxSOCA, read-only
    bit 9-8       01:     SOCAPRD, 01 = generate SOCA on first event
    bit 7-4       0000:   reserved
    bit 3-2       00:     INTCNT, don't care
    bit 1-0       00:     INTPRD, don't care
   */
   EPwm2Regs.ETSEL.all = 0x0A00;           // Enable SOCA to ADC
   /*
    bit 15        0:      SOCBEN, 0 = disable SOCB
    bit 14-12     000:    SOCBSEL, don't care
    bit 11        1:      SOCAEN, 1 = enable SOCA
    bit 10-8      010:    SOCASEL, 010 = SOCA on PRD event
    bit 7-4       0000:   reserved
    bit 3         0:      INTEN, 0 = disable interrupt
    bit 2-0       000:    INTSEL, don't care
   */
   EALLOW;
   PieVectTable.TINT0 = &cpu_timer0_isr;
   PieVectTable.ADCINT = &adc_isr;
   EDIS;
   InitCpuTimers();    // basic setup CPU Timer0, 1 and 2
   ConfigCpuTimer(&CpuTimer0,100,100000);
   PieCtrlRegs.PIEIER1.bit.INTx7 = 1;      // CPU Timer 0
   PieCtrlRegs.PIEIER1.bit.INTx6 = 1;      // ADC
   IER |=1;
   EINT;
   ERTM;
   CpuTimer0Regs.TCR.bit.TSS = 0;  // start timer0



   // In this case just initialise eCAN-A and eCAN-B
   // This function is in DSP280x_ECan.c
   InitECan();

   EALLOW;

// Step 5. User specific code:

   Uint16 MaxDCvoltL = 0;
   Uint16 MaxDCvoltM = 0;
   Uint16 MaxDCcurrL = 0;
   Uint16 MaxDCcurrM = 0;
   Uint16 Power_RefL = 0;
   Uint16 Power_RefM = 0;

/*   Uint16 ADC_voltL = 0;
   Uint16 ADC_voltM = 0;
   Uint16 ADC_currL = 0;
   Uint16 ADC_currM = 0;
   Uint16 ADC_tempL = 0;
   Uint16 ADC_tempM = 0;*/

/* Zero out (or initialise) the entire MBX RAM area */

   MBXwrA();

/* Configure Mailbox as Transmit (0) or Receive (1), as needed */

   ECanaShadow.CANMD.all = 0;
   ECanaShadow.CANMD.bit.MD0 = 1;
   ECanaShadow.CANMD.bit.MD1 = 1;
   ECanaShadow.CANMD.bit.MD2 = 1;
   ECanaShadow.CANMD.bit.MD3 = 1;
   ECanaShadow.CANMD.bit.MD4 = 1;
   ECanaShadow.CANMD.bit.MD5 = 1;
   ECanaShadow.CANMD.bit.MD6 = 0; //for BCM_Request
   ECanaShadow.CANMD.bit.MD7 = 0; //for Charger_status_1
   ECanaShadow.CANMD.bit.MD8 = 0; //for NLG5_Status (ID 0x610)
   ECanaShadow.CANMD.bit.MD9 = 0; //for Charger_status_2
   ECanaShadow.CANMD.bit.MD10 = 0; //for NLG5_int_values (ID 0x611)
   ECanaShadow.CANMD.bit.MD11 = 0; //for error and warning
   ECanaShadow.CANMD.bit.MD12 = 0; //for charger identification
   ECanaShadow.CANMD.bit.MD13 = 0; //for BCM_Status
   ECanaShadow.CANMD.bit.MD14 = 0; //for MOS ON-OFF status

   ECanaRegs.CANMD.all = ECanaShadow.CANMD.all;

/* Write to DLC field in Master Control reg */
// By default, in this code, the mailboxes are configured for 8-bytes

   ECanaMboxes.MBOX0.MSGCTRL.bit.DLC = 8;
   ECanaMboxes.MBOX6.MSGCTRL.bit.DLC = 4; //for BCM_Request (4 bytes)
   ECanaMboxes.MBOX7.MSGCTRL.bit.DLC = 8; //for Charger_status_1
   ECanaMboxes.MBOX8.MSGCTRL.bit.DLC = 4; //for NLG5_Status (ID 0x610)
   ECanaMboxes.MBOX9.MSGCTRL.bit.DLC = 8; //for Charger_status_2
   ECanaMboxes.MBOX10.MSGCTRL.bit.DLC = 8; //for NLG5_int_values (ID 0x611)
   ECanaMboxes.MBOX11.MSGCTRL.bit.DLC = 8; //for error and warning
   ECanaMboxes.MBOX12.MSGCTRL.bit.DLC = 8; //for charger identification
   ECanaMboxes.MBOX13.MSGCTRL.bit.DLC = 8;  //for BCM_Status
   ECanaMboxes.MBOX14.MSGCTRL.bit.DLC = 4; //for MOS ON-OFF status

/* Write to the Charger Control mailbox RAM field */
   // 0xFF as mask to remove all but 8 bits
   Power_RefL = Power_Refe & 0xFF;
   // Discards the lowest 8 bits by moving all bits 8 places to the right
   Power_RefM = (Power_Refe >> 8) & 0xFF;

   // 0xFF as mask to remove all but 8 bits
   MaxDCvoltL = IDCVoltMax & 0xFF;
   // Discards the lowest 8 bits by moving all bits 8 places to the right
   MaxDCvoltM = (IDCVoltMax >> 8) & 0xFF;

   // 0xFF as mask to remove all but 8 bits
   MaxDCcurrL = IDCCurrMax & 0xFF;
   // Discards the lowest 8 bits by moving all bits 8 places to the right
   MaxDCcurrM = (IDCCurrMax >> 8) & 0xFF;

   ECanaMboxes.MBOX0.MDL.byte.BYTE0 = Charger_EN;    // CHARGER_ENABLE
   ECanaMboxes.MBOX0.MDL.byte.BYTE1 = Power_RefL;    // CHARGER POWER REFERENCE LSB
   ECanaMboxes.MBOX0.MDL.byte.BYTE2 = Power_RefM;    // CHARGER_POWER_REFERENCE_MSB
   ECanaMboxes.MBOX0.MDL.byte.BYTE3 = MaxDCvoltL;    // CHARGER_MAX_DC_VOLT_LIMIT_LSB
   ECanaMboxes.MBOX0.MDH.byte.BYTE4 = MaxDCvoltM;    // CHARGER_MAX_DC_VOLT_LIMIT_MSB
   ECanaMboxes.MBOX0.MDH.byte.BYTE5 = MaxDCcurrL;    // CHARGER_MAX_DC_CURR_LIMIT_LSB
   ECanaMboxes.MBOX0.MDH.byte.BYTE6 = MaxDCcurrM;    // CHARGER_MAX_DC_CURR_LIMIT_MSB
   ECanaMboxes.MBOX0.MDH.byte.BYTE7 = Charger_Re;    // RESERVED

/* Choose what to do based on GPIO_val value */

   switch(GPIO_val)
   {
       case 0:             // Transmit 8 bytes @ 500 kbps       Extended ID
           Use_Ext_ID();
           _500_kbps();
           // Transmit loop for CAN-A
           while(1)
           //for(i=0; i < TXCOUNT; i++)                // Uncomment this line for finite transmissions
           {
               Transmit_A();
           }
           //asm(" ESTOP0");
           //break;

       case 1:             // Transmit 8 bytes @ 500 kbps      Standard ID
           Use_Std_ID();
           _500_kbps();
           // Transmit loop for CAN-A
           while(1)
           //for(i=0; i < TXCOUNT; i++)                // Uncomment this line for finite transmissions
           {
               Transmit_A();
           }
           //asm(" ESTOP0");
           //break;

       case 2:             // Receive @ 500 kbps
           Use_Ext_ID();
           _500_kbps();
           // Receive loop for CAN-A
           while(1)
           //for(i=0; i < TXCOUNT; i++)                // Uncomment this line for finite transmissions
           {
               Receive_A();
           }
           //asm(" ESTOP0");
           //break;

       case 3:             // Receive @ 500 kbps (node merely provides an ACK)
           Use_Std_ID();
           _500_kbps();
           // Receive loop for CAN-A
           while(1)
           //for(i=0; i < TXCOUNT; i++)                // Uncomment this line for finite transmissions
           {
               Receive_A();
           }
           //asm(" ESTOP0");
           //break;

       case 4:            // Transmit a remote frame @ 500 kbps    Standard ID
           _500_kbps();
           while(1)
           //for(i=0; i < TXCOUNT; i++)                // Uncomment this line for finite transmissions
           {
           Xmit_Remote_frame();
           }
           //asm(" ESTOP0");
           //break;

       case 5:            // Respond to a remote frame @ 500 kbps    Standard ID
           _500_kbps();
           Resp_Remote_frame();
           break;

       case 6:            // Tx a STD frame. Repeat the cycle, but give 0.5 sec delay between frames.
           _500_kbps();
           while(1)
           {
               CAN_traffic();     // Generates CAN traffic
               Time_Interrup_Count();   // Called Timer Interrupt
           }
       }

}

/*****************/
/* End of main() */
/*****************/

void Gpio_select(void)
{
    EALLOW;

    GpioCtrlRegs.GPAMUX1.all = 0;       // GPIO15 ... GPIO0 = General Purpose I/O

    //Set Direction Registers
    GpioCtrlRegs.GPADIR.bit.GPIO0 = 1;  // GPIO0 as output 47: PMW-A
    GpioCtrlRegs.GPADIR.bit.GPIO1 = 1;  // GPIO1 as output 44: PMW-B
    GpioCtrlRegs.GPADIR.bit.GPIO2 = 1;  // GPIO2 as output 45: PMW-C
    GpioCtrlRegs.GPADIR.bit.GPIO3 = 1;  // GPIO3 as output 48: PMW-D
    GpioCtrlRegs.GPADIR.bit.GPIO9 = 0; //GPIO9 as input
// changes
   // GpioCtrlRegs.GPADIR.bit.GPIO9 = 0;  //GPIO9 is the output from zero cross detector
   // GpioCtrlRegs.GPAQSEL1.bit.GPIO9 = 0; // XINT1 Synch to SYSCLKOUT only
    EDIS;
//changes
   // EALLOW;
   // GpioIntRegs.GPIOXINT1SEL.bit.GPIOSEL = 9; //Set XINT1 source to GPIO-pin
   // EDIS;
}

void Time_Interrup_Count(void)
{
    //###########################################################################
    // ADC
    if(CpuTimer0.InterruptCount <10) // wait for 100 ms
       {
           EALLOW;
           SysCtrlRegs.WDKEY = 0x55;       // Service watchdog #1
           EDIS;
       }
    else
    {
        CpuTimer0.InterruptCount = 0;
    }
}

void Use_Ext_ID(void)                           // Use Extended ID for the Mailbox
{
    Disable_Mailboxes();
    ECanaMboxes.MBOX0.MSGID.all = 0x95555555;   // Extended Identifier
    Enable_Mailboxes();
}

void Use_Std_ID(void)                           // Use Standard ID for the Mailbox
{
    Disable_Mailboxes();
    ECanaMboxes.MBOX0.MSGID.all = 0xBFC0000;    // IDE=0, AME=0, AAM=0, STD ID = x2FF Charger control Lab (32 bits - 4 bytes) leave 2 bits empty in beginning
    ECanaMboxes.MBOX1.MSGID.all = 0xC140000;    // eCAN-A -> Std identifier = 0x305 Status #1
    ECanaMboxes.MBOX2.MSGID.all = 0xC180000;    // eCAN-A -> Std identifier = 0x306 Status #2
    ECanaMboxes.MBOX3.MSGID.all = 0xC1C0000;    // eCAN-A -> Std identifier = 0x307 Errors / Warnings
    ECanaMboxes.MBOX4.MSGID.all = 0xC200000;    // eCAN-A -> Std identifier = 0x308 Serial Number - Charger Identification
    ECanaMboxes.MBOX5.MSGID.all = 0x13FC0000;    // IDE=0, AME=0, AAM=0, STD ID = x4FF Charger control of EV
    ECanaMboxes.MBOX6.MSGID.all = 0x10000000; // IDE=0, AME=0, AAM=0, STD ID = x400 BCM_Request 0x10000000
    ECanaMboxes.MBOX7.MSGID.all = 0x14140000; // IDE=0, AME=0, AAM=0, STD ID = x505 Charger_status_1 0x14140000
    ECanaMboxes.MBOX8.MSGID.all = 0x18400000; // IDE=0, AME=0, AAM=0, STD ID = x610 NLG5_Status 0x18400000
    ECanaMboxes.MBOX9.MSGID.all = 0x14180000; //Charger_status_2
    ECanaMboxes.MBOX10.MSGID.all = 0x18440000;    // IDE=0, AME=0, AAM=0, STD ID = x611 NLG5_Int_Values
    ECanaMboxes.MBOX11.MSGID.all = 0x141C0000; //Error and warning
    ECanaMboxes.MBOX12.MSGID.all = 0x14200000; //Charger identification
    ECanaMboxes.MBOX13.MSGID.all = 0x10400000; //BCM_status
    ECanaMboxes.MBOX14.MSGID.all = 0x16040000; // IDE=0, AME=0, AAM=0, STD ID = x581 MOS ON-OFF status

    Enable_Mailboxes();
}

void Transmit_A(void)                                   // Transmit Mailbox Data
{
    ECanaMboxes.MBOX0.MDL.byte.BYTE0 = 1;               // Turn On Power Charger

    ECanaShadow.CANTRS.all = 0;
    ECanaShadow.CANTRS.bit.TRS0 = 1;                    // Set TRS for Mailbox 0
    ECanaRegs.CANTRS.all = ECanaShadow.CANTRS.all;

    do
    {
        ECanaShadow.CANTA.all = ECanaRegs.CANTA.all;    // Wait for TA0 bit to be set..
    } while(ECanaShadow.CANTA.bit.TA0 == 0 );

    ECanaShadow.CANTA.all = 0;
    ECanaShadow.CANTA.bit.TA0 = 1;                      // Clear TA0
    ECanaRegs.CANTA.all = ECanaShadow.CANTA.all;

    TX_A_loopcount ++;
}

void Transmit_B(void)                                   // Transmit Mailbox Data
{
    ECanaMboxes.MBOX0.MDL.byte.BYTE0 = 0;               // Turn Off Power Charger

    ECanaShadow.CANTRS.all = 0;
    ECanaShadow.CANTRS.bit.TRS0 = 1;                    // Set TRS for Mailbox 0
    ECanaRegs.CANTRS.all = ECanaShadow.CANTRS.all;

    do
    {
        ECanaShadow.CANTA.all = ECanaRegs.CANTA.all;    // Wait for TA0 bit to be set..
    } while(ECanaShadow.CANTA.bit.TA0 == 0 );

    ECanaShadow.CANTA.all = 0;
    ECanaShadow.CANTA.bit.TA0 = 1;                      // Clear TA0
    ECanaRegs.CANTA.all = ECanaShadow.CANTA.all;
}
/*
void Transmit_C(void)                                   // Transmit Mailbox Data
{
    ECanaShadow.CANTRS.all = 0;
    ECanaShadow.CANTRS.bit.TRS6 = 1;                    // Set TRS for Mailbox 6
    ECanaRegs.CANTRS.all = ECanaShadow.CANTRS.all;

    do
    {
        ECanaShadow.CANTA.all = ECanaRegs.CANTA.all;    // Wait for TA6 bit to be set..
    } while(ECanaShadow.CANTA.bit.TA6 == 0 );

    ECanaShadow.CANTA.all = 0;
    ECanaShadow.CANTA.bit.TA6 = 1;                      // Clear TA6
    ECanaRegs.CANTA.all = ECanaShadow.CANTA.all;

    DELAY_US(30000);        // Delay for not use the Bus all the time
    TX_A_loopcount ++;
}*/

void Transmit_BCM_request(void)                                   // Transmit Mailbox Data BCM_Request
{
    ECanaShadow.CANTRS.all = 0;
    ECanaShadow.CANTRS.bit.TRS6 = 1;                    // Set TRS for Mailbox 6
    ECanaRegs.CANTRS.all = ECanaShadow.CANTRS.all;

    do
    {
        ECanaShadow.CANTA.all = ECanaRegs.CANTA.all;    // Wait for TA6 bit to be set..
    } while(ECanaShadow.CANTA.bit.TA6 == 0 );

    ECanaShadow.CANTA.all = 0;
    ECanaShadow.CANTA.bit.TA6 = 1;                      // Clear TA6
    ECanaRegs.CANTA.all = ECanaShadow.CANTA.all;

    DELAY_US(1000);        // Delay for not use the Bus all the time
    TX_A_loopcount ++;
}

void Transmit_chg_status1(void)                                   // Transmit Mailbox Charger Status 1
{
    ECanaShadow.CANTRS.all = 0;
    ECanaShadow.CANTRS.bit.TRS7 = 1;                    // Set TRS for Mailbox 7
    ECanaRegs.CANTRS.all = ECanaShadow.CANTRS.all;

    do
    {
        ECanaShadow.CANTA.all = ECanaRegs.CANTA.all;    // Wait for TA7 bit to be set..
    } while(ECanaShadow.CANTA.bit.TA7 == 0 );

    ECanaShadow.CANTA.all = 0;
    ECanaShadow.CANTA.bit.TA7 = 1;                      // Clear TA7
    ECanaRegs.CANTA.all = ECanaShadow.CANTA.all;

    DELAY_US(1000);        // Delay for not use the Bus all the time
    TX_A_loopcount ++;
}

void Transmit_NLG5_status(void)
{
    ECanaShadow.CANTRS.all = 0;
    ECanaShadow.CANTRS.bit.TRS8 = 1;                    // Set TRS for Mailbox 8
    ECanaRegs.CANTRS.all = ECanaShadow.CANTRS.all;

    do
    {
        ECanaShadow.CANTA.all = ECanaRegs.CANTA.all;    // Wait for TA8 bit to be set..
    } while(ECanaShadow.CANTA.bit.TA8 == 0 );

    ECanaShadow.CANTA.all = 0;
    ECanaShadow.CANTA.bit.TA8 = 1;                      // Clear TA8
    ECanaRegs.CANTA.all = ECanaShadow.CANTA.all;

    DELAY_US(1000);        // Delay for not use the Bus all the time
    TX_A_loopcount ++;
}

void Transmit_chg_status2(void)
{
    ECanaShadow.CANTRS.all = 0;
    ECanaShadow.CANTRS.bit.TRS9 = 1;                    // Set TRS for Mailbox 9
    ECanaRegs.CANTRS.all = ECanaShadow.CANTRS.all;

    do
    {
        ECanaShadow.CANTA.all = ECanaRegs.CANTA.all;    // Wait for TA9 bit to be set..
    } while(ECanaShadow.CANTA.bit.TA9 == 0 );

    ECanaShadow.CANTA.all = 0;
    ECanaShadow.CANTA.bit.TA9 = 1;                      // Clear TA9
    ECanaRegs.CANTA.all = ECanaShadow.CANTA.all;

    DELAY_US(1000);        // Delay for not use the Bus all the time
    TX_A_loopcount ++;
}

void Transmit_NLG5_int_values(void)
{
    ECanaShadow.CANTRS.all = 0;
    ECanaShadow.CANTRS.bit.TRS10 = 1;                    // Set TRS for Mailbox 10
    ECanaRegs.CANTRS.all = ECanaShadow.CANTRS.all;

    do
    {
        ECanaShadow.CANTA.all = ECanaRegs.CANTA.all;    // Wait for TA10 bit to be set..
    } while(ECanaShadow.CANTA.bit.TA10 == 0 );

    ECanaShadow.CANTA.all = 0;
    ECanaShadow.CANTA.bit.TA10 = 1;                      // Clear TA10
    ECanaRegs.CANTA.all = ECanaShadow.CANTA.all;

    DELAY_US(1000);        // Delay for not use the Bus all the time
    TX_A_loopcount ++;
}

void error_warning(void)
{
    ECanaShadow.CANTRS.all = 0;
    ECanaShadow.CANTRS.bit.TRS11 = 1;                    // Set TRS for Mailbox 11
    ECanaRegs.CANTRS.all = ECanaShadow.CANTRS.all;

    do
    {
        ECanaShadow.CANTA.all = ECanaRegs.CANTA.all;    // Wait for TA11 bit to be set..
    } while(ECanaShadow.CANTA.bit.TA11 == 0 );

    ECanaShadow.CANTA.all = 0;
    ECanaShadow.CANTA.bit.TA11 = 1;                      // Clear TA11
    ECanaRegs.CANTA.all = ECanaShadow.CANTA.all;

    DELAY_US(1000);        // Delay for not use the Bus all the time
    TX_A_loopcount ++;
}

void charger_identification(void)
{
    ECanaShadow.CANTRS.all = 0;
    ECanaShadow.CANTRS.bit.TRS12 = 1;                    // Set TRS for Mailbox 12
    ECanaRegs.CANTRS.all = ECanaShadow.CANTRS.all;

    do
    {
        ECanaShadow.CANTA.all = ECanaRegs.CANTA.all;    // Wait for TA12 bit to be set..
    } while(ECanaShadow.CANTA.bit.TA12 == 0 );

    ECanaShadow.CANTA.all = 0;
    ECanaShadow.CANTA.bit.TA12 = 1;                      // Clear TA12
    ECanaRegs.CANTA.all = ECanaShadow.CANTA.all;

    DELAY_US(1000);        // Delay for not use the Bus all the time
    TX_A_loopcount ++;
}

void Transmit_BCM_status(void)
{
    ECanaShadow.CANTRS.all = 0;
    ECanaShadow.CANTRS.bit.TRS13 = 1;                    // Set TRS for Mailbox 13
    ECanaRegs.CANTRS.all = ECanaShadow.CANTRS.all;

    do
    {
        ECanaShadow.CANTA.all = ECanaRegs.CANTA.all;    // Wait for TA13 bit to be set..
    } while(ECanaShadow.CANTA.bit.TA13 == 0 );

    ECanaShadow.CANTA.all = 0;
    ECanaShadow.CANTA.bit.TA13 = 1;                      // Clear TA13
    ECanaRegs.CANTA.all = ECanaShadow.CANTA.all;

    DELAY_US(1000);        // Delay for not use the Bus all the time
    TX_A_loopcount ++;
}

void Transmit_MOS(void)
{
    ECanaShadow.CANTRS.all = 0;
    ECanaShadow.CANTRS.bit.TRS14 = 1;                    // Set TRS for Mailbox 14
    ECanaRegs.CANTRS.all = ECanaShadow.CANTRS.all;

    do
    {
        ECanaShadow.CANTA.all = ECanaRegs.CANTA.all;    // Wait for TA14 bit to be set..
    } while(ECanaShadow.CANTA.bit.TA14 == 0 );

    ECanaShadow.CANTA.all = 0;
    ECanaShadow.CANTA.bit.TA14 = 1;                      // Clear TA14
    ECanaRegs.CANTA.all = ECanaShadow.CANTA.all;

    DELAY_US(1000);        // Delay for not use the Bus all the time
    TX_A_loopcount ++;
}


void Receive_A(void)                // Receive Data
{
    Disable_Mailboxes();
    Use_Std_ID();
    Enable_Mailboxes();

    status1();

    status2();

    errors();

    chargerID();

    currentBMS();

//    ECanaRegs.CANTRS.all = 0x0000001E;
//    while(ECanaRegs.CANRMP.all != 0x0000001E ) {}  // Wait for RMP4 to be set..

//    ECanaRegs.CANRMP.all = 0x0000001E;            // Clear RMP bit and start

    RX_A_loopcount++ ;                            // Receiver counter
}

void currentBMS(void)
{
    Uint16 ChargerbCu = 0;

    // Reading DLC from EV power charger
//     ChargerDLC = (int) ECanaMboxes.MBOX5.MSGCTRL.bit.DLC;

    // Getting current from BMS
    ChargerbCu = (Uint16) ECanaMboxes.MBOX5.MDH.byte.BYTE6; //MS byte current BMS
    Charger_Cu = (Uint16) ECanaMboxes.MBOX5.MDH.byte.BYTE5; //LS byte current BMS
    // from the BMS current value MSB:LSB
    Charger_Cu |= (ChargerbCu << 8);
}

void chargerID(void)
{
    Uint16 ChargerbID = 0;

    // Getting the Charger ID from CAN message "serial number"
    ChargerbID = (Uint16) ECanaMboxes.MBOX4.MDH.byte.BYTE7; //MS byte Charger ID
    ChargerSID = (Uint16) ECanaMboxes.MBOX4.MDH.byte.BYTE6; //LS byte Charger ID
    // form the CHarger ID from the MSB:LSB
    ChargerSID |= (ChargerbID << 8);
}

void status1(void)
{
    Uint16 ChargeDcVM = 0;
    Uint16 ChargeDcVL = 0;
    Uint16 ChargeDcAM = 0;
    Uint16 ChargeDcAL = 0;
    Uint16 ChargeMaAM = 0;
    Uint16 ChargeMaAL = 0;

    // Getting Charger Info from CAN message "Charger status 1"
    // Input voltage frequency measurement
    ChargerFrq = (Uint16) ECanaMboxes.MBOX1.MDH.byte.BYTE7; //Charger Frequency

    // Output voltage measurement
    ChargeDcVM = (Uint16) ECanaMboxes.MBOX1.MDH.byte.BYTE6; //MS byte DC V
    ChargeDcVL = (Uint16) ECanaMboxes.MBOX1.MDH.byte.BYTE5; //LS byte DC V
    // Output voltage measurement from the MSB:LSB
    // // Discards the highest 8 bits by moving all bits 8 places to the left
    ChargeDcVL |= (ChargeDcVM << 8);
    ChargeOutV = ChargeDcVL / (1.0 * 10);

    // Output current measurement
    ChargeDcAM = (Uint16) ECanaMboxes.MBOX1.MDH.byte.BYTE4; //MS byte DC A
    ChargeDcAL = (Uint16) ECanaMboxes.MBOX1.MDL.byte.BYTE3; //LS byte DC A
    // Output current measurement from the MSB:LSB
    ChargeDcAL |= (ChargeDcAM << 8);
    ChargeOutA = ChargeDcAL / (1.0 * 10);

    // Input current measurement
    ChargeMaAM = (Uint16) ECanaMboxes.MBOX1.MDL.byte.BYTE2; //MS Charger A
    ChargeMaAL = (Uint16) ECanaMboxes.MBOX1.MDL.byte.BYTE1; //LS Charger A
    // Input current measurement from the MSB:LSB
    ChargeMaAL |= ((ChargeMaAM << 8));
    ChargerInA = ChargeMaAL / (1.0 * 10);

    // Charger status
    ChargerSta = (Uint16) ECanaMboxes.MBOX1.MDL.byte.BYTE0; //Charger Status
}

void status2(void)
{
    Uint16 ChargMaxPM = 0;
    Uint16 ChargMaiVM = 0;

    // Getting Charger Info from CAN message "Charger status 2"
    // Maximum available power (may be less than max power due temp and mains derating)
    CharAvailP = (Uint16) ECanaMboxes.MBOX2.MDH.byte.BYTE6; //Charger Frequency
    CharAvailP = (CharAvailP / 2);

    // Maximum charger power (constant value)
    ChargMaxPM = (Uint16) ECanaMboxes.MBOX2.MDH.byte.BYTE5; //MS byte DC V
    ChargeMaxP = (Uint16) ECanaMboxes.MBOX2.MDH.byte.BYTE4; //LS byte DC V
    // Output power measurement from the MSB:LSB
    // // Discards the highest 8 bits by moving all bits 8 places to the left
    ChargeMaxP |= (ChargMaxPM << 8);

    // Input voltage measurement
    ChargMaiVM = (Uint16) ECanaMboxes.MBOX2.MDL.byte.BYTE3; //MS byte DC A
    ChargMainV = (Uint16) ECanaMboxes.MBOX2.MDL.byte.BYTE2; //LS byte DC A
    // Output current measurement from the MSB:LSB
    ChargMainV |= (ChargMaiVM << 8);

    // Secondary temperature measurement
    ChargeSeTe = (Uint16) ECanaMboxes.MBOX2.MDL.byte.BYTE1; //MS Charger A

    // Primary temperature measurement
    ChargePrTe = (Uint16) ECanaMboxes.MBOX2.MDL.byte.BYTE0; //Charger Status
}

void errors(void)
{
    Uint16 E_Char_by2 = 0;
    Uint16 E_Char_by1 = 0;
    Uint16 E_Char_by0 = 0;

    // Getting Charger Info from CAN message "Errors"
    // Control system communication timeout
    E_Char_by2 = (Uint16) ECanaMboxes.MBOX3.MDL.byte.BYTE2; //LS byte DC A
    // DCUVS DC under-voltage shutdown
    E_DCUVS_b0 = E_Char_by2 & 0x1;
    // E_CNTCOMMFAIL Control system communication timeout.
    E_CNTCOMMF = ((E_Char_by2 & 0x3) >> 1);

    E_Char_by1 = (Uint16) ECanaMboxes.MBOX3.MDL.byte.BYTE1; //LS byte DC A
    // MODFAIL Transformer failure.
    Er_MODFAIL = ((E_Char_by1 & 0x3) >> 1);

    E_Char_by0 = (Uint16) ECanaMboxes.MBOX3.MDL.byte.BYTE0; //LS byte DC A
    // DCOVS DC over-voltage shutdown
    E_DCOVS_b0 = E_Char_by0 & 0x1;
    // SCICOMMFAIL Secondary DSP is unable to communicate with primary DSP over SCI
    E_2DspCoFa = ((E_Char_by0 & 0x4) >> 2);
    // HIGHMAINS High mains shutdown
    E_HIGHMAIS = ((E_Char_by0 & 0x8) >> 3);
    // LOWMAINS Low mains shutdown
    E_LOWMAINS = ((E_Char_by0 & 0x10) >> 4);
    // HIGHTEMP High temperature shutdown (Primary or secondary)
    E_HIGHTEMP = ((E_Char_by0 & 0x20) >> 5);
    // LOWTEMP Low temperature shutdown (Primary or secondary)
    Er_LOWTEMP = ((E_Char_by0 & 0x40) >> 6);
    // CURRLIM Charger is in current limit when it derates due to low
    // input voltage or high temperature and the power demand is higher
    // than available power
    Er_CURRLIM = ((E_Char_by0 & 0x80) >> 7);
}

void _500_kbps(void)        // Configure the modules for 500 kbps
{
    CCR_Enable();
    ECanaShadow.CANBTC.bit.BRPREG = 19;
    CCR_Disable();
}

void Xmit_Remote_frame(void)                    // Transmit a Remote frame
{
    ECanaMboxes.MBOX4.MSGID.all  = 0xC200000;  // eCAN-A -> Std identifier (Will eventually transmit a Remote frame)
    ECanaMboxes.MBOX4.MSGCTRL.bit.RTR = 1;      // eCAN-A -> RTR bit will be cleared upon successful reception of data
    ECanaMboxes.MBOX4.MSGCTRL.bit.DLC = 8;
    Enable_Mailboxes();
    ECanaRegs.CANTRS.all = 0x00000010;          // MBOX4
    DELAY_US(1000000);
}

void Resp_Remote_frame(void)    // Respond to a Remote frame
{
    Enable_Mailboxes();
}

void CAN_traffic(void)          // Generates CAN traffic
{
    Disable_Mailboxes();
    Use_Std_ID();
    Enable_Mailboxes();
    Receive_A();

    //Disable_Mailboxes();
    //Use_Std_ID();
    //Enable_Mailboxes();
    //Transmit_C();

    Disable_Mailboxes();
    Use_Std_ID();
    Enable_Mailboxes();
    Transmit_BCM_request();

    Disable_Mailboxes();
    Use_Std_ID();
    Enable_Mailboxes();
    Transmit_chg_status1();

    Disable_Mailboxes();
    Use_Std_ID();
    Enable_Mailboxes();
    Transmit_NLG5_status();

    Disable_Mailboxes();
    Use_Std_ID();
    Enable_Mailboxes();
    Transmit_chg_status2();

    Disable_Mailboxes();
    Use_Std_ID();
    Enable_Mailboxes();
    Transmit_NLG5_int_values();

    Disable_Mailboxes();
    Use_Std_ID();
    Enable_Mailboxes();
    error_warning();

    Disable_Mailboxes();
    Use_Std_ID();
    Enable_Mailboxes();
    charger_identification();

    Disable_Mailboxes();
    Use_Std_ID();
    Enable_Mailboxes();
    Transmit_BCM_status();

    Disable_Mailboxes();
    Use_Std_ID();
    Enable_Mailboxes();
    Transmit_MOS();

    controLLC();
}


void controLLC(void)
{
    if (Charger_Cu >= 96)   //Current Between bigger than 9.6 [A]
         {
            Disable_Mailboxes();
            Use_Std_ID();
            Enable_Mailboxes();

            InitSysCtrl(); //disables the watchdog
            InitGpio();

            DINT;
            InitPieCtrl();// The purpose of this function is to clear all pending PIE-Interrupts and to disable all PIE interrupt lines. This is a useful step when we would like to initialize the PIE-unit.

            IER = 0x0000; //initialization IER register
            IFR = 0x0000;

            InitPieVectTable();

            EALLOW;
            GpioCtrlRegs.GPAMUX1.bit.GPIO0 = 0;
            GpioCtrlRegs.GPADIR.bit.GPIO0 = 1; //Output PWM A
            GpioDataRegs.GPACLEAR.bit.GPIO0 = 1; //


            GpioCtrlRegs.GPAMUX1.bit.GPIO1 = 0;
            GpioCtrlRegs.GPADIR.bit.GPIO1 = 1; //Output PWM B
            GpioDataRegs.GPASET.bit.GPIO1 = 1; //



            GpioCtrlRegs.GPAMUX1.bit.GPIO2 = 0;
            GpioCtrlRegs.GPADIR.bit.GPIO2 = 1; //Output PWM C
            GpioDataRegs.GPASET.bit.GPIO2 = 1; //



            GpioCtrlRegs.GPAMUX1.bit.GPIO3 = 0;
            GpioCtrlRegs.GPADIR.bit.GPIO3 = 1; //Output PWM D
            GpioDataRegs.GPASET.bit.GPIO3 = 1; //

            EDIS;

            EINT; // Enable Global interrupt INTM
            ERTM; // Enable Global realtime interrupt DBGM

            EALLOW;
            GpioCtrlRegs.GPAMUX1.bit.GPIO11 = 0; // GPIO
            GpioCtrlRegs.GPADIR.bit.GPIO11 = 0; // input ZERO cross is fed here
            GpioCtrlRegs.GPAQSEL1.bit.GPIO11 = 0; // XINT1 Synch to SYSCLKOUT only
            EDIS;

            EALLOW;
            GpioIntRegs.GPIOXINT1SEL.bit.GPIOSEL = 11; //Set XINT1 source to GPIO-pin
            EDIS;

            //DELAY_US(1*5);//5us
            XIntruptRegs.XINT1CR.bit.POLARITY = 1; // 0 - Falling edge interrupt | 1 - Rising edge interrupt
            XIntruptRegs.XINT1CR.bit.ENABLE = 1; // Enable XINT1

            EALLOW; // This is needed to write to EALLOW protected registers
            PieVectTable.XINT1 = &XINT1_ISR;
            EDIS; // This is needed to disable write to EALLOW protected registers

            PieCtrlRegs.PIECTRL.bit.ENPIE = 1; // Enable the PIE block
            PieCtrlRegs.PIEIER1.bit.INTx4 = 1; // Enable PIE Group 1 INT4
            IER |= M_INT1; // Enable CPU INT1


            }


//            Transmit_A();   // ON power charger
            //Logic low to the gate input for HS MOSFETs Q1 and Q2 (from driver)
            // GPASET put the port HIGH

            //interrup routine
/*
            XIntruptRegs.XINT1CR.bit.POLARITY = 1; // 0 - Falling edge interrupt | 1 - Rising edge interrupt
            XIntruptRegs.XINT1CR.bit.ENABLE = 1; // Enable XINT1

            EALLOW; // This is needed to write to EALLOW protected registers
            PieVectTable.XINT1 = &XINT1_ISR;
            EDIS; // This is needed to disable write to EALLOW protected registers

            PieCtrlRegs.PIECTRL.bit.ENPIE = 1; // Enable the PIE block
            PieCtrlRegs.PIEIER1.bit.INTx4 = 1; // Enable PIE Group 1 INT4
            IER |= M_INT1; // Enable CPU INT1
*/




        else
        {
            Disable_Mailboxes();
            Use_Std_ID();
            Enable_Mailboxes();
//            Transmit_B();   // OFF the power charger
            // GPACLEAR put the port in low
            GpioDataRegs.GPACLEAR.bit.GPIO0= 1;
           GpioDataRegs.GPACLEAR.bit.GPIO1= 1;

            GpioDataRegs.GPACLEAR.bit.GPIO2 = 1;
            GpioDataRegs.GPACLEAR.bit.GPIO3 = 1;

                     ECanaMboxes.MBOX14.MDL.byte.BYTE0 = 0x0;          // PWM A status
                     ECanaMboxes.MBOX14.MDL.byte.BYTE1 = 0x0;         //  PWM B status
                     ECanaMboxes.MBOX14.MDL.byte.BYTE2 = 0x0;         //  PWM C status
                     ECanaMboxes.MBOX14.MDL.byte.BYTE3 = 0x0;         //  PWM D status
        }

    //Transmit to the CAN Bus

         // 0xFF as mask to remove all but 8 bits
         ADC_voltL = Voltage_VR1 & 0xFF;
         // Discards the lowest 8 bits by moving all bits 8 places to the right
         ADC_voltM = (Voltage_VR1 >> 8) & 0xFF;

         // 0xFF as mask to remove all but 8 bits
         ADC_currL = Current_CR2 & 0xFF;
         // Discards the lowest 8 bits by moving all bits 8 places to the right
         ADC_currM = (Current_CR2 >> 8) & 0xFF;

         // 0xFF as mask to remove all but 8 bits
         ADC_tempL = Tempera_TR3 & 0xFF;
         // Discards the lowest 8 bits by moving all bits 8 places to the right
         ADC_tempM = (Tempera_TR3 >> 8) & 0xFF;

         //BCM_Request
         ECanaMboxes.MBOX6.MDL.byte.BYTE0 = 0x0D;          // hsc_bcm_enable, hsc_bcm_epo, hsc_bcm_mainc_close, hsc_bcm_leakage_ena
         ECanaMboxes.MBOX6.MDL.byte.BYTE1 = 0x0;         // no value
         ECanaMboxes.MBOX6.MDL.byte.BYTE2 = 0x0;         // no value
         ECanaMboxes.MBOX6.MDL.byte.BYTE3 = 0x0;         // no value
         ECanaMboxes.MBOX6.MDH.byte.BYTE4 = ADC_currM;    // LLC converter ADC current value MSB

         //Charger-status1
         ECanaMboxes.MBOX7.MDL.byte.BYTE0 = 0x02;  // Mailbox 7 - Charger status 1 - Byte 0 status (2 = Charge)
         ECanaMboxes.MBOX7.MDL.byte.BYTE1 = 0x79; //Byte 1 Charger mains current LSB (12 A)
         ECanaMboxes.MBOX7.MDL.byte.BYTE2 = 0x00; //Byte 2 Charger mains current MSB
         ECanaMboxes.MBOX7.MDL.byte.BYTE3 = 0x00; //Byte 3 Charger DC current LSB (Changes according to ADC input)
         ECanaMboxes.MBOX7.MDH.byte.BYTE4 = 0x00; //Byte 4 Charger DC current MSB
         ECanaMboxes.MBOX7.MDH.byte.BYTE5 = ADC_voltL; // Byte 5 Charger DC voltage LSB (Changes according to ADC input)
         ECanaMboxes.MBOX7.MDH.byte.BYTE6 = ADC_voltM; // Byte 6 Charger DC voltage MSB
         ECanaMboxes.MBOX7.MDH.byte.BYTE7 = 0x32; //Byte 7 Charger mains frequency (50 Hz)

         //NLG5_status id 0x610
         ECanaMboxes.MBOX8.MDL.byte.BYTE0 = 0x80;          // chg_en = 1
         ECanaMboxes.MBOX8.MDL.byte.BYTE1 = 0x0;         // no value
         ECanaMboxes.MBOX8.MDL.byte.BYTE2 = 0x0;         // no value
         ECanaMboxes.MBOX8.MDL.byte.BYTE3 = 0x0;         // no value

         //charger_status2
         ECanaMboxes.MBOX9.MDL.byte.BYTE0 = 0x0;
         ECanaMboxes.MBOX9.MDL.byte.BYTE1 = 0x0;
         ECanaMboxes.MBOX9.MDL.byte.BYTE2 = 0x0;
         ECanaMboxes.MBOX9.MDL.byte.BYTE3 = 0x0;
         ECanaMboxes.MBOX9.MDH.byte.BYTE4 = 0x0;
         ECanaMboxes.MBOX9.MDH.byte.BYTE5 = 0x0;
         ECanaMboxes.MBOX9.MDH.byte.BYTE6 = 0x0; //actual DC current
         ECanaMboxes.MBOX9.MDH.byte.BYTE7 = 0x0; //actual DC current

         //NLG5_int_values
         ECanaMboxes.MBOX10.MDL.byte.BYTE0 = 0x0;
         ECanaMboxes.MBOX10.MDL.byte.BYTE1 = 0x0;
         ECanaMboxes.MBOX10.MDL.byte.BYTE2 = 0x0;
         ECanaMboxes.MBOX10.MDL.byte.BYTE3 = 0x0;
         ECanaMboxes.MBOX10.MDH.byte.BYTE4 = 0x0;
         ECanaMboxes.MBOX10.MDH.byte.BYTE5 = 0x0;
         ECanaMboxes.MBOX10.MDH.byte.BYTE6 = 0x0;
         ECanaMboxes.MBOX10.MDH.byte.BYTE7 = 0x0;

         //error_warning
         ECanaMboxes.MBOX11.MDL.all = 0x0;

         //charger_identification
         ECanaMboxes.MBOX12.MDL.byte.BYTE0 = 0x0;
         ECanaMboxes.MBOX12.MDL.byte.BYTE1 = 0x0;
         ECanaMboxes.MBOX12.MDL.byte.BYTE2 = 0x0;
         ECanaMboxes.MBOX12.MDL.byte.BYTE3 = 0x0;
         ECanaMboxes.MBOX12.MDH.byte.BYTE4 = 0x0;
         ECanaMboxes.MBOX12.MDH.byte.BYTE5 = 0x0;
         ECanaMboxes.MBOX12.MDH.byte.BYTE6 = 0x0;
         ECanaMboxes.MBOX12.MDH.byte.BYTE7 = 0x0;

         //transmit_BCM_status
         ECanaMboxes.MBOX13.MDL.byte.BYTE0 = 0x0;
         ECanaMboxes.MBOX13.MDL.byte.BYTE1 = 0x0;
         ECanaMboxes.MBOX13.MDL.byte.BYTE2 = 0x0;
         ECanaMboxes.MBOX13.MDL.byte.BYTE3 = 0x0;
         ECanaMboxes.MBOX13.MDH.byte.BYTE4 = 0x0;
         ECanaMboxes.MBOX13.MDH.byte.BYTE5 = 0x0;
         ECanaMboxes.MBOX13.MDH.byte.BYTE6 = 0x0;
         ECanaMboxes.MBOX13.MDH.byte.BYTE7 = 0x0;
}

void CCR_Enable(void)
{
    ECanaShadow.CANMC.all = ECanaRegs.CANMC.all;
    ECanaShadow.CANMC.bit.CCR = 1 ;            // Set CCR = 1 Software request write access to CANBTC, CANGAM, LAM[0], LAM[3]
  //  ECanaShadow.CANMC.bit.SCB = 1;
    ECanaRegs.CANMC.all = ECanaShadow.CANMC.all;

    // Wait until the CPU has been granted permission to change the configuration registers
    do
    {
    ECanaShadow.CANES.all = ECanaRegs.CANES.all;
    } while(ECanaShadow.CANES.bit.CCE != 1 );       // Wait for CCE bit to be set..

    ECanaShadow.CANBTC.all = ECanaRegs.CANBTC.all;
}

void CCR_Disable(void)
{
    ECanaRegs.CANBTC.all = ECanaShadow.CANBTC.all;

    ECanaShadow.CANMC.all = ECanaRegs.CANMC.all;
    ECanaShadow.CANMC.bit.CCR = 0 ;            // Set CCR = 0 Software request normal operation
    ECanaRegs.CANMC.all = ECanaShadow.CANMC.all;

    // Wait until the CPU no longer has permission to change the configuration registers
    do
    {
        ECanaShadow.CANES.all = ECanaRegs.CANES.all;
    } while(ECanaShadow.CANES.bit.CCE != 0 );       // Wait for CCE bit to be  cleared..
}

void Enable_Mailboxes(void)         /* Enable Mailboxes */
{
    ECanaShadow.CANME.all = 0;
    ECanaShadow.CANME.bit.ME0 = 1;
    ECanaShadow.CANME.bit.ME1 = 1;
    ECanaShadow.CANME.bit.ME2 = 1;
    ECanaShadow.CANME.bit.ME3 = 1;
    ECanaShadow.CANME.bit.ME4 = 1;
    ECanaShadow.CANME.bit.ME5 = 1;
    ECanaShadow.CANME.bit.ME6 = 1;//for BCM_Request
    ECanaShadow.CANME.bit.ME7 = 1; //for Charger Status 1
    ECanaShadow.CANME.bit.ME8 = 1; //for NLG5_status
    ECanaShadow.CANME.bit.ME9 = 1; //for Charger Status 2
    ECanaShadow.CANME.bit.ME10 = 1;
    ECanaShadow.CANME.bit.ME11 = 1;
    ECanaShadow.CANME.bit.ME12 = 1;
    ECanaShadow.CANME.bit.ME13 = 1;
    ECanaShadow.CANME.bit.ME14 = 1;

    ECanaRegs.CANME.all = ECanaShadow.CANME.all;
}

void Disable_Mailboxes(void)         /* Disable Mailboxes */
{
    ECanaRegs.CANME.all = 0;
}

/* Zero-out the MBX RAM of CAN-A */

void MBXwrA()
    {
    int j;
    volatile struct MBOX *Mailbox  = (void *) 0x6100;

        for(j=0; j<32; j++)
        {
            Mailbox->MSGID.all = 0;
            Mailbox->MSGCTRL.all = 0;
            Mailbox->MDH.all = 0;
            Mailbox->MDL.all = 0;
            Mailbox = Mailbox + 1;

        }
    }


//###########################################################################
// ADC


interrupt void cpu_timer0_isr(void)
{
    CpuTimer0.InterruptCount++;
    EALLOW;
    SysCtrlRegs.WDKEY = 0xAA;   // service WD #2
    EDIS;
    PieCtrlRegs.PIEACK.all = PIEACK_GROUP1;
}



interrupt void adc_isr(void)
{
    float VR1;
    float CR2;
    float TR3;


    VR1 = (AdcMirror.ADCRESULT0 *600.0 / 4095); // Vin = D * 3 / 2^12 + ADCLO=0.0208789 (1.5V = 300V and 2V = 400V steps of 5mV)
    Voltage_VR1 = VR1;

    TR3 = (AdcMirror.ADCRESULT1 *300.0 / 4095); //
    Tempera_TR3 = TR3;

    CR2 = (AdcMirror.ADCRESULT2 *3000.0 / 8190) - 1258; // (0A = 2.492V steps of 0.02V)
    Current_CR2 = CR2;


    // Reinitialise for next ADC sequence
    AdcRegs.ADCTRL2.bit.RST_SEQ1 = 1;       // Reset SEQ1
    AdcRegs.ADCST.bit.INT_SEQ1_CLR = 1;     // Clear INT SEQ1 bit
    PieCtrlRegs.PIEACK.all = PIEACK_GROUP1; // Acknowledge interrupt to PIE


}


/*
interrupt void xint1(void)
{
  //  PieCtrlRegs.PIEACK.all = 0x0000;
    GpioDataRegs.GPATOGGLE.bit.GPIO0 = 1;  //PMW-A
GpioDataRegs.GPASET.bit.GPIO1 = 1;  //PMW-B

 GpioDataRegs.GPACLEAR.bit.GPIO2= 1; //PMW-C
 GpioDataRegs.GPASET.bit.GPIO3= 1; //PMW-D

 ECanaMboxes.MBOX14.MDL.byte.BYTE0 = 0x0;          // PWM A status
 ECanaMboxes.MBOX14.MDL.byte.BYTE1 = 0x1;         //  PWM B status
 ECanaMboxes.MBOX14.MDL.byte.BYTE2 = 0x0;         //  PWM C status
 ECanaMboxes.MBOX14.MDL.byte.BYTE3 = 0x1;         //  PWM D status

 PieCtrlRegs.PIEACK.all = PIEACK_GROUP1; // Acknowledge interrupt to PIE
}
*/

/*
--------------------------------------------------------------------------------
GPIO value  Decimal         Operation                       Comments
--------------------------------------------------------------------------------
0000        0       Transmit 8 bytes @ 500 kbps             Extended ID
0001        1       Transmit 8 bytes @ 500 kbps             Standard ID
0010        2       Receive @ 500 kbps                      Extended ID
0011        3       Receive @ 500 kbps                      Standard ID
0100        4       Transmit a remote frame @ 500 kbps      Standard ID
0101        5       Respond to a remote frame @ 1 Mbps      Standard ID
0110        6       Tx a STD frame followed by an Ext.frame @ 500 kbps. Repeat the cycle, but give 0.5 sec delay between frames.
 */

