//****************************************************************************
//
// Shared RAM management (RAM_management)
//
// This example shows how to assign shared RAM for use by both the cpu2 and
// cpu1 core.
//
// Shared RAM regions are defined in  both the cpu2 and cpu1 linker
// files.
// In the In this example S0 and S1 are assigned to/owned by c2.
//  - c2_r_w_array[] is mapped to shared RAM GS0
//  - c2_r_array[] is mapped to shared RAM GS2
//
//###########################################################################
// $TI Release: F2837xD Support Library v3.11.00.00 $
// $Release Date: Sun Oct  4 15:55:24 IST 2020 $
// $Copyright:
// Copyright (C) 2013-2020 Texas Instruments Incorporated - http://www.ti.com/
//
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions 
// are met:
// 
//   Redistributions of source code must retain the above copyright 
//   notice, this list of conditions and the following disclaimer.
// 
//   Redistributions in binary form must reproduce the above copyright
//   notice, this list of conditions and the following disclaimer in the 
//   documentation and/or other materials provided with the   
//   distribution.
// 
//   Neither the name of Texas Instruments Incorporated nor the names of
//   its contributors may be used to endorse or promote products derived
//   from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// $
//###########################################################################

//
// Included Files
//
#include "F28x_Project.h"
#include "F2837xD_Ipc_drivers.h"

//
// Globals
//
uint16_t c2_r_w_array[256];   // mapped to GS1 of shared RAM owned by CPU02
uint16_t c2_r_array[256];     // mapped to GS0 of shared RAM owned by CPU01
#pragma DATA_SECTION(c2_r_array,"SHARERAMGS1");
#pragma DATA_SECTION(c2_r_w_array,"SHARERAMGS0");

extern uint16_t isrfuncLoadStart;
extern uint16_t isrfuncLoadEnd;
extern uint16_t isrfuncRunStart;
extern uint16_t isrfuncLoadSize;
Uint16 cpu2test;
Uint16 cpu2test1;

//////////////////
// Globals for SPI
//////////////////
Uint16 sData_16[16];     // Send data buffer
Uint16 rData_16[16];     // Receive data buffer
Uint16 rData_point;  // Keep track of where we are
                     // in the data stream to check received data
Uint32 sData_32[8];
Uint32 rData_32[8];
Uint32 rData_flag;

Uint16 flag;
float temp;
float step;

float shareData1to2_32[8];
float shareData2to1_32[8];

#pragma DATA_SECTION(shareData1to2_32,"SHARERAMGS2");
#pragma DATA_SECTION(shareData2to1_32,"SHARERAMGS3");

//JNIE Command reference:
Uint32 Default = 0x00000000;
Uint32 Mode = 0x00001111;
Uint32 State = 0x00002222;
Uint32 PF = 0x00003333;
Uint32 P0 = 0x00004444;
Uint32 P1 = 0x00005555;
Uint32 Q0 = 0x00006666;
Uint32 Q1 = 0x00007777;
Uint32 OnOff = 0x00008888;
Uint32 Current = 0x00009999;
Uint32 kwh = 0x00001212;
Uint32 savedData;
float savedFloat;
Uint32 rData32_high = 0x00000000;
Uint32 rData32_low = 0x00000000;

//


// SPI Baud rate decision here:

#if CPU_FRQ_200MHZ
#define SPI_BRR        ((200E6 / 4) / 500E3) - 1
#endif

#if CPU_FRQ_150MHZ
#define SPI_BRR        ((150E6 / 4) / 500E3) - 1
#endif

#if CPU_FRQ_120MHZ
#define SPI_BRR        ((120E6 / 4) / 500E3) - 1
#endif



//
// Function Prototypes
//
__interrupt void cpu_timer0_isr(void);
#pragma CODE_SECTION(cpu_timer0_isr,"isrfunc")
void Shared_Ram_dataWrite_c2(void);

//for SPI
void spi_fifo_init(void);
interrupt void spiTxFifoIsr(void);
interrupt void spiRxFifoIsr(void);



//
// Main
//
void main(void)
{
    Uint16 i;
//
// Step 1. Initialize System Control:
// PLL, WatchDog, enable Peripheral Clocks
// This example function is found in the F2837xD_SysCtrl.c file.
//
   InitSysCtrl();

//
// Step 2. Initialize GPIO:
// This example function is found in the F2837xD_Gpio.c file and
// illustrates how to set the GPIO to it's default state.
//
// InitGpio();  // Skipped for this example

//
// Step 3. Clear all interrupts and initialize PIE vector table:
// Disable CPU interrupts
//
   DINT;

//
// Initialize the PIE control registers to their default state.
// The default state is all PIE interrupts disabled and flags
// are cleared.
// This function is found in the F2837xD_PieCtrl.c file.
//
   InitPieCtrl();


//
// Disable CPU interrupts and clear all CPU interrupt flags:
//
   IER = 0x0000;
   IFR = 0x0000;

//
// Initialize the PIE vector table with pointers to the shell Interrupt
// Service Routines (ISR).
// This will populate the entire table, even if the interrupt
// is not used in this example.  This is useful for debug purposes.
// The shell ISR routines are found in F2837xD_DefaultIsr.c.
// This function is found in F2837xD_PieVect.c.
//
   InitPieVectTable();

//
// Wait until Shared RAM is available.
//
   while(!( MemCfgRegs.GSxMSEL.bit.MSEL_GS0 &
            MemCfgRegs.GSxMSEL.bit.MSEL_GS14 ))
   {
   }

//
//  Copy ISR routine to a specified RAM location to determine the size
//
    memcpy(&isrfuncRunStart, &isrfuncLoadStart, (uint32_t)&isrfuncLoadSize);

//
// Interrupts that are used in this example are re-mapped to
// ISR functions found within this file.
//
   EALLOW;  // This is needed to write to EALLOW protected registers
   PieVectTable.TIMER0_INT = &cpu_timer0_isr;

   PieVectTable.SPIA_RX_INT = &spiRxFifoIsr;
   PieVectTable.SPIA_TX_INT = &spiTxFifoIsr;
   EDIS;    // This is needed to disable write to EALLOW protected registers

//
// Step 4. Initialize the Device Peripheral. This function can be
//         found in F2837xD_CpuTimers.c
//
   InitCpuTimers();   // For this example, only initialize the Cpu Timers
   spi_fifo_init();
//
// Configure CPU-Timer0 to interrupt every second:
// c2_FREQ in MHz, 1 second Period (in uSeconds)
//
   ConfigCpuTimer(&CpuTimer0, 200, 1000000);

//
// To ensure precise timing, use write-only instructions to write to the
// entire register.
//
   CpuTimer0Regs.TCR.all = 0x4000;

//

// Step 5. User specific code, enable interrupts: for SPI

//
// Enable CPU int1 which is connected to CPU-Timer 0
//
   IER |= M_INT1;

   //
   // Initialize the send data buffer
   //
      flag = 0;
      rData_flag = Default;
      temp = 1.11;
      for(i=0; i<8; i++)
      {
   //      sData_32[i] = 1;
          memcpy(&sData_32[i], &temp, sizeof(float));
          memcpy(&shareData2to1_32[i], &Default, sizeof(float));
      }
      rData_point = 0;

      step = 2.22;

   //
   // Enable interrupts required for this example  SPI
   //
      PieCtrlRegs.PIECTRL.bit.ENPIE = 1;     // Enable the PIE block
      PieCtrlRegs.PIEIER6.bit.INTx1 = 1;     // Enable PIE Group 6, INT 1
      PieCtrlRegs.PIEIER6.bit.INTx2 = 1;     // Enable PIE Group 6, INT 2
      IER|=0x20;                              // Enable CPU INT6

//
// Enable TINT0 in the PIE: Group 1 interrupt 7
//
   PieCtrlRegs.PIEIER1.bit.INTx7 = 1;

//
// Enable global Interrupts and higher priority real-time debug events:
//
   EINT;   // Enable Global interrupt INTM
   ERTM;   // Enable Global realtime interrupt DBGM

   cpu2test = 0;
   cpu2test1 = 0;

   while(1)
   {
       cpu2test += 2;

//        if(IPCRtoLFlagBusy(IPC_FLAG5) == 1)
//        {
//
//            //
//            // Read c2_r_array and modify c2_r_w_array
//            //
//            Shared_Ram_dataWrite_c2();
//
//            IPCRtoLFlagAcknowledge (IPC_FLAG5);
//         }
   }
}

//
// cpu_timer0_isr - CPU Timer0 ISR
//
__interrupt void cpu_timer0_isr(void)
{
   EALLOW;
   CpuTimer0.InterruptCount++;
   GpioDataRegs.GPATOGGLE.bit.GPIO31 = 1;
   EDIS;
   cpu2test1++;
   PieCtrlRegs.PIEACK.all = PIEACK_GROUP1;

   if(IPCRtoLFlagBusy(IPC_FLAG5) == 1)
   {
      //
      // Read c2_r_array and modify c2_r_w_array
      //
      Shared_Ram_dataWrite_c2();

      IPCRtoLFlagAcknowledge (IPC_FLAG5);
   }
}

//
// Shared_Ram_dataWrite_c2 - Read data from c2_r_array written by CPU01 and
//                           modify and write into c2_r_w_array
//                           c2_r_array[0] is used to hold the multiplier
//                           value.
//
void Shared_Ram_dataWrite_c2(void)
{
    uint16_t index;
    uint16_t multiplier;

    multiplier = c2_r_array[0];
    c2_r_w_array[0] = multiplier;

    for(index = 1; index < 256; index ++)
    {
        c2_r_w_array[index] = multiplier*c2_r_array[index];
    }
}


//SPI functions
void spi_fifo_init()
{
    //
    // Initialize SPI FIFO registers
    //
    SpiaRegs.SPIFFTX.all = 0xC030;    // Enable FIFOs, set TX FIFO level to 16
    SpiaRegs.SPIFFRX.all = 0x0030;    // Set RX FIFO level to 16
    SpiaRegs.SPIFFCT.all = 0x00;

    SpiaRegs.SPIFFTX.bit.TXFIFO=1;
    SpiaRegs.SPIFFRX.bit.RXFIFORESET=1;

    //
    // Initialize core SPI registers
    //
    InitSpi();
}


interrupt void spiTxFifoIsr(void)
{
    if(flag == 0) {
        Uint16 i;
        if(IPCRtoLFlagBusy(IPC_FLAG6) == 1)
        {
           //
           // Read shareData1to2_32
           //
            for(i = 0; i < 8; i ++)
            {
                //sData_32[i] = shareData1to2_32[i];
                memcpy(&sData_32[i],&shareData1to2_32[i], sizeof(float));
            }




            for (i = 0; i<8; i++)
            {
                sData_16[2*i]   = (Uint16) (sData_32[i] >> 16);                   // pick the higher 16 bit from sData_32
                sData_16[2*i+1] = (Uint16) (sData_32[i]);                           // pick the lower 16 bit from sData_32

            }
            for(i=0;i<16;i++)
            {
               SpiaRegs.SPITXBUF=sData_16[i];      // Send data
            }

            temp = temp + 1;
//            for(i=0;i<8;i++)                    // Increment data for next cycle
//            {
//                memcpy(&sData_32[i], &temp, sizeof(float));
//               //sData_32[i] = sData_32[i]+1;
//                //sData_32[i] = (Uint32)(sData_32[i] << 1);
//            }

            IPCRtoLFlagAcknowledge (IPC_FLAG6);
        }
        flag = 1;
    }

        SpiaRegs.SPIFFTX.bit.TXFFINTCLR=1;  // Clear Interrupt flag
        PieCtrlRegs.PIEACK.all|=PIEACK_GROUP6;       // Issue PIE ACK
}

//
// spiRxFifoIsr - ISR for SPI receive FIFO
//
interrupt void spiRxFifoIsr(void)
{
    Uint16 i;

    //Receive data at SPI receiving buffer (16 bit)
    for(i=0; i<16; i++)
    {
        rData_16[i]=SpiaRegs.SPIRXBUF;     // Read data
    }

    //Convert the received data to 32 bit
    for(i=0; i<16; i+=2)
    {
        Uint32 rData_high = ((Uint32) rData_16[i]) << 16;
        Uint32 rData_low  = rData_16[i+1];
        rData_32[i/2] = rData_high | rData_low;
    }
    //Put the received data into the correct space in the shared variable: shareData2to1
    for(i=0; i<8; i++)
    {
        if(rData_32[i] != rData_flag && rData_32[i] != Default){
            rData32_high = (Uint32)(rData_32[i] >> 16);
            //Decode the Mode Value
            if(rData32_high == Mode){
                rData_flag = rData_32[i];
                rData32_low = rData_32[i]-(Uint32)(Mode << 16);
                savedFloat = (float)rData32_low;
                memcpy(&shareData2to1_32[0],&savedFloat, sizeof(float));
            }
            //Decode the State Value
            else if(rData32_high == State){
                rData_flag = rData_32[i];
                rData32_low = rData_32[i]-(Uint32)(State << 16);
                savedFloat = ((float)rData32_low)-16;
                memcpy(&shareData2to1_32[1],&savedFloat, sizeof(float));
            }
            //Decode the PF
            else if(rData32_high == PF){
                rData_flag  = rData_32[i];
                rData32_low = rData_32[i]-(Uint32)(PF << 16);
                savedFloat = ((float) rData32_low/1000)-1;
                memcpy(&shareData2to1_32[2],&savedFloat, sizeof(float));
            }
            //Decode the Active Power
            else if(rData32_high == P0){
                rData_flag = rData_32[i];
                rData32_low = rData_32[i]-(Uint32)(P0 << 16);
                savedFloat = (float) rData32_low;
                memcpy(&shareData2to1_32[3],&savedFloat, sizeof(float));
            }
            else if(rData32_high == P1){
                rData_flag = rData_32[i];
                rData32_low = rData_32[i]-(Uint32)(P1 << 16);
                savedFloat = -((float) rData32_low);
                memcpy(&shareData2to1_32[3],&savedFloat, sizeof(float));
            }
            //Decode the Reactive Power
            else if(rData32_high == Q0){
                rData_flag = rData_32[i];
                rData32_low = rData_32[i]-(Uint32)(Q0 << 16);
                savedFloat = (float) rData32_low;
                memcpy(&shareData2to1_32[4],&savedFloat, sizeof(float));
            }
            else if(rData32_high == Q1){
                rData_flag = rData_32[i];
                rData32_low = rData_32[i]-(Uint32)(Q1 << 16);
                savedFloat = -((float) rData32_low);
                memcpy(&shareData2to1_32[4],&savedFloat, sizeof(float));
            }
            //Decode the Charger On and Off
            else if(rData32_high == OnOff){
                rData_flag = rData_32[i];
                rData32_low = rData_32[i]-(Uint32)(OnOff << 16);
                savedFloat = (float)rData32_low;
                memcpy(&shareData2to1_32[5],&savedFloat, sizeof(float));
            }
            //Decode the Current Value
            else if(rData32_high == Current){
                rData_flag = rData_32[i];
                rData32_low = rData_32[i]-(Uint32)(Current << 16);
                savedFloat = (float)rData32_low;
                memcpy(&shareData2to1_32[6],&savedFloat, sizeof(float));
            }
            //Decode the KWH Value
            else if(rData32_high == kwh){
                rData_flag = rData_32[i];
                rData32_low = rData_32[i]-(Uint32)(kwh << 16);
                savedFloat = (float)rData32_low;
                memcpy(&shareData2to1_32[7],&savedFloat, sizeof(float));
            }
        }

      }



//    for(i=0; i<8; i++)
//    {
//        step += 2;
//        memcpy(&shareData2to1_32[i], &step, sizeof(float));
//    }

//    for(i=0; i<8; i++)                  // Check received data
//    {
//        if(rdata[i] != rdata_point+i)
//        {
//            error();
//        }
//    }

    flag = 0;

    rData_point++;
    SpiaRegs.SPIFFRX.bit.RXFFOVFCLR=1;  // Clear Overflow flag
    SpiaRegs.SPIFFRX.bit.RXFFINTCLR=1;  // Clear Interrupt flag
    PieCtrlRegs.PIEACK.all|=0x20;       // Issue PIE ack
}

void InitSpi(void)
{
    // Initialize SPI-A

    // Set reset low before configuration changes
    // Clock polarity (0 == rising, 1 == falling)
    // 16-bit character
    // Enable loop-back
    SpiaRegs.SPICCR.bit.SPISWRESET = 0;
    SpiaRegs.SPICCR.bit.CLKPOLARITY = 0;
    SpiaRegs.SPICCR.bit.SPICHAR = (16-1);
    SpiaRegs.SPICCR.bit.SPILBK = 0;  //0h (R/W) = SPI loopback mode disabled. This is the default value after reset. 1h (R/W) = SPI loopback mode enabled, SIMO/SOMI lines areconnected internally. Used for module self-tests.

    // Enable master (0 == slave, 1 == master)
    // Enable transmission (Talk)
    // Clock phase (0 == normal, 1 == delayed)
    // SPI interrupts are disabled
    SpiaRegs.SPICTL.bit.MASTER_SLAVE = 0;
    SpiaRegs.SPICTL.bit.TALK = 1;
    SpiaRegs.SPICTL.bit.CLK_PHASE = 1;
    SpiaRegs.SPICTL.bit.SPIINTENA = 0;

    // Set the baud rate
    SpiaRegs.SPIBRR.bit.SPI_BIT_RATE = SPI_BRR;

    // Set FREE bit
    // Halting on a breakpoint will not halt the SPI
    SpiaRegs.SPIPRI.bit.FREE = 1;

    // Release the SPI from reset
    SpiaRegs.SPICCR.bit.SPISWRESET = 1;
}
//
// End of file
//
