//------------------------------------------------------------------------------
//	scriptSessionApp.js - Wrapper for all debugSession actions.
//------------------------------------------------------------------------------

var debugServer;
var scriptEnv;
var sessionParam;
var debugSession;
var System;
var ScriptingEnvironment;

var g_bRunCCS = undefined;
var g_cNewLine = "undefined";

function defineNewline()
{
	if( g_cNewLine === "undefined" ) {
		g_cNewLine = System.getProperty( "line.separator" );
	}
	return g_cNewLine;
}

function resetTarget()
{
	var aLine;

	print( "Apply power to the target board, then press the 'Enter' key:" );
	var stdin = new BufferedReader( new InputStreamReader( System["in"] ));
	try {
		aLine = stdin.readLine();
	}
	catch(err)
	{
		print( "Handle Error - runSystemReset" );
		aLine = 0;
	}
}

function traceWrite( sString )
{
	scriptEnv.traceWrite( sString );
}
function getSymbolAddress( sSymbol )
{
	return debugSession.symbol.getAddress( sSymbol );
}

function getSessionPC()
{
	return( debugSession.expression.evaluate( "PC" ) );
}

function memoryReadRegister( sRegName )
{
	return( debugSession.memory.readRegister( sRegName ));
}
function memoryReadData( pData, nDataSize )
{
	return( debugSession.memory.readData( 1, pData, nDataSize ));
}

function memoryWriteData( pData, nState, nDataSize )
{
	debugSession.memory.writeData( 1, pData, nState, nDataSize );
}

//	Breakpoint support functions:
function getBreakpointAddress( sBpLabel )
{
	var pBpAddr = undefined;
	try {
		pBpAddr = getSymbolAddress( sBpLabel );
	}
	catch( err ) {
		print( "Breakpoint ", sBpLabel, " could not be found." + g_cNewLine );
	}
	return(pBpAddr );
}

function addNewBreakpoint( sBpLabel )
{
	var bpID = undefined;
	try {
		bpID = debugSession.breakpoint.add( sBpLabel );
	}
	catch( err ) {
		print( "Breakpoint " + sBpLabel + " could not be added." + g_cNewLine );
	}
	return( bpID );
}

function defineSessionParams( bRunCCS )
{
	var dspAppName = "DssBkptTest";
//	var wsDir = "../../"Workspace/";
	var dspAppDir = "../../" + dspAppName + "/";
	var sessionLogName = dspAppName + "_SessionLog";
	var sessionTimeoutMs = 60000;

	g_bRunCCS = bRunCCS;
	sessionParam = { sDebugSessionLogName:sessionLogName,
		serverConfigName: dspAppDir + "/targetConfigs/TMS320F28379D.ccxml",
		DSPExecutableName: dspAppDir + "Debug/" + dspAppName + ".out",
		sessionTimeout: sessionTimeoutMs,
		runCCS: g_bRunCCS };
	print( "ExecutableName =           " + sessionParam.DSPExecutableName + "." );
	print( "SessionTimeout =           " + sessionParam.sessionTimeout + " msec." );
}

//	Start a debug session.
function sessionStart()
{
	OpenDebugSession();
}

// Insert delay to wait for a connection to be established.
function waitForTargetConnect()
{
	while( !debugSession.target.isConnected() ) {
		print( "Waiting for target connected..." );
	}
}

function OpenDebugSession()
{
	//	Terminate any residual debug session.
	if( debugSession !== undefined ) {
		debugSession.terminate();
	}

	if( debugServer !== undefined ) {
		debugServer.stop();
	}

	if( scriptEnv !== undefined ) {
		scriptEnv.traceEnd();
	}

	//	Define a new debug session.
	// Create a scripting environment object - which is the main entry point into any script
	// and the factory for creating other Scriptable servers and Sessions.
	scriptEnv = ScriptingEnvironment.instance();

	//	Experimental: Try calling traceBegin() before opening CCS.
	scriptEnv.traceBegin( sessionParam.sDebugSessionLogName, "DefaultStylesheet.xsl" );

	if( sessionParam.runCCS )
	{
		//	Open a CCS Session to bring up the CCS GUI which will share the same debug context as the script.
		//	The following is recommended by TI example script msp430F5529_ccsSession.js.
		//	NOTE: SEU v2.x uses a multi-core processor so the calling syntax requires
		//	specifying the core that is being debugged.
		//ccsServer = scriptEnv.getServer( "CCSServer.1" );
		//ccsSession = ccsServer.openSession( ".*/C28xx_CPU1" );

		//	The separate calls above cause a DSS crash. The combined call below
		//	works for SEU v1.x, as recommeneded in the TI web page:
		//	http://software-dl.ti.com/ccs/esd/documents/users_guide_to_launching_ccs_from_dss.html
		scriptEnv.getServer( "CCSServer.1" ).openSession( ".*/C28xx_CPU1" );
	}

	//	This is the way I called CCS from the script in the past. It seemed to work...
//	scriptEnv.traceBegin( sessionParam.sDebugSessionLogName, "DefaultStylesheet.xsl" );
//
//	scriptEnv.traceWrite( "SEU Selected: " + SEU_Options[SEU_Selected] );

	//	Start the DSS session.
	debugServer = scriptEnv.getServer( "DebugServer.1" );
	scriptEnv.traceWrite( "Get Server: " + debugServer );
	debugServer.setConfig( sessionParam.serverConfigName );
	scriptEnv.traceWrite( "Debugger Configured." );

	//	Open a debug session to core CPU1 only.
	//	For TMS320F28335, there is only one core on which to open a new session.
	debugSession = debugServer.openSession( ".*C28xx_CPU1" );
	scriptEnv.traceWrite( "Connecting to device: " + debugSession );
	debugSession.target.connect();
	scriptEnv.traceWrite( "Target is Connected." );

	print( "sessionParam.sessionTimeout = ", sessionParam.sessionTimeout, " msec.\n" );
	scriptEnv.setScriptTimeout( sessionParam.sessionTimeout );

	//	Download SEU.out to the target.
	try{
		scriptEnv.traceWrite( "Load Program: " + sessionParam.DSPExecutableName );
		// Make sure the External watchdog is disable before loading the code!!!!!
		// Load code into RAM only!!!!!!!
		debugSession.memory.loadProgram( sessionParam.DSPExecutableName );
	}
	catch(err)
	{
		scriptEnv.traceWrite( "DSP Load Program Error." );
	}

	debugSession.breakpoint.removeAll();

	date = new Date();
	scriptEnv.traceWrite( "START TIME: " + date.toTimeString() + "\n" );

	scriptEnv.traceWrite( "Restarting Target." );
	debugSession.target.restart();
}

function CloseDebugSession()
{
	debugSession.terminate();
	scriptEnv.traceWrite( "Session Terminated!" );
	debugServer.stop();
	scriptEnv.traceEnd();
}

function removeBreakpoint( nBpId )
{
	debugSession.breakpoint.remove( nBpId );
}

function removeAllBreakpoints()
{
	debugSession.breakpoint.removeAll();
}

function dspRun()
{
	debugSession.target.run();
}

function dspResume()
{
	scriptEnv.traceWrite( "Resuming DSP execution." );
	dspRun();
}

function targetIsHalted()
{
	return( debugSession.target.isHalted() );
}

function atBreakpoint( sBpLabel )
{
	return( debugSession.expression.evaluate( "PC" ) === getSymbolAddress( sBpLabel  ));
}

