*************************************************************
** CLEAR  Subroutine                                       **
*                                                          **
* TMS320F2XX Flash Utilities.                              **
*  Revision: 2.0,  9/10/97                                 **
*  Revision: 2.1,  1/31/98                                 **
*                                                          **
* Filename: sclr20.asm                                     **
* Changes20: Re-written to include latest flash algorithms.**
* Changes21: Cleaned up source for release to mass market. **
*            Initialize error flag to 0 (no error). Removed**
*            redundant call/ret to clear routine. Changed  **
*            variable names from B2_ to BASE_ since they're**
*            not necessarily stored in B2 RAM. Removed imr **
*            modifications. Created PRG_BYTE subroutine for**
*            use on both high and low bytes. Added more    **
*            comments.                                     **
*                                                          **
*                                                          **
* Called by : c2xx_bcx.asm or flash application programs.  **
*                                                          **
* !!CAUTION - INITIALIZE DP BEFORE CALLING THIS ROUTINE!!  **
*                                                          **
* Function  : Clears one or more contiguous segments of    **
*             flash array 0/1 as specified by the fol      **
*             -lowing variables.                           **
*             SEG_ST = Segment start address.              **
*             SEG_END= Segment end address.                **
*             PROTECT= Sector protect enable.              **
*                                                           *
* The algorithm used is "row-horizontal" which means that   *
* an entire flash row (32 words) are programed in parallel. *
* This method provides better uniformity of programming     *
* levels between adjacent bits then if each address were    *
* programed independently. The algorithm also uses a 3-read *
* check for VER0 margin (i.e. The flash location is read    *
* three times and the first two values are discarded.) This *
* provides low-freq read-back margin on programed bits. For *
* example, if the flash is programmed using a CLKOUT period *
* of 50ns, the flash can be reliably readback over the      *
* CLKOUT period range of 50ns to 150ns (6.67Mhz-20Mhz).     *
* The programming pulse-width is 100us, and a maximum of    *
* 150 pulses are applied per row.                           *
*                                                           *
* The following resources are used for temporary storage:   *
*        AR0  - Used for comparisons.                       *
*        AR1  - Used for pgm pulse count.                   *
*        AR2  - Used for row banz loop.                     *
*        AR6  - Parameter passed to Delay.                  *
*    FL_ADRS  - Used for flash address.                     *
*    FL_DATA  - Used for flash data.                        *
*      FL_ST  - Used for flash start address.               *
*     BASE_0  - Used for row-done flag.                     *
*     BASE_1  - Used for row start address.                 *
*      SPAD1  - Flash commands.                             *
*      SPAD2  - Flash commands.                             *
*                                                          **
*************************************************************
        .include "Svar20.h"
*                   
**clear definitions**
MAX_PGM .set    150             ;Only allow 150 pulses per row.
VER0    .set    010h            ;VER0 command.
WR_CMND .set    4               ;Write command.
WR_EXE  .set    045h            ;Write EXEBIN command.
STOP    .set    0               ;Reset command.
**erase definitions**
MAX_ER  .set    1000            ;Only allow 1000 erase pulses.
VER1    .set    8               ;VER1 command.
ER_CMND .set    2               ;ERASE COMMAND WORD
ER_EXE  .set    043h            ;ERASE EXEBIN COMMAND WORD
INV_ER  .set    018h            ;INVERSE ERASE COMMAND WORD
FL_WR   .set    6               ;FLASH WRITE COMMAND WORD
FLWR_EX .set    047h            ;FLASH WRITE EXEBIN COMMAND WORD

        .def    GCLR
        .def    GERS    
        
        .ref    PROTECT,SEG_ST,SEG_END
		.ref	DELAY,REGS,ARRAY

       .sect "alg"
************************************************
* GCLR:    This routine performs a clear opera *
* -tion on the flash array defined by the FL_ST*
* variable. The segments to be cleared are     *
* defined by the SEG_ST, SEG_END, and PROTECT  *
* variables.                                   *
* The following resources are used for temp    *
* storage:                                     *
*        AR0  - Used for comparisons.          *
*        AR1  - Used for pgm pulse count.      *
*        AR2  - Used for row banz loop.        *
*    FL_ADRS  - Used for flash address.        *
*    FL_DATA  - Used for flash data.           *
*      BASE_0 - Used for row-done flag.        *
*      BASE_1 - Used for row start address.    *
*      BASE_2 - Used for byte mask.            *
************************************************
GCLR:
        SETC    INTM            ;Disable all ints.
        CLRC    SXM             ;Disable sign extension.
        SPLK    #0,ERROR        ;Reset error flag
        LACL    SEG_ST          ;Get segment start address.
        SACL    FL_ADRS         ;Save as current address.
        AND     #04000h         ;Get array start address.
        SACL    FL_ST           ;Save array start address.
        LACL    FL_ADRS         ;Get segment start address.

NEWROW          ;********Begin a new row.*
        SACL    BASE_1          ;Save row start address.
        LAR     AR1,#0          ;Init pulse count to zero.

SAMEROW         ;********Same row, next pulse.*
        SPLK    #1,BASE_0       ;Set row done flag = 1(True).
        LACL    BASE_1          ;Get row start address.
        SACL    FL_ADRS         ;Save as current address.
        LAR     AR2,#31         ;Init row index.

********Repeat the following code 32 times until end of row.*
LOBYTE          ;********First do low-byte.*
        SPLK    #0FFh,BASE_2    ;Get lo-byte mask.
        CALL    PRG_BYTE        ;Check/Program lo-byte
        SPLK    #0FF00h,BASE_2  ;Get hi-byte mask.
        CALL    PRG_BYTE        ;Check/Program hi-byte.

NEXTWORD        ;********Next word in row.
        LACL    FL_ADRS         ;Load address for next word.
        ADD     #1              ;Increment address.
        SACL    FL_ADRS         ;Save as current address.
        MAR     *, AR2          ;Point to row index.
        BANZ    LOBYTE          ;Do next word,and dec AR2.

********Reached end of row. Check if row done. *
        BIT     BASE_0,15       ;Get row_done flag.
        BCND    ROW_DONE,TC     ;If 1 then row is done.
        MAR     *,AR1           ;Else, row is not done, so
        MAR     *+              ;inc row pulse count.
        LAR     AR0,#MAX_PGM    ;Check if passed allowable max.
        CMPR    2               ;If AR1>MAX_PGM then
        BCND    EXIT,TC         ;fail, don't continue.
        B       SAMEROW         ;else, go to beginning
                                ;of same row.

********If row done then check if Array done. *
ROW_DONE                        ;Check if end of array.
        SUB     SEG_END         ;Subtract segment end address.
        BCND    DONE, GEQ       ;If >0 then done.

********Else, goto next row. *
        LACL    FL_ADRS         ;Get current address.
        B       NEWROW          ;Start new row.

********If here then done.
DONE    CALL    ARRAY           ;Access flash in array mode.
        RET

********If here then unit failed to program. *
EXIT    SPLK    #1,ERROR        ;Update error flag.
        B       DONE            ;Get outa here.

	.page
************************************************************
* THIS SECTION PROGRAMS THE VALUE STORED IN FL_DATA INTO   *
* THE FLASH ADDRESS DEFINED BY FL_ADRS.                    *
*							   *
* The following resources are used for temporary storage:  *
*        AR6  - Parameter passed to Delay.                 *
*        SPAD1 - Flash program and STOP commands.          *
*        SPAD2 - Flash program + EXE command.              *
************************************************************
EXE_PGM                                                   ;*
	CALL	ARRAY		;ACCESS ARRAY		   *
* LOAD WADRS AND WDATA					  **
        LACL    FL_ADRS         ;ACC => PROGRAM ADRS       *
        TBLW    FL_DATA         ;LOAD WADRS AND WDATA      *
	CALL	REGS		;ACCESS FLASH REGS	   *
* SET-UP WRITE COMMAND WORDS				  **
        LACL    PROTECT         ;GET SEGMENT PROTECT MASK **
        OR      #WR_CMND        ;OR IN WRITE COMMAND      **
        SACL    SPAD1           ;SPAD1 = WRITE COMMAND    **
        OR      #WR_EXE         ;OR IN EXEBIN COMMAND     **
        SACL    SPAD2           ;SPAD2 = WRITE EXE COMMAND**
*							   *
        LACL    FL_ST           ;ACC => 0 (FLASH0)         *
* ACTIVATE WRITE BIT					  **
        TBLW    SPAD1           ;EXECUTE COMMAND          **
        LAR     AR6,#D10        ;SET DELAY                **
	CALL	DELAY,*,AR6	;WAIT			  **
* SET EXEBIN BIT					 ***
        TBLW    SPAD2           ;EXECUTE COMMAND         ***
        LAR     AR6,#D100       ;SET DELAY               ***
	CALL	DELAY,*,AR6	;WAIT			 ***
* STOP WRITE OPERATION					   *
        SPLK    #0,SPAD1        ;SHUTDONW WRITE OPERATION  *
        TBLW    SPAD1           ;EXECUTE COMMAND           *
        LAR     AR6,#D10        ;SET DELAY                 *
	CALL	DELAY,*,AR6	;WAIT			   *
*							   *
        RET                     ;RETURN TO CALLING SEQUENCE*
************************************************************
	.page
************************************************************
* ACTIVATE VER0 ON FLASH READS                             *
* LOADS FLASH WORD AT ADDR FL_ADRS TO FL_DATA.             *
* Uses SPAD1 for temporary storage of flash comands.       *
************************************************************
SET_RD_VER0                                               ;*
        CALL    REGS            ;ACCESS FLASH REGISTERS    *
        LACL    FL_ST           ;ACC => FLASH              *
        SPLK    #VER0,SPAD1     ;ACTIVATE VER0             *
        TBLW    SPAD1           ;EXECUTE COMMAND           *
        LAR     AR6,#D10        ;SET DELAY                 *
        CALL    DELAY,*,AR6     ;WAIT                      *
        CALL    ARRAY           ;ACCESS FLASH ARRAY        *
        LACL    FL_ADRS         ;POINT TO ADRS             *
        TBLR    FL_DATA         ;GET FLASH WORD 1x read    *
        TBLR    FL_DATA         ; 2x read                  *
        TBLR    FL_DATA         ; 3x read                  *
        CALL    REGS            ;ACCESS FLASH REGISTERS    *
        LACL    FL_ST           ;ACC => FLASH              *
        SPLK    #STOP,SPAD1     ;DEACTIVATE VER0           *
        TBLW    SPAD1           ;EXECUTE COMMAND           *
        LAR     AR6,#D10        ;SET DELAY                 *
        CALL    DELAY,*,AR6     ;WAIT                      *
        CALL    ARRAY           ;ACCESS FLASH ARRAY        *
        RET                     ;RETURN TO CALLING SEQUENCE*
************************************************************

************************************************
* PRG_BYTE: Programs hi or lo byte depending on*
*           byte mask (BASE_2).                *
************************************************
PRG_BYTE:
        CALL    SET_RD_VER0     ;Read word at VER0 level.
        LACL    BASE_2          ;Get lo/hi byte mask.
        AND     FL_DATA         ;Xor with read-back value.
        BCND    PB_DONE,EQ      ;If zero then done.
        XOR     #0FFFFh         ;else, mask off good bits.
        SACL    FL_DATA         ;New data.
        CALL    EXE_PGM         ;PGM Pulse.
        SPLK    #0,BASE_0       ;Set row done flag = 0(False).
PB_DONE RET
************************************************

*************************************************************
* ERASE subroutine                                         **
*                                                          **
* TMS320F2XX Flash Utilities.                              **
*  Revision: 2.0, 9/10/97                                  **
*                                                          **
* Filename: sera20.asm                                     **
* Changes20: Re-written to include latest flash algorithms.**
* Changes21: Cleaned up source for release to mass market. **
*            Removed redundant call/ret to XOR_erase       **
*            routine. Changed variable names from B2_ to   **
*            BASE_ since they're not necessarily stored in **
*            B2 RAM. Removed imr modifications. Merged     **
*            inc_erase sub-routine into erase routine.     **
*            Added more comments.                          **
*                                                          **
*                                                          **
* Called by : c2xx_bex.asm or flash application programs.  **
*                                                          **
* !!CAUTION - INITIALIZE DP BEFORE CALLING THIS ROUTINE!!  **
*                                                          **
* Function  : Erases one or more contiguos segments of     **
*             flash array 0/1 as specified by the          **
*             following variables.                         **
*             SEG_ST = Segment start address.              **
*             SEG_END= Segment end address.                **
*             PROTECT= Sector protect enable.              **
*                                                           *
* The algorithm used is "XOR-VER1" which means that in      *
* addition to the VER1 readmode, an XOR readback is used to *
* gain more margin. During the read portion of the erase to *
* reads are performed for each address; for the first read  *
* all address bits are complemented using a logical XOR with*
* the array end address. The data read during the first read*
* is discarded and the second read is performed on the      *
* actual address. This scheme simulates the worst-case      *
* branching condition for code executing from the flash     *
* array.                                                    *
* The erase pulse-width is 7ms, and a maximum of 1000 pulses*
* are applied to the array.                                 *
*                                                           *
* The following resources are used for temporary storage:   *
*        AR0  - Used for comparisons.                       *
*        AR1  - Used for erase pulse count.                 *
*        AR2  - Used for main banz loop.                    *
*        AR6  - Parameter passsed to DELAY.                 *
*      BASE_0 - Parameter passed to Set_mode.               *
*      BASE_1 - Used for flash address.                     *
*      BASE_2 - Used for flash data.                        *
*      BASE_3 - Used for flash checksum.                    *
*      BASE_4 - Used for segment size.                      *
*      BASE_5 - Flash Erase command.                        *
*      BASE_6 - Flash Erase+EXE command.                    *
*************************************************************
;        .include "svar20.h"     ;defines variables for flash0
                                ;or for flash1 array
*
;MAX_ER  .set    1000            ;Only allow 1000 erase pulses.
;VER1    .set    8               ;VER1 command.
;ER_CMND .set    2               ;ERASE COMMAND WORD
;ER_EXE  .set    043h            ;ERASE EXEBIN COMMAND WORD
;INV_ER  .set    018h            ;INVERSE ERASE COMMAND WORD
;FL_WR   .set    6               ;FLASH WRITE COMMAND WORD
;FLWR_EX .set    047h            ;FLASH WRITE EXEBIN COMMAND WORD
;STOP    .set    0               ;RESET REGISTER COMMAND WORD


;        .ref    PROTECT, SEG_ST,SEG_END
;	.ref	DELAY,REGS,ARRAY

************************************************
* GERS:  This routine performs an erase to     *
* xorver1 level. The Seg to erase is defined by*
* the vars SEG_ST and SEG_END. The following   *
* resources are used for temporary storage:    *
*        AR0  - Used for comparisons.          *
*        AR1  - Used for erase pulse count.    *
*        AR2  - Used for main banz loop.       *
*      BASE_0 - Parameter passed to Set_mode.  *
*      BASE_1 - Used for flash address.        *
*      BASE_2 - Used for flash data.           *
*      BASE_3 - Used for flash checksum.       *
*      BASE_4 - Used for segment size.         *
************************************************
GERS: 
************************************************************
* Code initialization section				   *
* Initialize test loop counters:			   *
*   AR1 is the number of ERASE pulses.                     *
************************************************************
        SETC    INTM            ;Disable all maskable ints.
        SETC    SXM             ;Enable sign extension.                     
        CLRC    OVM             ;Disable overflow mode.
        LACL    SEG_ST          ;Get segment start address.
        AND     #04000h         ;Get array start address.
        SACL    FL_ST           ;Save array start address.
        OR      #ARRYSZ         ;Get array end address.
        SACL    FL_END          ;Save array end address.
        SPLK    #0,ERROR        ;Reset error flag
        LAR     AR1,#0          ;Set erase count to 0.
        SPLK    #STOP, BASE_0   ;Stop command.
        CALL    SET_MODE        ;Disable any flash cmds.

XOR_ERASE
**** Compute checksum for flash, using address complementing.****
        LACC    SEG_END
        SUB     SEG_ST
        SACL    BASE_4          ;Segment length-1.
        LAR     AR2,BASE_4      ;load n-1 to loop n times.
        ADD     #1
        SACL    BASE_4          ;Segment length.
        SPLK    #VER1,BASE_0    ;VER1 command.
        CALL    SET_MODE        ;Set VER1 mode.
        MAR     *,AR2
        BLDD    #SEG_ST,BASE_1  ;Segment start address.
        SPLK    #0,BASE_3       ;Clear checksum.
RD1_LOOP                        ;For I = SEG_ST to SEG_END.
        LACC    BASE_1          ;ACC => CURRENT ADDR.
        XOR     FL_END          ;XOR addr with flash end addr.
        TBLR    BASE_2          ;Dummy Read.
        LACC    BASE_1          ;Get actual addr again.
        TBLR    BASE_2          ;True Read.
        ADD     #1              ;Increment flash addr.
        SACL    BASE_1          ;Store for next read.
        LACC    BASE_3          ;Get old check sum.
        ADD     BASE_2          ;ACC=>ACC+FL_DATA.
        SACL    BASE_3          ;Save new check sum.
        BANZ    RD1_LOOP,*-
        ADD     BASE_4          ;Should make ACC=0 for erased array.
        BCND    XOR_ERFIN,EQ    ;If BASE_3=0 finished.
                
***** If not erased, apply an erase pulse.
        CALL    ERASE_A         ;Else, pulse it again.
        MAR     *,AR1           ;ARP->AR1(Erase pulse count)
        MAR     *+              ;Increment Erase count.
        LAR     AR0,#MAX_ER
        CMPR    2               ;If AR1>MAX_ER then
        BCND    EXIT,TC         ;fail, don't continue erasing.
        B       XOR_ERASE       ;Else, check again.

***** If here, then erase passed, now check for depletion.
XOR_ERFIN
        SPLK    #STOP, BASE_0   ;Stop command.
        CALL    SET_MODE        ;Disable any flash cmds.
        CALL    INV_ERASE       ;Check for depletion.
		RET
;DONE    RET                     ;Return to calling code.

***** If here, then an error has occured.
;EXIT    SPLK    #1,ERROR        ;Update error flag
;        SPLK    #STOP, BASE_0   ;Stop command.
;        CALL    SET_MODE        ;Disable any flash cmds.
;        B       DONE            ;Get outa here.
************************************************************
        .page


*************************************************
* SET_MODE: This routine sets the flash in the  *
* mode specified by the contents of BASE_0. This*
* can be used for VER0,VER1,INVERASE, or STOP.  *
*        AR6  - Parameter passsed to DELAY.     *
*************************************************
SET_MODE
        CALL    REGS            ;ACCESS FLASH REGS    
        LACL    FL_ST           ;ACC => SEG_CTR.
        TBLW    BASE_0          ;Activate MODE.
        LAR     AR6,#D10        ;SET DELAY                
        CALL    DELAY,*,AR6     ;WAIT                     *
        CALL    ARRAY           ;ACCESS FLASH ARRAY       *
	RET

************************************************
* INV_ERASE: This routine is used to check for *
* depletion in the flash array.                *
*        AR2  - Used for main banz loop.       *
*      BASE_0 - Parameter passed to Set_mode.  *
*      BASE_1 - Used for flash address.        *
*      BASE_2 - Used for flash data.           *
************************************************
INV_ERASE
        SPLK    #INV_ER,BASE_0
        CALL    SET_MODE        ;Set inverse-erase mode.
        BLDD    #FL_ST,BASE_1   ;Array start address.
        LAR     AR2,#31         ;Loop count.
        MAR     *,AR2
NEXT_IVERS
        LACL    BASE_1          ;Get address.
        TBLR    BASE_2          ;Dummy read.
        TBLR    BASE_2          ;Read data.
        ADD     #1              ;Increment address.
        SACL    BASE_1          ;Save address.
        ZAC
        ADD     BASE_2          ;Add data.
        BCND    EXIT, NEQ       ;If ACC<>0, then fail.
*Else continue, until until done with row.
        BANZ    NEXT_IVERS      ;Loop 32 times.
        SPLK    #STOP,BASE_0    ;Stop command.
        CALL    SET_MODE        ;Disable any flash cmds.
        RET                     ;If here then test passed.
        .page


************************************************************
* ERASE_A: This sub-routine applies one erase pulse to the *
* flash array.                                             *
*							   *
* The following resources are used for temporary storage:  *
*       BASE_0 - Flash STOP command, and FFFF for WDATA.   *
*       BASE_5 - Flash erase command.                      *
*       BASE_6 - Flash erase + EXE command.                *
************************************************************
ERASE_A
* SETUP FLASH ERASE COMMANDS FOR PROTECT MASK.            **
        LACL    PROTECT         ;GET SEGMENT PROTECT MASK **
        OR      #ER_CMND        ;OR IN ERASE COMMAND      **
        SACL    BASE_5          ;BASE_5 = ERASE COMMAND   **
        OR      #ER_EXE         ;OR IN EXEBIN COMMAND     **
        SACL    BASE_6          ;BASE_6 = ERASE EXE COMMND**
*							   *
* MUST LOAD WDATA WITH FFFF.                              **
        SPLK    #0FFFFh,BASE_0  ;WDATA VALUE FOR ERASE    **
        LACC    FL_ST           ;ACC => FLASH             **
        TBLW    BASE_0          ;SET WDATA = FFFF          *
*							   *
* THIS SECTION ACTIVATES THE WRITE COMMAND.               **
        SPLK    #STOP,BASE_0    ;Stop command.            **
        CALL    SET_MODE        ;Disable any flash cmds.  **
	CALL	REGS		;ACCESS FLASH REGS	  **
        LACC    FL_ST           ;ACC => FLASH             **
        TBLW    BASE_5          ;ACTIVATE ERASE           **
        LAR     AR6,#D10        ;SET DELAY                **
	CALL	DELAY,*,AR6	;WAIT			  **
*							   *
* THIS SECTION ACTIVATES THE EXEBIN COMMAND.              **
        TBLW    BASE_6          ;START ERASURE            **
        LAR     AR6,#D7K        ;SET DELAY to 7ms         **
	CALL	DELAY,*,AR6	;WAIT			  **
        SPLK    #STOP,BASE_0    ;STOP COMMAND             **
        CALL    SET_MODE        ;STOP ERASE               **
        RET                     ;RETURN TO CALLING CODE   **
************************************************************
	.end

